//--------------------------------------------------------------
// File     : stm32_ub_lcd_ssd2119.c
// Datum    : 09.04.2013
// Version  : 1.0
// Autor    : UB
// EMail    : mc-4u(@)t-online.de
// Web      : www.mikrocontroller-4u.de
// Changes  : 2mils - angepasst auf SSD2119 04.2014
// CPU      : STM32F4
// IDE      : CooCox CoIDE 1.7.0
// Module   : GPIO,FSMC
// Funktion : Grafik-LCD Funktionen (Chip=SSD2119)
//            Der Zugriff erfolgt ueber den FSMC-Controller
//            Im 16bit 8080-Mode (5R6G5B = RGB565)
// Hinweis  : Das Display benutzt die CPU-Pins :
//             PBD13  -> LCD_Backlight   PE3  -> LCD_RS
//             PD0    -> LCD_DB2         PE7  -> LCD_DB4
//             PD1    -> LCD_DB3         PE8  -> LCD_DB5
//             PD4    -> LCD_RD          PE9  -> LCD_DB6
//             PD5    -> LCD_WR          PE10 -> LCD_DB7
//             PD7    -> LCD_CS          PE11 -> LCD_DB10
//             PD8    -> LCD_DB15        PE12 -> LCD_DB11
//             PD9    -> LCD_DB16        PE13 -> LCD_DB12
//             PD10   -> LCD_DB17        PE14 -> LCD_DB13
//             PD14   -> LCD_DB0         PE15 -> LCD_DB14
//             PD15   -> LCD_DB1		 PD3  -> LCD_RESET
//--------------------------------------------------------------

//--------------------------------------------------------------
// Includes
//--------------------------------------------------------------
#include "stm32_ub_lcd_ssd2119.h"


//--------------------------------------------------------------
// interne Funktionen
//--------------------------------------------------------------
void P_LCD2119_InitIO(void);
void P_LCD2119_InitFSMC(void);
void P_LCD2119_InitChip(uint16_t mode);
uint16_t P_LCD2119_ReadReg(uint8_t reg_adr);
void P_LCD2119_WriteReg(uint8_t reg_adr, uint16_t reg_value);
void P_LCD2119_Delay(volatile uint32_t nCount);


//--------------------------------------------------------------
// Init vom LCD-Display
// Return_wert :
//  -> ERROR   , wenn Display nicht gefunden wurde
//  -> SUCCESS , wenn Display OK (ID=9919)
//--------------------------------------------------------------
ErrorStatus UB_LCD_Init(void)
{ 
  ErrorStatus ret_wert=ERROR;
  uint16_t lcdid = 0;

  // IO-Lines initialisieren
  P_LCD2119_InitIO();
  // kleine Pause 
  UB_LCD_Backlight_On(); // (Missbraucht fr RESET!)
  P_LCD2119_Delay(LCD_SSD2119_PAUSE);
  // FSMC initialisieren
  P_LCD2119_InitFSMC();
  // kleine Pause
  P_LCD2119_Delay(LCD_SSD2119_PAUSE);
  
  // LCD-ID auslesen
  lcdid = P_LCD2119_ReadReg(LCD_SSD2119_REG_00);
  if((lcdid==LCD_SSD2119_ID)) {
    // Display gefunden
    ret_wert=SUCCESS;
    // LCD-Controller initialisieren (Mode=Portrait)
    P_LCD2119_InitChip(LCD_SSD2119_PORTRAIT);
    LCD_DISPLAY_MODE=PORTRAIT;
    //P_LCD2119_InitChip(LCD_SSD2119_LANDSCAPE);
    //LCD_DISPLAY_MODE=LANDSCAPE;
  }  

  LCD_WINDOW.xstart=0;
  LCD_WINDOW.ystart=0;
  LCD_WINDOW.xend=LCD_MAXX-1;
  LCD_WINDOW.yend=LCD_MAXY-1;
  LCD_WINDOW.pixel=LCD_PIXEL;

  return(ret_wert);
}


//--------------------------------------------------------------
// setzt den Cursor auf x,y Position
// und bereitet das schreiben ins Display-RAM vor
//--------------------------------------------------------------
void UB_LCD_SetCursor2Draw(uint16_t xpos, uint16_t ypos)
{
  // Cursor setzen
  P_LCD2119_WriteReg(LCD_SSD2119_REG_4E, xpos);
  P_LCD2119_WriteReg(LCD_SSD2119_REG_4F, ypos);

  // adresse anlegen
  LCD_REG=LCD_SSD2119_REG_22;
}


//--------------------------------------------------------------
// Fllt den Screen mit einer Farbe
//--------------------------------------------------------------
void UB_LCD_FillScreen(uint16_t color)
{
  uint32_t n = 0;

  // Cursor auf 0 setzen
  UB_LCD_SetCursor2Draw(LCD_WINDOW.xstart,LCD_WINDOW.ystart);
  // Komplettes Display beschreiben
  for(n=0;n<LCD_WINDOW.pixel;n++) {
    LCD_RAM = color;
  }
}


//--------------------------------------------------------------
// Backlight einschalten (Missbraucht fr RESET!)
//--------------------------------------------------------------
void UB_LCD_Backlight_On(void)
{
  //GPIOB->BSRRH = GPIO_Pin_0;
  GPIO_ResetBits(GPIOD, GPIO_Pin_3);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE);
  GPIO_SetBits(GPIOD, GPIO_Pin_3);

}


//--------------------------------------------------------------
// Backlight ausschalten
//--------------------------------------------------------------
/*void UB_LCD_Backlight_Off(void)
{
  GPIOB->BSRRL = GPIO_Pin_0;
}*/

//--------------------------------------------------------------
// Screen-Mode einstellen
// muss direkt nach dem Init gemacht werden
//
// Mode : [PORTRAIT=default, LANDSCAPE]
//--------------------------------------------------------------
void UB_LCD_SetMode(LCD_MODE_t mode)
{
  if(mode==PORTRAIT) {
    P_LCD2119_InitChip(LCD_SSD2119_PORTRAIT);
    LCD_DISPLAY_MODE=PORTRAIT;
  }
  else {
    P_LCD2119_InitChip(LCD_SSD2119_LANDSCAPE);
    LCD_DISPLAY_MODE=LANDSCAPE;
  }
}


//--------------------------------------------------------------
// stellt ein Display-Fenster zum zeichnen ein
// von xtart,ystart zu xend,yend
//--------------------------------------------------------------
void UB_LCD_SetWindow(uint16_t xstart, uint16_t ystart, uint16_t xend, uint16_t yend)
{
  uint16_t start,end;
  uint16_t xstart_end;

  start=(xstart&0x00FF);
  end=((xend&0x00FF)<<8);
  xstart_end=(start | end);
  
  P_LCD2119_WriteReg(LCD_SSD2119_REG_44, xstart_end);
  P_LCD2119_WriteReg(LCD_SSD2119_REG_45, ystart);
  P_LCD2119_WriteReg(LCD_SSD2119_REG_46, yend);

  LCD_WINDOW.xstart=xstart;
  LCD_WINDOW.ystart=ystart;
  LCD_WINDOW.xend=xend;
  LCD_WINDOW.yend=yend;
  LCD_WINDOW.pixel=(xend-xstart+1)*(yend-ystart+1);
}


//--------------------------------------------------------------
// interne Funktion
// Init aller IO-Pins fuer das Display
//--------------------------------------------------------------
void P_LCD2119_InitIO(void)
{
  GPIO_InitTypeDef GPIO_InitStructure;

  //-----------------------------------------
  // Clock Enable von Port-B, Port-D und Port-E
  //-----------------------------------------
  RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOB,ENABLE);
  RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOD,ENABLE);
  RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOE,ENABLE);
  
  //-----------------------------------------
  // Alle Pins von Port-B initialisieren
  //-----------------------------------------
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_3;              //PB0 -> enable Backlight
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_UP;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  // Config von Port-B
  //GPIO_Init(GPIOB, &GPIO_InitStructure);
  GPIO_Init(GPIOD, &GPIO_InitStructure);
  
  //-----------------------------------------
  // Alle Pins von Port-D initialisieren
  //-----------------------------------------
  GPIO_PinAFConfig(GPIOD, GPIO_PinSource0,  GPIO_AF_FSMC); // PD0=FSMC_D2   -> DB2
  GPIO_PinAFConfig(GPIOD, GPIO_PinSource1,  GPIO_AF_FSMC); // PD1=FSMC_D3   -> DB3
  GPIO_PinAFConfig(GPIOD, GPIO_PinSource4,  GPIO_AF_FSMC); // PD4=FSMC_NOE  -> RD
  GPIO_PinAFConfig(GPIOD, GPIO_PinSource5,  GPIO_AF_FSMC); // PD5=FSMC_NWE  -> WR
  GPIO_PinAFConfig(GPIOD, GPIO_PinSource7,  GPIO_AF_FSMC); // PD7=FSMC_NE1  -> CS
  GPIO_PinAFConfig(GPIOD, GPIO_PinSource8,  GPIO_AF_FSMC); // PD8=FSMC_D13  -> DB15
  GPIO_PinAFConfig(GPIOD, GPIO_PinSource9,  GPIO_AF_FSMC); // PD9=FSMC_D14  -> DB16
  GPIO_PinAFConfig(GPIOD, GPIO_PinSource10, GPIO_AF_FSMC); // PD10=FSMC_D15 -> DB17  
  GPIO_PinAFConfig(GPIOD, GPIO_PinSource14, GPIO_AF_FSMC); // PD14=FSMC_D0  -> DB0
  GPIO_PinAFConfig(GPIOD, GPIO_PinSource15, GPIO_AF_FSMC); // PD15=FSMC_D1  -> DB1
  
  // Structur fuer Port-D
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_0 | GPIO_Pin_1 | GPIO_Pin_4 | GPIO_Pin_5 |
                                GPIO_Pin_7 | GPIO_Pin_8 | GPIO_Pin_9 | GPIO_Pin_10 |
                                GPIO_Pin_14 | GPIO_Pin_15;
  
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_100MHz;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  // Config von Port-D
  GPIO_Init(GPIOD, &GPIO_InitStructure);   
  
  //-----------------------------------------
  // Alle Pins von Port-E initialisieren
  //-----------------------------------------
  GPIO_PinAFConfig(GPIOE, GPIO_PinSource3,  GPIO_AF_FSMC); // PE3=FSMC_A19  -> RS
  GPIO_PinAFConfig(GPIOE, GPIO_PinSource7,  GPIO_AF_FSMC); // PE7=FSMC_D4   -> DB4
  GPIO_PinAFConfig(GPIOE, GPIO_PinSource8,  GPIO_AF_FSMC); // PE8=FSMC_D5   -> DB5  
  GPIO_PinAFConfig(GPIOE, GPIO_PinSource9,  GPIO_AF_FSMC); // PE9=FSMC_D6   -> DB6
  GPIO_PinAFConfig(GPIOE, GPIO_PinSource10, GPIO_AF_FSMC); // PE10=FSMC_D7  -> DB7
  GPIO_PinAFConfig(GPIOE, GPIO_PinSource11, GPIO_AF_FSMC); // PE11=FSMC_D8  -> DB10
  GPIO_PinAFConfig(GPIOE, GPIO_PinSource12, GPIO_AF_FSMC); // PE12=FSMC_D9  -> DB11
  GPIO_PinAFConfig(GPIOE, GPIO_PinSource13, GPIO_AF_FSMC); // PE13=FSMC_D10 -> DB12
  GPIO_PinAFConfig(GPIOE, GPIO_PinSource14, GPIO_AF_FSMC); // PE14=FSMC_D11 -> DB13
  GPIO_PinAFConfig(GPIOE, GPIO_PinSource15, GPIO_AF_FSMC); // PE15=FSMC_D12 -> DB14

  // Structur fuer Port-E
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_3 | GPIO_Pin_7 | GPIO_Pin_8 | GPIO_Pin_9 |
                                GPIO_Pin_10 | GPIO_Pin_11 | GPIO_Pin_12 | GPIO_Pin_13 |
                                GPIO_Pin_14 | GPIO_Pin_15;
  
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_100MHz;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  // Config von Port-E
  GPIO_Init(GPIOE, &GPIO_InitStructure);
  
}

//--------------------------------------------------------------
// interne Funktion
// Init vom FSMC fuer das Display
// CS an PD7=FSMC_NE1 = Bank-1 / PSRAM-1
//--------------------------------------------------------------
void P_LCD2119_InitFSMC(void)
{
  FSMC_NORSRAMInitTypeDef        FSMC_NORSRAMInitStructure;
  FSMC_NORSRAMTimingInitTypeDef  FSMC_NORSRAMTimingInitStructure;
  
  //-----------------------------------------
  // Clock Enable von FSMC
  //-----------------------------------------
  RCC_AHB3PeriphClockCmd(RCC_AHB3Periph_FSMC, ENABLE);  
  
  //-----------------------------------------
  // Structure fuer das Timing
  //-----------------------------------------  
  FSMC_NORSRAMTimingInitStructure.FSMC_AddressSetupTime = LCD_SSD2119_FSMC_AST;
  FSMC_NORSRAMTimingInitStructure.FSMC_AddressHoldTime = 0;
  FSMC_NORSRAMTimingInitStructure.FSMC_DataSetupTime = LCD_SSD2119_FSMC_DST;
  FSMC_NORSRAMTimingInitStructure.FSMC_BusTurnAroundDuration = 0;
  FSMC_NORSRAMTimingInitStructure.FSMC_CLKDivision = 0;
  FSMC_NORSRAMTimingInitStructure.FSMC_DataLatency = 0;
  FSMC_NORSRAMTimingInitStructure.FSMC_AccessMode = FSMC_AccessMode_A;

  //-----------------------------------------
  // Structure fuer Bank-1 / PSRAM-1
  //-----------------------------------------
  FSMC_NORSRAMInitStructure.FSMC_Bank = FSMC_Bank1_NORSRAM1;
  FSMC_NORSRAMInitStructure.FSMC_DataAddressMux = FSMC_DataAddressMux_Disable;
  FSMC_NORSRAMInitStructure.FSMC_MemoryType = FSMC_MemoryType_SRAM;
  FSMC_NORSRAMInitStructure.FSMC_MemoryDataWidth = FSMC_MemoryDataWidth_16b;
  FSMC_NORSRAMInitStructure.FSMC_BurstAccessMode = FSMC_BurstAccessMode_Disable;
  FSMC_NORSRAMInitStructure.FSMC_AsynchronousWait = FSMC_AsynchronousWait_Disable;
  FSMC_NORSRAMInitStructure.FSMC_WaitSignalPolarity = FSMC_WaitSignalPolarity_Low;
  FSMC_NORSRAMInitStructure.FSMC_WrapMode = FSMC_WrapMode_Disable;
  FSMC_NORSRAMInitStructure.FSMC_WaitSignalActive = FSMC_WaitSignalActive_BeforeWaitState;
  FSMC_NORSRAMInitStructure.FSMC_WriteOperation = FSMC_WriteOperation_Enable;
  FSMC_NORSRAMInitStructure.FSMC_WaitSignal = FSMC_WaitSignal_Disable;
  FSMC_NORSRAMInitStructure.FSMC_ExtendedMode = FSMC_ExtendedMode_Disable;
  FSMC_NORSRAMInitStructure.FSMC_WriteBurst = FSMC_WriteBurst_Disable;
  FSMC_NORSRAMInitStructure.FSMC_ReadWriteTimingStruct = &FSMC_NORSRAMTimingInitStructure;
  FSMC_NORSRAMInitStructure.FSMC_WriteTimingStruct = &FSMC_NORSRAMTimingInitStructure;

  // Config vom FSMC
  FSMC_NORSRAMInit(&FSMC_NORSRAMInitStructure);

  // Enable Bank-1 / PSRAM-1
  FSMC_NORSRAMCmd(FSMC_Bank1_NORSRAM1, ENABLE);
}

#define SSD2119_SLEEP_MODE_1_REG		0x10

//--------------------------------------------------------------
// interne Funktion
// initialisiert den SSD2119-Controller im Display
// mode :  0x6830=Portrait, 0x6838=Landscape
//--------------------------------------------------------------
void P_LCD2119_InitChip(uint16_t mode)
{

	// Minimal Initialisierung
	P_LCD2119_WriteReg(0x07,0x0021);       // Page 83 of SSD2119 datasheet
	P_LCD2119_WriteReg(0x00,0x0001);       // Page 83 of SSD2119 datasheet
	P_LCD2119_WriteReg(0x07,0x0023);       // Page 83 of SSD2119 datasheet
	P_LCD2119_WriteReg(0x10,0x0000);       // Page 83 of SSD2119 datasheet

	P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
	P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
	P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
	P_LCD2119_Delay(LCD_SSD2119_PAUSE2);

	P_LCD2119_WriteReg(0x07,0x0033);       // Page 83 of SSD2119 datasheet

	//							VDD0DWNDTTIIA000|
	//							SFF emomYYDDM   |
	//							MMM noso1010    |
	//							o10 Mdyd        |
	//							d   oene        |
	//							e   d c         |
	//							    e           |
	P_LCD2119_WriteReg(0x0011,0b0110100000110000); //0x6830);
	P_LCD2119_WriteReg(0x0002,0x0600);     // Page 40-42 of SSD2119 datasheet

	//                          0RRGBST0        |
	//                           LEDGMB         |
	//                            V R           |
	P_LCD2119_WriteReg(0x0001,0b0011000011101111); //0x30EF);



	/*P_LCD2119_WriteReg(0x0028,0x0006);       // Page 55-56 of SSD2119 datasheet

	P_LCD2119_WriteReg(0x0000,0x0001);       // Page 36 of SSD2119 datasheet

	P_LCD2119_WriteReg(0x0010,0x0000);       // Page 49 of SSD2119 datasheet

	P_LCD2119_WriteReg(0x0001,0b0011000011101111); //0x32EF);       // Page 36-39 of SSD2119 datasheet

	P_LCD2119_WriteReg(0x0002,0x0600);       // Page 40-42 of SSD2119 datasheet

	P_LCD2119_WriteReg(0x0003,0x6A38);       // Page 43-44 of SSD2119 datasheet

	P_LCD2119_WriteReg(0x0011,0x6870);       // Page 50-52 of SSD2119 datasheet

	P_LCD2119_WriteReg(0X000F,0x0000);       // Page 49 of SSD2119 datasheet

	P_LCD2119_WriteReg(0X000B,0x5308);       // Page 45 of SSD2119 datasheet

	P_LCD2119_WriteReg(0x000C,0x0003);       // Page 47 of SSD2119 datasheet

	P_LCD2119_WriteReg(0x000D,0x000A);       // Page 48 of SSD2119 datasheet

	P_LCD2119_WriteReg(0x000E,0x2E00);       // Page 48 of SSD2119 datasheet

	P_LCD2119_WriteReg(0x001E,0x00BE);       // Page 53 of SSD2119 datasheet

	P_LCD2119_WriteReg(0x0025,0x8000);       // Page 53 of SSD2119 datasheet

	P_LCD2119_WriteReg(0x0026,0x7800);       // Page 54 of SSD2119 datasheet

	P_LCD2119_WriteReg(0x004E,0x0000);       // Page 58 of SSD2119 datasheet

	P_LCD2119_WriteReg(0x004F,0x0000);       // Page 58 of SSD2119 datasheet

	P_LCD2119_WriteReg(0x0012,0x08D9);       // Page 49 of SSD2119 datasheet

	// Gamma Control (R30h to R3Bh) -- Page 56 of SSD2119 datasheet
	P_LCD2119_WriteReg(0x0030,0x0000);

	P_LCD2119_WriteReg(0x0031,0x0104);

	P_LCD2119_WriteReg(0x0032,0x0100);

	P_LCD2119_WriteReg(0x0033,0x0305);

	P_LCD2119_WriteReg(0x0034,0x0505);

	P_LCD2119_WriteReg(0x0035,0x0305);

	P_LCD2119_WriteReg(0x0036,0x0707);

	P_LCD2119_WriteReg(0x0037,0x0300);

	P_LCD2119_WriteReg(0x003A,0x1200);

	P_LCD2119_WriteReg(0x003B,0x0800);

	P_LCD2119_WriteReg(0x0007,0x0033);       // Page 45 of SSD2119 datasheet*/

	/*
  P_LCD2119_WriteReg(SSD2119_DISPLAY_CTRL_REG,0x0021);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(SSD2119_OSC_START_REG,0x0001);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(SSD2119_DISPLAY_CTRL_REG,0x0023);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(SSD2119_SLEEP_MODE_1_REG,0x0000);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(SSD2119_DISPLAY_CTRL_REG,0x0033);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(SSD2119_ENTRY_MODE_REG,mode);   // Portrait, Landscape
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(SSD2119_LCD_DRIVE_AC_CTRL_REG,0x0600);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  //P_LCD2119_WriteReg(0x0012,0x6CEB);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0003,0xA8A4);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x000C,0x0000);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x000D,0x080C);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x000E,0x2B00);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x001E,0x00B0);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);*/



  //                          0RRGBST0
  //                           LEDGMB
  //                            V R
  //P_LCD2119_WriteReg(0x0001,0b0011000011101111); //0x30EF);
  /*P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0005,0x0000);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0006,0x0000);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0016,0xEF1C);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0017,0x0103);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x000B,0x0000);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x000F,0x0000);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0041,0x0000);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0042,0x0000);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0048,0x0000);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0049,0x00EF); //0x013F);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x004A,0x0000);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x004B,0x0000);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0044,0xEF00);  // Horizontal Start und Ende
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0045,0x0000);  // Vertikal Start
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0046,0x013F);  // Vertikal Ende
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0030,0x0707);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0031,0x0204);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0032,0x0204);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0033,0x0502);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0034,0x0507);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0035,0x0204);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0036,0x0204);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0037,0x0502);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x003A,0x0302);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x002F,0x12BE);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x003B,0x0302);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0023,0x0000);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0024,0x0000);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x0025,0x8000);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x004f,0x0000);
  P_LCD2119_Delay(LCD_SSD2119_PAUSE2);
  P_LCD2119_WriteReg(0x004e,0x0000);
  // kleine pause
  P_LCD2119_Delay(LCD_SSD2119_PAUSE);*/
}

//--------------------------------------------------------------
// interne Funktion
// einen Wert aus einem Register auslesen
//--------------------------------------------------------------
uint16_t P_LCD2119_ReadReg(uint8_t reg_adr)
{
  // adresse anlegen
  LCD_REG = reg_adr;
  // wert auslesen   
  return LCD_RAM;
}

//--------------------------------------------------------------
// interne Funktion
// einen Wert in eine Register schreiben
//--------------------------------------------------------------
void P_LCD2119_WriteReg(uint8_t reg_adr, uint16_t reg_value)
{
  // adresse anlegen
  LCD_REG = reg_adr;
  // wert schreiben
  LCD_RAM = reg_value;  
}


//--------------------------------------------------------------
// kleine Pause (ohne Timer)
//--------------------------------------------------------------
void P_LCD2119_Delay(volatile uint32_t nCount)
{
  /*while(nCount--)
  {
  }*/
}
