/**
 * \file
 *
 * \brief Provides the low-level initialization functions that called 
 * on chip startup.
 *
 * Copyright (c) 2011 - 2013 Atmel Corporation. All rights reserved.
 *
 * \asf_license_start
 *
 * \page License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 *    Atmel microcontroller product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * \asf_license_stop
 *
 */

#include "system_sam3x.h"
#include "sam3xa.h"

/* @cond 0 */
/**INDENT-OFF**/
#ifdef __cplusplus
extern "C" {
#endif
/**INDENT-ON**/
/* @endcond */

/* Clock settings (84MHz) */
// Start-up Time 8x8 = 64 SCK
#define SYS_BOARD_OSCOUNT   (CKGR_MOR_MOSCXTST(0x8))

// PLLA Multiplier(MULA) = 13+1
// Start-up Time = 63x8 = 504 SCK
// PLLA-Divisor (DIVA) = 1
// PLLA-Out = 168 MHz
#define SYS_BOARD_PLLAR     (CKGR_PLLAR_ONE \
							| CKGR_PLLAR_MULA(0xdUL) \
							| CKGR_PLLAR_PLLACOUNT(0x3fUL) \
							| CKGR_PLLAR_DIVA(0x1UL))

// PLLA-Clock Divider 2 = 84MhZ
#define SYS_BOARD_MCKR      (PMC_MCKR_PRES_CLK_2 | PMC_MCKR_CSS_PLLA_CLK)

/* Clock Definitions */
#define SYS_UTMIPLL     		(480000000UL)	/* UTMI PLL frequency */

#define SYS_CKGR_MOR_KEY_VALUE	CKGR_MOR_KEY(0x37) /* Key to unlock MOR register */

/* FIXME: should be generated by sock */
uint32_t SystemCoreClock = CHIP_FREQ_MAINCK_RC_4MHZ;

/**
 * \brief Setup the microcontroller system.
 * Initialize the System and update the SystemFrequency variable.
 */
void SystemInit(void)
{
	/* Set FWS according to SYS_BOARD_MCKR configuration */
	EFC0->EEFC_FMR = EEFC_FMR_FWS(4);
	EFC1->EEFC_FMR = EEFC_FMR_FWS(4);

	/**
	 * Seite 565 - PMC Clock Generator Main Oscillator Register
	 * MAINCK - Main Clock versorgt PLLA, UTMI PLL und Master Clock Controller (PMC_MCKR)
	 * CKGR_MOR_MOSCSEL = 0 => The Main On-Chip RC Oscillator is selected.
	 * CKGR_MOR_MOSCSEL = 1 => The Main Crystal Oscillator is selected.
	 */
	// Ist der Main Crystal Oscillator als Taktquelle ausgewähtl ?
	if (!(PMC->CKGR_MOR & CKGR_MOR_MOSCSEL)) {
		/**
		 * Umschalten auf RC Oscillator, Main Crystal Oscillator einschalten, XIN & XOUT müssen beschaltet sein. (Quarz)
		 * Start-up Zeit auf 64 Clocks setzen.
		 * Die Clocks beziehen sich auf den Slow-Clock, hier ist bei Reset der interne 32kHz RC Oscillator ausgewählt.
		 * Externer 32kHz-Quarz kann im Supply-Controller aktiviert werden. (SUPC - Seite 287)
		 */
		PMC->CKGR_MOR = SYS_CKGR_MOR_KEY_VALUE | SYS_BOARD_OSCOUNT | 
			                     CKGR_MOR_MOSCRCEN | CKGR_MOR_MOSCXTEN;

		/**
		 * Warten bis der Quarz stabil ist. 64 * 1 / 32,768kHz = 1,953125ms
		 */
		while (!(PMC->PMC_SR & PMC_SR_MOSCXTS)) {
		}
	}

	/**
	 * Main Clock um schalten auf Main Crystal Oscillator.
	 */
	PMC->CKGR_MOR = SYS_CKGR_MOR_KEY_VALUE | SYS_BOARD_OSCOUNT | 
	                           CKGR_MOR_MOSCRCEN | CKGR_MOR_MOSCXTEN | CKGR_MOR_MOSCSEL;

	/**
	 * Warten bis Main Crystal Oscillator als Taktquelle gesetzt wurde.
	 */
	while (!(PMC->PMC_SR & PMC_SR_MOSCSELS)) {
	}

	/**
	 * Main Clock als Taktquelle für Master Clock Controller auswählen.
	 * Die PLLA muss erst konfiguriert und eingerastet sein um als Taktquelle für den
	 * Master Clock Controller verwendet werden zu können.
	 */
	PMC->PMC_MCKR = (PMC->PMC_MCKR & ~(uint32_t)PMC_MCKR_CSS_Msk) |
		                     PMC_MCKR_CSS_MAIN_CLK;
	/**
	 * Warten bis Master-Clock bereit ist.
	 */
	while (!(PMC->PMC_SR & PMC_SR_MCKRDY)) {
	}

	/**
	 * PLLA initialisieren.
	 * DIVA = 1 Divider is bypassed, 504 Clocks bis PLL eingerastet ist, MULA = 13
	 * XTAL / DIVA * (MULA + 1) = 12MHz / 1 * (13 + 1) = 168 MHz
	 * 504 * 1 / 32,768kHz = 15,380859375ms Start-up
	 */
	PMC->CKGR_PLLAR = SYS_BOARD_PLLAR;
	/**
	 * Warten bis PLLA eingerastet ist.
	 */
	while (!(PMC->PMC_SR & PMC_SR_LOCKA)) {
	}

	/**
	 * Prescaler einstellen, aber Master-Clock bleibt auf Main Clock = 12MHz
	 * Prescaler = 1 => 84MHz
	 */
	PMC->PMC_MCKR = (SYS_BOARD_MCKR & ~PMC_MCKR_CSS_Msk) | PMC_MCKR_CSS_MAIN_CLK;
	/**
	 * Warten bis Master-Clock bereit ist.
	 */
	while (!(PMC->PMC_SR & PMC_SR_MCKRDY)) {
	}

	/**
	 * PLLA als Taktquelle für Master Clock Controller auswählen.
	 */
	PMC->PMC_MCKR = SYS_BOARD_MCKR;
	/**
	 * Warten bis Master-Clock bereit ist.
	 */
	while (!(PMC->PMC_SR & PMC_SR_MCKRDY)) {
	}

	SystemCoreClock = CHIP_FREQ_CPU_MAX;
}

void SystemCoreClockUpdate(void)
{
	/* Determine clock frequency according to clock register values */
	switch (PMC->PMC_MCKR & PMC_MCKR_CSS_Msk) {
	case PMC_MCKR_CSS_SLOW_CLK:	/* Slow clock */
		if (SUPC->SUPC_SR & SUPC_SR_OSCSEL) {
			SystemCoreClock = CHIP_FREQ_XTAL_32K;
		} else {
			SystemCoreClock = CHIP_FREQ_SLCK_RC;
		}
		break;
	case PMC_MCKR_CSS_MAIN_CLK:	/* Main clock */
		if (PMC->CKGR_MOR & CKGR_MOR_MOSCSEL) {
			SystemCoreClock = CHIP_FREQ_XTAL_12M;
		} else {
			SystemCoreClock = CHIP_FREQ_MAINCK_RC_4MHZ;

			switch (PMC->CKGR_MOR & CKGR_MOR_MOSCRCF_Msk) {
			case CKGR_MOR_MOSCRCF_4_MHz:
				break;
			case CKGR_MOR_MOSCRCF_8_MHz:
				SystemCoreClock *= 2U;
				break;
			case CKGR_MOR_MOSCRCF_12_MHz:
				SystemCoreClock *= 3U;
				break;
			default:
				break;
			}
		}
		break;
	case PMC_MCKR_CSS_PLLA_CLK:	/* PLLA clock */
	case PMC_MCKR_CSS_UPLL_CLK:	/* UPLL clock */
		if (PMC->CKGR_MOR & CKGR_MOR_MOSCSEL) {
			SystemCoreClock = CHIP_FREQ_XTAL_12M;
		} else {
			SystemCoreClock = CHIP_FREQ_MAINCK_RC_4MHZ;

			switch (PMC->CKGR_MOR & CKGR_MOR_MOSCRCF_Msk) {
			case CKGR_MOR_MOSCRCF_4_MHz:
				break;
			case CKGR_MOR_MOSCRCF_8_MHz:
				SystemCoreClock *= 2U;
				break;
			case CKGR_MOR_MOSCRCF_12_MHz:
				SystemCoreClock *= 3U;
				break;
			default:
				break;
			}
		}
		if ((PMC->PMC_MCKR & PMC_MCKR_CSS_Msk) == PMC_MCKR_CSS_PLLA_CLK) {
			SystemCoreClock *= ((((PMC->CKGR_PLLAR) & CKGR_PLLAR_MULA_Msk) >> 
				                            CKGR_PLLAR_MULA_Pos) + 1U);
			SystemCoreClock /= ((((PMC->CKGR_PLLAR) & CKGR_PLLAR_DIVA_Msk) >> 
				                             CKGR_PLLAR_DIVA_Pos));
		} else {
			SystemCoreClock = SYS_UTMIPLL / 2U;
		}
		break;
	}

	if ((PMC->PMC_MCKR & PMC_MCKR_PRES_Msk) == PMC_MCKR_PRES_CLK_3) {
		SystemCoreClock /= 3U;
	} else {
		SystemCoreClock >>= ((PMC->PMC_MCKR & PMC_MCKR_PRES_Msk) >> 
			                          PMC_MCKR_PRES_Pos);
	}
}

/** 
 * Initialize flash.
 */
void system_init_flash(uint32_t ul_clk)
{
	/* Set FWS for embedded Flash access according to operating frequency */
	if (ul_clk < CHIP_FREQ_FWS_0) {
		EFC0->EEFC_FMR = EEFC_FMR_FWS(0);
		EFC1->EEFC_FMR = EEFC_FMR_FWS(0);
	} else if (ul_clk < CHIP_FREQ_FWS_1) {
		EFC0->EEFC_FMR = EEFC_FMR_FWS(1);
		EFC1->EEFC_FMR = EEFC_FMR_FWS(1);
	} else if (ul_clk < CHIP_FREQ_FWS_2) {
		EFC0->EEFC_FMR = EEFC_FMR_FWS(2);
		EFC1->EEFC_FMR = EEFC_FMR_FWS(2);
	} else if (ul_clk < CHIP_FREQ_FWS_3) {
		EFC0->EEFC_FMR = EEFC_FMR_FWS(3);
		EFC1->EEFC_FMR = EEFC_FMR_FWS(3);
	} else if (ul_clk < CHIP_FREQ_FWS_4) {
		EFC0->EEFC_FMR = EEFC_FMR_FWS(4);
		EFC1->EEFC_FMR = EEFC_FMR_FWS(4);	
	} else {
		EFC0->EEFC_FMR = EEFC_FMR_FWS(5);
		EFC1->EEFC_FMR = EEFC_FMR_FWS(5);
	}
}

/* @cond 0 */
/**INDENT-OFF**/
#ifdef __cplusplus
}
#endif
/**INDENT-ON**/
/* @endcond */
