/*
 * Copyright (c) 2013 - 2014, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "board.h"
#include "i2c/fsl_i2c_slave_driver.h"
#include "device/fsl_device_registers.h"
#include "port/hal/fsl_port_hal.h"
#include "utilities/fsl_debug_uart.h"
#include "clock/fsl_clock_manager.h"
#include <string.h>
#include <math.h>
#include <stdio.h>
#include "gpio/hal/fsl_gpio_hal.h"
#include "utilities/sw_timer.h"

void init_hardware(void);
void LED_toggle_master(void);
void LED_turnon_master(void);
void LED_toggle_slave(void);
void LED_turnon_slave(void);

enum _subaddress_index_e
{
    Subaddress_Index_0 = 0x00,
    Subaddress_Index_1 = 0x01,
    Subaddress_Index_2 = 0x02,
    Subaddress_Index_3 = 0x03,
    Subaddress_Index_4 = 0x04,
    Subaddress_Index_5 = 0x05,
    Subaddress_Index_6 = 0x06,
    Subaddress_Index_7 = 0x07,
    Invalid_Subaddress_Index,
    Max_Subaddress_Index

};

//u8SinkData is received from I2C master
uint8_t u8SinkData      = 0x00;
//u8SourceData will be sent to I2C master
uint8_t u8SourceData    = 0xCD;

uint8_t u8SubaddressIndex = Invalid_Subaddress_Index;
uint8_t u8SlaveDataBuffer[Max_Subaddress_Index]   = {0};

static status_t data_sink(uint8_t sinkByte)
{
    if (u8SubaddressIndex == Invalid_Subaddress_Index)
    {
        //the first byte received is subaddress
        if (sinkByte < Invalid_Subaddress_Index)
        {
            u8SubaddressIndex = sinkByte;
        }
    }
    else
    {
        //the seconde byte received is data
        u8SlaveDataBuffer[u8SubaddressIndex] = sinkByte;

        //reset index as invalid for the next data tx/rx circle.
        u8SubaddressIndex = Invalid_Subaddress_Index;
    }
    
    //u8SinkData = sinkByte;
    return kStatus_Success;
}

static status_t data_source(uint8_t * sourceByte)
{

    *sourceByte = u8SlaveDataBuffer[u8SubaddressIndex];
    return kStatus_Success;
}

static void on_error(status_t error)
{
    switch (error)
    {
        case kStatus_I2C_SlaveTxUnderrun:
            //printf("I2C Error (0x%X) - Slave TX Underrun.\r\n", error);
            break;

        case kStatus_I2C_SlaveRxOverrun:
            //printf("I2C Error (0x%X) - Slave RX Overrun.\r\n", error);
            break;

        case kStatus_I2C_AribtrationLost:
            //printf("I2C Error (0x%X) - Arbitration Lost.\r\n", error);
            break;

        default:
            //printf("I2C Error (0x%X) - Unknown Error.\r\n", error);
            break;
    }
}

void delay(unsigned int delay_value)
{
    unsigned int i, j, k;

    for (i=0; i<delay_value; i++)
    {
        for (j=0; j<0xF; j++)
            for (k=0; k<0xFFFF; k++)
            {
                __ASM volatile("nop");
            }
    }

}

////////////////////////////////////////////////////////////////////////////////
// Code
////////////////////////////////////////////////////////////////////////////////
void main(void)
{
    i2c_slave_user_config_t slave =
    {
        .data_source = data_source,
        .data_sink = data_sink,
        .on_error = on_error,
        .slaveAddress = 0x3A
    };  

    init_hardware();

    sw_timer_init_service();

    //Init I2C1 driver, I2C instance 1, PTC10/PTC11.
    // Configure pinmux for this peripheral.
    if (&configure_i2c_pin_mux != 0)
    {
        configure_i2c_pin_mux(HW_I2C1);
    }

    i2c_slave_init(HW_I2C1, &slave);

    printf("\r\n====== I2C Slave ======\r\n\r\n");

    while (u8SinkData != 0xAB)
    {
        //waiting for data received from i2c master
        //LED blinks slowly to indicate i2c communication idle
        delay(5);
        LED_toggle_slave();
    }

    //turn LED-Orange and Yellow on to indicates I2C communication OK
    LED_turnon_slave();
    LED_turnon_master();

    while (1)
    {
        delay(10);
    }
}

////////////////////////////////////////////////////////////////////////////////
// EOF
////////////////////////////////////////////////////////////////////////////////


