/*
 * Copyright (c) 2013 - 2014, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "board.h"
#include "fsl_i2c_slave_driver.h"
#include "fsl_device_registers.h"
#include "fsl_port_hal.h"
#include "fsl_debug_uart.h"
#include "fsl_clock_manager.h"
#include <string.h>
#include <math.h>
#include <stdio.h>
#include "fsl_gpio_hal.h"
#include "fsl_smc_manager.h"
#include "fsl_smc_hal.h"
#include "fsl_interrupt_manager.h"
#include "sw_timer.h"


#define UPPER_VALUE_LIMIT       (1)     /*! This value/10 is going to be added to current Temp to set the upper boundary*/
#define LOWER_VALUE_LIMIT       (1)     /*! This Value/10 is going to be subtracted from current Temp to set the lower boundary*/
#define LPTMR_COMPARE_VALUE     (500)   /*! Low Power Timer interrupt time in miliseconds */
#define UPDATE_BOUNDARIES_TIME  (20)    /*! This value indicates the number of cycles needed to update boundaries. To know the Time it will take, multiply this value times LPTMR_COMPARE_VALUE*/


void LED_toggle_master(void)
{
    gpio_toggle_pin_output(kGpioLED1);
}
void LED_turnon_master(void)
{
    gpio_clear_pin_output(kGpioLED1);
}
void LED_turnoff_master(void)
{
    gpio_set_pin_output(kGpioLED1);
}
void LED_toggle_slave(void)
{
    gpio_toggle_pin_output(kGpioLED2);
}
void LED_turnon_slave(void)
{
    gpio_clear_pin_output(kGpioLED2);
}
void LED_turnoff_slave(void)
{
    gpio_set_pin_output(kGpioLED2);
}

enum _subaddress_index_e
{
    Subaddress_Index_0 = 0x00,
    Subaddress_Index_1 = 0x01,
    Subaddress_Index_2 = 0x02,
    Subaddress_Index_3 = 0x03,
    Subaddress_Index_4 = 0x04,
    Subaddress_Index_5 = 0x05,
    Subaddress_Index_6 = 0x06,
    Subaddress_Index_7 = 0x07,
    Invalid_Subaddress_Index,
    Max_Subaddress_Index
};


//u8SinkData is received from I2C master
uint8_t u8SinkData      = 0x00;
//u8SourceData will be sent to I2C master
uint8_t u8SourceData    = 0xCD;

uint8_t u8SubaddressIndex = Invalid_Subaddress_Index;
uint8_t u8SlaveDataBuffer[Max_Subaddress_Index]   = {'I', '2', 'C', '-', 'C', 'O', 'M', 'M', };

static i2c_status_t data_sink(uint8_t sinkByte)
{
    if (u8SubaddressIndex == Invalid_Subaddress_Index)
    {
        //the first byte received is subaddress
        if (sinkByte < Invalid_Subaddress_Index)
        {
            u8SubaddressIndex = sinkByte;
        }

        LED_toggle_slave();
    }
    else
    {
        //the seconde byte received is data
        u8SlaveDataBuffer[u8SubaddressIndex] = sinkByte;

        //reset index as invalid for the next data tx/rx circle.
        u8SubaddressIndex = Invalid_Subaddress_Index;

        LED_toggle_slave();
    }

    return kStatus_I2C_Success;
}

static i2c_status_t data_source(uint8_t * sourceByte)
{

    *sourceByte = u8SlaveDataBuffer[u8SubaddressIndex];
    
    //reset index as invalid for the next data tx/rx circle.
    u8SubaddressIndex = Invalid_Subaddress_Index;

    LED_toggle_slave();
    
    return kStatus_I2C_Success;
}


static void on_error(i2c_status_t error)
{
    switch (error)
    {
        case kStatus_I2C_SlaveTxUnderrun:
            //printf("I2C Error (0x%X) - Slave TX Underrun.\r\n", error);
            break;

        case kStatus_I2C_SlaveRxOverrun:
            //printf("I2C Error (0x%X) - Slave RX Overrun.\r\n", error);
            break;

        case kStatus_I2C_AribtrationLost:
            //printf("I2C Error (0x%X) - Arbitration Lost.\r\n", error);
            break;

        default:
            //printf("I2C Error (0x%X) - Unknown Error.\r\n", error);
            break;
    }
}

void delay(unsigned int delay_value)
{
    unsigned int i, j, k;

    for (i=0; i<delay_value; i++)
    {
        for (j=0; j<0xF; j++)
            for (k=0; k<0xFFFF; k++)
            {
                __ASM volatile("nop");
            }
    }

}

////////////////////////////////////////////////////////////////////////////////
// Code
////////////////////////////////////////////////////////////////////////////////
void main(void)
{
    i2c_slave_user_config_t slave =
    {
        .data_source = data_source,
        .data_sink = data_sink,
        .on_error = on_error,
        .slaveAddress = 0x3A
    };  

    // Low Power Configuration
    smc_power_mode_config_t smcConfig;
    smc_power_mode_protection_config_t smcProtConfig;

    hardware_init();
    dbg_uart_init();

    gpio_init(0, ledPins);

    // Initiate I2C instance 1 module
    i2c_slave_init(BOARD_I2C_COMM_INSTANCE, &slave);

    printf("\r\n====== I2C Slave ======\r\n\r\n");

    //turn LED_slave on to indicate I2C slave status is waiting for date receiving.
    LED_turnon_slave();
    LED_turnoff_master();
    delay(5);

    printf("\r\n I2C slave enters low power mode...\r\n");

    // set to allow entering specific modes
    smcProtConfig.vlpProt = true;
    smc_hal_config_power_mode_protection(&smcProtConfig);
    
    // set power mode to specific Run mode
    smcConfig.lpwuiOption = true;
    smcConfig.lpwuiOptionValue = kSmcLpwuiEnabled;
    smcConfig.porOption = false;
    smcConfig.powerModeName = kPowerModeWait;

    // Entry to Low Power Mode
    smc_set_power_mode(&smcConfig);

    //LED_slave is still on during low power mode until I2C master send data to slave.
    //Turn off LED_slave to indicate MCU wake up by I2C address matching interrupt
    LED_turnoff_slave();
    printf("\r\n I2C slave wakes up from low power mode by I2C address matching.\r\n");
    

    while(1);
}

////////////////////////////////////////////////////////////////////////////////
// EOF
////////////////////////////////////////////////////////////////////////////////


