/*
 * Copyright (c) 2014, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>

#include "fsl_clock_manager.h"
#include "board.h"
#include "fsl_debug_uart.h"
#include "fsl_i2c_master_driver.h"
#include "fsl_uart_driver.h"
#include "fsl_i2c_slave_driver.h"
#include "fsl_smc_manager.h"
#include "i2c_rtos.h"
#include "fsl_adc_driver.h"
extern uint32_t gSlaveId;

/*******************************************************************************
 * Defination
 ******************************************************************************/

/* These values are used covert temperature. DO NOT MODIFY */
#define VTEMP25_ADC             (14219)
#define K                       (10000)
#define M1                      (250000)
#define M2                      (311)
/*******************************************************************************
 * Prototypes
 ******************************************************************************/
extern void task_slave(void *param);

/********************************************************************************
 * Variables
 ******************************************************************************/
/* ADC Channel Configuration */
adc_channel_config_t tempSnseChannelConfig =
{
    .channelId = kAdcChannelTemperature,
    .muxSelect = kAdcChannelMuxA,
    .isDifferentialEnabled = false,
    .isInterruptEnabled = false
};

/* global handler var */
task_handler_t hTaskSlave;
/* task define */
FSL_RTOS_TASK_DEFINE(task_slave, TASK_SLAVE_STACK_SIZE, "slave", false);
 /*******************************************************************************
 * Code
 ******************************************************************************/

/*!
 * Get the 32bits temperature in byte
 * designed for BM version of I2C_RTOS demo
 * from the ISR context
 */
uint8_t get_temp_in_byte(uint32_t index)
{
    uint32_t adcValue;
    uint32_t Temperature;
    static uint8_t *pTemp;
    pTemp = (uint8_t*)&Temperature;
    /* ADC starts conversion */
    adc_start_conversion(HW_ADC1, &tempSnseChannelConfig);
    /* poll to complete status and read back result */
    adcValue = adc_get_conversion_value(HW_ADC1, &tempSnseChannelConfig);
    /* ADC stop conversion */
    adc_stop_conversion(HW_ADC1, &tempSnseChannelConfig);
    /* convert to temperature */
    Temperature = (M1 - (adcValue - VTEMP25_ADC) * M2)/K;
    return pTemp[index];
}

void main(void)
{
    hardware_init();
    gpio_init(NULL, ledPins);

    /* ADC Configuration */
    adc_user_config_t tempSnseAdcConfig =
    {
        .clockSourceMode = kAdcClockSourceAsynchrounous,
        .clockSourceDividerMode = kAdcClockDivider8,
        .resolutionMode = kAdcSingleDiff16,
        .referenceVoltageMode = kAdcVoltageVref,
        .isContinuousEnabled = false
    };

    /* ADC Extended Configuration */
    adc_extend_config_t adcExtConfig =
    {
        .isLowPowerEnabled      = false,
        .isLongSampleEnabled    = false,
        .hwLongSampleMode       = kAdcLongSampleExtra20,
        .isHighSpeedEnabled     = false,
        .isAsynClockEnabled     = true,
        .isHwTriggerEnabled     = false,
        .isHwCompareEnabled     = false,
        .isHwCompareGreaterEnabled = false,
        .isHwCompareRangeEnabled= false,
        .hwCompareValue1        = 0,
        .hwCompareValue2        = 0,
        .isHwAverageEnabled     = true,
        .hwAverageSampleMode    = kAdcHwAverageCount32,
        .isDmaEnabled           = false
    };

    /* Initialize ADC */
    adc_init(HW_ADC1, &tempSnseAdcConfig);
    /* Initialize ADC Extended */
    adc_init_extend(HW_ADC1, &adcExtConfig);
    
    /* SMC Power mode protection configurations */
    smc_power_mode_protection_config_t smc_power_prot_cfg = 
    {
        .vlpProt = true,
        .llsProt = false,
        .vllsProt = false
    };
    
    /* Configure the power mode protection */
    smc_hal_config_power_mode_protection(&smc_power_prot_cfg);
    
    /* get cpu uid low value for slave */
    gSlaveId = HW_SIM_UIDL_RD();
    
    /* enable sw timer for delay in BM */
    sw_timer_init_service();
    
    /* enable printf */
    dbg_uart_init();
    
    printf("i2c_rtos_slave_bm demo\r\n");
    
    /* task list initialize*/
    POLL_init();
    
    /* create task(in BM: only the first registered task can be executed) */
    task_create(task_slave, 512, (void *)0, &hTaskSlave);
    
    while(1)
    {
        Poll();
    }

}

/*******************************************************************************
 * EOF
 ******************************************************************************/