/*
 * Copyright (c) 2013, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>

#include "fsl_uart_hal.h"
#include "clock/fsl_clock_manager.h"
#include "gpio_pins.h"
#include "low_power_demo.h"

/*******************************************************************************
 * Variables
 ******************************************************************************/
extern uint8_t uartEnabled; /*!< indicates whether the uart port is currently enabled or not */

/*******************************************************************************
 * Code
 ******************************************************************************/

/*!
 * @brief Set up the LLWU for wakeup the MCU from LLS and VLLSx modes 
 * from the selected pin or module.
 *  
 * @param
 * pin_en - unsigned integer, bit position indicates the pin is enabled.  
 *          More than one bit can be set to enable more than one pin at a time.  
 * @param 
 * rise_fall - 0x00 = External input disabled as wakeup
 *             0x01 - External input enabled as rising edge detection
 *             0x02 - External input enabled as falling edge detection
 *             0x03 - External input enablge as any edge detection
 * @param
 * module_en - unsigned char, bit position indicates the module is enabled.  
 *             More than one bit can be set to enabled more than one module                   
 *  
 * for example:  if bit 0 and 1 need to be enabled as rising edge detect call this  routine with
 * pin_en = 0x0003 and rise_fall = 0x02
   
 * Note: to set up one set of pins for rising and another for falling, 2 calls to this 
 *       function are required, 1st for rising then the second for falling.
 */
void llwu_configure(uint32_t pinEn, uint8_t riseFall, uint8_t moduleEn ) 
{
    uint8_t temp;
    
    temp = LLWU_PE1;
    if( pinEn & 0x0001)
    {      
        temp |= LLWU_PE1_WUPE0(riseFall);
        printf("LLWU configured pins PTE1/LLWU_P0 is LLWU wakeup source\n\r");
        LLWU_F1 |= LLWU_F1_WUF0_MASK;   /* write one to clear the flag */     
    }
    if( pinEn & 0x0002)
    {      
        temp |= LLWU_PE1_WUPE1(riseFall);
        printf("LLWU configured pins PTE2/LLWU_P1 is LLWU wakeup source\n\r");
        LLWU_F1 |= LLWU_F1_WUF1_MASK;   /* write one to clear the flag */  
    }
    if( pinEn & 0x0004)
    {
        temp |= LLWU_PE1_WUPE2(riseFall);
        printf("LLWU configured pins PTE4/LLWU_P2 is LLWU wakeup source\n\r");
        LLWU_F1 |= LLWU_F1_WUF2_MASK;   /* write one to clear the flag */  
    }
    if( pinEn & 0x0008)
    {
        temp |= LLWU_PE1_WUPE3(riseFall);
        printf("LLWU configured pins PTA4/LLWU_P3 is LLWU wakeup source\n\r");
        LLWU_F1 |= LLWU_F1_WUF3_MASK;   /* write one to clear the flag */  
    }
    LLWU_PE1 = temp;

    temp = LLWU_PE2;
    if( pinEn & 0x0010)
    {
        temp |= LLWU_PE2_WUPE4(riseFall);
        printf("LLWU configured pins PTA13/LLWU_P4 is LLWU wakeup source\n\r");
        LLWU_F1 |= LLWU_F1_WUF4_MASK;   /* write one to clear the flag */  
    }
    if( pinEn & 0x0020)
    {
        temp |= LLWU_PE2_WUPE5(riseFall);
        printf("LLWU configured pins PTB0/LLWU_P5 is LLWU wakeup source\n\r");
        LLWU_F1 |= LLWU_F1_WUF5_MASK;   /* write one to clear the flag */
    }
    if( pinEn & 0x0040)
    {
        temp |= LLWU_PE2_WUPE6(riseFall);
        printf("LLWU configured pins PTC1/LLWU_P6 is LLWU wakeup source\n\r");
        LLWU_F1 |= LLWU_F1_WUF6_MASK;   /* write one to clear the flag */
    }
    if( pinEn & 0x0080)
    {
        temp |= LLWU_PE2_WUPE7(riseFall);
        printf("LLWU configured pins PTC3/LLWU_P7 is LLWU wakeup source\n\r");
        LLWU_F1 |= LLWU_F1_WUF7_MASK;   /* write one to clear the flag */
    }
    LLWU_PE2 = temp;

    temp = LLWU_PE3;
    if( pinEn & 0x0100)
    {
        temp |= LLWU_PE3_WUPE8(riseFall);
        printf("LLWU configured pins PTC4/LLWU_P8 is LLWU wakeup source\n\r");
        LLWU_F2 |= LLWU_F2_WUF8_MASK;   /* write one to clear the flag */
    }
    if( pinEn & 0x0200)
    {
        temp |= LLWU_PE3_WUPE9(riseFall);
        printf("LLWU configured pins PTC5/LLWU_P9 is LLWU wakeup source\n\r");
        LLWU_F2 |= LLWU_F2_WUF9_MASK;   /* write one to clear the flag */
    }
    if( pinEn & 0x0400)
    {
        temp |= LLWU_PE3_WUPE10(riseFall);
        printf("LLWU configured pins PTC6/LLWU_P10 to be an LLWU wakeup source\n\r");
        LLWU_F2 |= LLWU_F2_WUF10_MASK;   /* write one to clear the flag */
    }
    if( pinEn & 0x0800)
    {
        temp |= LLWU_PE3_WUPE11(riseFall);
        printf("LLWU configured pins PTC11/LLWU_P11 to be an LLWU wakeup source\n\r");
        LLWU_F2 |= LLWU_F2_WUF11_MASK;   /* write one to clear the flag */
    }
    LLWU_PE3 = temp;

    temp = LLWU_PE4;
    if( pinEn & 0x1000)
    {
        temp |= LLWU_PE4_WUPE12(riseFall);
        printf("LLWU configured pins PTD0/LLWU_P12 to be an LLWU wakeup source\n\r");
        LLWU_F2 |= LLWU_F2_WUF12_MASK;   /* write one to clear the flag */
    }
    if( pinEn & 0x2000)
    {
        temp |= LLWU_PE4_WUPE13(riseFall);
        printf("LLWU configured pins PTD2/LLWU_P13 to be an LLWU wakeup source\n\r");
        LLWU_F2 |= LLWU_F2_WUF13_MASK;   /* write one to clear the flag */
    }
    if( pinEn & 0x4000)
    {
        temp |= LLWU_PE4_WUPE14(riseFall);
        printf("LLWU configured pins PTD4/LLWU_P14 is LLWU wakeup source\n\r");
        LLWU_F2 |= LLWU_F2_WUF14_MASK;   /* write one to clear the flag */
    }
    if( pinEn & 0x8000)
    {
        temp |= LLWU_PE4_WUPE15(riseFall);
        printf("LLWU configured pins PTD6/LLWU_P15 is LLWU wakeup source\n\r");
        LLWU_F2 |= LLWU_F2_WUF15_MASK;   /* write one to clear the flag */
    }
    LLWU_PE4 = temp;
    if (moduleEn == 0){
      LLWU_ME = 0;
    }else  {
    LLWU_ME |= moduleEn;  /*Set up more modules to wakeup up */
    printf("LLWU configured modules as LLWU wakeup sources = 0x%02X\n\r",(LLWU_ME));

    }
} /* End LLWU Configuration */
/*******************************************************************************
 * EOF
 ******************************************************************************/
