/*
 * Copyright (c) 2013 - 2014, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "shell.h"
#include "board.h"

#include "fsl_uart_hal.h"
#include "clock/fsl_clock_manager.h"
#include "fsl_uart_driver.h"
/*******************************************************************************
 * Defination
 ******************************************************************************/

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
static uint8_t Getchar(void);
static void Putchar(uint8_t ch);
extern const cmd_tbl_t CommandFun_Help;
extern const cmd_tbl_t CommandFun_Hist;
static const cmd_tbl_t CommandFun_Test;
static int DoTest(int argc, char * const argv[]);

/********************************************************************************
 * Variables
 ******************************************************************************/
static uart_state_t gUARTState;
static uart_user_config_t gShellUARTConfig =        //!< global uart config struct
{
    115200,
    kUartParityDisabled,
    kUartOneStopBit,
    kUart8BitsPerChar
};
static shell_io_install_t Shell_IOInstallStruct1 =
{
    .sh_getc = Getchar,
    .sh_putc = Putchar,
};

 /*******************************************************************************
 * Code
 ******************************************************************************/

 /*!
 * @brief get a char
 */
static uint8_t Getchar(void)
{
    uint8_t ch = 0;
    uart_receive_data(&gUARTState, &ch, 1, 1);
    return ch;
}
 /*!
 * @brief put a char
 */
static void Putchar(uint8_t ch)
{
    uart_send_data(&gUARTState, &ch, 1, 1);
}

void main(void)
{
    uint8_t i;
    /* UART init */
    for (i = 0; i < HW_PORT_INSTANCE_COUNT; i++)
    {
        clock_manager_set_gate(kClockModulePORT, i, true);
    }
    /* Init UART */
    configure_uart_pin_mux(BOARD_DEBUG_UART_INSTANCE);
#ifdef CPU_MKL25Z128VLK4
    SIM->SOPT2 |= SIM_SOPT2_PLLFLLSEL_MASK; // set PLLFLLSEL to select the PLL for this clock source
    SIM->SOPT2 |= SIM_SOPT2_UART0SRC(1);    // select the PLLFLLCLK as UART0 clock source
#endif
    uart_init(BOARD_DEBUG_UART_INSTANCE, &gUARTState, &gShellUARTConfig);

    /* install shell IO */
    shell_io_install(&Shell_IOInstallStruct1);
#ifdef SHELL_CONFIG_USE_STDIO
    printf("\r\nIf you see this string, It means that UART%d & SDTOUT is OK!\r\n", BOARD_DEBUG_UART_INSTANCE);
#else
    shell_printf("\r\nIf you see this string, It means SDTOUT is disabled and UART%d is OK in shell.\r\n", BOARD_DEBUG_UART_INSTANCE);
#endif /* SHELL_CONFIG_USE_STDIO */

#ifdef SHELL_CONFIG_AUTO_COMPLETE
    shell_printf("auto complete is enabled in shell\r\n");
#else
    shell_printf("auto complete is disabled in shell\r\n");
#endif /* SHELL_CONFIG_AUTO_COMPLETE */

#ifdef SHELL_CONFIG_USE_HIST
    shell_printf("history function is enabled in shell\r\n");
#else
    shell_printf("history function is disabled in shell\r\n");
#endif /* SHELL_CONFIG_USE_HIST */

    /* register user functions */
#ifdef SHELL_CONFIG_USE_HIST
    shell_register_function(&CommandFun_Hist);
#endif /* SHELL_CONFIG_USE_HIST */
    shell_register_function(&CommandFun_Help);
    shell_register_function(&CommandFun_Test);
    /* main loop */
    while (1)
    {
        shell_main_loop("SHELL>>");
    }
}

 /*!
 * @brief shell command register type
 */
static const cmd_tbl_t CommandFun_Test =
{
    .name = "test",
    .maxargs = 5,
    .repeatable = 1,
    .cmd = DoTest,
    .usage = "help - app test function",
    .complete = NULL,
    .help = "long help - app test function",

};

 /*!
 * @brief the app implementation function
 */
static int DoTest(int argc, char * const argv[])
{
    shell_printf("I am the test funtion\r\n");
    while (argc--)
    {
        shell_printf("argv[%d]:%s\r\n", argc, argv[argc]);
    }
    return CMD_RET_SUCCESS;
}

/*******************************************************************************
 * EOF
 ******************************************************************************/


