/*
 * Copyright (c) 2013 - 2014, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include "fsl_wdog_driver.h"
#include "fsl_gpio_driver.h"
#include "board.h"

#define SLOW_BLINK_TIME         8
#define FAST_BLINK_TIME         7

void set_busyled_Off()
{
    gpio_set_pin_output(kGpioLED1);
}

void set_busyled_on()
{
    gpio_clear_pin_output(kGpioLED1);
}

void toggle_active_led()
{
    gpio_toggle_pin_output(kGpioLED2);
}

uint32_t is_key_pressed()
{
    uint32_t pinState;
    pinState = gpio_read_pin_input(kGpioSW1);
    return (!pinState);
}

/*! 
 * @brief Print chip reset reason
 *
 * print chip reset reson to the terminal
 */
void print_reset_reason()
{
    /* Determine the last cause(s) of reset */
    printf("\r\n\r\n********************************");
    if (RCM->SRS1 & RCM_SRS1_SW_MASK)
        printf("\r\nSoftware Reset");
    if (RCM->SRS1 & RCM_SRS1_LOCKUP_MASK)
        printf("\r\nCore Lockup Event Reset");
    if (RCM->SRS1 & RCM_SRS1_JTAG_MASK)
        printf("\r\nJTAG Reset");
    if (RCM->SRS0 & RCM_SRS0_POR_MASK)
        printf("\r\nPower-on Reset");
    if (RCM->SRS0 & RCM_SRS0_PIN_MASK)
        printf("\r\nExternal Pin Reset");
    if (RCM->SRS0 & RCM_SRS0_WDOG_MASK)
        printf("\r\nWatchdog(COP) Reset");
    if (RCM->SRS0 & RCM_SRS0_LOC_MASK)
        printf("\r\nLoss of Clock Reset");
    if (RCM->SRS0 & RCM_SRS0_LVD_MASK)
        printf("\r\nLow-voltage Detect Reset");
    if (RCM->SRS0 & RCM_SRS0_WAKEUP_MASK)
        printf("\r\nLLWU Reset");
}

/*!
 * @brief Delay some time
 *
 * used to simulate the program's execution
 *
 * @param[in] t         the time that would be delayed
 */
void time_delay(uint32_t t)
{
    uint32_t i, j;
    
    for (i = 0; i < t; i++)
    {
        for (j = 0; j < 100000; j++)
        {
            __ASM volatile("nop");
        }
    }
}

void wdog_irq_handler(void)
{
    /* Message to user */
    printf("\nThis is the Watchdog Interrupt Routine!\n");  
    /* Clear Watchdog Interrupt Flag.  This also gets cleared on a system reset */  
    //WDOG->STCTRLL |= WDOG_STCTRLL_INTFLG_MASK;
    wdog_hal_clear_interrupt_flag();
}

/*!
 * @brief Watchdog main routine 
 *
 * Run a simple application which enables watchdog, then
 * continuously refreshes the watchdog to prevent CPU reset
 * Upon SW1 button push, the watchdog will expire after approximately 2 seconds and issue reset
 * To debug, use the OpenSDA debugger connection to allow recovery after watchdog reset, this way
 * the debug connection can be resumed after reset.  
 * Recovery not possible when debugging with JLink and powering through OpenSDA.
 *
 */
void main(void)
{
    uint32_t delay, watchdog_demo_loop_count = 0;

    hardware_init();
    dbg_uart_init();
    gpio_init(switchPins, ledPins);

    /* print chip reset reason */
    print_reset_reason();     
    
    /* if not wdog reset, clear reset count */
    if (!(RCM->SRS0 & RCM_SRS0_WDOG_MASK)) 
    {
        wdog_clear_reset_count();
    }

    printf("\r\nWatchdog(cop) reset count: %d", wdog_get_reset_count());      

    wdog_user_config_t wdoginit = 
    {
        .timeOutValue = 2048,	/* Watchdog overflow time is 2s */
        .windowValue = 0,
        .clockPrescaler = kWdogClockPrescaler1,
        .wdogCallbackFunc = wdog_irq_handler,
        .updateRegisterEnable = true,
        .clockSource = kWdogDedicatedClock,
        .cpuWaitModeEnable = true,
        .cpuStopModeEnable = true,
        .cpuDebugModeEnable = false,
    };
    wdog_init(&wdoginit);
    
    /* First enable LED to show we are running */
    set_busyled_on();
    /*******************************************************************/
    /* Continue to run in loop to refresh watchdog until SW1 is pushed */
    /*******************************************************************/
    while (1) 
    {
        /* LED toggle shows we are refreshing the watchdog */
        toggle_active_led();

        /* delay before we refresh the watchdog, check for SW1 button push which forces watchog to expire */
        for (delay = 0; delay < 10; delay++)
        {
            /* Check for SW1 button push.  Pin is grounded when button is pushed */
            if (is_key_pressed())
            {
              while (1) 
              {                
                /* button has been pushed, blink LED rapidly showing that the watchdog is about to expire */
                toggle_active_led();
                time_delay(FAST_BLINK_TIME);
              } 
            }

            time_delay(SLOW_BLINK_TIME);
        }
        /* Refresh the watchdog so we don't reset */
        wdog_refresh();        
        
        /* Print total loops */
        printf("\n\rWatchdog example running, Loop #: %d, press <SW> to start watchdog timeout...", 
               watchdog_demo_loop_count++);
    } 
}

/*************************************************************
 * EOF
 *************************************************************/
