#define __STDIO_FDEVOPEN_COMPAT_12
#include <stdio.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include "uart.h"
#include "config.h"
#include "fifo.h"

// FIFO-Objekte und Puffer fr die Ein- und Ausgabe 
#define BUFSIZE_IN  0x40
uint8_t inbuf[BUFSIZE_IN];
fifo_t infifo;

#define BUFSIZE_OUT 0x40
uint8_t outbuf[BUFSIZE_OUT];
fifo_t outfifo;

// Empfangene Zeichen werden in die Eingabgs-FIFO gespeichert und warten dort 
SIGNAL (SIG_UART_RECV)
{
    _inline_fifo_put (&infifo, UDR);
}

// Ein Zeichen aus der Ausgabe-FIFO lesen und ausgeben 
// Ist das Zeichen fertig ausgegeben, wird ein neuer SIG_UART_DATA-IRQ getriggert 
// Ist die FIFO leer, deaktiviert die ISR ihren eigenen IRQ. 
SIGNAL (SIG_UART_DATA)
{
    if (outfifo.count > 0)
       UDR = _inline_fifo_get (&outfifo);
    else
        UCSRB &= ~(1 << UDRIE);
}

void uart_init(void)
{
    uint8_t sreg = SREG;
    uint16_t ubrr = (uint16_t) ((uint32_t) F_CPU/(16*BAUDRATE) - 1);

    UBRRH = (uint8_t) (ubrr>>8);
    UBRRL = (uint8_t) (ubrr);

    // Interrupts kurz deaktivieren 
    cli();

    // UART Receiver und Transmitter anschalten, Receive-Interrupt aktivieren 
    // Data mode 8N1, asynchron 
    UCSRB = (1 << RXEN) | (1 << TXEN) | (1 << RXCIE);
    UCSRC = (1 << URSEL) | (1 << UCSZ1) | (1 << UCSZ0);

    // Flush Receive-Buffer (entfernen evtl. vorhandener ungltiger Werte) 
    do
    {
        // UDR auslesen (Wert wird nicht verwendet) 
        UDR;
    }
    while (UCSRA & (1 << RXC));

    // Rcksetzen von Receive und Transmit Complete-Flags 
    UCSRA = (1 << RXC) | (1 << TXC);

    // Global Interrupt-Flag wieder herstellen 
    SREG = sreg;

    // FIFOs fr Ein- und Ausgabe initialisieren 
    fifo_init (&infifo,   inbuf, BUFSIZE_IN);
    fifo_init (&outfifo, outbuf, BUFSIZE_OUT);
    
	//ffnet einen Kanal fr printf (STDOUT)
	fdevopen (uart_direct_putc, 0, 0);
}

int uart_putc (const uint8_t c)
{
    int ret = fifo_put (&outfifo, c);
	
    UCSRB |= (1 << UDRIE);
	 
    return ret;
}

int uart_getc_nowait (void)
{
    return fifo_get_nowait (&infifo);
}

uint8_t uart_getc_wait (void)
{
    return fifo_get_wait (&infifo);
}

// Einen 0-terminierten String bertragen. 
void uart_puts (const char *s)
{
    do
    {
        uart_putc (*s);
    }
    while (*s++);
}

// Einen 0-terminierten String bertragen. 
void uart_direct_puts(const char *s)
{
    do
    {
    	uart_direct_putc (*s);
    }
    while (*s++);
}

void uart_fifo_flush (void)
{
	fifo_flush (&infifo);
}
