
#ifndef signal_t_h
#define signal_t_h

#include "nios.h"

#define FIR_LENGTH 	15
#define SINC_LENGTH	25

/**
	Methods for digital signal processing like filtering, 
	sprectral analysis, interpolation and any other form
	of signal manipulation.
*/

class Signal
{
	protected:
	
	static const short _FIR_COEFFS[3][FIR_LENGTH];
	static const short _SINC_COEFFS[4][SINC_LENGTH];

	static int _FirLookup[FIR_LENGTH][256];
	static int _SincLookup[SINC_LENGTH][256];

	static unsigned char _SincLength;				// filter length, number of coefficients, order + 1, depends on cutoff/samplerate
	static unsigned char _SincIntWidth;				// integer size in bit

	/* trigonometric tables for FFT */
	static int _SinTab[4096];
	static int _CosTab[4096];
	static int _RevTab[4096];
	static int _WinTab[4096];

	static int _FFT_Length_old;
	static int _FFT_Windex_old;

	
	public:	
		
	static unsigned char _FFT_Mode; // FFT output mode
	static int _FFT_Length;		// 512/1024/2048/4096
	static int _FFT_Level;		// FFT_Level = ld(FFT_Length) 9/10/11/12
	static int _FFT_Windex;		// FFT window index default is Rect, -1 - 8
		
	
 
	static void processSignal(void); 								// BF new level 1 routine for signal processing
													// and math functions

	//----------------------------------------------------------------------------------------------------------------------------------------------------------
	static void USTB_clearBuffer(void);
	static void USTB_resetShiftBuffer(void);
	static void USTB_insertSample(short invert, unsigned char ADC_VirtualZero, unsigned char *USTB_buff ,unsigned char *Signal);
	static void USTB_invertSignal(unsigned char ADC_VirtualZero ,unsigned char *Signal);
	static void USTB_recalcMath(void);
	//----------------------------------------------------------------------------------------------------------------------------------------------------------

    	static void interpolate(int timebase, int start, unsigned char *SignalData, unsigned char *InterpolateResult);	//BF new interpolation routine
	//static void interpolateLinear(int x1, int y1, int x2, int y2, unsigned char *interpolation);	

	// Math calculation
	static void calcMathMul(int start, int stop, int stepwidth ,unsigned char *SignalData1, unsigned char *SignalData2, unsigned char *MathData);
	static void calcMathSub(int start, int stop, int stepwidth ,unsigned char *SignalData1, unsigned char *SignalData2, unsigned char *MathData);
	static void calcMathAdd(int start, int stop, int stepwidth ,unsigned char *SignalData1, unsigned char *SignalData2, unsigned char *MathData);

	static void generateTestSignal(void);


	static void FFT_buildTrigoTables(void);
	static void FFT_generateSin(unsigned int points, int *SinTab);
	static void FFT_generateCos(unsigned int points, int *CosTab);
	static void FFT_generateWindows(unsigned int points, int WinType, int *WinTab);
	static void FFT_calcSpectrum(unsigned char *Signal, int *Spectrum);

	//----------------------------------------------------------------------------------------------------------------------------------------------------------
	//static void LowPassFilter(unsigned char *RawData, int start, int stop, unsigned char *DisplayData);

	static void calcMinMax(int32_t start, int32_t stop, int32_t stepwidth ,uint8_t *SignalData1, uint32_t *min, uint32_t *max);

	static void BuildScaleLookupTable(void);
	static void FFT_buildScaleLookupTable(void);
	static void LowPassFilter(uint32_t start, uint32_t stop, uint32_t strength, uint8_t *input, uint8_t *output);
	static void IIR_Filter(int start, int stop, int stage, unsigned char *input, unsigned char *output);

	static void LOGA_LogicProcessor(void);

	static void FIR_NoiseFilter(int start, int stop, unsigned char *input, unsigned char *output);
	static void FIR_SincFilter(int start, int stop, unsigned char *input, unsigned char *output);
	static void FIR_buildNoiseFilterLookupTable(void);
	static void FIR_buildSincLookupTable(void);

	static void CalcFilterCutoff(char *string);
	
	static void PeakDetect(void);
	
	
};


// interpolate as inline function
inline void interpolateLinear(int x1, int y1, int x2, int y2, unsigned char *interpolation)
{
	
	//--------------------------------------------
	//special case horizontal line
	//--------------------------------------------
	if(y1 == y2)
	{
		if(x1 < x2)
		for(; x1 != x2; x1++)	{ interpolation[x1] = (char)y1; }
		else
		for(; x2 != x1; x2++)	{ interpolation[x2] = (char)y1; }
		return;
	}	

	//--------------------------------------------
	//special case vertical line
	//--------------------------------------------
	else if(x1 == x2)
	{
		if(y1 < y2)
		for(; y1 != y2; y1++)	{ interpolation[x1] = (char)y1; }
		else
		for(; y2 != y1; y2++)	{ interpolation[x1] = (char)y2; }
		return;
	}	

	//-------------------------------------------------------------
	// Bresenham algorithm for drawing diagonal lines
	//-------------------------------------------------------------
	int dy = y2 - y1;
        int dx = x2 - x1;
        int stepx, stepy;

	//check if delta is positive or negative
	if (dy < 0) { dy = -dy;  stepy = -1; } else { stepy = 1; }
        if (dx < 0) { dx = -dx;  stepx = -1; } else { stepx = 1; }

	//special case exact 45 degrees delta x / delta y = 1
	if (dx == dy)
	{
		//horizontal and vertical component increment constant with 1 
		if(x1 < x2) 
		{
			if(y1 < y2)
			{ 
				for(; x1 != x2; x1++, y1++)
				{ interpolation[x1] = (char)y1; }
				return;
			}
			else if(y1 > y2)
			{
				for(; x1 != x2; x1++, y1--)
				{ interpolation[x1] = (char)y1; }
				return;
			}
			
		}
		else if(x1 > x2)
		{
			if(y1 < y2) 
			{
				for(; x1 != x2; x1--, y1++)
				{ interpolation[x1] = (char)y1; }
				return;
			}
			else if(y1 > y2)
			{
				for(; x1 != x2; x1--, y1--)
				{ interpolation[x1] = (char)y1; }
				return;
			}
		}
	}

	
        dy <<= 1;                              		// dy is now 2*dy
        dx <<= 1;                              		// dx is now 2*dx
	
	//set starting point
	interpolation[x1] = (char)y1;
	
	//---------------------------------------------
	//between horizontal and 45 degrees
	//---------------------------------------------
	if (dx > dy)
	{
		int fraction = dy - (dx >> 1);   	// same as 2*dy - dx
		while (x1 != x2) {
			if (fraction >= 0) {
			y1 += stepy;
			fraction -= dx;              	// same as fraction -= 2*dx
			}
			x1 += stepx;
			fraction += dy;          	// same as fraction -= 2*dy
			interpolation[x1] = (char)y1;
		}
        	return;
	}

	//---------------------------------------------
	//between vertical and 45 degrees
	//---------------------------------------------
	if (dx < dy)
	{
		int fraction = dx - (dy >> 1);
		while (y1 != y2) {
			if (fraction >= 0) {
			x1 += stepx;
			fraction -= dy;
			}
			y1 += stepy;
			fraction += dx;
			interpolation[x1] = (char)y1;
		}
		return;
        }

}


#endif //signal_t_h
