#include <avr/io.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include "rf01.h"

#define RF_PORT	PORTB
#define RF_DDR	DDRB
#define RF_PIN	PINB

#define SDI		4	// SDI	0
#define SCK		3	// SCK	2
#define CS		2	// nSEL	3
#define SDO		1	// SDO	1

// nFFS: 1-10k Pullup an Vcc

#ifndef cbi
#define cbi(sfr, bit)     (_SFR_BYTE(sfr) &= ~_BV(bit)) 
#endif
#ifndef sbi
#define sbi(sfr, bit)     (_SFR_BYTE(sfr) |= _BV(bit))  
#endif

static unsigned char sdrssi, sgain;

void rf01_trans(unsigned short wert)
{	unsigned char i;
	cbi(RF_PORT, SCK);
	cbi(RF_PORT, CS);
	for (i=0; i<16; i++)
	{	if (wert&32768)
			sbi(RF_PORT, SDI);
		else
			cbi(RF_PORT, SDI);
		sbi(RF_PORT, SCK);
		wert<<=1;
		_delay_us(0.2);
		cbi(RF_PORT, SCK);
	}
	sbi(RF_PORT, CS);
}

void rf01_init(void)
{	unsigned char i;
	RF_PORT |= (1<<CS);
	RF_DDR |= (1<<SDI)|(1<<SCK)|(1<<CS);
	RF_DDR &= ~(1<<SDO);

	for (i=0; i<11; i++)
		_delay_ms(10);			// wait until POR done
	
	rf01_trans(0x0000);			// Status Read
	rf01_trans(0x913A);			// 868Mhz, 134khz, 10p (913A) 
	rf01_trans(0xC6E5);			// AFC settings: autotuning: -10kHz...+7,5kHz (C6E7) E1
	rf01_trans(0xC42B);			// Data Filter: internal (C42B)
	rf01_trans(0xC2E0);			// AVR CLK: 10MHz
	rf01_trans(0xE000);			// disable wakeuptimer
	rf01_trans(0xCC00);			// disable low duty cycle
	rf01_trans(0xC0C0);			// Rx setting
	rf01_trans(0xC0C1);			// Rx on
}

void rf01_setreceiver(unsigned char gain, unsigned char drssi)
{
	sdrssi=drssi;
	sgain=gain;
}

void rf01_setfreq(unsigned short freq)
{	if (freq<96)				// 860MHz
		freq=96;
	else if (freq>3903)			// 888MHz
		freq=3903;
	rf01_trans(0xA000|freq);
}

void rf01_setbaud(unsigned short baud)
{
	if (baud<336)
		return;
	if (baud<5400)				// Baudrate= 344827,58621/(R+1)/(1+CS*7)
		rf01_trans(0xC880|((43104/baud)-1));
	else
		rf01_trans(0xC800|((344828UL/baud)-1));
}

void rf01_rxdata(unsigned char *data, unsigned char number)
{	unsigned char i,j,c;

	rf01_trans(0xC0C1|((sgain&3)<<4)|((sdrssi&7)<<1));	// RX on
	rf01_trans(0xCE85);			// set FIFO mode CE89 (CE8D) CE85
	rf01_trans(0xCE87);			// enable FIFO CE8B	(CE8F) CE87
	
	cbi(RF_PORT, SDI);
	cbi(RF_PORT, SCK);
	cbi(RF_PORT, CS);
	
	for (i=0; i<number; i++)
	{	
		cbi(RF_PORT, CS);
		while (!(RF_PIN&(1<<SDO))); // wait until data in FIFO
		for (j=0; j<16; j++)	// read and discard status register
		{	sbi(RF_PORT, SCK);
			asm("nop");
			cbi(RF_PORT, SCK);
		}
		c=0;
		for (j=0; j<8; j++)
		{	c<<=1;
			if (RF_PIN&(1<<SDO))
				c|=1;
			sbi(RF_PORT, SCK);
			_delay_us(0.2);
			cbi(RF_PORT, SCK);
		}
		*data++=c;
		sbi(RF_PORT, CS);
	}
	rf01_trans(0xC0C0|((sgain&3)<<4)|((sdrssi&7)<<1));	// RX off
}

