/*----------------------------------------------------------*/
/*                                                          */
/* S/W-Text VGA Terminal (VT52) mit einem ATMega1284P       */
/*                                                          */
/* Basierend auf dem dmm_vga Projekt von                    */
/* Jan Roesler (jan.roesler@tu-bs.de) und                   */
/* Sebastian Brueckner (se.brueckner@tu-bs.de)              */
/*                                                          */
/* Hans Georg Giese, df2au@gmx.de                           */
/*                                                          */
/* Copyright (C) 2015 unter ALAS - frei fr                 */
/* nichtkommerzielle Nutzung                                */
/*                                                          */
/* Version 0.1: noch offen: Scrollen, Tastatur nur          */
/*                          rudimentr dekodiert            */
/*                                                          */
/* Hinweise:                                                */
/* - Auf der originalen Leiterplatte muss die Verbindung    */
/* zwischen ATMega-Pin18 und Max232 getrennt werden und     */
/* eine Brcke von ATMega-Pin1 zu Pin18 gelegt werden.      */
/* - Der Code passt auch in einen ATMega644A. Da das RAM    */
/* dann aber zu 99% belegt ist, ist ein sicherer Betrieb    */
/* zweifelhaft (Stack Kollision)                            */
/*                                                          */
/*----------------------------------------------------------*/

#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <stdint.h>
#include "global.h"
#include "keyboard.h"
#include "framebuffer.h"
#include "vt52.h"
#include "uart.h"

static volatile uint8_t g_vsync; // flag indicating begin of vertical sync period


#define VST_SYNC   0
#define VST_BACK   1
#define VST_ACTIVE 2
#define VST_FRONT  3

//
// VSYNC ISR
//
// Activated by active video ISR at the end of every frame
//
ISR(TIMER2_COMPA_vect) {

    static uint8_t state = VST_SYNC, next_state = VST_SYNC;

    switch(state) {
      case VST_SYNC:
        VSYNC_PORT |= (1<<VSYNC_PIN);
        OCR2A = V_SYNC * H_TOTAL * 8 / 64;
        g_vsync = 1;
        next_state = VST_BACK;
        break;

      case VST_BACK:
        VSYNC_PORT &= ~(1<<VSYNC_PIN);
        OCR2A = ((V_BACK) * H_TOTAL * 8 / 64) - 1;
        next_state = VST_ACTIVE;
        break;

      case VST_ACTIVE:
        g_vsync = 0;
        // deactivate myself
        TCCR2B &= ~(1<<CS22);
        OCR2A = V_FRONT * H_TOTAL * 8 / 64 - 1; // prepare for next invocation of this ISR
        TCNT2 = 0;
        // activate active video ISR
        TIFR1 = (1<<OCF1B);
        TIMSK1 |= (1<<OCIE1B);
        next_state = VST_SYNC;
        break;

      default:
        state = next_state = VST_SYNC;
      }

    state = next_state;
    }


//--------------------------------------
// Active video ISR
//
// This function generates the whole video line
//
ISR(TIMER1_COMPB_vect) {
    static uint16_t line_no = 0;
    static uint8_t *frame_p = (uint8_t*)framebuf;

    uint8_t char_l = line_no % 8;

    if(line_no == V_ACTIVE) {
        frame_p = (uint8_t*)framebuf;
        line_no = 0;
        TIMSK1 &= ~(1<<OCIE1B);         // deactivate myself
        TCCR2B |= (1<<CS22);            // activate VSYNC interrupt, prescaler=64
        return;
        }

    // For discussion of the inline assembly segment:
    // http://www.avrfreaks.net/index.php?name=PNphpBB2&file=viewtopic&p=595112#595112

    uint8_t *temp_font;  // dummy variables - only needed to tell compiler that input variables will be changed
    uint8_t *temp_frame; // they should be discarded during optimization

    asm volatile(
        "add r31, %[char_l]"           "\n\t"
        "jmp 1f"                       "\n\t"
        ".text 2"                      "\n\t"
        "1:"                           "\n\t"
        ".rept 80"                     "\n\t"
        "ld r30, Y+"                   "\n\t"
        "lpm __tmp_reg__, Z"           "\n\t"
        "out %[color], __tmp_reg__"    "\n\t"
        "out %[shift], __zero_reg__"   "\n\t"
        "out %[shift], %[set_pl]"      "\n\t"
        ".endr"                        "\n\t"
        "jmp 2f"                       "\n\t"
        ".text"                        "\n\t"
        "2:"                           "\n\t"
        :            "=y" (temp_frame),               // <-- output operands
                     "=z" (temp_font)
        : [shift]    "I" (_SFR_IO_ADDR(SHIFT_PORT)),
          [color]    "I" (_SFR_IO_ADDR(COLOR_PORT)),  // <-- input operands
          [set_pl]   "r" (1<<SHIFT_PL),
          [char_l]   "r" (char_l),
                     "y" (frame_p),
                     "z" (font)
    );

    if(char_l == 0x07)
        frame_p += H_CHARS;

    line_no++;

    // Check for characters from RS-232
    if(UCSR0A & (1<<RXC0)) {
        uart0_buf[uart0_index++] = UDR0;
        }

    // Check for Characters from Keyboard
    if(UCSR1A & (1<<RXC1)) {
        kb_buffer[kb_bufpos++] = UDR1;
        }
    }


//--------------------------------------
uint8_t counter;

void inline vsync_blinkcursor() {

    counter++;
    if(counter >= 12) {
        counter = 0;
        cursor_blink();
        }
    }


//--------------------------------------
void inline vsync() {

    uint8_t i;

    vsync_blinkcursor();

    for (i = 0; i < uart0_index; i++) { // Display characters received from RS-232
        vt_handle_input(uart0_buf[i]);
        }
    uart0_index = 0;

    for (i = 0 ; i < kb_bufpos ; i++) { // decode and display keyboard input
        decode(kb_buffer[i]);
        }
    kb_bufpos = 0;

    if (buffcnt && uart_putsts()) {     // Tastatur Eingaben auf RS232 geben
        buffcnt--;
        uart_putchar(*outpt);
        outpt++;
        if (outpt >= (kb_buffer + KB_BUFF_SIZE)) {
            outpt = kb_buffer;
            }
        }
    }


//--------------------------------------
int main(void) {

    uint16_t i;

    fb_clear();

/*
for (i = 0; i < 80; i++) {
    fb_putchar_xy(i, 0, 'T');
    fb_putchar_xy(i, 49, 'B');
    }
for (i = 0; i < 49; i++) {
    fb_putchar_xy(0, i, 'L');
    fb_putchar_xy(79, i, 'R');
    }
*/
/*
    for (i = 0; i < 256; i++) {
        vt_putchar(i);
        }
*/
// initialize ports

    COLOR_DDR = 0xFF;
    COLOR_PORT = 0x00;

    SHIFT_DDR |= (1<<SHIFT_CE)|(1<<SHIFT_PL);
    SHIFT_PORT = (1<<SHIFT_PL);

    HSYNC_DDR |= (1<<HSYNC_PIN);
    VSYNC_DDR |= (1<<VSYNC_PIN);


// Initialize Timer 1: HSYNC, active video
// Timer 1 will call the active video interrupt function via OCR1B (isr.s)
// and output the HSYNC pulse via OC1A

    TCCR1B = 0;                     // stop timer
    TCCR1B = (1<<WGM13)|(1<<WGM12); // Mode 14: Fast PWM counting up to ICR1
    TCCR1A = (1<<WGM11)|(1<<COM1A1)|(1<<COM1A0);// INVERTED PWM output
    TIMSK1 = (1<<OCIE1B);           // execute ISR on OC1B match
    ICR1   = H_TOTAL-1;             // timer period is one full scanline minus 1 for interrupt latency
    OCR1A  = H_SYNC-1;
    // ISR start to output of first pixel: 51 cycles
    // wake up from idle and enter ISR: ~10 cycles     => OCR1B - 8
    // HSYNC to ACTIVE should be 3.19 us
    OCR1B  = H_SYNC + H_BACK - 9 - 1;   // substract some cycles for initialisation in ISR
    TCNT1 = 0;

    // Initialize Timer 2: VSYNC
    // Timer 2 will be enabled from active video ISR after end of active frame

    TCCR2B = 0;
    TCCR2A = (1<<WGM21); // Mode 2: CTC
    TIMSK2 = (1<<OCIE2A);
    TCNT2 = 0;
    OCR2A = V_FRONT;

    // Initialize UART0 (RS-232)
    // 9600 baud, 8n1
    //
    // 25 MHz:
    // --------------
    // UBRR0 = F_CPU / 16 / BAUD - 1 = 161.76 = 162
    // BAUD = F_CPU / 16 / (UBRR0 + 1) = 9585.89
    // error = BAUD / 9600 * 100% = 0.14%
    //
    // maximum transmission speed:
    // UART0_BUFLEN * video refresh rate
    // = 16 byte * 70 Hz = 1120 byte/sec = 8960 baud
    //
    UCSR0A = 0;
    UCSR0B = (1<<RXEN0)|(1<<TXEN0);
    UCSR0C = (1<<UCSZ01)|(1<<UCSZ00);
    UBRR0 = 162; // 25 MHz
    // UBRR0 = 129; // 20 MHz

    // Initialize UART1 (Keyboard)
    kbInit();

    // Start Timer 1
    sei();
    TCCR1B |= (1<<CS11); // prescaler = 8

    //
    // SLEEP mode is necessary to eliminate interrupt jitter:
    // due to processor instructions taking 2 or 3 clock cycles
    // ISRs can't always execute "on time".
    // When waking up from SLEEP modes always takes a CONSTANT
    // number of cycles --> problem solved
    //
    set_sleep_mode(SLEEP_MODE_IDLE);

    while(1) {
        sleep_enable();
        sei();
        sleep_cpu();
        sleep_disable();

        if(g_vsync) {
            vsync();
            g_vsync = 0;
            }
        }

    return 0;
    }
