#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include <util/delay.h>
#include <stdio.h>
#include <stdbool.h>

/* my little helpers */
#include "midi.h" //midi uart
#include "switch.h"
/* INPUTS */

/* FRONT SWITCHES just for demo*/
#define LEAD PD2
#define RYTHM PD3
#define CLEAN PD4
#define BOOST PB1



/* OUTPUTS */
#define LEAD_ON PORTC |= (1<<PC2);
#define LEAD_OFF PORTC &= ~(1<<PC2);

#define RYTHM_ON PORTC |= (1<<PC3);
#define RYTHM_OFF PORTC &= ~(1<<PC3);

#define CLEAN_ON PORTC |= (1<<PC4);
#define CLEAN_OFF PORTC &= ~(1<<PC4);

#define BOOST_ON PORTC |= (1<<PC1);
#define BOOST_OFF PORTC &= ~(1<<PC1);

/* position in program data */
#define LEAD_BIT 0
#define RYTHM_BIT 1
#define CLEAN_BIT 2
#define BOOST_BIT 3

/* globals */
#define PAUSE 200
#define ADR_GLOBAL 128 //adress where globals are saved
#define ADR_DETECT 129 //test byte, if eeprom is init


#define OMNI_ON 0 //listen on all midi channels if 1, else only channel 0
#define SW_MODE_ON 0 //use switches instead of pushbuttons 1=switches

static unsigned char globals = 0x00; //global data byte, unused in thi example

static unsigned char  currentProgram = 128; //0 - 127 
static unsigned char currentData = 0x00; // program data for 


/* position in program data */
#define LEAD_BIT 0
#define RYTHM_BIT 1
#define CLEAN_BIT 2
#define BOOST_BIT 3


void lead(bool on)
{
 	if (on) {
		LEAD_ON
		currentData |= (1<<LEAD_BIT);
	}
	else {
		LEAD_OFF
		currentData &= ~(1<<LEAD_BIT);
	}
}
void rythm(bool on)
{
 	if (on) {
		RYTHM_ON
		currentData |= (1<<RYTHM_BIT);
	}
	else {
		RYTHM_OFF
		currentData &= ~(1<<RYTHM_BIT);
	}
}
void clean(bool on)
{
 	if (on) {
		CLEAN_ON
		currentData |= (1<<CLEAN_BIT);
	}
	else {
		CLEAN_OFF
		currentData &= ~(1<<CLEAN_BIT);
	}
}
void boost(bool on)
{
 	if (on) {
		BOOST_ON
		currentData |= (1<<BOOST_BIT);
	}
	else {
		BOOST_OFF
		currentData &= ~(1<<BOOST_BIT);
	}
}
void loadProgamData()
{
	for (uint8_t i = 0;i < 4; i++) {
		uint8_t value = (currentData & (1 << i));
		if (i == BOOST_BIT) {
			boost(value);
			setState(PIND,BOOST,value);//inits the switch return
		}
		else if (i == LEAD_BIT) {
			lead(value);
			setState(PIND,LEAD,value);
		}
		else if (i == RYTHM_BIT) {
			rythm(value);
			setState(PIND,RYTHM,value);
		}
		else if (i == CLEAN_BIT) {
			clean(value);
			setState(PIND,CLEAN,value);
		}
	}
}
void initOutputs()
{
	DDRC |= (1<<PC5); //bypass
	DDRC |= (1<<PC4); // clean
	DDRC |= (1<<PC3); // rythm
	
	DDRC |= (1<<PC2); // lead
	DDRC |= (1<<PC3); // rythm
	DDRC |= (1<<PC4); // clean
	DDRC |= (1<<PC1); //boost
}
void initInputs()
{
	/* LEAD */
	DDRD &= ~(1<<PD2);
	PORTD |= (1<<PD2);
	/* RYTHM */
	DDRD &= ~(1<<PD3);
	PORTD |= (1<<PD3); 	

	/* CLEAN */
	DDRD &= ~(1<<PD4);
	PORTD |= (1<<PD4); 
	/* BOOST */
	DDRB &= ~(1<<PB1);
	PORTB |= (1<<PB1); 

}

void checkEeprom()
{
	uint8_t byte = eeprom_read_byte((uint8_t*)ADR_DETECT);
	if (byte != 0x33) {//eeprom not init
		byte = 0x02; //clean, all others off
		for (int i = 0;i < ADR_GLOBAL;i++) {
			eeprom_write_byte((uint8_t*)i,byte);
		} 
		byte = 0x00;
		eeprom_write_byte((uint8_t*)ADR_GLOBAL,byte);
		byte = 0x33;
		eeprom_write_byte((uint8_t*)ADR_DETECT,byte);
	}
}

void programChanged(int newProgram)
{
	uint8_t prgData = eeprom_read_byte((uint8_t*)newProgram);
	if (currentProgram == newProgram) {
		if (currentData != prgData) {
			eeprom_write_byte((uint8_t*)newProgram,currentData);
			//show write effect clean blinks
			for (int i = 1; i < 5;i++) {			
				clean(true);
				_delay_ms(PAUSE - 100);	
				clean(false);
			}
		}
		else return;//program already loaded
	}
	else currentData = prgData;
	currentProgram = newProgram;
	loadProgamData();

}
void toggleOmni()
{
	globals ^=  (1 << 0);
	eeprom_write_byte((uint8_t*)ADR_GLOBAL,globals);
}
void toggleSW_Mode()
{
	globals ^=  (1 << 1);
	eeprom_write_byte((uint8_t*)ADR_GLOBAL,globals);
}
int main(void)
{
	checkEeprom();//int eeprom, if necessary

	//init
	initOutputs();
	initInputs();
	initSwitch();
	initMidiUart(1);	//1 = receive only, 0 also send
	sei();//enable IRQ

	//set fuse bit eesave for eeprom write

	//no global data in this example, but can be useful, if OMNI is switchable by botton....
	globals = eeprom_read_byte((uint8_t*)ADR_GLOBAL);

	programChanged(0);//init to program 0


	while (1) {
		if (bytesAvailable() ) {
			unsigned char byte = readByte();
			unsigned char mask = byte;
			if (OMNI_ON) mask = (byte & 0xF0);
			if ( mask == MIDI_PROG_CHANGE ) {//OMni mode
				
				while (!bytesAvailable() )  {;}//wait for program number, todo -> add a timeout
				byte = readByte();
				programChanged(  byte);
			}
			// * else if  ( mask == MIDI_NOTE_ON) playNote() .....most midi commands send 3 bytes!;
			// * else more Midi
			else {
				readByte();//read and ignore, empties the buffer
			}

		}		
		else {//no midi data, do the usually stuff
			
			/* Just front switches/buttons */
			if (SW_MODE_ON) {//switches
				lead(switchBtn_d(LEAD) );
				rythm(switchBtn_d(RYTHM) );
				clean(switchBtn_d(CLEAN) );
				boost(switchBtn_d(BOOST) );
			}
			else {//pushbutton
				lead(toggleBtn_d(LEAD) );
				rythm(toggleBtn_d(RYTHM) );
				clean(toggleBtn_d(CLEAN) );
				boost(toggleBtn_b(BOOST) );
			}

		}
	}
}
