/////////////////////////////////////////////////////////////////////////////////////////
//
// FAT
//
// --------------------------------------------------------------------------------------
//
// Filename:      fat.c
// Version:       1.0
// Date:          15/10/2011
// Author:        Joel Guittet - http://myfreescalewebpage.free.fr
//
/////////////////////////////////////////////////////////////////////////////////////////
//
// Description
//
// This file implements the functions used to access FAT file system on the SD Card.
// This file is based on AN4115 from Freescale, on which some improvements have been made.
//
/////////////////////////////////////////////////////////////////////////////////////////
//
// Revisions
//
// Version	| Author		| Description
// --------------------------------------------------------------------------------------
//			|				| 
//
/////////////////////////////////////////////////////////////////////////////////////////


//---------------------------------------------------------------------------------------
// Includes
//---------------------------------------------------------------------------------------

#include "fat.h"


//---------------------------------------------------------------------------------------
// Global Variables
//---------------------------------------------------------------------------------------

/* File Handlers */
static t_ReadFileHandler g_FAT_ReadFileHandler;
static t_WriteFileHandler g_FAT_WriteFileHandler;

/* File Buffers */
static uint8_t g_FAT_ReadBuffer[512];
static uint8_t g_FAT_WriteBuffer[512];

/* Other variables */
static uint16_t g_FAT_SectorSize;
static uint16_t g_FAT_ClusterSize;
static uint16_t g_FAT_FatBase;
static uint16_t g_FAT_RootBase;
static uint16_t g_FAT_DataBase;
static uint16_t g_FAT_MainOffset;
static uint32_t g_FAT_FileSize;


//---------------------------------------------------------------------------------------
// Name:        FAT_ReadBootSector
// Param:		-
// Return:      -
//
// Description:	This function read the Boot Sector of the SD Card. It must be called
//				before using the SD Card.
//---------------------------------------------------------------------------------------
void FAT_ReadBootSector(void)
{
	t_PartitionBootSector * pxBootSector;

	g_FAT_MainOffset = 0;
	g_FAT_FileSize = 0;
    
    /* Search the Boot Sector (begin with 0xEB, 0x3C, 0x90) */
    while (g_FAT_ReadBuffer[0] != 0xEB && g_FAT_ReadBuffer[1] != 0x3C && g_FAT_ReadBuffer[2] != 0x90) 
    {
        (void)SD_ReadBlock(g_FAT_MainOffset++, g_FAT_ReadBuffer);
    }
    
    g_FAT_MainOffset--;

    /* Get data from the Boot Sector */
    pxBootSector = (t_PartitionBootSector*)g_FAT_ReadBuffer;
    g_FAT_ClusterSize = pxBootSector->SectorsPerCluster;
    g_FAT_SectorSize = SWAP16(pxBootSector->BytesPerSector);
    g_FAT_FatBase = g_FAT_MainOffset + SWAP16(pxBootSector->ReservedSectors);
    g_FAT_RootBase = (SWAP16(pxBootSector->SectorsPerFat) << 1) + g_FAT_FatBase;
    g_FAT_DataBase = (SWAP16(pxBootSector->RootDirectoryEntries) >> 4) + g_FAT_RootBase;
}


//---------------------------------------------------------------------------------------
// Name:        FAT_FileOpen
// Param:		Filename: name of the file to open, 8.3 format
//				Function: READ, MODIFY or CREATE
// Return:      File error code: FILE_FOUND, FILE_NOT_FOUND, FILE_CREATE_OK,
//				NO_FILE_ENTRY_AVAILABLE or NO_FAT_ENTRY_AVAIlABLE
//
// Description:	This function opens a file. This function must be called before reading,
//				modifying or creating a file on the SD Card
//---------------------------------------------------------------------------------------
uint8_t FAT_FileOpen(uint8_t * pu8FileName, uint8_t u8Function)
{
    uint16_t u16Temporal;
    uint8_t u8FileName[11];
    uint8_t u8Counter = 0;
    uint8_t bFlag = 0;
    uint16_t u16Index;
    uint16_t u16Block;
    uint16_t u16BlockNum = g_FAT_DataBase - g_FAT_RootBase;
    uint8_t u8ErrorCode = ERROR_IDLE;
    uint8_t * pu8Pointer;
    t_RootEntries * xFileStructure;
    
    /* Format the filename */
    FAT_FileNameOrganizer(pu8FileName, &u8FileName[0]);
    
    u16Block = 0;
    
    /* Parse the SD Card content */
    while (u16Block < u16BlockNum && u8ErrorCode == ERROR_IDLE)
    {
    	/* Get block */
    	(void)SD_ReadBlock(g_FAT_RootBase + u16Block, g_FAT_ReadBuffer);
        xFileStructure = (t_RootEntries*)g_FAT_ReadBuffer;

        u16Index = 0;
        
        /* Parse the block */
        while (u16Index < g_FAT_SectorSize && u8ErrorCode == ERROR_IDLE)    
        {
            /* Read or modify an existing file */
            if (u8Function == READ || u8Function == MODIFY)
            {
            	/* Look for the file */ 
				if (xFileStructure->FileName[0] == FILE_CLEAR)
				{
                    u8ErrorCode = FILE_NOT_FOUND;
				}
        
				/* First character is the same than the filename given in parameter, maybe it's the wanted file ? */
                if (xFileStructure->FileName[0] == u8FileName[0])
                {
					bFlag = 1;
                    u8Counter = 0;
                    while (bFlag == 1 && u8Counter < 10)
                    {
                        u8Counter++;
                        if (xFileStructure->FileName[u8Counter] != u8FileName[u8Counter])
                        {
                            bFlag = 0;
                        }
                    }
                    
                    /* Check if file found */
                    if (bFlag == 1)
                    {
                        /* Reading an existing file */
                        if (u8Function == READ)
                        {
                        	/* Set Read Handler content */ 
                        	g_FAT_ReadFileHandler.Dir_Entry = (u16Block * sizeof(t_RootEntries)) + (u16Index / sizeof(t_RootEntries));
                        	g_FAT_ReadFileHandler.File_Size = SWAP32(xFileStructure->SizeofFile);
                        	g_FAT_ReadFileHandler.FAT_Entry = SWAP16(xFileStructure->ClusterNumber);
                        	g_FAT_ReadFileHandler.SectorOffset = 0;
                        	g_FAT_FileSize = g_FAT_ReadFileHandler.File_Size;
                            u8ErrorCode = FILE_FOUND;
                        } 
                        /* Modifying an existing file */
                        else
                        {
                        	/* Set Write Handler content */
                            pu8Pointer = g_FAT_WriteFileHandler.FileName;
                            for (u8Counter = 0; u8Counter < 11; u8Counter++)
                            {
								*pu8Pointer++ = u8FileName[u8Counter];
                            }
                            g_FAT_WriteFileHandler.Dir_Entry = (u16Block * sizeof(t_RootEntries)) + (u16Index / sizeof(t_RootEntries));
                            g_FAT_WriteFileHandler.File_Size = SWAP32(xFileStructure->SizeofFile);
                            g_FAT_WriteFileHandler.BaseFatEntry = SWAP16(xFileStructure->ClusterNumber);
                            
                            if (g_FAT_WriteFileHandler.BaseFatEntry != 0)
                            {
                                u16Temporal = g_FAT_WriteFileHandler.BaseFatEntry;
                                do
                                {
                                	g_FAT_WriteFileHandler.CurrentFatEntry = g_FAT_WriteFileHandler.BaseFatEntry;
                                	g_FAT_WriteFileHandler.BaseFatEntry = FAT_Entry(g_FAT_WriteFileHandler.CurrentFatEntry, 0, NEXT_ENTRY);
                                }
                                while (g_FAT_WriteFileHandler.BaseFatEntry != 0xFFFF);
                                
                                g_FAT_WriteFileHandler.BaseFatEntry = u16Temporal;
                            } 
                            else
                            {
                            	g_FAT_WriteFileHandler.BaseFatEntry = FAT_SearchAvailableFAT(0);
                            	g_FAT_WriteFileHandler.CurrentFatEntry = g_FAT_WriteFileHandler.BaseFatEntry;
                            }
                            
                            u16Temporal = (uint16_t)g_FAT_WriteFileHandler.File_Size % (g_FAT_SectorSize << 4);
                            g_FAT_WriteFileHandler.ClusterIndex = u16Temporal / g_FAT_SectorSize;
                            g_FAT_WriteFileHandler.SectorIndex = u16Temporal % g_FAT_SectorSize;
                            g_FAT_FileSize = g_FAT_WriteFileHandler.File_Size;
                            u8ErrorCode = FILE_FOUND;
                        }
                    }
                }
            }

            /* Creating a new file */
            if (u8Function == CREATE)
            {
            	/* We need an empty root entry */
                if (xFileStructure->FileName[0] == FILE_CLEAR || xFileStructure->FileName[0] == FILE_ERASED) 
                {
                	/* Empty root entry found, set Write Handler content */
                    pu8Pointer = g_FAT_WriteFileHandler.FileName;
                    for (u8Counter = 0; u8Counter < 11; u8Counter++)
                    {
                        *pu8Pointer++ = u8FileName[u8Counter];
                    }

                    g_FAT_WriteFileHandler.Dir_Entry = (u16Block * sizeof(t_RootEntries)) + (u16Index / sizeof(t_RootEntries));
                    g_FAT_WriteFileHandler.File_Size = 0;
                    g_FAT_WriteFileHandler.BaseFatEntry = FAT_SearchAvailableFAT(0);
                    g_FAT_WriteFileHandler.CurrentFatEntry = g_FAT_WriteFileHandler.BaseFatEntry;
                    g_FAT_WriteFileHandler.ClusterIndex = 0;
                    g_FAT_WriteFileHandler.SectorIndex = 0;
        
                    g_FAT_FileSize = 0;
                    
                    if (g_FAT_WriteFileHandler.BaseFatEntry)
                        u8ErrorCode = FILE_CREATE_OK;
                    else
                        u8ErrorCode = NO_FAT_ENTRY_AVAIlABLE;
                }
            }
            
            /* Next root entry */
            xFileStructure++;
            u16Index += sizeof(t_RootEntries);
        }
        
        /* Next block */
        u16Block++;
    }
    
    if (u16BlockNum == u16Block)
    {
    	/* All block checked, no file entry found */
        u8ErrorCode = NO_FILE_ENTRY_AVAILABLE;
    }
    
    return u8ErrorCode;
}


//---------------------------------------------------------------------------------------
// Name:        FAT_FileRead
// Param:		UserBuffer: 512 bytes buffer to get file data from the SD Card
// Return:      Number of bytes read and returned in the user buffer
//
// Description:	This function is used to read a file on the SD Card. The file must be
//				previously successful opened with FAT_FileOpen function and using READ
//				function.
//---------------------------------------------------------------------------------------
uint16_t FAT_FileRead(uint8_t * pu8UserBuffer)
{
	uint16_t u16BufferSize;
	uint32_t u32SectorToRead;
    
    if (g_FAT_ReadFileHandler.File_Size == 0)
	{
    	return 0;
	}
   
    /* Read SD Card */
    u32SectorToRead = g_FAT_DataBase + ((g_FAT_ReadFileHandler.FAT_Entry - 2) * g_FAT_ClusterSize) + g_FAT_ReadFileHandler.SectorOffset;
    (void)SD_ReadBlock(u32SectorToRead, pu8UserBuffer);
    
    /* Update Read Handler */
    /* Update the remaining size not read */
    if (g_FAT_ReadFileHandler.File_Size > g_FAT_SectorSize)
    {
    	g_FAT_ReadFileHandler.File_Size -= g_FAT_SectorSize;
        u16BufferSize = 512;
    }
    else
    {
        u16BufferSize = (uint16_t)g_FAT_ReadFileHandler.File_Size;
        g_FAT_ReadFileHandler.File_Size = 0;
    }
    
    /* Update the sector offset */
    if (g_FAT_ReadFileHandler.SectorOffset < (g_FAT_ClusterSize - 1))
    {
    	g_FAT_ReadFileHandler.SectorOffset++;
    }
    else
    {
    	g_FAT_ReadFileHandler.SectorOffset = 0;
    	
    	/* Get next FAT entry */
    	g_FAT_ReadFileHandler.FAT_Entry = FAT_Entry(g_FAT_ReadFileHandler.FAT_Entry, 0, NEXT_ENTRY);
    }
    
    return u16BufferSize;    
}


//---------------------------------------------------------------------------------------
// Name:        FAT_FileWrite
// Param:		DataPointer: data to write in the file
//				Size: number of bytes to write  
// Return:      -
//
// Description:	This function is used to write in a file on the SD Card. The file must be
//				previously successful opened with FAT_FileOpen function and using CREATE
//				or MODIFY function.
//---------------------------------------------------------------------------------------
void FAT_FileWrite(uint8_t * pu8DataPointer, uint32_t u32Size)
{
    uint32_t u32SectorToWrite;
    uint8_t * pu8ArrayPointer;
    uint16_t u16TempFat;
    uint8_t u8ChangeSector = 1;

    while (u32Size)
    {
    	/* First read the SD Card and calculate where to write */
        if (u8ChangeSector)
        {
            u32SectorToWrite = g_FAT_DataBase + g_FAT_WriteFileHandler.ClusterIndex + (g_FAT_WriteFileHandler.CurrentFatEntry - 2) * g_FAT_ClusterSize;
            (void)SD_ReadBlock(u32SectorToWrite, g_FAT_WriteBuffer); 
            pu8ArrayPointer = g_FAT_WriteBuffer + g_FAT_WriteFileHandler.SectorIndex;
            u8ChangeSector = 0;
        }
        
        /* Copy data */
        while ((g_FAT_WriteFileHandler.SectorIndex < g_FAT_SectorSize) && (u32Size > 0))
        {
            u32Size--;
            g_FAT_WriteFileHandler.SectorIndex++;
            g_FAT_WriteFileHandler.File_Size++;
            *pu8ArrayPointer++ = *pu8DataPointer++;    
        }
        
        /* Write data in the SD Card */
        (void)SD_WriteBlock(u32SectorToWrite, g_FAT_WriteBuffer);
    
        /* Check Sector Size */
        if (g_FAT_WriteFileHandler.SectorIndex == g_FAT_SectorSize)
        {
        	g_FAT_WriteFileHandler.SectorIndex = 0;
        	g_FAT_WriteFileHandler.ClusterIndex++;    
            u8ChangeSector = 1;
        }
    
        /* Check Cluster Size */
        if (g_FAT_WriteFileHandler.ClusterIndex == g_FAT_ClusterSize)
        {
        	g_FAT_WriteFileHandler.ClusterIndex = 0;
            u16TempFat = FAT_SearchAvailableFAT(g_FAT_WriteFileHandler.CurrentFatEntry);   
            (void)FAT_Entry(g_FAT_WriteFileHandler.CurrentFatEntry, u16TempFat, WRITE_ENTRY);
            g_FAT_WriteFileHandler.CurrentFatEntry = u16TempFat;
            u8ChangeSector = 1;
        }
    }
}


//---------------------------------------------------------------------------------------
// Name:        FAT_FileClose
// Param:		-  
// Return:      -
//
// Description:	This function closes a file previously opened with CREATE or MODIFY
//				function. Do not call this function if you have opened the file using READ
//				function (because there is no need to close the file in this case).
//---------------------------------------------------------------------------------------
void FAT_FileClose(void)
{
    uint8_t u8Counter;
    uint32_t u32Sector;
    uint16_t u16Offset;
    uint16_t * pu16FatPointer;
    t_RootEntries * xFileStructure;
    
    /* Directory Entry*/
    u32Sector = g_FAT_WriteFileHandler.Dir_Entry / (g_FAT_SectorSize >> 5);
    u16Offset = g_FAT_WriteFileHandler.Dir_Entry % (g_FAT_SectorSize >> 5);
    
    /* Read the root entry of the file */
    (void)SD_ReadBlock(g_FAT_RootBase + u32Sector, g_FAT_ReadBuffer);
    xFileStructure = (t_RootEntries*)g_FAT_ReadBuffer;
    xFileStructure += u16Offset;

    /* Copy Filename */
    for (u8Counter = 0; u8Counter < 8; u8Counter++)
    {
    	xFileStructure->FileName[u8Counter] = g_FAT_WriteFileHandler.FileName[u8Counter];
    }

    /* Copy Extension */
    for (u8Counter = 0; u8Counter < 3; u8Counter++)
    {
        xFileStructure->Extension[u8Counter] = g_FAT_WriteFileHandler.Extension[u8Counter];
    }

    /* Copy Attributes */
    xFileStructure->Attributes = FILE_AT_ARCHIVE;
    
    /* Copy Case */
    xFileStructure->Case = 0x18;
    
    /* Date & Time Information */
    xFileStructure->MiliSeconds = 0xC6;
    xFileStructure->CreationTime = 0x2008;
    xFileStructure->CreationDate = 0x2136;
    xFileStructure->AccessDate = 0x2136;
    xFileStructure->ModificationTime = 0x2008;
    xFileStructure->ModificationDate = 0x2136;
    
    /* Fat entry and file Size */
    xFileStructure->ClusterNumber = SWAP16(g_FAT_WriteFileHandler.BaseFatEntry);
    xFileStructure->SizeofFile = SWAP32(g_FAT_WriteFileHandler.File_Size); 

    /* Write the updated root entry in the SD Card */
    (void)SD_WriteBlock(g_FAT_RootBase + u32Sector, g_FAT_ReadBuffer);
    
    /* Update FAT table */
    u32Sector = g_FAT_WriteFileHandler.CurrentFatEntry / (g_FAT_SectorSize >> 1);
    u16Offset = g_FAT_WriteFileHandler.CurrentFatEntry % (g_FAT_SectorSize >> 1);

    (void)SD_ReadBlock(g_FAT_FatBase + u32Sector, g_FAT_ReadBuffer);
    
    pu16FatPointer = (uint16_t*)g_FAT_ReadBuffer;
    pu16FatPointer += u16Offset;
    *pu16FatPointer = 0xFFFF;    

    (void)SD_WriteBlock(g_FAT_FatBase + u32Sector, g_FAT_ReadBuffer);
    
    g_FAT_FileSize = 0;
}


//---------------------------------------------------------------------------------------
// Name:        FAT_GetFileSize
// Param:		-  
// Return:      Size of the file in bytes
//
// Description:	This function returns the size of the currently opened file. It returns
//				0 if there is no file opened.
//---------------------------------------------------------------------------------------
uint32_t FAT_GetFileSize(void)
{
	return g_FAT_FileSize;
}


//---------------------------------------------------------------------------------------
// Name:        FAT_FileNameOrganizer
// Param:		FileName: filename from user application
//				Destiny: filename formated in order to be used with the SD Card
// Return:      -
//
// Description:	This function is used by FAT_FileOpen function and formats the filename
//				given by user application in order to be used with the SD Card.
//---------------------------------------------------------------------------------------
static void FAT_FileNameOrganizer(uint8_t * pu8FileName, uint8_t * pu8Destiny)
{
	uint8_t u8Counter = 0;
    
    while (u8Counter < 12)
    {
    	if (*pu8FileName == 0)
    	{
    		*pu8Destiny++ = 0x20;    	
    	}
    	else if (*pu8FileName == '.')
    	{
    		if (u8Counter < 8)
				*pu8Destiny++ = 0x20;
			else
				pu8FileName++;
    	}
    	else if ((*pu8FileName >= 'a') && (*pu8FileName <= 'z'))
    	{
    		*pu8Destiny++ = *pu8FileName++ - 0x20;
    	}
    	else
        {
            *pu8Destiny++ = *pu8FileName++;
        } 
        
        u8Counter++;
    }
}


//---------------------------------------------------------------------------------------
// Name:        FAT_Entry
// Param:		FatEntry: FAT entry to read or write
//				FatValue: value to write (if function is WRITE_ENTRY)
//				Function: NEXT_ENTRY or WRITE_ENTRY 
// Return:      Pointer to the next cluster if function is NEXT_ENTRY, 0 if WRITE_ENTRY
//
// Description:	Read or write the pointer to the next cluster for the current FAT entry 
//---------------------------------------------------------------------------------------
static uint16_t FAT_Entry(uint16_t u16FatEntry, uint16_t u16FatValue, uint8_t u8Function)
{
    uint8_t u8Offset;
    uint16_t u16Block;
    uint16_t * pu16DataPointer;
    
    u16Block = u16FatEntry / (g_FAT_SectorSize >> 1);
    u8Offset = (uint8_t)(u16FatEntry % (g_FAT_SectorSize >> 1));

    (void)SD_ReadBlock(g_FAT_FatBase + u16Block, g_FAT_ReadBuffer);
    pu16DataPointer = (uint16_t*)g_FAT_ReadBuffer;
    pu16DataPointer += u8Offset;

    if (u8Function == NEXT_ENTRY)
    {
        return (SWAP16(*pu16DataPointer));
    }
    
    if (u8Function == WRITE_ENTRY)
    {
        *pu16DataPointer = SWAP16(u16FatValue);
        (void)SD_WriteBlock(g_FAT_FatBase + u16Block, g_FAT_ReadBuffer);
    }
    
    return 0;
}


//---------------------------------------------------------------------------------------
// Name:        FAT_SearchAvailableFAT
// Param:		CurrentFAT: current FAT entry used
// Return:      FAT entry found 
//
// Description:	Search an available FAT entry, except the one given in parameter
//				Return 0 if there is no more entry available
//---------------------------------------------------------------------------------------
static uint16_t FAT_SearchAvailableFAT(uint16_t u16CurrentFAT)
{
    uint16_t u16ByteSector;
    uint16_t * pu16DataPointer;
    uint16_t u16FatEntry = 0;
    uint16_t u16Sector = g_FAT_FatBase;

    /* Parse the FAT */
    while (u16Sector < (((g_FAT_RootBase - g_FAT_FatBase) >> 1) + g_FAT_MainOffset))
    {
    	(void)SD_ReadBlock(u16Sector++, g_FAT_ReadBuffer);
        pu16DataPointer = (uint16_t*)g_FAT_ReadBuffer;
        u16ByteSector = 0;
        
        /* Parse the block */
        while (u16ByteSector < g_FAT_SectorSize)
        {
        	if ((*pu16DataPointer == 0x0000) && (u16FatEntry != u16CurrentFAT))
        	{
        		/* Available Fat Entry found */
        		return u16FatEntry;
        	}
        	
        	/* Next entry */
            pu16DataPointer++;
            u16FatEntry++;
            u16ByteSector++;
        }
    }
    
    /* Return 0 if no more FAT positions available */
    return 0;
}
