/////////////////////////////////////////////////////////////////////////////////////////
//
// Wiznet W5100 HTTP Server
//
// --------------------------------------------------------------------------------------
//
// Filename:      http.c
// Version:       1.0
// Date:          15/10/2011
// Author:        Joel Guittet - http://myfreescalewebpage.free.fr
//
/////////////////////////////////////////////////////////////////////////////////////////
//
// Description
//
// This file implements a Web Server which uses W5100 device for network communication
// and SD Card for files storage. It supports HTML, Javascript, and some images file 
// formats. It has been tested successfully with Firefox 9.0, and on Android smartphones.
//
/////////////////////////////////////////////////////////////////////////////////////////
//
// Revisions
//
// Version	| Author		| Description
// --------------------------------------------------------------------------------------
//			|				| 
//
/////////////////////////////////////////////////////////////////////////////////////////


//---------------------------------------------------------------------------------------
// Includes
//---------------------------------------------------------------------------------------

#include "http.h"
#include "onewire.h"
#include "ds18x20.h"
#include "USART.h"
#include "sd.h"



//---------------------------------------------------------------------------------------
// Constants
//---------------------------------------------------------------------------------------

/* Methods */
static const char c_HTTP_Method_GET[] = "GET";
static const char c_HTTP_Method_POST[] = "POST";

/* Status Codes */
static const char c_HTTP_200_OK[] = "HTTP/1.0 200 OK\r\n";
static const char c_HTTP_404_NotFound[] = "HTTP/1.0 404 Not Found\r\n";

/* Cache Control */
static const char c_HTTP_CacheControl_NoCache[] = "Pragma: no-cache\r\nExpires: Fri, 01 Jan 1990 00:00:00 GMT\r\nCache-Control: no-cache, must-revalidate\r\n";

/* Content Types */
static const char c_HTTP_Content_HTML[] = "Content-Type: text/html\r\n";
static const char c_HTTP_Content_Javascript[] = "Content-Type: text/javascript\r\n";
static const char c_HTTP_Content_Bmp[] = "Content-Type: image/bmp\r\n";
static const char c_HTTP_Content_Ico[] = "Content-Type: image/ico\r\n";
static const char c_HTTP_Content_Jpeg[] = "Content-Type: image/jpeg\r\n";
static const char c_HTTP_Content_Gif[] = "Content-Type: image/gif\r\n";
static const char c_HTTP_Content_Png[] = "Content-Type: image/png\r\n";

/* Content Length */
static const char c_HTTP_Content_Length[] = "Content-Length: ";

/* Empty Line */
static const char c_HTTP_EmptyLine[] = "\r\n";

/* File Extensions */
static const char c_HTTP_FileExtension_Htm[] = "htm";
static const char c_HTTP_FileExtension_Js[] = "js";
static const char c_HTTP_FileExtension_Bmp[] = "bmp";
static const char c_HTTP_FileExtension_Ico[] = "ico";
static const char c_HTTP_FileExtension_Jpg[] = "jpg";
static const char c_HTTP_FileExtension_Gif[] = "gif";
static const char c_HTTP_FileExtension_Png[] = "png";

/* Functions */
static const char c_HTTP_Function_GetTemperature[] = "GetTemperature";

/* Pages stored in the Flash memory */
static const char c_HTTP_Page_404[] = "<html><body><div id=\"title\" style=\"width: 800px; margin: 0 auto; text-align:center\"><h1>TheUno WebServer - Error 404</h1></div><br /><div id=\"text\" style=\"width: 800px; margin: 0 auto; text-align:center\"><p>The page requested was not found. Please click <a href=\"index.htm\">this link</a> to return on the main page of the WebServer.</p></div></body></html>\r\n";


//---------------------------------------------------------------------------------------
// Global variables
//---------------------------------------------------------------------------------------

/* Buffers used to communicate with the client */
/* Note that depending of your application, it is possible to declare and use only one buffer, saving memory */
static char g_HTTP_RecvMsgBuffer[HTTP_RX_BUFFER_SIZE];
static char g_HTTP_SendMsgBuffer[HTTP_TX_BUFFER_SIZE];

/* Timeout used to close the client socket */
static char g_HTTP_SockEstablishedTimeout = 0;

/* DS18B20 sensors (see main.c) */
/*extern t_DS18B20Device g_MAIN_DS18B20Device1;
extern t_DS18B20Device g_MAIN_DS18B20Device2;*/
extern char temperatures [MAXSENSORS][6];


//---------------------------------------------------------------------------------------
// Name:        HTTP_Init
// Param:		-
// Return:      -
//
// Description:	This function initializes the HTTP Server (W5100 and SD Card)
//---------------------------------------------------------------------------------------
void HTTP_Init(void)
{
	/* Initialize the W5100 device */
	W5100_Init();
	printf( NEWLINESTR "W5100-Init done");
	
#if (W5100_DHCP == 1)
	
	/* Get IP Address using DHCP */
	while (DHCP_GetIPAddress() == 0);
	
#endif
	
	/* Initialize the SD Card */
	while (SD_Init() != OK);
	printf( NEWLINESTR "SD-Init done");

	/* Initialize the FAT system */
	FAT_ReadBootSector();
	printf( NEWLINESTR "FAT-Init done");

}


//---------------------------------------------------------------------------------------
// Name:        HTTP_Server
// Param:		-
// Return:      -
//
// Description:	This function runs the HTTP Server
//---------------------------------------------------------------------------------------
void HTTP_Server(void)
{
	uint8_t u8SocketStatus;
	int16_t s16RecvMsgSize;
	uint8_t * u8Temp;
	uint16_t u16FileLength;
	
	/* Get socket status */
	u8SocketStatus = SOCKET_GetStatus(HTTP_SOCKET);
	
	switch (u8SocketStatus)
	{
		/* The socket is currently closed */
		case Sn_SR_SOCK_CLOSED:
			/* Open TCP socket and listen for clients */
			if (SOCKET_Open(HTTP_SOCKET, Sn_MR_TCP, HTTP_PORT, 0) == 1)
			{
				(void)SOCKET_Listen(HTTP_SOCKET);
			}
			break;

		/* Waiting for client */
		case Sn_SR_SOCK_LISTEN:
			break;
			
		/* The client is currently connected */
		case Sn_SR_SOCK_ESTABLISHED:
			/* Get incoming message */
			s16RecvMsgSize = SOCKET_Recv(HTTP_SOCKET, g_HTTP_RecvMsgBuffer, HTTP_RX_BUFFER_SIZE);
			
			/* Check number of bytes received */
			/* Take care: when the socket is established, there is a small time before receiving the data */
			if (s16RecvMsgSize > 0) 
			{
				/* Get request method type */
				u8Temp = strtok(g_HTTP_RecvMsgBuffer, " /&\r\n");
				
				/* Check request method type */
				if (strncmp(u8Temp, c_HTTP_Method_GET, strlen(c_HTTP_Method_GET)) == 0)
				{
					/* Get file requested */
					u8Temp = strtok(NULL, " /&\r\n");
					
					/* Special treatment for the index page */
					if (strncmp(u8Temp, "HTTP", strlen("HTTP")) == 0)
					{
						(void)strcpy(u8Temp, "index.htm");
					}
					
					/* Function "GetTemperature" */
					if (strncmp(u8Temp, c_HTTP_Function_GetTemperature, strlen(c_HTTP_Function_GetTemperature)) == 0)
					{
						/* Send the result of "GetTemperature" function */
						(void)SOCKET_Send(HTTP_SOCKET, (uint8_t*)c_HTTP_200_OK, strlen(c_HTTP_200_OK));
						(void)SOCKET_Send(HTTP_SOCKET, (uint8_t*)c_HTTP_CacheControl_NoCache, strlen(c_HTTP_CacheControl_NoCache));
						(void)SOCKET_Send(HTTP_SOCKET, (uint8_t*)c_HTTP_Content_HTML, strlen(c_HTTP_Content_HTML));
						
						/* Add an empty line */
						(void)SOCKET_Send(HTTP_SOCKET, (uint8_t*)c_HTTP_EmptyLine, strlen(c_HTTP_EmptyLine));
								
						/* Send the reply */
						/* Note that the format of the temperatures is: S|2^5|2^4|2^3|2^2|2^1|2^0|2^-1|2^-2|2^-3|2^-4 */
/*						(void)sprintf(g_HTTP_SendMsgBuffer, "%d.%d;%d.%d\r\n",
								(g_MAIN_DS18B20Device1.Temperature >> 4), 25 * ((g_MAIN_DS18B20Device1.Temperature >> 2) & 0x0003),
								(g_MAIN_DS18B20Device2.Temperature >> 4), 25 * ((g_MAIN_DS18B20Device2.Temperature >> 2) & 0x0003));
*/								strcat(g_HTTP_SendMsgBuffer, temperatures[0]);
								strcat(g_HTTP_SendMsgBuffer, ";");
								strcat(g_HTTP_SendMsgBuffer, temperatures[1]);
								strcat(g_HTTP_SendMsgBuffer, ";");
								strcat(g_HTTP_SendMsgBuffer, temperatures[2]);
								strcat(g_HTTP_SendMsgBuffer, ";");
								strcat(g_HTTP_SendMsgBuffer, temperatures[3]);
								strcat(g_HTTP_SendMsgBuffer, ";");
								strcat(g_HTTP_SendMsgBuffer, temperatures[4]);
								strcat(g_HTTP_SendMsgBuffer, ";");
								strcat(g_HTTP_SendMsgBuffer, temperatures[5]);
								strcat(g_HTTP_SendMsgBuffer, ";");
								strcat(g_HTTP_SendMsgBuffer, temperatures[6]);								
								strcat(g_HTTP_SendMsgBuffer, ";");
								strcat(g_HTTP_SendMsgBuffer, temperatures[7]);								
								strcat(g_HTTP_SendMsgBuffer, ";");
								strcat(g_HTTP_SendMsgBuffer, temperatures[8]);								
								strcat(g_HTTP_SendMsgBuffer, ";");
								strcat(g_HTTP_SendMsgBuffer, temperatures[9]);							
						(void)SOCKET_Send(HTTP_SOCKET, (uint8_t*)g_HTTP_SendMsgBuffer, strlen(g_HTTP_SendMsgBuffer));
					}
					/* Try to open the file on the SD Card */
					else if (FAT_FileOpen(u8Temp, READ) == FILE_FOUND)
					{
						/* File found, send HTTP Header */
						(void)SOCKET_Send(HTTP_SOCKET, (uint8_t*)c_HTTP_200_OK, strlen(c_HTTP_200_OK));
						
						/* No cache */
						(void)SOCKET_Send(HTTP_SOCKET, (uint8_t*)c_HTTP_CacheControl_NoCache, strlen(c_HTTP_CacheControl_NoCache));
						
						/* Get file extension */
						u8Temp = strtok(u8Temp, ".");
						u8Temp = strtok(NULL, ".");
						
						/* Send content type */
						if (strncmp(u8Temp, c_HTTP_FileExtension_Htm, strlen(c_HTTP_FileExtension_Htm)) == 0)
							(void)SOCKET_Send(HTTP_SOCKET, (char*)c_HTTP_Content_HTML, strlen(c_HTTP_Content_HTML));
						else if (strncmp(u8Temp, c_HTTP_FileExtension_Js, strlen(c_HTTP_FileExtension_Js)) == 0)
							(void)SOCKET_Send(HTTP_SOCKET, (char*)c_HTTP_Content_Javascript, strlen(c_HTTP_Content_Javascript));
						else if (strncmp(u8Temp, c_HTTP_FileExtension_Bmp, strlen(c_HTTP_FileExtension_Bmp)) == 0)
							(void)SOCKET_Send(HTTP_SOCKET, (char*)c_HTTP_Content_Bmp, strlen(c_HTTP_Content_Bmp));
						else if (strncmp(u8Temp, c_HTTP_FileExtension_Ico, strlen(c_HTTP_FileExtension_Ico)) == 0)
							(void)SOCKET_Send(HTTP_SOCKET, (char*)c_HTTP_Content_Ico, strlen(c_HTTP_Content_Ico));
						else if (strncmp(u8Temp, c_HTTP_FileExtension_Jpg, strlen(c_HTTP_FileExtension_Jpg)) == 0)
							(void)SOCKET_Send(HTTP_SOCKET, (char*)c_HTTP_Content_Jpeg, strlen(c_HTTP_Content_Jpeg));
						else if (strncmp(u8Temp, c_HTTP_FileExtension_Gif, strlen(c_HTTP_FileExtension_Gif)) == 0)
							(void)SOCKET_Send(HTTP_SOCKET, (char*)c_HTTP_Content_Gif, strlen(c_HTTP_Content_Gif));
						else if (strncmp(u8Temp, c_HTTP_FileExtension_Png, strlen(c_HTTP_FileExtension_Png)) == 0)
							(void)SOCKET_Send(HTTP_SOCKET, (char*)c_HTTP_Content_Png, strlen(c_HTTP_Content_Png));
						
						/* Send content length */
						(void)sprintf(g_HTTP_SendMsgBuffer, "Content-Length: %ld\r\n", FAT_GetFileSize());
						(void)SOCKET_Send(HTTP_SOCKET, (char*)g_HTTP_SendMsgBuffer, strlen(g_HTTP_SendMsgBuffer));
												
						/* Add an empty line */
						(void)SOCKET_Send(HTTP_SOCKET, (char*)c_HTTP_EmptyLine, strlen(c_HTTP_EmptyLine));
						
						/* Send the file content */
						do
						{
							u16FileLength = FAT_FileRead(g_HTTP_SendMsgBuffer);
							if (u16FileLength > 0) (void)SOCKET_Send(HTTP_SOCKET, g_HTTP_SendMsgBuffer, u16FileLength);
						}
						while (u16FileLength > 0);
					}
					/* File not found - Send 404 */
					else
					{
						/* File not found (Error 404) */
						(void)SOCKET_Send(HTTP_SOCKET, (char*)c_HTTP_404_NotFound, strlen(c_HTTP_404_NotFound));
						(void)SOCKET_Send(HTTP_SOCKET, (char*)c_HTTP_CacheControl_NoCache, strlen(c_HTTP_CacheControl_NoCache));
						(void)SOCKET_Send(HTTP_SOCKET, (char*)c_HTTP_Content_HTML, strlen(c_HTTP_Content_HTML));
						(void)SOCKET_Send(HTTP_SOCKET, (char*)c_HTTP_EmptyLine, strlen(c_HTTP_EmptyLine));
						(void)SOCKET_Send(HTTP_SOCKET, (char*)c_HTTP_Page_404, strlen(c_HTTP_Page_404));
					}
				}
				else if (strncmp(u8Temp, c_HTTP_Method_POST, strlen(c_HTTP_Method_POST)) == 0)
				{
					/* Nothing defined today */
				}

				/* Finally disconnect the socket */
				/* REMOVED -> Permit the client to stay connected and ask immediately another resource, work better */
				/* Connection will be closed on timeout (see below) */
				/*SOCKET_Disconnect(HTTP_SOCKET);*/
				
				/* Reset the timeout */
				g_HTTP_SockEstablishedTimeout = 0;
			}
			else
			{
				/* Socket established but nothing received */
				if (g_HTTP_SockEstablishedTimeout < HTTP_SOCK_ESTABLISHED_TIMEOUT)
				{
					/* Increment the timeout value */
					g_HTTP_SockEstablishedTimeout++;
				}
				else
				{
					/* Nothing received and timeout elapsed, disconnect the socket */
					SOCKET_Disconnect(HTTP_SOCKET);
					g_HTTP_SockEstablishedTimeout = 0;
				}
			}
			break;

		/* Other status */
		case Sn_SR_SOCK_FIN_WAIT:
		case Sn_SR_SOCK_CLOSING:
		case Sn_SR_SOCK_TIME_WAIT:
		case Sn_SR_SOCK_CLOSE_WAIT:
		case Sn_SR_SOCK_LAST_ACK:
			/* Force closing the socket */
			SOCKET_Close(HTTP_SOCKET);
			break;
		
		default:
			break;
    }
}
