/////////////////////////////////////////////////////////////////////////////////////////
//
// SD Card Low-Level Interface
//
// --------------------------------------------------------------------------------------
//
// Filename:      sd.c
// Version:       1.0
// Date:          15/10/2011
// Author:        Joel Guittet - http://myfreescalewebpage.free.fr
//
/////////////////////////////////////////////////////////////////////////////////////////
//
// Description
//
// This file is used to communicate with SD Card over the SPI interface.
// This file is based on AN4115 from Freescale, on which some improvements have been made.
//
/////////////////////////////////////////////////////////////////////////////////////////
//
// Revisions
//
// Version	| Author		| Description
// --------------------------------------------------------------------------------------
//			|				| 
//
/////////////////////////////////////////////////////////////////////////////////////////


//---------------------------------------------------------------------------------------
// Includes
//---------------------------------------------------------------------------------------

#include "sd.h"
#include "USART.h"


//---------------------------------------------------------------------------------------
// Name:        SD_Init
// Param:		-
// Return:      SD Card error code depending of initialization result 
//
// Description:	This function is used to know if the SD card is present
//---------------------------------------------------------------------------------------
uint8_t SD_Init(void) 
{
	uint8_t u8Index;
	
	/* Check if the SD card is present */
	if (SPI_IsSDCardPresent() == 0)
	{
		return NO_SD_CARD;
	}
  
	/* Initialize the SPI module for SD Card initialization (low rate required) */
	SPI_Init_LowRate();

	SPI_SelectSDCard();
	
	/* Send at least 75 SPI clock cycles to initialize the internal SD Card state machine */
	for (u8Index = 0; u8Index < 10; u8Index++)
	{
		SPI_Write(0xFF);
	}
	
	SPI_UnselectSDCard();
	
	/* Send at least 16 (?) SPI clock cycles with SD Card unselected */
	for (u8Index = 0; u8Index < 10; u8Index++)
	{
		SPI_Write(0xFF);
	}

	SPI_SelectSDCard();
    
	/* Send RESET command to restart SD Card in SPI mode */
	if (SD_SendCommand(SD_CMD0, 0, SD_IDLE) != OK)
    {
		SPI_UnselectSDCard();
		return INIT_FAILS;      
    }
	
	SPI_UnselectSDCard();
    
	/* Dummy SPI cycle */
    (void)SPI_Read();
    
    SPI_SelectSDCard();
    	
    /* Sends host capacity support information and activates the card's initialization process */
    while (SD_SendCommand(SD_CMD1, 0, SD_OK) != OK);
    
    SPI_UnselectSDCard();
    
    /* Dummy SPI cycle */
	(void)SPI_Read();
	
	SPI_SelectSDCard();
	    
    /* Send Block Length */
    if (SD_SendCommand(SD_CMD16, SD_BLOCK_SIZE, SD_OK) != OK)
    {
    	SPI_UnselectSDCard();
    	return INIT_FAILS;      
	}
    
    SPI_UnselectSDCard();
    
    /* Initializer the SPI module for SD Card normal operation (high rate, up to 25MHz) */  
    SPI_Init_HighRate();

    /* Dummy SPI Cycle */
    SPI_Write(0x00);
    SPI_Write(0x00);
    
    return OK;
}


//---------------------------------------------------------------------------------------
// Name:        SD_WriteBlock
// Param:		Block: number of the block to write
//				DataPointer: point on data to write in the SD Card
// Return:      SD Card error code depending of SD Card response
//
// Description:	This function writes a block of data in the SD Card
//---------------------------------------------------------------------------------------
uint8_t SD_WriteBlock(uint32_t u32Block, uint8_t * pu8DataPointer) 
{
    uint16_t u16Counter;

    /* Check if the SD-Card is not write protected */
	if (SPI_IsSDCardWriteProtected() == 1)
	{
		return WP_SD_CARD;
	}
        
    SPI_SelectSDCard();

    /* Send block number */
    if (SD_SendCommand(SD_CMD24, u32Block * SD_BLOCK_SIZE, SD_OK) != OK)
    {
        SPI_UnselectSDCard();
        return WRITE_COMMAND_FAILS;      
    }
    
    SPI_Write(0xFE);
    
    /* Write data */
    for (u16Counter = 0; u16Counter < SD_BLOCK_SIZE; u16Counter++)
    {
        SPI_Write(*pu8DataPointer++);
    }

    /* Checksum not required */
    SPI_Write(0xFF);
    SPI_Write(0xFF);
    
    if ((SPI_Read() & 0x0F) != 0x05)
    {
    	SPI_UnselectSDCard();
		return WRITE_COMMAND_FAILS;      
	}

    /* Dummy cycles */
    while (SPI_Read() == 0x00);
    
    SPI_UnselectSDCard();
    
    return OK;
}


//---------------------------------------------------------------------------------------
// Name:        SD_ReadBlock
// Param:		Block: number of the block to read
//				DataPointer: point on buffer to get the data read in the SD Card
// Return:      SD Card error code depending of SD Card response
//
// Description:	This function reads a block of data in the SD Card
//---------------------------------------------------------------------------------------
uint8_t SD_ReadBlock(uint32_t u32Block, uint8_t * pu8DataPointer) 
{
    uint16_t u16Counter;

    SPI_SelectSDCard();   

    if (SD_SendCommand(SD_CMD17, u32Block * SD_BLOCK_SIZE, SD_OK) != OK)
    {
        SPI_UnselectSDCard();
        return READ_COMMAND_FAILS;      
    }
    
    while (SPI_Read() != 0xFE);
    
    /* Read data */
    for (u16Counter = 0; u16Counter < SD_BLOCK_SIZE; u16Counter++)
    {
        *pu8DataPointer++ = SPI_Read();
    }
    
    /* Dummy cycles */
    (void)SPI_Read();
    (void)SPI_Read();
    
    SPI_UnselectSDCard();
  
    /* Dummy cycle */
    (void)SPI_Read();

    return OK;
}


//---------------------------------------------------------------------------------------
// Name:        SD_SendCommand
// Param:		Command: command to send
//				Argument: argument for the previous command
//				Response: response waited from SD Card
// Return:      SD Card error code depending of SD Card response
//
// Description:	This function sends a command to the SD Card and wait for the waited response
//---------------------------------------------------------------------------------------
static uint8_t SD_SendCommand(uint8_t u8Command, uint32_t u32Argument, uint8_t u8Response) 
{
    uint8_t u8Counter, u8Temp;

    /* Send Start byte */
    SPI_Write(u8Command | 0x40);

    /* Send Argument */
    SPI_Write((uint8_t)((u32Argument & 0xFF000000) >> 24));
    SPI_Write((uint8_t)((u32Argument & 0x00FF0000) >> 16));
    SPI_Write((uint8_t)((u32Argument & 0x0000FF00) >> 8));
    SPI_Write((uint8_t)(u32Argument & 0x000000FF));
    
    /* Send CRC */
    SPI_Write(0x95);
  
    /* Response RHandler */
    u8Counter = 10;
    do
    {
        u8Temp = SPI_Read();
        u8Counter--;
    }
    while ((u8Temp != u8Response) && (u8Counter > 0));
    
    return ((u8Temp == u8Response) ? OK : COMMAND_FAILS);
}
