/////////////////////////////////////////////////////////////////////////////////////////
//
// Wiznet W5100 Sockets Interface
//
// --------------------------------------------------------------------------------------
//
// Filename:      socket.c
// Version:       1.0
// Date:          15/10/2011
// Author:        Joel Guittet - http://myfreescalewebpage.free.fr
//
/////////////////////////////////////////////////////////////////////////////////////////
//
// Description
//
// This file is used to manipulate sockets on W5100 device.
//
/////////////////////////////////////////////////////////////////////////////////////////
//
// Revisions
//
// Version	| Author		| Description
// --------------------------------------------------------------------------------------
//			|				| 
//
/////////////////////////////////////////////////////////////////////////////////////////


//---------------------------------------------------------------------------------------
// Includes
//---------------------------------------------------------------------------------------

#include "socket.h"
#include "USART.h"


//---------------------------------------------------------------------------------------
// Name:        SOCKET_Open
// Param:		Socket: socket number
//				Protocol: socket protocol
//				Port: source port (0 for client socket)
//				Flag: options (Sn_MR_ND, Sn_MR_MULTI) 
// Return:      TRUE if socket initialization succeeded (Sn_SR_SOCK_INIT status), FALSE if an error occurred
//
// Description:	This function open a new socket
//---------------------------------------------------------------------------------------
uint8_t SOCKET_Open(uint8_t u8Socket, uint8_t u8Protocol, uint16_t u16Port, uint8_t u8Flag)
{
	uint8_t u8Status;
	uint8_t bResult = 0;

	
	/* Check parameters */
	if ((u8Protocol == Sn_MR_TCP) || (u8Protocol == Sn_MR_UDP) || (u8Protocol == Sn_MR_IPRAW) || (u8Protocol == Sn_MR_MACRAW) || (u8Protocol == Sn_MR_PPPOE))
	{
		/* Close the socket, if already opened */
		SOCKET_Close(u8Socket);
		
		/* Set socket option and protocol */
		W5100_WriteByte(W5100_Sn_MR(u8Socket), u8Protocol | u8Flag);
		
		/* Port */
		W5100_WriteByte(W5100_Sn_PORT0(u8Socket), (uint8_t)(u16Port >> 8));
		W5100_WriteByte(W5100_Sn_PORT1(u8Socket), (uint8_t)(u16Port & 0x00FF));
		
		/* Open socket */
		W5100_WriteByte(W5100_Sn_CR(u8Socket), Sn_CR_OPEN);
		
		/* Wait */
		while (W5100_ReadByte(W5100_Sn_CR(u8Socket)));
		
		/* Check status */
		u8Status = W5100_ReadByte(W5100_Sn_SR(u8Socket));
		if ((u8Status == Sn_SR_SOCK_INIT) || (u8Status == Sn_SR_SOCK_UDP) || (u8Status == Sn_SR_SOCK_IPRAW) || (u8Status == Sn_SR_SOCK_MACRAW))
		{
			bResult = 1;
		}
		
		/* If an error occurred, close the socket */
		if (bResult == 0)
		{
			SOCKET_Close(u8Socket);
		}
	}

	return bResult;
}


//---------------------------------------------------------------------------------------
// Name:        SOCKET_Close
// Param:		Socket: socket number
// Return:      -
//
// Description:	This function closes a socket
//---------------------------------------------------------------------------------------
void SOCKET_Close(uint8_t u8Socket)
{
	/* Close socket */
	W5100_WriteByte(W5100_Sn_CR(u8Socket), Sn_CR_CLOSE);
	
	/* Wait */
	while (W5100_ReadByte(W5100_Sn_CR(u8Socket)));
}


//---------------------------------------------------------------------------------------
// Name:        SOCKET_Connect
// Param:		Socket: socket number
//				Address: destination IP address array (4 bytes)
//				Port: destination port 
// Return:      TRUE if the connection is established (Sn_SR_SOCK_ESTABLISHED status), FALSE if an error occurred
//
// Description:	This function is used to establish a connection with a remote server
//				It waits until the connection is established or closed 
//---------------------------------------------------------------------------------------
uint8_t SOCKET_Connect(uint8_t u8Socket, uint8_t * u8Address, uint16_t u16Port)
{
	uint8_t u8Status;
	uint8_t bResult = 0;
	
	if (((u8Address[0] != 0xFF) || (u8Address[1] != 0xFF) || (u8Address[2] != 0xFF) || (u8Address[3] != 0xFF)) &&
		((u8Address[0] != 0x00) || (u8Address[1] != 0x00) || (u8Address[2] != 0x00) || (u8Address[3] != 0x00)) && (u16Port != 0x00))
	{
		/* Destination IP */
		W5100_WriteByte(W5100_Sn_DIPR0(u8Socket), u8Address[0]);
		W5100_WriteByte(W5100_Sn_DIPR1(u8Socket), u8Address[1]);
		W5100_WriteByte(W5100_Sn_DIPR2(u8Socket), u8Address[2]);
		W5100_WriteByte(W5100_Sn_DIPR3(u8Socket), u8Address[3]);
		
		/* Port */
		W5100_WriteByte(W5100_Sn_DPORT0(u8Socket), (uint8_t)(u16Port >> 8));
		W5100_WriteByte(W5100_Sn_DPORT1(u8Socket), (uint8_t)(u16Port & 0x00FF));
		
		/* Establish connection */
		W5100_WriteByte(W5100_Sn_CR(u8Socket), Sn_CR_CONNECT);
		
		/* Wait */
		while (W5100_ReadByte(W5100_Sn_CR(u8Socket)));
		
		/* Wait while connection is not established */
		do
		{
			u8Status = W5100_ReadByte(W5100_Sn_SR(u8Socket));
		}
		while (u8Status != Sn_SR_SOCK_ESTABLISHED && u8Status != Sn_SR_SOCK_CLOSED && u8Status != Sn_SR_SOCK_CLOSE_WAIT);

		bResult = (u8Status == Sn_SR_SOCK_ESTABLISHED);
	}
	
	return bResult;
}


//---------------------------------------------------------------------------------------
// Name:        SOCKET_Disconnect
// Param:		Socket: socket number
// Return:      -
//
// Description:	This function terminates a connection
//---------------------------------------------------------------------------------------
void SOCKET_Disconnect(uint8_t u8Socket)
{
	/* Disconnect the socket */
	W5100_WriteByte(W5100_Sn_CR(u8Socket), Sn_CR_DISCON);
	
	/* Wait */
	while (W5100_ReadByte(W5100_Sn_CR(u8Socket)));
}


//---------------------------------------------------------------------------------------
// Name:        SOCKET_GetStatus
// Param:		Socket: socket number
// Return:      Socket status
//
// Description:	This function returns the socket status from W5100 device
//---------------------------------------------------------------------------------------
uint8_t SOCKET_GetStatus(uint8_t u8Socket)
{
	return W5100_ReadByte(W5100_Sn_SR(u8Socket));
}


//---------------------------------------------------------------------------------------
// Name:        SOCKET_Listen
// Param:		Socket: socket number
// Return:      TRUE in case of success, FALSE if an error occurred
//
// Description:	This function establishes the connection in passive (server) mode
//---------------------------------------------------------------------------------------
uint8_t SOCKET_Listen(uint8_t u8Socket)
{
	uint8_t bResult = 0;

	/* Check socket status */
	if (W5100_ReadByte(W5100_Sn_SR(u8Socket)) == Sn_SR_SOCK_INIT)
	{
		/* Send Listen command */
		W5100_WriteByte(W5100_Sn_CR(u8Socket), Sn_CR_LISTEN);
		
		/* Wait */
		while (W5100_ReadByte(W5100_Sn_CR(u8Socket)));
		
		/* Check socket status */
		bResult = (W5100_ReadByte(W5100_Sn_SR(u8Socket)) == Sn_SR_SOCK_LISTEN);
		
		/* If an error occurred, close the socket */
		if (bResult == 0)
		{
			SOCKET_Close(u8Socket);
		}
	}

	return bResult;
}


//---------------------------------------------------------------------------------------
// Name:        SOCKET_Send
// Param:		Socket: socket number
//				Buffer: contains the data to send
//				Length: size of the buffer
// Return:      Number of bytes sent
//
// Description:	This function is used to send data in TCP mode
//---------------------------------------------------------------------------------------
int16_t SOCKET_Send(uint8_t u8Socket, uint8_t * u8Buffer, uint16_t u16Length)
{
	uint16_t u16TxPointer;
	uint16_t u16TxBaseAddress;
	uint16_t u16TxBufferSize;
	uint16_t u16Index;
	int16_t s16BytesSent = -1;
	
	/* Base Address and Buffer Size in W5100 device to transmit data */
	switch (u8Socket)
	{
		case 0: u16TxBaseAddress = W5100_TX_MEMORY_SOCKET_0_ADDRESS; u16TxBufferSize = W5100_TX_SOCKET_0_SIZE_BYTES; break;
		case 1: u16TxBaseAddress = W5100_TX_MEMORY_SOCKET_1_ADDRESS; u16TxBufferSize = W5100_TX_SOCKET_1_SIZE_BYTES; break;
		case 2: u16TxBaseAddress = W5100_TX_MEMORY_SOCKET_2_ADDRESS; u16TxBufferSize = W5100_TX_SOCKET_2_SIZE_BYTES; break;
		case 3: u16TxBaseAddress = W5100_TX_MEMORY_SOCKET_3_ADDRESS; u16TxBufferSize = W5100_TX_SOCKET_3_SIZE_BYTES; break;
		default: break;
	}
	
	/* Check Socket status */
	if (W5100_ReadByte(W5100_Sn_SR(u8Socket)) == Sn_SR_SOCK_ESTABLISHED)
	{
		/* Get TX pointer */
		u16TxPointer = W5100_ReadByte(W5100_Sn_TX_WR0(u8Socket)) << 8;
		u16TxPointer += W5100_ReadByte(W5100_Sn_TX_WR1(u8Socket));
				
		for (u16Index = 0; u16Index < u16Length && u16Length < u16TxBufferSize; u16Index++)
		{
			/* Write byte */
			W5100_WriteByte(u16TxBaseAddress + (u16TxPointer & (u16TxBufferSize - 1)), u8Buffer[u16Index]);
			
			/* Increment TX Pointer */
			u16TxPointer++;
		}
		
		/* Write TX Pointer */
		W5100_WriteByte(W5100_Sn_TX_WR0(u8Socket), (uint8_t)(u16TxPointer >> 8));
		W5100_WriteByte(W5100_Sn_TX_WR1(u8Socket), (uint8_t)(u16TxPointer & 0x00FF));
				
		/* Send data */
		W5100_WriteByte(W5100_Sn_CR(u8Socket), Sn_CR_SEND);
		
		/* Wait */
		while (W5100_ReadByte(W5100_Sn_CR(u8Socket)));
		
		/* Update number of bytes sent */
		s16BytesSent = (int16_t)u16Index;
	}
	
	/* Return number of bytes sent */
	return s16BytesSent;
}


//---------------------------------------------------------------------------------------
// Name:        SOCKET_Recv
// Param:		Socket: socket number
//				Buffer: buffer to get data from W5100 device
//				Length: maximum number of bytes to get from W5100 device 
// Return:      Number of bytes read from the W5100 device, -1 if nothing received
//
// Description:	This function is used to receive data in TCP mode
//---------------------------------------------------------------------------------------
int16_t SOCKET_Recv(uint8_t u8Socket, uint8_t * u8Buffer, uint16_t u16Length)
{
	uint16_t u16RxRecvSize;
	uint16_t u16RxPointer;
	uint16_t u16RxBaseAddress;
	uint16_t u16RxBufferSize;
	uint16_t u16Index;
	int16_t s16BytesReceived = -1;

	/* Base Address and Buffer Size in W5100 device to transmit data */
	switch (u8Socket)
	{
		case 0: u16RxBaseAddress = W5100_RX_MEMORY_SOCKET_0_ADDRESS; u16RxBufferSize = W5100_RX_SOCKET_0_SIZE_BYTES; break;
		case 1: u16RxBaseAddress = W5100_RX_MEMORY_SOCKET_1_ADDRESS; u16RxBufferSize = W5100_RX_SOCKET_1_SIZE_BYTES; break;
		case 2: u16RxBaseAddress = W5100_RX_MEMORY_SOCKET_2_ADDRESS; u16RxBufferSize = W5100_RX_SOCKET_2_SIZE_BYTES; break;
		case 3: u16RxBaseAddress = W5100_RX_MEMORY_SOCKET_3_ADDRESS; u16RxBufferSize = W5100_RX_SOCKET_3_SIZE_BYTES; break;
		default: break;
	}
	
	/* Check Socket status */
	if (W5100_ReadByte(W5100_Sn_SR(u8Socket)) == Sn_SR_SOCK_ESTABLISHED)
	{
		/* Get received size */
		u16RxRecvSize = W5100_ReadByte(W5100_Sn_RX_RSR0(u8Socket)) << 8;
		u16RxRecvSize += W5100_ReadByte(W5100_Sn_RX_RSR1(u8Socket));
		
		if (u16RxRecvSize > 0)
		{
			/* Get RX pointer */
			u16RxPointer = W5100_ReadByte(W5100_Sn_RX_RD0(u8Socket)) << 8;
			u16RxPointer += W5100_ReadByte(W5100_Sn_RX_RD1(u8Socket));
			
			for (u16Index = 0; u16Index < u16RxRecvSize && u16Index < u16Length; u16Index++)
			{
				/* Read byte */
				u8Buffer[u16Index] = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
				
				/* Increment RX Pointer */
				u16RxPointer++;
			}
			
			/* Save new RX Pointer */
			W5100_WriteByte(W5100_Sn_RX_RD0(u8Socket), (uint8_t)(u16RxPointer >> 8));
			W5100_WriteByte(W5100_Sn_RX_RD1(u8Socket), (uint8_t)(u16RxPointer & 0x00FF));
			
			/* Send Receive Command */
			W5100_WriteByte(W5100_Sn_CR(u8Socket), Sn_CR_RECV);
			
			/* Wait */
			while (W5100_ReadByte(W5100_Sn_CR(u8Socket)));
			
			/* Update number of bytes received */
			s16BytesReceived = (int16_t)u16Index;
		}
	}

	return s16BytesReceived;
}


//---------------------------------------------------------------------------------------
// Name:        SOCKET_SendTo
// Param:		Socket: socket number
//				Buffer: contains the data to send
//				Length: size of the buffer
//				Address: destination IP address
//				Port: port on which to send the data
// Return:      Number of bytes sent
//
// Description:	This function is used to send data in UDP, IP_RAW and MAC_RAW modes
//---------------------------------------------------------------------------------------
int16_t SOCKET_SendTo(uint8_t u8Socket, uint8_t * u8Buffer, uint16_t u16Length, uint8_t * u8Address, uint16_t u16Port)
{
	uint8_t u8Status;
	uint16_t u16TxPointer;
	uint16_t u16TxBaseAddress;
	uint16_t u16TxBufferSize;
	uint16_t u16Index;
	int16_t s16BytesSent = -1;
	
	/* Base Address and Buffer Size in W5100 device to transmit data */
	switch (u8Socket)
	{
		case 0: u16TxBaseAddress = W5100_TX_MEMORY_SOCKET_0_ADDRESS; u16TxBufferSize = W5100_TX_SOCKET_0_SIZE_BYTES; break;
		case 1: u16TxBaseAddress = W5100_TX_MEMORY_SOCKET_1_ADDRESS; u16TxBufferSize = W5100_TX_SOCKET_1_SIZE_BYTES; break;
		case 2: u16TxBaseAddress = W5100_TX_MEMORY_SOCKET_2_ADDRESS; u16TxBufferSize = W5100_TX_SOCKET_2_SIZE_BYTES; break;
		case 3: u16TxBaseAddress = W5100_TX_MEMORY_SOCKET_3_ADDRESS; u16TxBufferSize = W5100_TX_SOCKET_3_SIZE_BYTES; break;
		default: break;
	}

	if (((u8Address[0] != 0x00) || (u8Address[1] != 0x00) || (u8Address[2] != 0x00) || (u8Address[3] != 0x00)) && (u16Port != 0x00)) 
 	{
		/* Check Socket status */
		u8Status = W5100_ReadByte(W5100_Sn_SR(u8Socket));
		
		if ((u8Status == Sn_SR_SOCK_UDP) || (u8Status == Sn_SR_SOCK_IPRAW) || (u8Status == Sn_SR_SOCK_MACRAW))
		{
			/* Destination IP address */
			W5100_WriteByte(W5100_Sn_DIPR0(u8Socket), u8Address[0]);
			W5100_WriteByte(W5100_Sn_DIPR1(u8Socket), u8Address[1]);
			W5100_WriteByte(W5100_Sn_DIPR2(u8Socket), u8Address[2]);
			W5100_WriteByte(W5100_Sn_DIPR3(u8Socket), u8Address[3]);
		   
			/* Port */
			W5100_WriteByte(W5100_Sn_DPORT0(u8Socket), (uint8_t)(u16Port >> 8));
			W5100_WriteByte(W5100_Sn_DPORT1(u8Socket), (uint8_t)(u16Port && 0x00FF));
					
			/* Get TX pointer */
			u16TxPointer = W5100_ReadByte(W5100_Sn_TX_WR0(u8Socket)) << 8;
			u16TxPointer += W5100_ReadByte(W5100_Sn_TX_WR1(u8Socket));
			
			for (u16Index = 0; u16Index < u16Length && u16Length < u16TxBufferSize; u16Index++)
			{
				/* Write byte */
				W5100_WriteByte(u16TxBaseAddress + (u16TxPointer & (u16TxBufferSize - 1)), u8Buffer[u16Index]);
				
				/* Increment TX Pointer */
				u16TxPointer++;
			}
			
			/* Save new TX Pointer */
			W5100_WriteByte(W5100_Sn_TX_WR0(u8Socket), (uint8_t)(u16TxPointer >> 8));
			W5100_WriteByte(W5100_Sn_TX_WR1(u8Socket), (uint8_t)(u16TxPointer & 0x00FF));
					
			/* Send data */
			W5100_WriteByte(W5100_Sn_CR(u8Socket), Sn_CR_SEND);
			
			/* Wait */
			while (W5100_ReadByte(W5100_Sn_CR(u8Socket)));
					
			/* Update number of bytes sent */
			s16BytesSent = (int16_t)u16Index;
		}
 	}
		
	/* Return number of bytes sent */
	return s16BytesSent;
}


//---------------------------------------------------------------------------------------
// Name:        SOCKET_RecvFrom
// Param:		Socket: socket number
//				Buffer: buffer to get data from W5100 device
//				Length: maximum number of bytes to get from W5100 device
//				Address: sender IP address
//				Port: port on which data are received
// Return:      Number of bytes read from the W5100 device, -1 if nothing received
//
// Description:	This function is used to receive data in UDP, IP_RAW and MAC_RAW modes
//---------------------------------------------------------------------------------------
int16_t SOCKET_RecvFrom(uint8_t u8Socket, uint8_t * u8Buffer, uint16_t u16Length, uint8_t * u8Address, uint16_t * u16Port)
{
	uint8_t u8Status;
	uint16_t u16RxRecvSize;
	uint16_t u16RxPointer;
	uint16_t u16RxBaseAddress;
	uint16_t u16RxBufferSize;
	uint16_t u16DataLength;
	uint16_t u16Index;
	int16_t s16BytesReceived = -1;

	/* Base Address and Buffer Size in W5100 device to transmit data */
	switch (u8Socket)
	{
		case 0: u16RxBaseAddress = W5100_RX_MEMORY_SOCKET_0_ADDRESS; u16RxBufferSize = W5100_RX_SOCKET_0_SIZE_BYTES; break;
		case 1: u16RxBaseAddress = W5100_RX_MEMORY_SOCKET_1_ADDRESS; u16RxBufferSize = W5100_RX_SOCKET_1_SIZE_BYTES; break;
		case 2: u16RxBaseAddress = W5100_RX_MEMORY_SOCKET_2_ADDRESS; u16RxBufferSize = W5100_RX_SOCKET_2_SIZE_BYTES; break;
		case 3: u16RxBaseAddress = W5100_RX_MEMORY_SOCKET_3_ADDRESS; u16RxBufferSize = W5100_RX_SOCKET_3_SIZE_BYTES; break;
		default: break;
	}
	
	/* Check Socket status */
	u8Status = W5100_ReadByte(W5100_Sn_SR(u8Socket));
			
	if ((u8Status == Sn_SR_SOCK_UDP) || (u8Status == Sn_SR_SOCK_IPRAW) || (u8Status == Sn_SR_SOCK_MACRAW))
	{
		/* Get received size */
		u16RxRecvSize = W5100_ReadByte(W5100_Sn_RX_RSR0(u8Socket)) << 8;
		u16RxRecvSize += W5100_ReadByte(W5100_Sn_RX_RSR1(u8Socket));
		
		if (u16RxRecvSize > 0)
		{
			/* Get RX pointer */
			u16RxPointer = W5100_ReadByte(W5100_Sn_RX_RD0(u8Socket)) << 8;
			u16RxPointer += W5100_ReadByte(W5100_Sn_RX_RD1(u8Socket));
			
			/* Check Protocol */
			switch (W5100_ReadByte(W5100_Sn_MR(u8Socket)) & 0x07)
			{
				case Sn_MR_UDP:
					/* Get sender IP address */
					u8Address[0] = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
					u16RxPointer++;
					u8Address[1] = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
					u16RxPointer++;
					u8Address[2] = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
					u16RxPointer++;
					u8Address[3] = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
					u16RxPointer++;
										
					/* Get port */
					*u16Port = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1))) << 8;
					u16RxPointer++;
					*u16Port += W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
					u16RxPointer++;
					
					/* Get Data length */
					u16DataLength = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1))) << 8;
					u16RxPointer++;
					u16DataLength += W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
					u16RxPointer++;
					
					for (u16Index = 0; u16Index < u16DataLength && u16Index < u16Length; u16Index++)
					{
						/* Read byte */
						u8Buffer[u16Index] = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
						
						/* Increment RX Pointer */
						u16RxPointer++;
					}
					
					/* Save new RX Pointer */
					W5100_WriteByte(W5100_Sn_RX_RD0(u8Socket), (uint8_t)(u16RxPointer >> 8));
					W5100_WriteByte(W5100_Sn_RX_RD1(u8Socket), (uint8_t)(u16RxPointer & 0x00FF));
					
					/* Send Receive Command */
					W5100_WriteByte(W5100_Sn_CR(u8Socket), Sn_CR_RECV);
					
					/* Wait */
					while (W5100_ReadByte(W5100_Sn_CR(u8Socket)));
					
					/* Update number of bytes received */
					s16BytesReceived = (int16_t)u16Index;
				
					break;
			
				case Sn_MR_IPRAW:
					/* Get sender IP address */
					u8Address[0] = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
					u16RxPointer++;
					u8Address[1] = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
					u16RxPointer++;
					u8Address[2] = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
					u16RxPointer++;
					u8Address[3] = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
					u16RxPointer++;
					
					/* Get Data length */
					u16DataLength = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1))) << 8;
					u16RxPointer++;
					u16DataLength += W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
					u16RxPointer++;
					
					for (u16Index = 0; u16Index < u16DataLength && u16Index < u16Length; u16Index++)
					{
						/* Read byte */
						u8Buffer[u16Index] = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
						
						/* Increment RX Pointer */
						u16RxPointer++;
					}
					
					/* Save new RX Pointer */
					W5100_WriteByte(W5100_Sn_RX_RD0(u8Socket), (uint8_t)(u16RxPointer >> 8));
					W5100_WriteByte(W5100_Sn_RX_RD1(u8Socket), (uint8_t)(u16RxPointer & 0x00FF));
					
					/* Send Receive Command */
					W5100_WriteByte(W5100_Sn_CR(u8Socket), Sn_CR_RECV);
					
					/* Wait */
					while (W5100_ReadByte(W5100_Sn_CR(u8Socket)));
					
					/* Update number of bytes received */
					s16BytesReceived = (int16_t)u16Index;
					
					break;
					
				case Sn_MR_MACRAW:
					/* Get Data length */
					u16DataLength = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1))) << 8;
					u16RxPointer++;
					u16DataLength += W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
					u16RxPointer++;
					
					for (u16Index = 0; u16Index < u16DataLength && u16Index < u16Length; u16Index++)
					{
						/* Read byte */
						u8Buffer[u16Index] = W5100_ReadByte(u16RxBaseAddress + (u16RxPointer & (u16RxBufferSize - 1)));
						
						/* Increment RX Pointer */
						u16RxPointer++;
					}
					
					/* Save new RX Pointer */
					W5100_WriteByte(W5100_Sn_RX_RD0(u8Socket), (uint8_t)(u16RxPointer >> 8));
					W5100_WriteByte(W5100_Sn_RX_RD1(u8Socket), (uint8_t)(u16RxPointer & 0x00FF));
					
					/* Send Receive Command */
					W5100_WriteByte(W5100_Sn_CR(u8Socket), Sn_CR_RECV);
					
					/* Wait */
					while (W5100_ReadByte(W5100_Sn_CR(u8Socket)));
					
					/* Update number of bytes received */
					s16BytesReceived = (int16_t)u16Index;
					
					break;
				
				default:
					break;
			}
		}
	}
	
	return s16BytesReceived;
}
