/////////////////////////////////////////////////////////////////////////////////////////
//
// SPI interface
//
// --------------------------------------------------------------------------------------
//
// Filename:      spi.c
// Version:       1.0
// Date:          15/10/2011
// Author:        Joel Guittet - http://myfreescalewebpage.free.fr
//
/////////////////////////////////////////////////////////////////////////////////////////
//
// Description
//
// This file is used to communicate with the SPI devices (W5100 and SD Card). It manages
// communication on the SPI interface, but is also used to get SD Card status (presence
// and write protection).
// This file can be modified to match hardware requirements, for example to modify pins
// or in order to add components on the SPI bus. 
//
/////////////////////////////////////////////////////////////////////////////////////////
//
// Revisions
//
// Version	| Author		| Description
// --------------------------------------------------------------------------------------
//			|				| 
//
/////////////////////////////////////////////////////////////////////////////////////////


//---------------------------------------------------------------------------------------
// Includes
//---------------------------------------------------------------------------------------

#include "spi.h"



void SPI_IO_Init(void)
//######################################################
{

	
	DDR_SPI |= ( 1 << DD_MOSI) | ( 1 << DD_SCK) | ( 1 << DD_HW_CS);		//MOSI and SCK set to output
	DDR_SPI &= ~( 1 << DD_MISO );						//MISO set to input

	DDR_HW_CS |= ( 1 << DDR_HW_CS);	// HW CS set to output
	PORT_HW_CS |= ( 1 << PIN_HW_CS);	// HW CS set to 1
	
	DDR_MMC_CS |= ( 1 << DDR_MMC_CS);	// MMC CS set to output
	PORT_MMC_CS |= ( 1 << PIN_MMC_CS);	// MMC CS set to 1
	
	DDR_W51_CS |= ( 1 << DDR_W51_CS);	// W51 CS set to output
	PORT_W51_CS |= ( 1 << PIN_W51_CS);	// W51 CS set to 1
}

//---------------------------------------------------------------------------------------
// Name:        SPI_Init_LowRate
// Param:		-
// Return:      -
//
// Description:	This function initialized the SPI interface with SPI clock set to 375kHz
//				It is used to initialized the SD Card
//---------------------------------------------------------------------------------------
void SPI_Init_LowRate(void)
{
	uint8_t i;
	// Init SPI with a very slow transfer rate first !

	// SPCR SPI Controlregister
	// SPIE=0; //No SPI Interrupt
	// SPE=1;  //SPI Enable
	// DORD=0; //Send MSB first
	// MSTR=1; //I am the master !
	// CPOL=0; //SCK low if IDLE
	// CPHA=0; //SPI Mode 0
	// SPR1=1; //SPI Clock = f/128 = 125kHz @16MHz Clock
	// SPR0=1; //or f/64 if SPI2X = 1 in SPSR register
	
	PRR0 &= ~(1 << PRSPI);	//turn off Powerreduction
	SPI_CTRL_REGISTER = (0<<SPIE) | (1<<SPE) | (0<<DORD) | (1<<MSTR) | (0<<CPOL) | (0<<CPHA) | (1<<SPR1) | (1<<SPR0);

	// SPSR SPI Statusregister
	// SPI2X=0; // No double speed
	SPI_STATUS_REGISTER = 0x00;
	
	i=10;
	do
	{
		SPI_WRITE(DUMMY_WRITE);
		SPI_WAIT();
		i--;
	}while(i);

}


//---------------------------------------------------------------------------------------
// Name:        SPI_Init_HighRate
// Param:		-
// Return:      -
//
// Description:	This function initialized the SPI interface at the maximum SPI clock
//				It is used for normal operation
//---------------------------------------------------------------------------------------
void SPI_Init_HighRate(void)
{
	uint8_t i;
	// SPI initialization
	// SPI Type: Master
	// SPI Clock Rate: 4000,000 kHz
	// SPI Clock Phase: Cycle Start
	// SPI Clock Polarity: Low
	// SPI Data Order: MSB First
	PRR0 &= ~(1 << PRSPI);	//turn off Powerreduction
	SPI_CTRL_REGISTER = (0<<SPIE) | (1<<SPE) | (0<<DORD) | (1<<MSTR) | (0<<CPOL) | (0<<CPHA) | (0<<SPR1) | (0<<SPR0);
	SPI_CTRL_REGISTER &= ~( 1 <<SPR1);
	SPI_CTRL_REGISTER &= ~( 1 <<SPR0);
	//SPI_CTRL_REGISTER = (1<<SPE) | (1<<MSTR) | (1<<SPR1) | (1<<SPR0);
	SPI_STATUS_REGISTER = 0x00;	//(0<<SPI2X);
	
	
	i=10;
	do
	{
		SPI_WRITE(DUMMY_WRITE);
		SPI_WAIT();
		i--;
	}while(i);
	
}		


//---------------------------------------------------------------------------------------
// Name:        SPI_Write
// Param:		Byte: byte to write on SPI interface
// Return:      -
//
// Description:	This function is used to write a byte on SPI interface
//---------------------------------------------------------------------------------------
uint8_t SPI_Write(uint8_t u8Byte)
{
	SPI_SelectW5100Device();
	SPI_DATA_REGISTER = u8Byte;				/* Write byte */
  // Wait for transmission complete
  while ( !(SPSR & (1<<SPIF))) {;}
  return SPI_DATA_REGISTER;
  SPI_UnselectW5100Device();
}


//---------------------------------------------------------------------------------------
// Name:        SPI_Read
// Param:		-
// Return:      Byte received on SPI interface
//
// Description:	This function is used to read a byte from the SPI device
//---------------------------------------------------------------------------------------
uint8_t SPI_Read(void)
{
	SPI_DATA_REGISTER = 0xFF;				/* To Read, you have to write */
	while ( !(SPSR & (1<<SPIF))) {;}		/* Wait */
	return SPI_DATA_REGISTER;				/* Return the byte received */
}


//---------------------------------------------------------------------------------------
// Name:        SPI_WriteAndRead
// Param:		Byte: byte to write on SPI interface
// Return:      Byte received on SPI interface
//
// Description:	This function is used to write a byte on SPI interface and read the one
//				sent by the SPI device
//---------------------------------------------------------------------------------------
uint8_t SPI_WriteAndRead(uint8_t u8Byte)
{
	SPI_DATA_REGISTER = u8Byte;				/* Write byte */
	while ( !(SPSR & (1<<SPIF))) {;}		/* Wait */
	return SPI_DATA_REGISTER;				/* Return the byte received */ 
}


//---------------------------------------------------------------------------------------
// Name:        SPI_SelectW5100Device
// Param:		-
// Return:      -
//
// Description:	This function selects the W5100 device
//---------------------------------------------------------------------------------------
void SPI_SelectW5100Device(void)
{
	PORT_MMC_CS |= ( 1 << PIN_MMC_CS);	// MMC CS set to 1
	PORT_W51_CS &= ~( 1 << PIN_W51_CS);	// W51 CS set to 1
	_delay_us(5);
}


//---------------------------------------------------------------------------------------
// Name:        SPI_UnselectW5100Device
// Param:		-
// Return:      -
//
// Description:	This function unselects the W5100 device
//---------------------------------------------------------------------------------------
void SPI_UnselectW5100Device(void)
{
	PORT_MMC_CS |= ( 1 << PIN_MMC_CS);	// MMC CS set to 1
	PORT_W51_CS |= ( 1 << PIN_W51_CS);	// W51 CS set to 1
}


//---------------------------------------------------------------------------------------
// Name:        SPI_SelectSDCard
// Param:		-
// Return:      -
//
// Description:	This function selects the SD Card
//---------------------------------------------------------------------------------------
void SPI_SelectSDCard(void)
{
	PORT_MMC_CS &= ~( 1 << PIN_MMC_CS);	// MMC CS set to 0
	PORT_W51_CS |= ( 1 << PIN_W51_CS);	// W51 CS set to 1
}


//---------------------------------------------------------------------------------------
// Name:        SPI_UnselectSDCard
// Param:		-
// Return:      -
//
// Description:	This function unselects the SD Card
//---------------------------------------------------------------------------------------
void SPI_UnselectSDCard(void)
{
	PORT_MMC_CS |= ( 1 << PIN_MMC_CS);	// MMC CS set to 1
	PORT_W51_CS |= ( 1 << PIN_W51_CS);	// W51 CS set to 1
}


//---------------------------------------------------------------------------------------
// Name:        SPI_IsSDCardPresent
// Param:		-
// Return:      TRUE if the SD Card is present, FALSE if not
//
// Description:	This function is used to know if the SD card is present
//---------------------------------------------------------------------------------------
uint8_t SPI_IsSDCardPresent(void)
{
	return 1; /* Not used on the board, return default value */
}


//---------------------------------------------------------------------------------------
// Name:        SPI_IsSDCardWriteProtected
// Param:		-
// Return:      TRUE if the SD Card is write protected, FALSE if not
//
// Description:	This function is used to know if the SD card is write protected
//---------------------------------------------------------------------------------------
uint8_t SPI_IsSDCardWriteProtected(void)
{
	return 0; /* Not used on the board, return default value */
}
