/////////////////////////////////////////////////////////////////////////////////////////
//
// Wiznet W5100 Low-Level Interface
//
// --------------------------------------------------------------------------------------
//
// Filename:      w5100.c
// Version:       1.0
// Date:          15/10/2011
// Author:        Joel Guittet - http://myfreescalewebpage.free.fr
//
/////////////////////////////////////////////////////////////////////////////////////////
//
// Description
//
// This file is used to communicate with the W5100 device over the SPI interface. 
//
/////////////////////////////////////////////////////////////////////////////////////////
//
// Revisions
//
// Version	| Author		| Description
// --------------------------------------------------------------------------------------
//			|				| 
//
/////////////////////////////////////////////////////////////////////////////////////////


//---------------------------------------------------------------------------------------
// Includes
//---------------------------------------------------------------------------------------

#include "w5100.h"
#include "Wait.h"
#include "spi.h"
#include "USART.h"

//---------------------------------------------------------------------------------------
// Name:        W5100_Init
// Param:		-
// Return:      -
//
// Description:	This function initializes the W5100 device
//---------------------------------------------------------------------------------------
void W5100_Init(void)
{
	/* Select W5100 */
	SPI_SelectW5100Device();
		
	/* Small delay - Added to leave time to W5100 on start up - Do not forget to reset the watchdog (if used) */
	WaitMs(50);
		
	/* Initialize the SPI interface */
	SPI_Init_HighRate();

	/* Reset W5100 device */
	W5100_WriteByte(W5100_MR, MR_RST);
	
	WaitMs(1);
	
	/* Set memory size for each socket (memory size are defined in w5100.h) */
	W5100_WriteByte(W5100_RMSR, Sn_RMSR(0, W5100_RX_SOCKET_0_SIZE) | Sn_RMSR(1, W5100_RX_SOCKET_1_SIZE) | Sn_RMSR(2, W5100_RX_SOCKET_2_SIZE) | Sn_RMSR(3, W5100_RX_SOCKET_3_SIZE));
	W5100_WriteByte(W5100_TMSR, Sn_TMSR(0, W5100_TX_SOCKET_0_SIZE) | Sn_TMSR(1, W5100_TX_SOCKET_1_SIZE) | Sn_TMSR(2, W5100_TX_SOCKET_2_SIZE) | Sn_TMSR(3, W5100_TX_SOCKET_3_SIZE));
		
	/* Set MAC Address */
	W5100_WriteByte(W5100_SHAR0, W5100_MAC_ADDRESS_0);
	W5100_WriteByte(W5100_SHAR1, W5100_MAC_ADDRESS_1);
	W5100_WriteByte(W5100_SHAR2, W5100_MAC_ADDRESS_2);
	W5100_WriteByte(W5100_SHAR3, W5100_MAC_ADDRESS_3);
	W5100_WriteByte(W5100_SHAR4, W5100_MAC_ADDRESS_4);
	W5100_WriteByte(W5100_SHAR5, W5100_MAC_ADDRESS_5);
	
#if (W5100_DHCP == 0)
	
	/* Set Gateway IP Address  */
	W5100_WriteByte(W5100_GAR0, W5100_GTW_IP_ADDRESS_0);
	W5100_WriteByte(W5100_GAR1, W5100_GTW_IP_ADDRESS_1);
	W5100_WriteByte(W5100_GAR2, W5100_GTW_IP_ADDRESS_2);
	W5100_WriteByte(W5100_GAR3, W5100_GTW_IP_ADDRESS_3);
	
	/* Set Subnet Mask */
	W5100_WriteByte(W5100_SUBR0, W5100_SUBNET_MASK_0);
	W5100_WriteByte(W5100_SUBR1, W5100_SUBNET_MASK_1);
	W5100_WriteByte(W5100_SUBR2, W5100_SUBNET_MASK_2);
	W5100_WriteByte(W5100_SUBR3, W5100_SUBNET_MASK_3);
	
	/* Set my (static) IP Address */
	W5100_WriteByte(W5100_SIPR0, W5100_OWN_IP_ADDRESS_0);
	W5100_WriteByte(W5100_SIPR1, W5100_OWN_IP_ADDRESS_1);
	W5100_WriteByte(W5100_SIPR2, W5100_OWN_IP_ADDRESS_2);
	W5100_WriteByte(W5100_SIPR3, W5100_OWN_IP_ADDRESS_3);
	
#else
	
	/* Set Gateway IP Address  */
	W5100_WriteByte(W5100_GAR0, 0x00);
	W5100_WriteByte(W5100_GAR1, 0x00);
	W5100_WriteByte(W5100_GAR2, 0x00);
	W5100_WriteByte(W5100_GAR3, 0x00);
	
	/* Set Subnet Mask */
	W5100_WriteByte(W5100_SUBR0, 0x00);
	W5100_WriteByte(W5100_SUBR1, 0x00);
	W5100_WriteByte(W5100_SUBR2, 0x00);
	W5100_WriteByte(W5100_SUBR3, 0x00);
	
	/* Set my (static) IP Address */
	W5100_WriteByte(W5100_SIPR0, 0x00);
	W5100_WriteByte(W5100_SIPR1, 0x00);
	W5100_WriteByte(W5100_SIPR2, 0x00);
	W5100_WriteByte(W5100_SIPR3, 0x00);
	
#endif
	/* Unselect W5100 */
	SPI_UnselectW5100Device();
}


//---------------------------------------------------------------------------------------
// Name:        W5100_WriteByte
// Param:		Address: address where to write
//				WriteData: data to write
// Return:      -
//
// Description:	This function is used to send data to the W5100 device throw the SPI interface
//---------------------------------------------------------------------------------------
void W5100_WriteByte(uint16_t u16Address, uint8_t u8WriteData)
{
	uint8_t u8ReadData;
	
	/* Select W5100 */
	SPI_SelectW5100Device();
	
	/* Write Command, W5100 return 0x00 */
	u8ReadData = SPI_WriteAndRead(W5100_WRITE);
	printf(" %i ", SPI_DATA_REGISTER);
	printf("   %i   ", SPI_STATUS_REGISTER);
	printf("   %i   ", SPI_CTRL_REGISTER);
	printf("   %i   ", PORTB);
	printf("   %i   ", PORTG);
	
	/* Write address (high byte), W5100 returns 0x01 */
	u8ReadData = SPI_WriteAndRead((u16Address & 0xFF00) >> 8);
	printf(" %i ", u8ReadData);
	
	/* Write address (low byte), W5100 returns 0x02 */
	u8ReadData = SPI_WriteAndRead(u16Address & 0x00FF);
	printf(" %i ", u8ReadData);
	
	/* Write data, W5100 returns 0x03 */
	u8ReadData = SPI_WriteAndRead(u8WriteData);	
	printf(" %i ", u8ReadData);
	
	/* Unselect W5100 */
	SPI_UnselectW5100Device();
}


//---------------------------------------------------------------------------------------
// Name:        W5100_ReadByte
// Param:		Address: address where to read
// Return:      Data returned by W5100 device
//
// Description:	This function is used to get data from the W5100 device throw the SPI interface
//---------------------------------------------------------------------------------------
uint8_t W5100_ReadByte(uint16_t u16Address)
{
	uint8_t u8ReadData;
			
	/* Select W5100 */
	SPI_SelectW5100Device();
	
	/* Write Command, W5100 return 0x00 */
	u8ReadData = SPI_WriteAndRead(W5100_READ);
	printf("%i", u8ReadData);
		
	/* Write address (high byte), W5100 returns 0x01 */
	u8ReadData = SPI_WriteAndRead((u16Address & 0xFF00) >> 8);
	printf("%i", u8ReadData);
		
	/* Write address (low byte), W5100 returns 0x02 */
	u8ReadData = SPI_WriteAndRead(u16Address & 0x00FF);
	printf("%i", u8ReadData);
		
	/* Write data (any value), W5100 returns data */
	u8ReadData = SPI_WriteAndRead(0);
	printf("%i", u8ReadData);
		
	/* Unselect W5100 */
	SPI_UnselectW5100Device();
	
	return (u8ReadData);
}
