/////////////////////////////////////////////////////////////////////////////////////////
//
// Wiznet W5100 Low-Level Interface
//
// --------------------------------------------------------------------------------------
//
// Filename:      w5100.h
// Version:       1.0
// Date:          15/10/2011
// Author:        Joel Guittet - http://myfreescalewebpage.free.fr
//
/////////////////////////////////////////////////////////////////////////////////////////
//
// Revisions
//
// Version	| Author		| Description
// --------------------------------------------------------------------------------------
//			|				| 
//
/////////////////////////////////////////////////////////////////////////////////////////


#ifndef W5100_H_
#define W5100_H_


//---------------------------------------------------------------------------------------
// Includes
//---------------------------------------------------------------------------------------

#include <avr/io.h>
#include <stdint.h>
#include "spi.h"


//---------------------------------------------------------------------------------------
// W5100 Sockets Size Definitions (1KB, 2KB, 4KB or 8KB, can be modified to match software requirements)
//---------------------------------------------------------------------------------------

#define W5100_TX_SOCKET_0_SIZE				W5100_SOCKET_SIZE_2KB
#define W5100_TX_SOCKET_1_SIZE				W5100_SOCKET_SIZE_2KB
#define W5100_TX_SOCKET_2_SIZE				W5100_SOCKET_SIZE_2KB
#define W5100_TX_SOCKET_3_SIZE				W5100_SOCKET_SIZE_2KB

#define W5100_RX_SOCKET_0_SIZE				W5100_SOCKET_SIZE_2KB
#define W5100_RX_SOCKET_1_SIZE				W5100_SOCKET_SIZE_2KB
#define W5100_RX_SOCKET_2_SIZE				W5100_SOCKET_SIZE_2KB
#define W5100_RX_SOCKET_3_SIZE				W5100_SOCKET_SIZE_2KB


//---------------------------------------------------------------------------------------
// W5100 MAC, DHCP, IP and GTW definitions (can be modified to match network requirements)
//---------------------------------------------------------------------------------------

/* MAC Address: 00:08:DC:00:00:01 */
#define W5100_MAC_ADDRESS_0					(0x00)
#define W5100_MAC_ADDRESS_1					(0x08)
#define W5100_MAC_ADDRESS_2					(0xDC)
#define W5100_MAC_ADDRESS_3					(0x00)
#define W5100_MAC_ADDRESS_4					(0x00)
#define W5100_MAC_ADDRESS_5					(0x01)

/* Using DHCP - Set to 1 to use DHCP, or 0 to configure network settings yourself */
#define W5100_DHCP 							(1)

/* The following parameters are defined only if you are not using DHCP */
#if (W5100_DHCP == 0)

/* Gateway IP Address: 10.0.0.138  */
#define W5100_GTW_IP_ADDRESS_0				(10)
#define W5100_GTW_IP_ADDRESS_1				(0)
#define W5100_GTW_IP_ADDRESS_2				(0)
#define W5100_GTW_IP_ADDRESS_3				(138)

/* Subnet Mask: 255.255.255.0 */
#define W5100_SUBNET_MASK_0					(255)
#define W5100_SUBNET_MASK_1					(255)
#define W5100_SUBNET_MASK_2					(255)
#define W5100_SUBNET_MASK_3					(0)

/* My (static) IP Address: 10.0.0.17 */
#define W5100_OWN_IP_ADDRESS_0				(10)
#define W5100_OWN_IP_ADDRESS_1				(0)
#define W5100_OWN_IP_ADDRESS_2				(0)
#define W5100_OWN_IP_ADDRESS_3				(17)

#endif


//---------------------------------------------------------------------------------------
// W5100 Registers Addresses Definitions
//---------------------------------------------------------------------------------------

/* Common Registers address */
#define W5100_COMMON_REGISTERS_ADDRESS		(0x0000)

/* Mode register */
#define W5100_MR							(W5100_COMMON_REGISTERS_ADDRESS + 0x0000)

/* Gateway Address registers */
#define W5100_GAR0							(W5100_COMMON_REGISTERS_ADDRESS + 0x0001)
#define W5100_GAR1							(W5100_COMMON_REGISTERS_ADDRESS + 0x0002)
#define W5100_GAR2							(W5100_COMMON_REGISTERS_ADDRESS + 0x0003)
#define W5100_GAR3							(W5100_COMMON_REGISTERS_ADDRESS + 0x0004)

/* Subnet Mask Address registers */
#define W5100_SUBR0							(W5100_COMMON_REGISTERS_ADDRESS + 0x0005)
#define W5100_SUBR1							(W5100_COMMON_REGISTERS_ADDRESS + 0x0006)
#define W5100_SUBR2							(W5100_COMMON_REGISTERS_ADDRESS + 0x0007)
#define W5100_SUBR3							(W5100_COMMON_REGISTERS_ADDRESS + 0x0008)

/* Source Hardware Address (MAC) registers */
#define W5100_SHAR0							(W5100_COMMON_REGISTERS_ADDRESS + 0x0009)
#define W5100_SHAR1							(W5100_COMMON_REGISTERS_ADDRESS + 0x000A)
#define W5100_SHAR2							(W5100_COMMON_REGISTERS_ADDRESS + 0x000B)
#define W5100_SHAR3							(W5100_COMMON_REGISTERS_ADDRESS + 0x000C)
#define W5100_SHAR4							(W5100_COMMON_REGISTERS_ADDRESS + 0x000D)
#define W5100_SHAR5							(W5100_COMMON_REGISTERS_ADDRESS + 0x000E)

/* Source IP Address registers */
#define W5100_SIPR0							(W5100_COMMON_REGISTERS_ADDRESS + 0x000F)
#define W5100_SIPR1							(W5100_COMMON_REGISTERS_ADDRESS + 0x0010)
#define W5100_SIPR2							(W5100_COMMON_REGISTERS_ADDRESS + 0x0011)
#define W5100_SIPR3							(W5100_COMMON_REGISTERS_ADDRESS + 0x0012)

/* Interrupt register */
#define W5100_IR							(W5100_COMMON_REGISTERS_ADDRESS + 0x0015)

/* Interrupt Mask register */
#define W5100_IMR							(W5100_COMMON_REGISTERS_ADDRESS + 0x0016)

/* Retry Time registers */
#define W5100_RTR0							(W5100_COMMON_REGISTERS_ADDRESS + 0x0017)
#define W5100_RTR1							(W5100_COMMON_REGISTERS_ADDRESS + 0x0018)

/* Retry Count register */
#define W5100_RCR							(W5100_COMMON_REGISTERS_ADDRESS + 0x0019)

/* Receive Memory Size register */
#define W5100_RMSR							(W5100_COMMON_REGISTERS_ADDRESS + 0x001A)

/* Transmit Memory Size register */
#define W5100_TMSR							(W5100_COMMON_REGISTERS_ADDRESS + 0x001B)

/* Authentication Type In PPPoE registers */
#define W5100_PATR0							(W5100_COMMON_REGISTERS_ADDRESS + 0x001C)
#define W5100_PATR1							(W5100_COMMON_REGISTERS_ADDRESS + 0x001D)

/* PPP LCP registers */
#define W5100_PTIMER 						(W5100_COMMON_REGISTERS_ADDRESS + 0x0028)
#define W5100_PMAGIC 						(W5100_COMMON_REGISTERS_ADDRESS + 0x0029)

/* Unreachable IP Address registers */
#define W5100_UIPR0							(W5100_COMMON_REGISTERS_ADDRESS + 0x002A)
#define W5100_UIPR1							(W5100_COMMON_REGISTERS_ADDRESS + 0x002B)
#define W5100_UIPR2							(W5100_COMMON_REGISTERS_ADDRESS + 0x002C)
#define W5100_UIPR3							(W5100_COMMON_REGISTERS_ADDRESS + 0x002D)

/* Unreachable Port registers */
#define W5100_UPORT0						(W5100_COMMON_REGISTERS_ADDRESS + 0x002E)
#define W5100_UPORT1						(W5100_COMMON_REGISTERS_ADDRESS + 0x002F)

/* Socket Registers address */
#define W5100_SOCKET_REGISTERS_ADDRESS		(W5100_COMMON_REGISTERS_ADDRESS + 0x0400)

/* Size of Sockets registers map */
#define W5100_SOCKET_REGISTERS_SIZE			(0x0100)

/* Socket Mode register */
#define W5100_Sn_MR(u8Socket)				(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0000)

/* Socket Command register */
#define W5100_Sn_CR(u8Socket)				(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0001)

/* Socket Interupt register */
#define W5100_Sn_IR(u8Socket)				(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0002)

/* Socket Status register */
#define W5100_Sn_SR(u8Socket)				(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0003)

/* Socket Source Port registers */
#define W5100_Sn_PORT0(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0004)
#define W5100_Sn_PORT1(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0005)

/* Socket Destination Hardware Address registers */
#define W5100_Sn_DHAR0(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0006)
#define W5100_Sn_DHAR1(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0007)
#define W5100_Sn_DHAR2(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0008)
#define W5100_Sn_DHAR3(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0009)
#define W5100_Sn_DHAR4(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x000A)
#define W5100_Sn_DHAR5(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x000B)

/* Socket Destination IP Address registers */
#define W5100_Sn_DIPR0(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x000C)
#define W5100_Sn_DIPR1(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x000D)
#define W5100_Sn_DIPR2(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x000E)
#define W5100_Sn_DIPR3(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x000F)

/* Socket Destination Port registers */
#define W5100_Sn_DPORT0(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0010)
#define W5100_Sn_DPORT1(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0011)

/* Socket Maximum Segment Size registers */
#define W5100_Sn_MSSR0(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0012)
#define W5100_Sn_MSSR1(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0013)

/* Socket Protocol in IP Raw Mode register */
#define W5100_Sn_PROTO(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0014)

/* Socket IP Type Of Service (TOS) register */
#define W5100_Sn_TOS(u8Socket)				(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0015)

/* Socket IP Time To Live (TTL) register */
#define W5100_Sn_TTL(u8Socket)				(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0016)

/* Socket TX Free Size registers */
#define W5100_Sn_TX_FSR0(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0020)
#define W5100_Sn_TX_FSR1(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0021)

/* Socket TX Read Pointer registers */
#define W5100_Sn_TX_RD0(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0022)
#define W5100_Sn_TX_RD1(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0023)

/* Socket TX Write Pointer registers */
#define W5100_Sn_TX_WR0(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0024)
#define W5100_Sn_TX_WR1(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0025)

/* Socket RX Received Size registers */
#define W5100_Sn_RX_RSR0(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0026)
#define W5100_Sn_RX_RSR1(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0027)

/* Socket RX Read Pointer registers */
#define W5100_Sn_RX_RD0(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0028)
#define W5100_Sn_RX_RD1(u8Socket)			(W5100_SOCKET_REGISTERS_ADDRESS + u8Socket * W5100_SOCKET_REGISTERS_SIZE + 0x0029)

/* Sockets Transmit Size definitions in bytes */
#define W5100_TX_SOCKET_0_SIZE_BYTES		(1024 << W5100_TX_SOCKET_0_SIZE)
#define W5100_TX_SOCKET_1_SIZE_BYTES		(1024 << W5100_TX_SOCKET_1_SIZE)
#define W5100_TX_SOCKET_2_SIZE_BYTES		(1024 << W5100_TX_SOCKET_2_SIZE)
#define W5100_TX_SOCKET_3_SIZE_BYTES		(1024 << W5100_TX_SOCKET_3_SIZE)

/* Sockets Receive Size definitions in bytes */
#define W5100_RX_SOCKET_0_SIZE_BYTES		(1024 << W5100_RX_SOCKET_0_SIZE)
#define W5100_RX_SOCKET_1_SIZE_BYTES		(1024 << W5100_RX_SOCKET_1_SIZE)
#define W5100_RX_SOCKET_2_SIZE_BYTES		(1024 << W5100_RX_SOCKET_2_SIZE)
#define W5100_RX_SOCKET_3_SIZE_BYTES		(1024 << W5100_RX_SOCKET_3_SIZE)

/* Transmit Address in the Memory Map */
#define W5100_TX_MEMORY_BASE_ADDRESS		(0x4000)
#define W5100_TX_MEMORY_SOCKET_0_ADDRESS	(W5100_TX_MEMORY_BASE_ADDRESS)
#define W5100_TX_MEMORY_SOCKET_1_ADDRESS	(W5100_TX_MEMORY_BASE_ADDRESS + W5100_TX_SOCKET_0_SIZE_BYTES)
#define W5100_TX_MEMORY_SOCKET_2_ADDRESS	(W5100_TX_MEMORY_BASE_ADDRESS + W5100_TX_SOCKET_0_SIZE_BYTES + W5100_TX_SOCKET_1_SIZE_BYTES)
#define W5100_TX_MEMORY_SOCKET_3_ADDRESS	(W5100_TX_MEMORY_BASE_ADDRESS + W5100_TX_SOCKET_0_SIZE_BYTES + W5100_TX_SOCKET_1_SIZE_BYTES + W5100_TX_SOCKET_2_SIZE_BYTES)

/* Receive Address in the Memory Map */
#define W5100_RX_MEMORY_BASE_ADDRESS		(0x6000)
#define W5100_RX_MEMORY_SOCKET_0_ADDRESS	(W5100_RX_MEMORY_BASE_ADDRESS)
#define W5100_RX_MEMORY_SOCKET_1_ADDRESS	(W5100_RX_MEMORY_BASE_ADDRESS + W5100_RX_SOCKET_0_SIZE_BYTES)
#define W5100_RX_MEMORY_SOCKET_2_ADDRESS	(W5100_RX_MEMORY_BASE_ADDRESS + W5100_RX_SOCKET_0_SIZE_BYTES + W5100_RX_SOCKET_1_SIZE_BYTES)
#define W5100_RX_MEMORY_SOCKET_3_ADDRESS	(W5100_RX_MEMORY_BASE_ADDRESS + W5100_RX_SOCKET_0_SIZE_BYTES + W5100_RX_SOCKET_1_SIZE_BYTES + W5100_RX_SOCKET_2_SIZE_BYTES)


//---------------------------------------------------------------------------------------
// W5100 Common Registers Definitions
//---------------------------------------------------------------------------------------

/* Mode register values */
#define MR_RST								(0x80)					/* Reset */
#define MR_PB								(0x10) 					/* Ping block mode */
#define MR_PPPOE							(0x08) 					/* PPPoE mode */
#define MR_AI								(0x02) 					/* Address Auto-Increment in Indirect Bus I/F */
#define MR_IND								(0x01) 					/* Enable Indirect Bus I/F mode */

/* IR register values */
#define IR_CONFLICT							(0x80) 					/* IP Confict */
#define IR_UNREACH							(0x40) 					/* Destination Unreachable */
#define IR_PPPoE							(0x20) 					/* PPPoE Connection Close */
#define IR_SOCK(u8Socket)					(0x01 << u8Socket) 		/* Socket Interrupt */

/* Receive and Transmit Socket Memory size */
#define W5100_SOCKET_SIZE_1KB				(0x00)					/* 1KB */
#define W5100_SOCKET_SIZE_2KB				(0x01)					/* 2KB */
#define W5100_SOCKET_SIZE_4KB				(0x02)					/* 4KB */
#define W5100_SOCKET_SIZE_8KB				(0x03)					/* 8KB */

/* Receive Memory Size register values */
#define Sn_RMSR(u8Socket, u8Size)			(u8Size << 2 * u8Socket)

/* Transmit Memory Size register */	
#define Sn_TMSR(u8Socket, u8Size)			(u8Size << 2 * u8Socket)

/* Sn_MR register values */
#define Sn_MR_CLOSE							(0x00)					/* Unused u8Socket */
#define Sn_MR_TCP							(0x01)					/* TCP */
#define Sn_MR_UDP							(0x02)					/* UDP */
#define Sn_MR_IPRAW							(0x03)					/* IP LAYER RAW SOCK */
#define Sn_MR_MACRAW						(0x04)					/* MAC LAYER RAW SOCK */
#define Sn_MR_PPPOE							(0x05)					/* PPPoE */
#define Sn_MR_ND							(0x20)					/* Enable No Delayed ACK option */
#define Sn_MR_MULTI							(0x80)					/* Enable multicasting */

/* Sn_CR register values */
#define Sn_CR_OPEN							(0x01)					/* Open u8Socket */
#define Sn_CR_LISTEN						(0x02)					/* Wait connection request (TCP server mode) */
#define Sn_CR_CONNECT						(0x04)					/* Send connection request (TCP client mode) */
#define Sn_CR_DISCON						(0x08)					/* Send closing connection request (TCP mode) */
#define Sn_CR_CLOSE							(0x10)					/* Close u8Socket */
#define Sn_CR_SEND							(0x20)					/* Send data */
#define Sn_CR_SEND_MAC						(0x21)					/* Send data with MAC address, so without ARP process (UDP mode) */
#define Sn_CR_SEND_KEEP						(0x22)					/* Send keep alive message, it checks the connection status (TCP mode) */
#define Sn_CR_RECV							(0x40)					/* Receive data */

/* Sn_IR register values */
#define Sn_IR_SEND_OK						(0x10)					/* Complete sending */
#define Sn_IR_TIMEOUT						(0x08)					/* Assert timeout */
#define Sn_IR_RECV							(0x04)					/* Receiving data */
#define Sn_IR_DISCON						(0x02)					/* Closed u8Socket */
#define Sn_IR_CON							(0x01)					/* Established connection */

/* Sn_SR register values */
#define Sn_SR_SOCK_CLOSED					(0x00)					/* Closed */
#define Sn_SR_SOCK_ARP						(0x01)					/* ARP request sent */
#define Sn_SR_SOCK_INIT 					(0x13)					/* Initialization status (TCP connection) */
#define Sn_SR_SOCK_LISTEN					(0x14)					/* Listen status (TCP connection) */
#define Sn_SR_SOCK_SYNSENT	   				(0x15)					/* Connecting status */
#define Sn_SR_SOCK_SYNRECV		   			(0x16)					/* Connecting status */
#define Sn_SR_SOCK_ESTABLISHED				(0x17)					/* Connected (TCP connection) */
#define Sn_SR_SOCK_FIN_WAIT					(0x18)					/* Closing status */
#define Sn_SR_SOCK_CLOSING		   			(0x1A)					/* Closing status */
#define Sn_SR_SOCK_TIME_WAIT				(0x1B)					/* Closing status */
#define Sn_SR_SOCK_CLOSE_WAIT				(0x1C)					/* Closing status */
#define Sn_SR_SOCK_LAST_ACK					(0x1D)					/* closing status */
#define Sn_SR_SOCK_UDP				  		(0x22)					/* Open command received (UDP connection) */
#define Sn_SR_SOCK_IPRAW			   		(0x32)					/* Open command received (IPRAW connection) */
#define Sn_SR_SOCK_MACRAW			   		(0x42)					/* Open command received (MACRAW connection) */
#define Sn_SR_SOCK_PPPOE					(0x5F)					/* Open command received (PPPoE connection) */


//---------------------------------------------------------------------------------------
// W5100 Command Definitions
//---------------------------------------------------------------------------------------

#define W5100_READ        					(0x0F)
#define W5100_WRITE       					(0xF0)


//---------------------------------------------------------------------------------------
// Prototypes
//---------------------------------------------------------------------------------------

void W5100_Init(void);
void W5100_WriteByte(uint16_t u16Address, uint8_t u8WriteData);
uint8_t W5100_ReadByte(uint16_t u16Address);


#endif
