

#include "hardware_t.h"
#include "tc_vars.h"
#include "flash_t.h"
#include "display_t.h"
#include "userif_t.h"
#include "commif_t.h"
#include "signal_t.h"
#include "opsys_t.h"

unsigned long AMDFlash::_ConfigSlot;									// slot address offset
//unsigned int AMDFlash::_SlotNumber;									// only needed for debugging
bool AMDFlash::_BackupConfig = false;

//#####################################################################################################################################################
/* Get the manufacturer ID and the Device ID

	AMD 	  01
	Macronix  C2
*/
// Return value: Device ID and manufacturer ID in the first and second byte of the 16 bit return value
unsigned short AMDFlash::GetDeviceID(void)
{
	volatile unsigned char *fb = (unsigned char *) nasys_main_flash;
	unsigned short result = 0;
	unsigned char ManID;
	unsigned char DevID;

 	//autoselect    XXX     AA      XXX     55      XXX     90
	*fb = 0xAA; 	// 1st cycle - unlock
	*fb = 0x55; 	// 2nd cycle - unlock
	*fb = 0x90; 	// 3rd cycle - autoselect

	//nr_delay_us(10);

	//get manufacturer ID
	ManID = fb[0];
	
	//get device ID
	DevID = fb[1];

	//reset flash      XXX     F0
	*fb = 0xF0;

	result = (short)ManID;
	result = result << 8;
	result = result | (short)DevID;

	//printf("ManID %x    DevID %x \r\n",ManID, DevID);

  	return result;
}


//#####################################################################################################################################################
// Unlock bypass mode, enabling fast writes.
static inline void unlock_bypass_mode(void)
{
	volatile unsigned char *flash_base = (unsigned char *) nasys_main_flash;

	flash_base[0xAAA] = 0xAA; // unlock bypass command - cycle 1
	flash_base[0x555] = 0x55; // unlock bypass command - cycle 2
	flash_base[0xAAA] = 0x20; // unlock bypass command - cycle 3
}
//#####################################################################################################################################################
// Turn bypass mode off, disabling fast writes and enabling normal function.
static inline void reset_bypass_mode(void)
{
	volatile unsigned char *flash_base = (unsigned char *) nasys_main_flash;

  	*flash_base = 0x90; // exit unlock bypass reset command - cycle 1
  	*flash_base = 0x0;  // exit unlock bypass reset command - cycle 2
}
//#####################################################################################################################################################
// Read the given address until two successive reads yield the
// same value.
// Return value: 1 on success, 0 on timeout failure.
static inline int await_write_complete(volatile unsigned char *addr, const unsigned char correct_value)
{
        // Typical programming time in accellerated mode is 4µs,
        // maximum is 120µs. With safety margin we wait 150µs max.
	unsigned long timeout = 150;
	int return_value = 0; // Pessimistic return value.
	

	while (timeout)
	{
		// While the flash is working on program data, read accesses return
		// "status" instead of data.  Status bit 7 is the complement of the
		// data being written.  When the program operation is complete and
		// successful, the written data is returned.  So, read the written
		// address until it equals the data written.
		nr_delay_us(1);

		if (*addr == correct_value) break;
		timeout--;
	}

	if (timeout) return_value = 1;		//BF return with success
	else
	printf("byte write error");

	return return_value;
}
//#####################################################################################################################################################
// Write val to the given flash address, in bypass mode (assumes
// that bypass mode has been enabled already).
// Return value: 1 on success, 0 on failure.
static inline int nr_flash_write_bypass(volatile unsigned char *addr, unsigned char val)
{
	volatile unsigned char *flash_base = (unsigned char *) nasys_main_flash;
	int result = -1;
	int timeout;
	
	nm_dcache_invalidate_line(addr);
	nm_icache_invalidate_line(addr);
		
	*flash_base = 0xA0;   // unlock bypass program command - 1st cycle
	*addr = val;          // program address and data    - 2nd cycle

	result = await_write_complete(addr,val);

	return result;
}
//#####################################################################################################################################################
//BF #010 redesigned
// Erase the flash sector at sector_address.
// Return value: 1 on success, 0 on failure.
int AMDFlash::nr_flash_erase_sector(unsigned long *sector_address)
{
	volatile unsigned char *fb = (unsigned char *) nasys_main_flash;
	volatile unsigned char *sa = (unsigned char *) (sector_address + na_flash_8MByte_base);
	int result = 0;
	int i;
	int timeout;

	//printf("AMDFlash::nr_flash_erase_sector\r\n");	

	// Disable the Interrupts is recommended by data sheet to avoid
	// errors in the command sequence -> AM29LV065d
	// -> causes problems with USTB-Mode -> timer 2

//	Hardware::Disable_All_Interrupts();	

	*fb = 0xAA; 	// 1st cycle - unlock
	*fb = 0x55; 	// 2nd cycle - unlock
	*fb = 0x80; 	// 3rd cycle - set-up
	*fb = 0xAA; 	// 4th cycle - unlock
	*fb = 0x55; 	// 5th cycle - unlock

	*sa = 0x30; 	// 6th cycle - sector erase command
	
//	Hardware::Setup_Interrupts();				// setup interrupts to normal working mode needs about 10ns
/*
	// new check routine from OSOZ driver
	if (na_flash_ry_by->np_piodata == 0)	// check if command was successful -> flash busy
	{
		// wait until flash ready/busy is high again -> flash ready
		for (timeout = 15000;timeout > 0 && na_flash_ry_by->np_piodata == 0; timeout--)
		{ nr_delay(1);	}	// wait 1 ms
	
		if (timeout > 0) result = 0;	// timeout not reached  -> erase is complete	
	}
	else
	{ result = -1; }
*/

	// check if memory contains 0xFF
	for ( timeout=15000;timeout > 0 && sa[0x7777] != 0xFF;timeout --)		// check if memory contains 0xFF
	{ nr_delay(1); }	// wait 1 ms

	if (timeout > 0) result = 1;	// timeout not reached  -> erase is complete	

	if(result == 0){ printf("Error while erasing flash sector\r\n"); }

  	return result;
}
//#####################################################################################################################################################
// Write a buffer of data to the flash, using bypass mode.
// Return value: 1 on success, 0 on failure.
// Note: the integer "size" is given as a number of half-words to
// write.   How convenient.  We write this 8-bit-wide flash one byte
// at a time (of course).
int AMDFlash::nr_flash_write_buffer(unsigned long *start_address, unsigned long *buffer, int size)
{
	unsigned char *sa  = (unsigned char *) start_address;
	unsigned char *buf = (unsigned char *) buffer;
	int num_bytes = size * 4;
	int i;
	int result;

	unlock_bypass_mode();

	for (i = 0, result = 1; i < num_bytes && result == 1; ++i)
	{ result = nr_flash_write_bypass(sa + i, buf[i]); }
	
	reset_bypass_mode();

	if(result == 0){ printf("Error while writing buffer to flash\r\n"); }

	return result;
}
//#####################################################################################################################################################
/* BF not needed at this time
// Write val to the given flash address.
// Return value: 1 on success, 0 on failure.
int AMDFlash::amd29lv065d_flash_write_byte(unsigned char *flash_base, unsigned char *addr, unsigned char val)
{

    volatile unsigned char *fb = (unsigned char *) flash_base;
    volatile unsigned char *a = (unsigned char *) addr;
    unsigned char us1, us2;
    int result = 1;

//printf("AMDFlash::amd29lv065d_flash_write_byte\n");
    nm_dcache_invalidate_line(a);
    nm_icache_invalidate_line(a);

#ifdef nasys_main_flash
    if (-1 == (int)fb) fb = (unsigned char *) nasys_main_flash;
#endif // nasys_main_flash

    while (result)
    {
        fb[0x555] = 0xAA; // 1st cycle  addr = XXX, data = AA
        fb[0x333] = 0x55; // 2nd cycle  addr = XXX, data = 55
        fb[0x555] = 0xA0; // 3rd cycle  addr = XXX, data = A0

        *a = val;     // 4th cycle  addr = PA, data = PD

        result = await_write_complete(a,val);
    }
    //if(result) return result;

    us1 = *a;
    if (us1 != val) result = -1;

    return result;
}
*/
//#####################################################################################################################################################
/* BF not needed at this time
int AMDFlash::nr_flash_write(unsigned long *flash_base, unsigned long *addr, unsigned long val)
{
    unsigned char* fb       = (unsigned char *) flash_base;
    unsigned char* a        = (unsigned char *) addr;
    unsigned char  byte_val = val & 0xff;
    int result;
//printf("AMDFlash::nr_flash_write\n");
    result = amd29lv065d_flash_write_byte (fb, a,   byte_val);
     // Nonzero result means error.
    if (result) return result;

    byte_val = (val >> 8) & 0xff;
    result = amd29lv065d_flash_write_byte (fb, a+1, byte_val);
     // Nonzero result means error.
    if (result) return result;

    byte_val = (val >> 16) & 0xff;
    result = amd29lv065d_flash_write_byte (fb, a+2, byte_val);
     // Nonzero result means error.
    if (result) return result;

    byte_val = (val >> 24) & 0xff;
    result = amd29lv065d_flash_write_byte (fb, a+3, byte_val);

    return result;
}
*/
//#####################################################################################################################################################
/* BF not needed at this time
void AMDFlash::Clear_Data_FLASH(void)
{
	volatile unsigned long fb  = (unsigned long) nasys_main_flash;
	int result = 0;
//printf("AMDFlash::Clear_Data_FLASH\n");		
	unsigned long  Data_Index = 0x0000007F;//0x00000070;
	unsigned long  i;
	
	unsigned long  sectorOffset[] =  // AM29LV065d sector map
	{
		0x000000, 0x010000, 0x020000, 0x030000,
		0x040000, 0x050000, 0x060000, 0x070000,
		0x080000, 0x090000, 0x0A0000, 0x0B0000,
		0x0C0000, 0x0D0000, 0x0E0000, 0x0F0000,
		0x100000, 0x110000, 0x120000, 0x130000,
		0x140000, 0x150000, 0x160000, 0x170000,
		0x180000, 0x190000, 0x1A0000, 0x1B0000,
		0x1C0000, 0x1D0000, 0x1E0000, 0x1F0000,
		0x200000, 0x210000, 0x220000, 0x230000,
		0x240000, 0x250000, 0x260000, 0x270000,
		0x280000, 0x290000, 0x2A0000, 0x2B0000,
		0x2C0000, 0x2D0000, 0x2E0000, 0x2F0000,
		0x300000, 0x310000, 0x320000, 0x330000,
		0x340000, 0x350000, 0x360000, 0x370000,
		0x380000, 0x390000, 0x3A0000, 0x3B0000,
		0x3C0000, 0x3D0000, 0x3E0000, 0x3F0000,
		0x400000, 0x410000, 0x420000, 0x430000,
		0x440000, 0x450000, 0x460000, 0x470000,
		0x480000, 0x490000, 0x4A0000, 0x4B0000,
		0x4C0000, 0x4D0000, 0x4E0000, 0x4F0000,
		0x500000, 0x510000, 0x520000, 0x530000,
		0x540000, 0x550000, 0x560000, 0x570000,
		0x580000, 0x590000, 0x5A0000, 0x5B0000,
		0x5C0000, 0x5D0000, 0x5E0000, 0x5F0000,
		0x600000, 0x610000, 0x620000, 0x630000,
		0x640000, 0x650000, 0x660000, 0x670000,
		0x680000, 0x690000, 0x6A0000, 0x6B0000,
		0x6C0000, 0x6D0000, 0x6E0000, 0x6F0000,
		0x700000, 0x710000, 0x720000, 0x730000,
		0x740000, 0x750000, 0x760000, 0x770000,
		0x780000, 0x790000, 0x7A0000, 0x7B0000,
		0x7C0000, 0x7D0000, 0x7E0000, 0x7F0000,
	};
	
	for (i = Data_Index; i < sizeof(sectorOffset) / sizeof(*sectorOffset); ++i)
	{
#ifdef _Debug_	
		printf("Deleting Sector : %x\n", sectorOffset[i]);
#endif		
		volatile unsigned long *sector = (unsigned long *) sectorOffset[i];

		result = nr_flash_erase_sector((unsigned long *) nasys_main_flash, (unsigned long *) sector);		// Clear Sectors
		
		if(result) break;
	}
}
*/
//#####################################################################################################################################################
// BF Save/Recall -> save trace and all settings
// Return value: 1 on success, 0 on failure.
int AMDFlash::WriteTraceToFlash(unsigned long *flash_address)
{
	unsigned long cx;
	unsigned long buf[0x1000];
	int led_value = 0x0000;	

	if (USTB_Mode == USTB_OFF)		// if USTB mode is not active, adjust signal pointers to 16k signal buffers
	{
		S1Ptr = SIGNAL1;
		S2Ptr = SIGNAL2;
		S3Ptr = SIGNAL3;
		S4Ptr = SIGNAL4;
		SMPtr = SIGNALM;
	}

	
	//printf("AMDFlash::WriteTraceToFlash Address %x\n", flash_address);

	if (!nr_flash_erase_sector(flash_address))	// erase config area 
	{ return 0; }
	
	//BF save signal 1
	for (cx = 0; cx < 0x1000; cx++)
	{
		buf[cx] = *(S1Ptr + (cx * 4) + 0) + (*(S1Ptr + (cx * 4) + 1) << 8) + (*(S1Ptr + (cx * 4) + 2) << 16) + (*(S1Ptr + (cx * 4) + 3) << 24);
	}

	if (!nr_flash_write_buffer(flash_address, buf, 0x1000)) { return 0; }

	//BF save signal 2
	for (cx = 0; cx < 0x1000; cx++)
	{
		buf[cx] = *(S2Ptr + (cx * 4) + 0) + (*(S2Ptr + (cx * 4) + 1) << 8) + (*(S2Ptr + (cx * 4) + 2) << 16) + (*(S2Ptr + (cx * 4) + 3) << 24);
	}

	if (!nr_flash_write_buffer(flash_address + 0x1000, buf, 0x1000)) { return 0; }

	if (NumberOfChannels == 4)
	{
		//BF save signal 3
		for (cx = 0; cx < 0x1000; cx++)
		{
			buf[cx] = *(S3Ptr + (cx * 4) + 0) + (*(S3Ptr + (cx * 4) + 1) << 8) + (*(S3Ptr + (cx * 4) + 2) << 16) + (*(S3Ptr + (cx * 4) + 3) << 24);
		}

		if (!nr_flash_write_buffer(flash_address + 0x2000, buf, 0x1000)) { return 0; }

		//BF save signal 4
		for (cx = 0; cx < 0x1000; cx++)
		{
			buf[cx] = *(S4Ptr + (cx * 4) + 0) + (*(S4Ptr + (cx * 4) + 1) << 8) + (*(S4Ptr + (cx * 4) + 2) << 16) + (*(S4Ptr + (cx * 4) + 3) << 24);
		}
	}	

	// save the settings at the end of signal 4 memory area (300 * 4 byte = 1200 byte)
	if (LED_ON[0] == 1) led_value = led_value | 0x0001;		// Select Channel 1
	if (LED_ON[1] == 1) led_value = led_value | 0x0002;		// Select Channel 2
	if (LED_ON[2] == 1) led_value = led_value | 0x0004;		// Select Channel 3
	if (LED_ON[3] == 1) led_value = led_value | 0x0008;		// Select Channel 4
	if (LED_ON[4] == 1) led_value = led_value | 0x0010;		// Select Channel Math
	if (LED_ON[5] == 1) led_value = led_value | 0x0020;		// General
	if (LED_ON[6] == 1) led_value = led_value | 0x0040;		// Cursor						
	if (LED_ON[7] == 1) led_value = led_value | 0x0080;		// Quick Measure
	if (LED_ON[8] == 1) led_value = led_value | 0x0100;		// Edge
	if (LED_ON[9] == 1) led_value = led_value | 0x0200;		// Pattern - Not Used
	if (LED_ON[10] == 1) led_value = led_value | 0x0400;		// Pulse Width
	if (LED_ON[11] == 1) led_value = led_value | 0x0800;		// More - Not Used
	if (LED_ON[12] == 1) led_value = led_value | 0x1000;		// Run/Stop Green
	if (LED_ON[13] == 1) led_value = led_value | 0x2000;		// Run/Stop Red
	if (LED_ON[14] == 1) led_value = led_value | 0x4000;		// Single Red					
	if (LED_ON[15] == 1) led_value = led_value | 0x8000;		// Single Green	
	
	//buf[0x1000 - 0] = (unsigned long) 0x00;			// this address is not writable! Don't use!
	//buf[0x1000 - 1] = (unsigned long) 0xAACC5523;			// Configuration Start Sign
	buf[0x1000 - 2] = (unsigned long) 0x22334455;
	//buf[0x1000 - 3] = (unsigned long) 0x00;		
	//buf[0x1000 - 4] = (unsigned long) tc_version;			// Quattro Version
	//buf[0x1000 - 5] = (unsigned long) tc_subversion;		// SubVerision
	//buf[0x1000 - 6] = (unsigned long) tc_dev_version;		// Revision
	//buf[0x1000 - 7] = (unsigned long) tc_dev_subversion;		// Compile Status
	//buf[0x1000 - 8] = (unsigned long) 0x00;   	
	
	// Timebase
	buf[0x1000 - 10] = (unsigned long) MenuStatus[MENU_TIMEBASE][0];
	buf[0x1000 - 11] = (unsigned long) MenuStatus[MENU_TIMEBASE][1];
	buf[0x1000 - 12] = (unsigned long) MenuStatus[MENU_TIMEBASE][2];
	buf[0x1000 - 13] = (unsigned long) MenuStatus[MENU_TIMEBASE][3];
	buf[0x1000 - 14] = (unsigned long) MenuStatus[MENU_TIMEBASE][4];
	//buf[0x1000 - 15] = (unsigned long) MenuStatus[MENU_DISPLAY][2];	
	buf[0x1000 - 16] = (unsigned long) XY_Mode;
	buf[0x1000 - 17] = (unsigned long) MainTimebase;	
	buf[0x1000 - 18] = (unsigned long) USTB_Mode;
	buf[0x1000 - 19] = (unsigned long) USTB_Mode_bak;
	buf[0x1000 - 20] = (unsigned long) USTB_Dir;	
	buf[0x1000 - 21] = (unsigned long) USTB_Disp;
	buf[0x1000 - 22] = (unsigned long) DelayedTimebase;
	buf[0x1000 - 23] = (unsigned long) VirtualTimebase;
	buf[0x1000 - 24] = (unsigned long) dmode_Window_Offset_Pos;
	buf[0x1000 - 25] = (unsigned long) MenuStatus[MENU_USTB][2];					// USTB buffer size
//	buf[0x1000 - 26] = (unsigned long) Cursor_Delayed_1;
//	buf[0x1000 - 27] = (unsigned long) Cursor_Delayed_2;
//	buf[0x1000 - 28] = (unsigned long) Cursor_Delayed_Size;
	buf[0x1000 - 29] = (unsigned long) MenuStatus[MENU_USTB][0];				     	// ultra slow timebase mode
	buf[0x1000 - 30] = (unsigned long) MenuStatus[MENU_USTB][1];
//    	BF buf[0x1000 - 31] = (unsigned long) ;		-> LED 1					// Trigger LED	
//  	BF buf[0x1000 - 32] = (unsigned long) ;		-> LED 2					// Trigger LED	
    	//buf[0x1000 - 33] = (unsigned long) free;
	//buf[0x1000 - 34] = (unsigned long) free	
	
	// Trigger
	buf[0x1000 - 35] = (unsigned long) MenuStatus[MENU_TRIGGERMODE][0];				// Trigger - Mode
	buf[0x1000 - 36] = (unsigned long) MenuStatus[MENU_TRIGGERMODE][1];				// Trigger - Coupling
	//buf[0x1000 - 37] = (unsigned long) free;				// 
	buf[0x1000 - 38] = (unsigned long) MenuStatus[MENU_TRIGGERMODE][4];				// Triggering - External Probe = 1 : 1  ( muß noch als funktion integriert werden )
	buf[0x1000 - 39] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][3];				// Triggering - TV
	buf[0x1000 - 40] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][0];				// Triggering - positive edge
	buf[0x1000 - 41] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][1];				// Triggering - Source channel
	buf[0x1000 - 42] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][2];				// external triggering
	buf[0x1000 - 43] = (unsigned long) MenuStatus[MENU_PULSEWIDTH][0];				// Triggering - Source Channel
	buf[0x1000 - 44] = (unsigned long) MenuStatus[MENU_PULSEWIDTH][1];				// Triggering - Negative Pulse
	buf[0x1000 - 45] = (unsigned long) MenuStatus[MENU_PULSEWIDTH][2];				// Triggering - bigger - smaller - between
	buf[0x1000 - 46] = (unsigned long) Hardware::_TriggerSource;					// Trigger - Source
//BF del	buf[0x1000 - 47] = (unsigned long) SelectedEdgeExtern;          //BF not used  Trigger Edge by extern triggering
	buf[0x1000 - 48] = (unsigned long) MenuBtnFocus[MENU_PULSEWIDTH]; 				//PulseToggle;		// Pulse width button focus
	//buf[0x1000 - 49] = (unsigned long) free;	

	buf[0x1000 - 50] = (unsigned long) FFT_TriggerMode_bak;					     //BF Backup for Triggermode
	buf[0x1000 - 51] = (unsigned long) Trigger_Pos_CH1;                                          // Trigger Level Channel 1
	buf[0x1000 - 52] = (unsigned long) Trigger_Pos_CH2;                                          // Trigger Level Channel 2
	buf[0x1000 - 53] = (unsigned long) Trigger_Pos_CH3;                                          // Trigger Level Channel 3
	buf[0x1000 - 54] = (unsigned long) Trigger_Pos_CH4;                                          // Trigger Level Channel 4
	//buf[0x1000 - 55] = (unsigned long) trg_val_reg;                                              // Trigger Value
	//buf[0x1000 - 56] = (unsigned long) trg_hyst_reg;                                             // Trigger Hysterese -> gibt's nicht
	//buf[0x1000 - 57] = (unsigned long) trig_range_reg;                                           // Trigger Range
	//buf[0x1000 - 58] = (unsigned long) trig_holdoff_reg;                                         // Trigger Holdoff
	//buf[0x1000 - 59] = (unsigned long) trig_width_reg;                                           // Trigger Width

	//buf[0x1000 - 60] = (unsigned long) 
	//buf[0x1000 - 61] = (unsigned long) 
	//buf[0x1000 - 62] = (unsigned long) 
	//buf[0x1000 - 63] = (unsigned long) 
	//buf[0x1000 - 64] = (unsigned long) 
	//buf[0x1000 - 65] = (unsigned long) 
	buf[0x1000 - 66] = (unsigned long) TriggerWay;                                               // Triggerway
	buf[0x1000 - 67] = (unsigned long) Trigger_Pos_CHE;                                          // Trigger extern value	
	
	//---------------------------------------------------
	// 		Channel 1
	//---------------------------------------------------
	buf[0x1000 - 68] = (unsigned long) MenuStatus[1][0];                                         // Channel - Coupling
	buf[0x1000 - 69] = (unsigned long) MenuStatus[1][1];                                         // Channel - BW Linit
	buf[0x1000 - 70] = (unsigned long) MenuStatus[1][2];                                         // Channel - Invert
	buf[0x1000 - 71] = (unsigned long) MenuStatus[1][3];                                         // Channel - Probe
	buf[0x1000 - 72] = (unsigned long) SwitchesCH1;                                              // Switches for Channel 1 and for the other the AC bit
	buf[0x1000 - 73] = (unsigned long) Selected_Voltage_CH1;                                     // Voltage 5V
	//BF del buf[0x1000 - 74] = (unsigned long) Trigger_Pos_CH1_Old;
	//BF del buf[0x1000 - 75] = (unsigned long) NoDisplay[0];;
	buf[0x1000 - 76] = (unsigned long) thres_low[0];						// QM thresholds
	buf[0x1000 - 77] = (unsigned long) thres_mdl[0];						// QM thresholds
	buf[0x1000 - 78] = (unsigned long) thres_up[0];							// QM thresholds
	buf[0x1000 - 79] = (unsigned long) thres_type[1];							// QM threshold type

	buf[0x1000 - 80] = (unsigned long) Selected_Voltage_CH1_bak; 
	buf[0x1000 - 81] = (unsigned long) Trigger_Pos_CH1_bak;
	buf[0x1000 - 82] = (unsigned long) CH1_DAC_Offset;                                           // Channel DAC
	buf[0x1000 - 83] = (unsigned long) ZeroLevelCH1;                                             // ZeroLevel
	buf[0x1000 - 84] = (unsigned long) Virtual_ZeroLevelCH1;                                     // Virtual ZeroLevel
    	//buf[0x1000 - 85] = 
    	//buf[0x1000 - 85] = 

	//---------------------------------------------------
	// 		Channel 2
	//---------------------------------------------------
	buf[0x1000 - 86] = (unsigned long) MenuStatus[2][0];                                         // Channel - Coupling
	buf[0x1000 - 87] = (unsigned long) MenuStatus[2][1];                                         // Channel - BW Linit
	buf[0x1000 - 88] = (unsigned long) MenuStatus[2][2];                                         // Channel - Invert
	buf[0x1000 - 89] = (unsigned long) MenuStatus[2][3];                                         // Channel - Probe

	buf[0x1000 - 90] = (unsigned long) SwitchesCH2;                                              // Switches for Channel
	buf[0x1000 - 91] = (unsigned long) Selected_Voltage_CH2;                                     // Voltage 5V
	//BF del buf[0x1000 - 92] = (unsigned long) Trigger_Pos_CH2_Old;
	//BF del buf[0x1000 - 93] = (unsigned long) NoDisplay[1];;
	buf[0x1000 - 94] = (unsigned long) thres_low[1];						// QM thresholds
	buf[0x1000 - 95] = (unsigned long) thres_mdl[1];						// QM thresholds
	buf[0x1000 - 96] = (unsigned long) thres_up[1];							// QM thresholds
	buf[0x1000 - 97] = (unsigned long) thres_type[1];							// QM threshold type
	
	buf[0x1000 - 98] = (unsigned long) Selected_Voltage_CH2_bak;
	buf[0x1000 - 99] = (unsigned long) Trigger_Pos_CH2_bak;
	buf[0x1000 - 100] = (unsigned long) CH2_DAC_Offset;                                           // Channel DAC
	buf[0x1000 - 101] = (unsigned long) ZeroLevelCH2;                                             // ZeroLevel
	buf[0x1000 - 102] = (unsigned long) Virtual_ZeroLevelCH2;                                     // Virtual ZeroLevel
    	//buf[0x1000 - 103] = 
    	//buf[0x1000 - 103] = 

	//---------------------------------------------------
	// 		Channel 3
	//---------------------------------------------------
	buf[0x1000 - 104] = (unsigned long) MenuStatus[3][0];                                         // Channel - Coupling
	buf[0x1000 - 105] = (unsigned long) MenuStatus[3][1];                                         // Channel - BW Linit
	buf[0x1000 - 106] = (unsigned long) MenuStatus[3][2];                                         // Channel - Invert
	buf[0x1000 - 107] = (unsigned long) MenuStatus[3][3];                                         // Channel - Probe
	buf[0x1000 - 108] = (unsigned long) SwitchesCH3;                                              // Switches for Channel
	buf[0x1000 - 109] = (unsigned long) Selected_Voltage_CH3;                                     // Voltage 5V

	//BF del buf[0x1000 - 110] = (unsigned long) Trigger_Pos_CH3_Old; 
	//BF del buf[0x1000 - 111] = (unsigned long) NoDisplay[2];;
	buf[0x1000 - 112] = (unsigned long) thres_low[2];						// QM thresholds
	buf[0x1000 - 113] = (unsigned long) thres_mdl[2];						// QM thresholds
	buf[0x1000 - 114] = (unsigned long) thres_up[2];						// QM thresholds
	buf[0x1000 - 115] = (unsigned long) thres_type[2];						// QM threshold type

	buf[0x1000 - 116] = (unsigned long) Selected_Voltage_CH3_bak;
	buf[0x1000 - 117] = (unsigned long) Trigger_Pos_CH3_bak;
	buf[0x1000 - 118] = (unsigned long) CH3_DAC_Offset;                                           // Channel DAC
	buf[0x1000 - 119] = (unsigned long) ZeroLevelCH3;                                             // ZeroLevel
	buf[0x1000 - 120] = (unsigned long) Virtual_ZeroLevelCH3;                                     // Virtual ZeroLevel
    	//buf[0x1000 - 121] = (unsigned long) free;
    	//buf[0x1000 - 121] = (unsigned long)

	//---------------------------------------------------
	// 		Channel 4
	//---------------------------------------------------
	buf[0x1000 - 122] = (unsigned long) MenuStatus[4][0];                                         // Channel - Coupling
	buf[0x1000 - 123] = (unsigned long) MenuStatus[4][1];                                         // Channel - BW Linit

	buf[0x1000 - 124] = (unsigned long) MenuStatus[4][2];                                         // Channel - Invert
	buf[0x1000 - 125] = (unsigned long) MenuStatus[4][3];                                         // Channel - Probe
	buf[0x1000 - 126] = (unsigned long) SwitchesCH4;                                              // Switches for Channel
	buf[0x1000 - 127] = (unsigned long) Selected_Voltage_CH4;                                     // Voltage 5V
	//BF del buf[0x1000 - 128] = (unsigned long) Trigger_Pos_CH4_Old;
	//BF del buf[0x1000 - 129] = (unsigned long) NoDisplay[3];;
	buf[0x1000 - 130] = (unsigned long) thres_low[3];						// QM thresholds
	buf[0x1000 - 131] = (unsigned long) thres_mdl[3];						// QM thresholds
	buf[0x1000 - 132] = (unsigned long) thres_up[3];						// QM thresholds
	buf[0x1000 - 133] = (unsigned long) thres_type[3];						// QM threshold type

	buf[0x1000 - 134] = (unsigned long) Selected_Voltage_CH4_bak;
	buf[0x1000 - 135] = (unsigned long) Trigger_Pos_CH4_bak;
	buf[0x1000 - 136] = (unsigned long) CH4_DAC_Offset;                                           // Channel DAC
	buf[0x1000 - 137] = (unsigned long) ZeroLevelCH4;                                             // ZeroLevel
	buf[0x1000 - 138] = (unsigned long) Virtual_ZeroLevelCH4;                                     // Virtual ZeroLevel
    	//buf[0x1000 - 139] = (unsigned long)

	//Math Settings
	buf[0x1000 - 140] = (unsigned long) Channel[eChM].active;                                      // Math channel off
	buf[0x1000 - 141] = (unsigned long) MenuStatus[MENU_MATH][0];                                        // scale
	buf[0x1000 - 142] = (unsigned long) MenuStatus[MENU_MATH][2];                                        // standart math 1*2 of
	buf[0x1000 - 143] = (unsigned long) MenuStatus[MENU_MATH][3];                                        // standart math 1-2 off
	buf[0x1000 - 144] = (unsigned long) MenuStatus[MENU_MATH][4];                                        // standart math 1+2 off
	
	//FFT Settings
	buf[0x1000 - 145] = (unsigned long) Signal::_FFT_Mode;
	buf[0x1000 - 146] = (unsigned long) Signal::_FFT_Length;
	buf[0x1000 - 147] = (unsigned long) Signal::_FFT_Level;
	buf[0x1000 - 148] = (unsigned long) Signal::_FFT_Windex;
//BF not used	buf[0x1000 - 149] = (unsigned long) FFT_Scale;
	
	buf[0x1000 - 150] = (unsigned long) MenuStatus[MENU_FFT][0]; 
	buf[0x1000 - 151] = (unsigned long) MenuStatus[MENU_FFT][1];
	buf[0x1000 - 152] = (unsigned long) MenuStatus[MENU_FFT][2];
	buf[0x1000 - 153] = (unsigned long) MenuStatus[MENU_FFT][3];

    	// Cursors    	
	buf[0x1000 - 154] = (unsigned long) MenuStatus[MENU_CURSOR][0];                   // Cursor source
	buf[0x1000 - 155] = (unsigned long) Cursor_Enabled;
	buf[0x1000 - 156] = (unsigned long) Cursor_Both_Selected;
	buf[0x1000 - 157] = (unsigned long) SelectedCursor;

	buf[0x1000 - 158] = (unsigned long) Cursor_Horizontal_Active;
	buf[0x1000 - 159] = (unsigned long) Cursor_Horizontal_Position_Real1;
	buf[0x1000 - 160] = (unsigned long) Cursor_Horizontal_Position_Real2;

	buf[0x1000 - 161] = (unsigned long) Cursor_Vertical_Active;
	buf[0x1000 - 162] = (unsigned long) Cursor_Vertical_Position_TY_1;
	buf[0x1000 - 163] = (unsigned long) Cursor_Vertical_Position_TY_2;
	
	// Quick Measure
	//buf[0x1000 - 164] = (unsigned long) QM_Enabled;
	//buf[0x1000 - 165] = (unsigned long) QM_Type[0];
	//buf[0x1000 - 166] = (unsigned long) QM_Type[1];
	//buf[0x1000 - 167] = (unsigned long) QM_Type[2];
	//buf[0x1000 - 168] = (unsigned long) QM_Channel[0];
	//buf[0x1000 - 169] = (unsigned long) QM_Channel[1];
	
	//buf[0x1000 - 170] = (unsigned long) QM_Channel[2];
	//buf[0x1000 - 171] = (unsigned long) QM_Horizontal_Active;
	//buf[0x1000 - 172] = (unsigned long) QM_Vertical_Active;
	//buf[0x1000 - 173] = (unsigned long) Quick_Measure_Threshold_btn_select;
	
	// Display
	//buf[0x1000 - 174] = (unsigned long) GridColorArrayPos;                                // Normal Grid
	//buf[0x1000 - 175] = (unsigned long) GridColor_Val;
	//buf[0x1000 - 176] = (unsigned long) MenuStatus[MENU_DISPLAY][3];			// no display
	//buf[0x1000 - 177] = (unsigned long) dotted_grid;
	buf[0x1000 - 178] = (unsigned long) MenuStatus[MENU_DISPLAY][0];  			// Display - Persist
	//buf[0x1000 - 179] = (unsigned long) MenuStatus[MENU_DISPLAY][4];  			// Display - Draw Mode

	buf[0x1000 - 180] = (unsigned long) MenuStatus[MENU_ACQUIRE][0];          		// Acquire - Logic
        buf[0x1000 - 181] = (unsigned long) MenuStatus[MENU_ACQUIRE][1];        		// Acquire - Averaging
        buf[0x1000 - 183] = (unsigned long) MenuStatus[MENU_ACQUIRE][2];               		// Acquire - Filter Setting
	buf[0x1000 - 184] = (unsigned long) MenuStatus[MENU_ACQUIRE][3];        		// Acquire - Interpolation
	
	//buf[0x1000 - 185] = (unsigned long) MenuStatus[MENU_AUTOSCALE][1];          		// Autoscale - Slow TB

//	buf[0x1000 - 186] = (unsigned long) MenuStatus[MENU_HARDWARE][1];			//BF gain of input stage

	buf[0x1000 - 187] = (unsigned long) MenuPopupStatus[8][0];				//BF extern trigger setting
	buf[0x1000 - 188] = (unsigned long) MenuPopupStatus[8][1];
	buf[0x1000 - 189] = (unsigned long) MenuPopupStatus[8][2];
	buf[0x1000 - 190] = (unsigned long) MenuPopupStatus[11][0];				//BF TV trigger setting
	buf[0x1000 - 191] = (unsigned long) MenuPopupStatus[11][1];
	
	buf[0x1000 - 192] = (unsigned long) led_value;
	buf[0x1000 - 193] = (unsigned long) Channel[eCh1].active;
	buf[0x1000 - 194] = (unsigned long) Channel[eCh2].active;
	buf[0x1000 - 195] = (unsigned long) Channel[eCh3].active;
	buf[0x1000 - 196] = (unsigned long) Channel[eCh4].active;

	buf[0x1000 - 197] = (unsigned long) FFT_grid;	

	//buf[0x1000 - 198] = (unsigned long) free;
	
	buf[0x1000 - 199] = (unsigned long) HoldOff_Value;
	buf[0x1000 - 200] = (unsigned long) HoldOff_Expo;
	buf[0x1000 - 201] = (unsigned long) HoldOff_Steps;
	
	buf[0x1000 - 202] = (unsigned long) Pulse11_Value;
	buf[0x1000 - 203] = (unsigned long) Pulse11_Expo;
	buf[0x1000 - 204] = (unsigned long) Pulse11_Steps;
	
	buf[0x1000 - 205] = (unsigned long) Pulse12_Value;
	buf[0x1000 - 206] = (unsigned long) Pulse12_Expo;
	buf[0x1000 - 207] = (unsigned long) Pulse12_Steps;
	
	buf[0x1000 - 208] = (unsigned long) Pulse21_Value;
	buf[0x1000 - 209] = (unsigned long) Pulse21_Expo;
	buf[0x1000 - 210] = (unsigned long) Pulse21_Steps;
	
	buf[0x1000 - 211] = (unsigned long) Pulse22_Value;
	buf[0x1000 - 212] = (unsigned long) Pulse22_Expo;
	buf[0x1000 - 213] = (unsigned long) Pulse22_Steps;
	
//BF new Math values
	buf[0x1000 - 214] = (unsigned int) Math_Offset;
	buf[0x1000 - 215] = (unsigned int) Math_Mul_Offset;
	buf[0x1000 - 216] = (unsigned int) Math_Sub_Offset;
	buf[0x1000 - 217] = (unsigned int) Math_Add_Offset;
	buf[0x1000 - 218] = (unsigned int) math_mul_scale;
	buf[0x1000 - 219] = (unsigned int) math_mul_scale;
	buf[0x1000 - 220] = (unsigned int) math_sub_scale;
	buf[0x1000 - 221] = (unsigned int) math_add_scale;
//BF end

	//buf[0x1000 - 222] = (unsigned long)
	//buf[0x1000 - 223] = (unsigned long)
	//buf[0x1000 - 224] = (unsigned long)
	//buf[0x1000 - 225] = (unsigned long) 
//	buf[0x1000 - 226] = (unsigned long) 


	buf[0x1000 - 229] = (unsigned long) Trig_Pos_Mem;
	buf[0x1000 - 230] = (unsigned long) Trig_Pos_Display;
	//buf[0x1000 - 231] = (unsigned long) not used;
	buf[0x1000 - 232] = (unsigned long) MemWinStart;
	
	buf[0x1000 - 233] = (unsigned long) Trig_Pos_Mem_old;
	buf[0x1000 - 234] = (unsigned long) Trig_Pos_Display_old;
	//buf[0x1000 - 235] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][4];
	buf[0x1000 - 236] = (unsigned long) MemWinStart_old;
	
	buf[0x1000 - 237] = (unsigned long) ZoomFactor;
	buf[0x1000 - 238] = (unsigned long) ZoomFactor_old;
	buf[0x1000 - 239] = (unsigned long) ZoomFactorDel;

	//buf[0x1000 - 240] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][0];
	//buf[0x1000 - 241] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][1];
	//buf[0x1000 - 242] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][2];

	//buf[0x1000 - 243] = (unsigned long) QM_Source;
	//buf[0x1000 - 244] = (unsigned long)
	//buf[0x1000 - 245] = (unsigned long) 
	//buf[0x1000 - 246] = (unsigned long)
	
	//buf[0x1000 - 247] = (unsigned long)   ;

// BF new zerolevels for XY
	buf[0x1000 - 248] = (unsigned int) Virtual_ZeroLevelXYCH1;
	buf[0x1000 - 249] = (unsigned int) Virtual_ZeroLevelXYCH2;
	buf[0x1000 - 250] = (unsigned int) Virtual_ZeroLevelXYCH3;
	buf[0x1000 - 251] = (unsigned int) Virtual_ZeroLevelXYCH4;
// BF end


	//BF backup values must be saved too
	buf[0x1000 - 252] = (unsigned long) ZeroLevelCH1_bak;
	buf[0x1000 - 253] = (unsigned long) ZeroLevelCH2_bak;
	buf[0x1000 - 254] = (unsigned long) ZeroLevelCH3_bak;
	buf[0x1000 - 255] = (unsigned long) ZeroLevelCH4_bak;

	buf[0x1000 - 256] = (unsigned long) Virtual_ZeroLevelCH1_bak;
	buf[0x1000 - 257] = (unsigned long) Virtual_ZeroLevelCH2_bak;
	buf[0x1000 - 258] = (unsigned long) Virtual_ZeroLevelCH3_bak;
	buf[0x1000 - 259] = (unsigned long) Virtual_ZeroLevelCH4_bak;

//	buf[0x1000 - 260] = (unsigned long) free;

	buf[0x1000 - 261] = (unsigned long) Cursor_Vertical_Position_XY_1;
	buf[0x1000 - 262] = (unsigned long) Cursor_Vertical_Position_XY_2;
	buf[0x1000 - 263] = (unsigned long) Cursor_Vertical_Position_FFT_1;
	buf[0x1000 - 264] = (unsigned long) Cursor_Vertical_Position_FFT_2;
	//buf[0x1000 - 265] = (unsigned long) SavePointer;

	buf[0x1000 - 266] = (unsigned long) ChannelStatusOld[eCh1];
	buf[0x1000 - 267] = (unsigned long) ChannelStatusOld[eCh2];
	buf[0x1000 - 268] = (unsigned long) ChannelStatusOld[eCh3];
	buf[0x1000 - 269] = (unsigned long) ChannelStatusOld[eCh4];
	buf[0x1000 - 270] = (unsigned long) ChannelStatusOld[eChM];
	
	//buf[0x1000 - 271] = (unsigned long) DrawMono;
	buf[0x1000 - 272] = (unsigned long) MainTimebase_bak;
	//buf[0x1000 - 273] = (unsigned long) free;
	//buf[0x1000 - 274] = (unsigned long) free;

	buf[0x1000 - 275] = (unsigned long) ZeroLevelCH1_Main_bak;
	buf[0x1000 - 276] = (unsigned long) ZeroLevelCH2_Main_bak;
	buf[0x1000 - 277] = (unsigned long) ZeroLevelCH3_Main_bak;
	buf[0x1000 - 278] = (unsigned long) ZeroLevelCH4_Main_bak;

	buf[0x1000 - 279] = (unsigned long) Virtual_ZeroLevelCH1_Main_bak;
	buf[0x1000 - 280] = (unsigned long) Virtual_ZeroLevelCH2_Main_bak;
	buf[0x1000 - 281] = (unsigned long) Virtual_ZeroLevelCH3_Main_bak;
	buf[0x1000 - 282] = (unsigned long) Virtual_ZeroLevelCH4_Main_bak;


	//buf[0x1000 - 283] = (unsigned long)
	//buf[0x1000 - 284] = (unsigned long) 
	//buf[0x1000 - 285] = (unsigned long)
	//buf[0x1000 - 286] = (unsigned long)
	//buf[0x1000 - 287] = (unsigned long) 
	//buf[0x1000 - 288] = (unsigned long) 

	//buf[0x1000 - 289] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][0];			// grid color
	//buf[0x1000 - 290] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][1];			// grid layout
	//buf[0x1000 - 291] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][2];			// status layout
	//buf[0x1000 - 292] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][3];			// QM Cursors
	//buf[0x1000 - 293] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][4];			// OSS on/off
	//buf[0x1000 - 294] = (unsigned long)MenuStatus[MENU_DISPL_SETUP][5];			// not used

	//buf[0x1000 - 295] = (unsigned long) thres_low[4];						// QM thresholds
	//buf[0x1000 - 296] = (unsigned long) thres_mdl[4];						// QM thresholds
	//buf[0x1000 - 297] = (unsigned long) thres_up[4];						// QM thresholds
	//buf[298] = (unsigned long) thres_type[4];					// QM threshold type

	buf[0x1000 - 299] = 0x23232323;	
	
	//write signal 4 and the settings to flash
	if (!nr_flash_write_buffer(flash_address + 0x3000, buf, 0x1000)) { return 0; }

	return 1;
}
//#####################################################################################################################################################
//BF Save/Recall -> load configuration of corresponding trace into memory
// Return value: 1 on success, 0 on failure.
int AMDFlash::LoadTraceSettings(unsigned long *flash_address)
{
	uint32_t lX;
	uint32_t buf[300];	
	int32_t rcode = 0;

	for (lX = 0; lX < 300; lX++) buf[lX] = flash_address[0x4000 - lX];		
	
	if (buf[2] == 0x22334455 && buf[299] == 0x23232323)
	{

		Hardware::Stop_Timer2();
		Hardware::Stop_Record();
		ADC_DataAvailable = 0;	
		FFT_NewData = 0;
		ACQ_NewData = 0;


		// Timebase      	
		MenuStatus[MENU_TIMEBASE][0] = (short) buf[10];
		MenuStatus[MENU_TIMEBASE][1] = (short) buf[11];
		MenuStatus[MENU_TIMEBASE][2] = (short) buf[12];
		MenuStatus[MENU_TIMEBASE][3] = (short) buf[13];
		MenuStatus[MENU_TIMEBASE][4] = (short) buf[14];
		//MenuStatus[MENU_DISPLAY][2] = (short) buf[15];	//grid brightness
		XY_Mode = (char) buf[16];
		MainTimebase = (int32_t) buf[17];
		USTB_Mode = (short) buf[18];
		USTB_Mode_bak = (short) buf[19];
		USTB_Dir = (short) buf[20];   
		USTB_Disp = (short) buf[21];
		DelayedTimebase = (int32_t) buf[22];
		VirtualTimebase = (int32_t)buf[23];
		dmode_Window_Offset_Pos = (int) buf[24];
		MenuStatus[MENU_USTB][2] = (short) buf[25];
	
		// werden frisch berrechnet
		//Cursor_Delayed_1 = (int) buf[26];
		//Cursor_Delayed_2 = (int) buf[27];
		//Cursor_Delayed_Size = (int) buf[28];
		MenuStatus[MENU_USTB][0] = (short) buf[29];					//ultra slow timebase mode
		MenuStatus[MENU_USTB][1] = (short) buf[30];
	//  		 = (volatile char) buf[31];		-> LED 1			// Trigger LED	
	//      	= (volatile char) buf[32];		-> LED 2			// Trigger LED	
		//free = (int) buf[33];
		// free = (unsigned short) buf[34];
		
		// Trigger	
		MenuStatus[MENU_TRIGGERMODE][0] = (short) buf[35];				// Triggering - Mode
		MenuStatus[MENU_TRIGGERMODE][1] = (short) buf[36];				// Triggering - Coupling
		// free = (short) buf[37];				// 
		MenuStatus[MENU_TRIGGERMODE][4] = (short) buf[38];				// Triggering - External Probe
		MenuStatus[MENU_TRIGGEREDGE][3] = (short) buf[39];				// Triggering - TV
		MenuStatus[MENU_TRIGGEREDGE][0] = (short) buf[40];				// Triggering - positive edge
		MenuStatus[MENU_TRIGGEREDGE][1] = (short) buf[41];				// Triggering - Source 
		MenuStatus[MENU_TRIGGEREDGE][2] = (short) buf[42];				// Triggering - no externel triggering
		MenuStatus[MENU_PULSEWIDTH][0] = (short) buf[43];				// Triggering - Source 
		//MenuStatus[MENU_PULSEWIDTH][1] = (short) buf[44];				// Triggering - Negative Pulse
		MenuStatus[MENU_PULSEWIDTH][2] = (short) buf[45];				// Triggering - bigger - smaller - then
		Hardware::_TriggerSource = (trigSrc_t) buf[46];					// Trigger - Source
//BF del		SelectedEdgeExtern = (char) buf[47]; 	//BF not used  Trigger Edge by extern triggering
		//PulseToggle = (char) buf[48];							// Pulse width button selection   
		MenuBtnFocus[MENU_PULSEWIDTH] = (unsigned char) buf[48];			// Pulse width button focus
		//free = (long) buf[49];						
		
		FFT_TriggerMode_bak = (short) buf[50];     					//BF backup trigger mode
		Trigger_Pos_CH1  = (int) buf[51];                                          // Trigger Level Channel 1
		Trigger_Pos_CH2  = (int) buf[52];                                          // Trigger Level Channel 2
		Trigger_Pos_CH3  = (int) buf[53];                                          // Trigger Level Channel 3
		Trigger_Pos_CH4  = (int) buf[54];                                          // Trigger Level Channel 4
		//trg_val_reg      = (unsigned int) buf[55];                                              // Trigger Value
		//trg_hyst_reg     = (unsigned int) buf[56];                                             // Trigger Hysterese
		//trig_range_reg   = (unsigned int) buf[57];                                           // Trigger Range
		//trig_holdoff_reg = (unsigned long) buf[58];                                         // Trigger Holdoff
		//trig_width_reg   = (unsigned int) buf[59];                                           // Trigger Width
		
		// free = (unsigned int) buf[60];                                                 // Control register
		//= (unsigned int) buf[61];                                             // ADC Control register
		// = (unsigned int) buf[62];                                                  // PreTrigger Value
		// = (unsigned int) buf[63];					// control register FPGA 1
		// = (unsigned long) buf[64]; 				// control register FPGA 1  
		// free = (unsigned int) buf[65];                                               // Triggering
		TriggerWay = (char) buf[66];                                               // Triggerway
		Trigger_Pos_CHE = (int) buf[67];                                                // Trigger extern value
		
		//---------------------------------------------------
		// 		Channel 1
		//---------------------------------------------------
		MenuStatus[1][0] = (short) buf[68];                                         // Channel - Coupling
		MenuStatus[1][1] = (short) buf[69];                                         // Channel - BW Linit
		MenuStatus[1][2] = (short) buf[70];                                         // Channel - Invert
		MenuStatus[1][3] = (short) buf[71];                                         // Channel - Probe
		SwitchesCH1 = (unsigned short) buf[72];                                              // Switches for Channel 1 and for the other the AC bit
		Selected_Voltage_CH1 = (volatile char) buf[73];                                     // Voltage
		//Trigger_Pos_CH1_Old = (int) buf[74];
		//NoDisplay[0] = (unsigned char) buf[75];
		thres_low[0] = (short) buf[76];						// QM thresholds;
		thres_mdl[0] = (short) buf[77];						// QM thresholds;
		thres_up[0]  = (short) buf[78];						// QM thresholds;
		thres_type[0] = (short) buf[79];					// QM threshold type
		
		Selected_Voltage_CH1_bak = (char) buf[80];
		Trigger_Pos_CH1_bak = (int) buf[81];
		CH1_DAC_Offset       = (int) buf[82];                                           // Channel DAC
		ZeroLevelCH1         = (int) buf[83];                                             // ZeroLevel
		Virtual_ZeroLevelCH1 = (int) buf[84];                                     // Virtual ZeroLevel
		//free             = (int) buf[85];
		//   = (int) buf[85];                                        

		//---------------------------------------------------
		// 		Channel 2
		//---------------------------------------------------
		MenuStatus[2][0] = (short) buf[86];                                         // Channel - Coupling
		MenuStatus[2][1] = (short) buf[87];                                         // Channel - BW Linit
		MenuStatus[2][2] = (short) buf[88];                                         // Channel - Invert
		MenuStatus[2][3] = (short) buf[89];                                         // Channel - Probe
	
		SwitchesCH2 = (unsigned short) buf[90];                                              // Switches for Channel 2
		Selected_Voltage_CH2 = (volatile char) buf[91];                                     // Voltage
		//Trigger_Pos_CH2_Old = (int) buf[92];
		//NoDisplay[1] = (unsigned char) buf[93];
		thres_low[1] = (short) buf[94];						// QM thresholds;
		thres_mdl[1] = (short) buf[95];						// QM thresholds;
		thres_up[1]  = (short) buf[96];						// QM thresholds;
		thres_type[1] = (short) buf[97];					// QM threshold type

		Selected_Voltage_CH2_bak = (char) buf[98];
		Trigger_Pos_CH2_bak = (int) buf[99];

		CH2_DAC_Offset       = (int) buf[100];                                           // Channel DAC
		ZeroLevelCH2         = (int) buf[101];                                             // ZeroLevel
		Virtual_ZeroLevelCH2 = (int) buf[102];                                     // Virtual ZeroLevel
		//         = (int) buf[103];                                          
		//   = (int) buf[103];                                         
	
		//---------------------------------------------------
		// 		Channel 3
		//---------------------------------------------------
		MenuStatus[3][0] = (short) buf[104];                                         // Channel - Coupling
		MenuStatus[3][1] = (short) buf[105];                                         // Channel - BW Linit
		MenuStatus[3][2] = (short) buf[106];                                         // Channel - Invert
		MenuStatus[3][3] = (short) buf[107];                                         // Channel - Probe
		SwitchesCH3 = (unsigned short) buf[108];                                              // Switches for Channel 3
		Selected_Voltage_CH3 = (volatile char) buf[109];                                     // Voltage
		//Trigger_Pos_CH3_Old = (int) buf[110];
		//NoDisplay[2] = (unsigned char) buf[111];
		thres_low[2] = (short) buf[112];						// QM thresholds;
		thres_mdl[2] = (short) buf[113];						// QM thresholds;
		thres_up[2]  = (short) buf[114];						// QM thresholds;
		thres_type[2] = (short) buf[115];						// QM threshold type
	
		Selected_Voltage_CH3_bak = (char) buf[116];
		Trigger_Pos_CH1_bak = (int) buf[117];
		CH3_DAC_Offset = (int) buf[118];                                           // Channel DAC
		ZeroLevelCH3 = (int) buf[119];                                             // ZeroLevel
		Virtual_ZeroLevelCH3 = (int) buf[120];                                     // Virtual ZeroLevel
		//          = (int) buf[121];                                        
		// = (int) buf[121];                                         
	
		//---------------------------------------------------
		// 		Channel 4
		//---------------------------------------------------
		MenuStatus[4][0] = (short) buf[122];                                         // Channel - Coupling
		MenuStatus[4][1] = (short) buf[123];                                         // Channel - BW Linit
		MenuStatus[4][2] = (short) buf[124];                                         // Channel - Invert
		MenuStatus[4][3] = (short) buf[125];                                         // Channel - Probe
		SwitchesCH4 = (unsigned short) buf[126];                                              // Switches for Channel 4
		Selected_Voltage_CH4 = (volatile char) buf[127];                                     // Voltage
		//Trigger_Pos_CH4_Old = (int) buf[128];                       
		//NoDisplay[3] = (unsigned char) buf[129];                        
		thres_low[3] = (short) buf[130];						// QM thresholds;
		thres_mdl[3] = (short) buf[131];						// QM thresholds;
		thres_up[3]  = (short) buf[132];						// QM thresholds;
		thres_type[3] = (short) buf[133];						// QM threshold type

		Selected_Voltage_CH4_bak = (char) buf[134];
		Trigger_Pos_CH1_bak = (int) buf[135];
		CH4_DAC_Offset       = (int) buf[136];                                           // Channel DAC
		ZeroLevelCH4         = (int) buf[137];                                             // ZeroLevel
		Virtual_ZeroLevelCH4 = (int) buf[138];                                     // Virtual ZeroLevel
		//             = (int) buf[139];                                        
		// = (int) buf[139];                                          
		
        	Channel[eChM].active = (uint32_t) buf[140];                                      	// Math channel
        	MenuStatus[MENU_MATH][0] = (short) buf[141];                                        // scale
        	MenuStatus[MENU_MATH][2] = (short) buf[142];                                        // standart math 1*2 
        	MenuStatus[MENU_MATH][3] = (short) buf[143];                                        // standart math 1-2 
        	MenuStatus[MENU_MATH][4] = (short) buf[144];                                        // standart math 1+2 

    		//FFT Settings
     		Signal::_FFT_Mode   = (int)buf[145];
    		Signal::_FFT_Length = (int)buf[146];
    		Signal::_FFT_Level  = (int)buf[147];
   		Signal::_FFT_Windex = (int)buf[148];
    		//FFT_Scale  = (int)buf[149];

		MenuStatus[MENU_FFT][0] = (short)buf[150]; 
		MenuStatus[MENU_FFT][1] = (short)buf[151];
		MenuStatus[MENU_FFT][2] = (short)buf[152];
		MenuStatus[MENU_FFT][3] = (short)buf[153];

		// Cursors    	
/*		MenuStatus[16][0] = (short) buf[154];                                        // Cursor source
		//Cursor_Enabled = (char) buf[155];
	
		if (Cursor_Enabled) Cursor_Data_First_Draw = 1;
	
		Cursor_Both_Selected = (char) buf[156];
		SelectedCursor = (char) buf[157];

		Cursor_Horizontal_Active = (char) buf[158];
		Cursor_Horizontal_Position_Real1 = (long) buf[159];
		Cursor_Horizontal_Position_Real2 = (long) buf[160];

		Cursor_Vertical_Active = (char) buf[161];
		Cursor_Vertical_Position_TY_1 = (long) buf[162];
		Cursor_Vertical_Position_TY_2 = (long) buf[163];
	
		// Quick Measure
		//QM_Enabled = 0; //QM_Enabled = (char) buf[164];
	
		//if (QM_Enabled) Quick_Measure_First_Draw = 1;
	
		QM_Type[0] = (char) buf[165];
		QM_Type[1] = (char) buf[166];
		QM_Type[2] = (char) buf[167];
		QM_Channel[0] = (char) buf[168];
		QM_Channel[1] = (char) buf[169];
	
		QM_Channel[2] = (char) buf[170];
		QM_Horizontal_Active = (char) buf[171];
		QM_Vertical_Active = (char) buf[172];
		Quick_Measure_Threshold_btn_select = (char) buf[173];
*/	
		// Display
		//GridColorArrayPos = (unsigned char) buf[174];
		//GridColor_Val     = (unsigned char) buf[175];
		// MenuStatus[MENU_DISPLAY][3] = (unsigned char) buf[176];				// no display
		//dotted_grid       = (unsigned char) buf[177];
		//MenuStatus[MENU_DISPLAY][0] = (short) buf[178];                                         // Display - Persist
		//MenuStatus[MENU_DISPLAY][4] = (short) buf[179];                                         // Display - Draw Mode
	
		MenuStatus[MENU_ACQUIRE][0] = (short) buf[180];                                         // Acquire - Logic
		MenuStatus[MENU_ACQUIRE][1] = (short) buf[181];                                         // Acquire - Averaging
		MenuStatus[MENU_ACQUIRE][2] = (short) buf[183];                                         // Acquire - Filter Setting
		MenuStatus[MENU_ACQUIRE][3] = (short) buf[184];                                         // Acquire - Interpolation
	
		//MenuStatus[MENU_AUTOSCALE][1] = (short) buf[185];                                        // Autoscale - Slow TB
//		 = (short) buf[186];
	
		MenuPopupStatus[8][0] = (short) buf[187];		                    //BF popup for external trigger
		MenuPopupStatus[8][1] = (short) buf[188];
		MenuPopupStatus[8][2] = (short) buf[189];
		MenuPopupStatus[11][0] = (short) buf[190];		                    //BF popup for TV trigger
		MenuPopupStatus[11][1] = (short) buf[191];

		// 192 BF -> reserved s.o. ledval

		Channel[eCh1].active = (uint32_t) buf[193];
		Channel[eCh2].active = (uint32_t) buf[194];
		Channel[eCh3].active = (uint32_t) buf[195];
		Channel[eCh4].active = (uint32_t) buf[196];

		FFT_grid = (unsigned char) buf[197];		

		// free = (volatile short) buf[198];
	
		HoldOff_Value = (int) buf[199];
		HoldOff_Expo  = (int) buf[200];
		HoldOff_Steps = (int) buf[201];
	
		Pulse11_Value = (int) buf[202];
		Pulse11_Expo  = (int) buf[203];
		Pulse11_Steps = (int) buf[204];
	
		Pulse12_Value = (int) buf[205];
		Pulse12_Expo  = (int) buf[206];
		Pulse12_Steps = (int) buf[207];
	
		Pulse21_Value = (int) buf[208];
		Pulse21_Expo  = (int) buf[209];
		Pulse21_Steps = (int) buf[210];
	
		Pulse22_Value = (int) buf[211];
		Pulse22_Expo  = (int) buf[212];
		Pulse22_Steps = (int) buf[213];
	
	//BF new Math values
		Math_Offset     = (int) buf[214]; 
		Math_Mul_Offset = (int) buf[215]; 
		Math_Sub_Offset = (int) buf[216];
		Math_Add_Offset = (int) buf[217];
		math_scale      = (int) buf[218];
		math_mul_scale  = (int) buf[219];
		math_sub_scale  = (int) buf[220];
		math_add_scale  = (int) buf[221];
	//BF end
	
		//MenuStatus[MENU_EXTUTILS][0] = (short) buf[222];	//BF #001
		//MenuStatus[MENU_EXTUTILS][1] = (short) buf[223];	//BF #001
		//MenuStatus[MENU_EXTUTILS][2] = (short) buf[224];	//BF #001
		//MenuStatus[MENU_EXTUTILS][3] = (short) buf[225];	//BF #001
//		MenuStatus[MENU_HARDWARE][0] = (short) buf[226];	//BF #002

		Trig_Pos_Mem     = (int) buf[229];
		Trig_Pos_Display = (int) buf[230];
		//not used    = (float) buf[231];
		MemWinStart      = (int) buf[232];
		
		Trig_Pos_Mem_old = (int) buf[233];
		Trig_Pos_Display_old = (int) buf[234];
		//MenuStatus[MENU_QUICKMEASURE][4] = (short) buf[235];
		MemWinStart_old = (int) buf[236];
		
		ZoomFactor        = (float) buf[237];
		ZoomFactor_old    = (float) buf[238];
		ZoomFactorDel     = (float) buf[239];

		//MenuStatus[MENU_QUICKMEASURE][0] = (short) buf[240];
		//MenuStatus[MENU_QUICKMEASURE][1] = (short) buf[241];
		//MenuStatus[MENU_QUICKMEASURE][2] = (short) buf[242];
		
		 //QM_Source = (short) buf[243];	
		// = (char) buf[244];	//BF #001
		// = (char) buf[245];	//BF #001
		// = (char) buf[246];	//BF #001
	
		//  = (short) buf[247];	

	// BF new zerolevels for XY
		Virtual_ZeroLevelXYCH1 = (int)buf[248];
		Virtual_ZeroLevelXYCH2 = (int)buf[249];
		Virtual_ZeroLevelXYCH3 = (int)buf[250];
		Virtual_ZeroLevelXYCH4 = (int)buf[251];
	// BF end

		//BF backup values must be restored too
		ZeroLevelCH1_bak = (int)buf[252];
		ZeroLevelCH2_bak = (int)buf[253];
		ZeroLevelCH3_bak = (int)buf[254];
		ZeroLevelCH4_bak = (int)buf[255];

		Virtual_ZeroLevelCH1_bak = (int)buf[256];
		Virtual_ZeroLevelCH2_bak = (int)buf[257];
		Virtual_ZeroLevelCH3_bak = (int)buf[258];
		Virtual_ZeroLevelCH4_bak = (int)buf[259];

		//free = (short)buf[260];
/*
		Cursor_Vertical_Position_XY_1 = (long)buf[261];
		Cursor_Vertical_Position_XY_2 = (long)buf[262];
		Cursor_Vertical_Position_FFT_1 = (long)buf[263];
		Cursor_Vertical_Position_FFT_2 = (long)buf[264];
		//SavePointer = (char)buf[265];
*/
		ChannelStatusOld[eCh1] = (uint32_t)buf[266];
		ChannelStatusOld[eCh2] = (uint32_t)buf[267];
		ChannelStatusOld[eCh3] = (uint32_t)buf[268];
		ChannelStatusOld[eCh4] = (uint32_t)buf[269];
		ChannelStatusOld[eChM] = (uint32_t)buf[270];

		// DrawMono  = (unsigned char)buf[271];
		MainTimebase_bak   = (int)buf[272];
		// free   = (short)buf[273];
		// free = (short)buf[274];

		ZeroLevelCH1_Main_bak = (int)buf[275];
		ZeroLevelCH2_Main_bak = (int)buf[276];
		ZeroLevelCH3_Main_bak = (int)buf[277];
		ZeroLevelCH4_Main_bak = (int)buf[278];

		Virtual_ZeroLevelCH1_Main_bak = (int)buf[279];
		Virtual_ZeroLevelCH2_Main_bak = (int)buf[280];
		Virtual_ZeroLevelCH3_Main_bak = (int)buf[281];
		Virtual_ZeroLevelCH4_Main_bak = (int)buf[282];

		 //  	= (unsigned short)buf[283];
		 //	= (unsigned short)buf[284];
		 //	= buf[285];
		//	= buf[286];
		//    = buf[287];		-> unused registers
		//   = buf[288];		-> unused registers

		//MenuStatus[MENU_DISPL_SETUP][0] = (short) buf[289];			// grid color
		//MenuStatus[MENU_DISPL_SETUP][1] = (short) buf[290];			// grid layout
		//MenuStatus[MENU_DISPL_SETUP][2] = (short) buf[291];			// status layout
		//MenuStatus[MENU_DISPL_SETUP][3] = (short) buf[292];			// QM Cursors
		//MenuStatus[MENU_DISPL_SETUP][4] = (short) buf[293];			// OSS on/off
		//MenuStatus[MENU_DISPL_SETUP][5] = (short) buf[294];			// not used
	
		//thres_low[4] = (short) buf[295];					// QM thresholds;
		//thres_mdl[4] = (short) buf[296];					// QM thresholds;
		//thres_up[4]  = (short) buf[297];					// QM thresholds;
		//thres_type[4] = (short) buf[298];					// QM threshold type


		LED_ON[0] = 0;
		LED_ON[1] = 0;
		LED_ON[2] = 0;
		LED_ON[3] = 0;
		LED_ON[4] = 0;
		LED_ON[5] = 1;
		//LED_ON[6] = 0;
		//LED_ON[7] = 0;
		LED_ON[8] = 0;
		LED_ON[9] = 0;
		LED_ON[10] = 0;
		LED_ON[11] = 0;
		//LED_ON[12] = 0;
		//LED_ON[13] = 0;
		LED_ON[14] = 0;
		LED_ON[15] = 0;

		//decode LED settings
		if (((int) buf[192] & 0x0001) == 0x0001) LED_ON[0] = 1;
		if (((int) buf[192] & 0x0002) == 0x0002) LED_ON[1] = 1;
		if (((int) buf[192] & 0x0004) == 0x0004) LED_ON[2] = 1;
		if (((int) buf[192] & 0x0008) == 0x0008) LED_ON[3] = 1;
	        if (((int) buf[192] & 0x0010) == 0x0010) LED_ON[4] = 1;  // Math
		//if (((int) buf[192] & 0x0020) == 0x0020) LED_ON[5] = 1; // General
		//if (((int) buf[192] & 0x0040) == 0x0040) LED_ON[6] = 1;  // cursor
		//if (((int) buf[192] & 0x0080) == 0x0080) LED_ON[7] = 1; // Quick Measure
		if (((int) buf[192] & 0x0100) == 0x0100) LED_ON[8] = 1;
		if (((int) buf[192] & 0x0200) == 0x0200) LED_ON[9] = 1;
		if (((int) buf[192] & 0x0400) == 0x0400) LED_ON[10] = 1;
		if (((int) buf[192] & 0x0800) == 0x0800) LED_ON[11] = 1;

		UserIF::Buttonhandler(BTN_STOP);			// switch to stop mode	

		switch(Pulse11_Expo)
		{
			case 1:
			{
				Pulse11.Write_Value((float) Pulse11_Value / (float) 1000000000);
				break;
			}
			case 2:
			{
				Pulse11.Write_Value((float) Pulse11_Value / (float) 1000000);
				break;
			}
			case 3:
			{
				Pulse11.Write_Value((float) Pulse11_Value / (float) 1000);
				break;
			}
		}
		Pulse11.Write_StepSize(Pulse11_Steps);
		Pulse11.RenderText();

		switch(Pulse12_Expo)
		{
			case 1:
			{
				Pulse12.Write_Value((float) Pulse12_Value / (float) 1000000000);
				break;
			}
			case 2:
			{
				Pulse12.Write_Value((float) Pulse12_Value / (float) 1000000);
				break;
			}
			case 3:
			{
				Pulse12.Write_Value((float) Pulse12_Value / (float) 1000);
				break;
			}
		}
		Pulse12.Write_StepSize(Pulse12_Steps);
		Pulse12.RenderText();
	
		switch(Pulse21_Expo)
		{
			case 1:
			{
				Pulse21.Write_Value((float) Pulse21_Value / (float) 1000000000);
				break;
			}
			case 2:
			{
				Pulse21.Write_Value((float) Pulse21_Value / (float) 1000000);
				break;
			}
			case 3:
			{
				Pulse21.Write_Value((float) Pulse21_Value / (float) 1000);
				break;
			}
		}
		Pulse21.Write_StepSize(Pulse21_Steps);
	        Pulse21.RenderText();

		switch(Pulse22_Expo)
		{
			case 1:
			{
				Pulse22.Write_Value((float) Pulse22_Value / (float) 1000000000);
				break;
			}
			case 2:
			{
				Pulse22.Write_Value((float) Pulse22_Value / (float) 1000000);
				break;
			}
			case 3:
			{
				Pulse22.Write_Value((float) Pulse22_Value / (float) 1000);
				break;
			}
		}
		Pulse22.Write_StepSize(Pulse22_Steps);
		Pulse22.RenderText();


//BF changed
		//Math_Offset_mul.Write_StepSize(1);
	//	Math_Offset_mul.Write_Value((float)(Math_Mul_Offset / 50000 ));
	//	Math_Offset_mul.RenderText();
	
	
		//Math_Offset_sub.Write_StepSize(1);
	//	Math_Offset_sub.Write_Value((float)(Math_Sub_Offset / 50000 ));
	//	Math_Offset_sub.RenderText();
	
	
		//Math_Offset_add.Write_StepSize(0.001);
	//	Math_Offset_add.Write_Value((float)(Math_Add_Offset / 50000 ));
	//	Math_Offset_add.RenderText();
//BF end	

		UserIF::UpdateChannel(eCh1, Channel[eCh1].active);	
		UserIF::UpdateChannel(eCh2, Channel[eCh2].active);
					
		VoltageChangedCh1 = 1;
		VoltageChangedCh2 = 1;

		RotarySteps = 0;
		UserIF::ON_Zero_Channel_1();
		RotarySteps = 0;
		UserIF::ON_Zero_Channel_2();
	
		Hardware::SetSwitches(eCh1, Selected_Voltage_CH1);
		Hardware::SetSwitches(eCh2, Selected_Voltage_CH2);

		if (NumberOfChannels > 2)
		{
	
			UserIF::UpdateChannel(eCh3, Channel[eCh3].active);	
			UserIF::UpdateChannel(eCh4, Channel[eCh4].active);

			VoltageChangedCh3 = 1;
			VoltageChangedCh4 = 1;

			RotarySteps = 0;
			UserIF::ON_Zero_Channel_3();
			RotarySteps = 0;
			UserIF::ON_Zero_Channel_4();
		
			Hardware::SetSwitches(eCh3, Selected_Voltage_CH3);
			Hardware::SetSwitches(eCh4, Selected_Voltage_CH4);
	
		}
		else
		{
			UserIF::UpdateChannel(eCh3, 0);	
			UserIF::UpdateChannel(eCh4, 0);
		}


		TriggerLevelChanged = 1;
		TriggerModeChanged = 1;	
		TriggerWayChanged = 1;
		TimebaseChanged = 1;
		
		//MenuStatusChanged = 1;
		MenuAktive = 1;
	
		//Cursor_Data_First_Draw = 0;
		//Quick_Measure_First_Draw = 0;
	
		Display::RecalcTimeParameters();
		Hardware::SetupTrigger();
		Display::CALCPRETRIGGER();

//		GainIdx = MenuStatus[MENU_HARDWARE][1] - 230;	//BF #004
//		Signal::Build_ScaleLookupTable();

		ACQ_Average = (MenuStatus[MENU_ACQUIRE][1] - AVRG_OFF);

		if (Memory_Window_visible) 
		Display::CloseMemoryBrowser();		//close memory window
	
		//Display::DrawInitialScreen();
		//Display::RefreshScreen();
	
		if(Signal::_FFT_Mode != FFT_OFF)
		{
			Display::FFT_CalcCursorData();
			Signal::FFT_buildScaleLookupTable();
		}
		else
		Display::CALCCURSORDATA();
	
		Hardware::Send_LED();		

		if (USTB_Mode != USTB_OFF)		// if USTB mode was active bevor prepare buffers and pointerrs
		Signal::USTB_clearBuffer();
		else
		{
			S1Ptr = SIGNAL1;		// adjust signal pointers to 16k signal buffers
			S2Ptr = SIGNAL2;
			S3Ptr = SIGNAL3;
			S4Ptr = SIGNAL4;
			SMPtr = SIGNALM;
		}
		
		rcode = 1;	//success
	}

	return rcode;
}

//#####################################################################################################################################################
//BF Save/Recall -> load sample data from flash
void AMDFlash::LoadTracePoints(unsigned long *flash_address)
{
	unsigned long lX, bufdata;
	unsigned long buf[300];	

	S1Ptr = SIGNAL1;
	S2Ptr = SIGNAL2;
	S3Ptr = SIGNAL3;
	S4Ptr = SIGNAL4;
	SMPtr = SIGNALM;
	
	for (lX = 0; lX < 300; lX++) buf[lX] = flash_address[0x4000 - lX];		
	
	if (buf[2] == 0x22334455 && buf[299] == 0x23232323)
	{

		for (lX = 0; lX < 0x1000; lX++)
		{
			bufdata = *(flash_address + lX + 0x0000);
			S1Ptr[(lX * 4) + 0] = (unsigned char) ((bufdata & 0x000000FF) >> 0);	
			S1Ptr[(lX * 4) + 1] = (unsigned char) ((bufdata & 0x0000FF00) >> 8);
			S1Ptr[(lX * 4) + 2] = (unsigned char) ((bufdata & 0x00FF0000) >> 16);
			S1Ptr[(lX * 4) + 3] = (unsigned char) ((bufdata & 0xFF000000) >> 24);
		
			bufdata = *(flash_address + lX + 0x1000);
			S2Ptr[(lX * 4) + 0] = (unsigned char) ((bufdata & 0x000000FF) >> 0);	
			S2Ptr[(lX * 4) + 1] = (unsigned char) ((bufdata & 0x0000FF00) >> 8);
			S2Ptr[(lX * 4) + 2] = (unsigned char) ((bufdata & 0x00FF0000) >> 16);
			S2Ptr[(lX * 4) + 3] = (unsigned char) ((bufdata & 0xFF000000) >> 24);

		}	

		if (NumberOfChannels == 4)
		{
			for (lX = 0; lX < 0x1000; lX++)
			{
				bufdata = *(flash_address + lX + 0x2000);
				S3Ptr[(lX * 4) + 0] = (unsigned char) ((bufdata & 0x000000FF) >> 0);	
				S3Ptr[(lX * 4) + 1] = (unsigned char) ((bufdata & 0x0000FF00) >> 8);
				S3Ptr[(lX * 4) + 2] = (unsigned char) ((bufdata & 0x00FF0000) >> 16);
				S3Ptr[(lX * 4) + 3] = (unsigned char) ((bufdata & 0xFF000000) >> 24);
			
				bufdata = *(flash_address + lX + 0x3000);
				S4Ptr[(lX * 4) + 0] = (unsigned char) ((bufdata & 0x000000FF) >> 0);	
				S4Ptr[(lX * 4) + 1] = (unsigned char) ((bufdata & 0x0000FF00) >> 8);
				S4Ptr[(lX * 4) + 2] = (unsigned char) ((bufdata & 0x00FF0000) >> 16);
				S4Ptr[(lX * 4) + 3] = (unsigned char) ((bufdata & 0xFF000000) >> 24);
			}	

			//delete last 1200 byte which are used for the settings
			for(int i=0;i < 1200;i++) S4Ptr[0x4000-i] = ADC_ZERO;
		}


	}
}
//#####################################################################################################################################################
//BF Save/Recall -> load trace
void AMDFlash::ReadTraceFromFlash(unsigned long *flash_address)
{
	if (LoadTraceSettings(flash_address)) { LoadTracePoints(flash_address); }	// load trace points if load of settings where successful
}

//#####################################################################################################################################################
/* BF not needed at this time
void AMDFlash::copy_flash_to_planes(void)
{
	unsigned long lX;
#ifdef _Debug_	
	printf("Restore the planes \n");
#endif

//printf("AMDFlash::copy_flash_to_planes\n");	
	for (lX = 0; lX < 9600; lX++)
	{	
		UI_Plane1[lX] = *(UI_Plane1_Flash + lX);	
		Buffer_UI2Plane[lX] = *(UI_Plane2_Flash + lX);	
		UI_Plane3[lX] = *(UI_Plane3_Flash + lX);	
		UI_Plane4[lX] = *(UI_Plane4_Flash + lX);	
		UI_Plane5[lX] = *(UI_Plane5_Flash + lX);	
		Channel_Plane1[lX] = *(Channel_Plane1_Flash + lX);	
		Channel_Plane2[lX] = *(Channel_Plane2_Flash + lX);	
		Channel_Plane3[lX] = *(Channel_Plane3_Flash + lX);	
		Channel_Plane4[lX] = *(Channel_Plane4_Flash + lX);	
		Channel_Math_Plane[lX] = *(Channel_Math_Plane_Flash + lX);	
		Buffer_Plane1[lX] = *(Channel_Plane1_Flash + lX);	
		Buffer_Plane2[lX] = *(Channel_Plane2_Flash + lX);	
		Buffer_Plane3[lX] = *(Channel_Plane3_Flash + lX);	
		Buffer_Plane4[lX] = *(Channel_Plane4_Flash + lX);	
		Buffer_PlaneM[lX] = *(Channel_Math_Plane_Flash + lX);			
		Memory_Plane1[lX] = *(Memory_Plane1_Flash + lX);	
		Memory_Plane2[lX] = *(Memory_Plane2_Flash + lX);	
		Memory_Plane3[lX] = *(Memory_Plane3_Flash + lX);	
		Marker_Plane1[lX] = *(Marker_Plane1_Flash + lX);	
		Marker_Plane2[lX] = *(Marker_Plane2_Flash + lX);	
		Grid_Plane[lX] = *(Grid_Plane_Flash + lX);	
	}
	UpdateMenuTextPlane = true;
}
*/
//#####################################################################################################################################################
// BF needed for copying splash logo to screen
void AMDFlash::copy_flash_to_plane(unsigned long *flash_address, unsigned long *plane)
{
	unsigned long lX;
	//printf("AMDFlash::copy_flash_to_plane\n");		
	for (lX = 0; lX < 9600; lX++)
	{	
		plane[lX] = flash_address[lX];	
	}
}
//#####################################################################################################################################################
// save actual configuration to flash
void AMDFlash::WriteConfig(void)
{
	if (ConfigChanged == 0) return;

	unsigned long *sect_addr;
	unsigned long buf[CONFIG_SLOT_SIZE_INT];			// define size of configuration buffer
	int led_value = 0x0000;
	char write_error = 0;

	//if (AS_request){ ConfigChanged = 0; return; }		//don't write config if autoscale is active

	//printf("AMDFlash::Write_Config_Flash\r\n");

	// switch to next free config slot
	_ConfigSlot += CONFIG_SLOT_SIZE_INT;

	// check if sector has to be erased
	if (_ConfigSlot + CONFIG_SLOT_SIZE_INT >= FLASH_SECTOR_SIZE_INT)
	{
	        //printf("Config sector full, erase\r\n");
		if (!nr_flash_erase_sector(CONFIG_FLASH))		// erase config area
		{ write_error = true; }
		else
		{ _ConfigSlot = 0; }
	}


	//backup LED setting
	if (LED_ON[0] == 1) led_value = led_value | 0x0001;		// Select Channel 1
	if (LED_ON[1] == 1) led_value = led_value | 0x0002;		// Select Channel 2
	if (LED_ON[2] == 1) led_value = led_value | 0x0004;		// Select Channel 3
	if (LED_ON[3] == 1) led_value = led_value | 0x0008;		// Select Channel 4
	if (LED_ON[4] == 1) led_value = led_value | 0x0010;		// Select Channel Math
	if (LED_ON[5] == 1) led_value = led_value | 0x0020;		// General
	if (LED_ON[6] == 1) led_value = led_value | 0x0040;		// Cursor						
	if (LED_ON[7] == 1) led_value = led_value | 0x0080;		// Quick Measure
	if (LED_ON[8] == 1) led_value = led_value | 0x0100;		// Edge
	if (LED_ON[9] == 1) led_value = led_value | 0x0200;		// BF Trigger armed
	if (LED_ON[10] == 1) led_value = led_value | 0x0400;		// Pulse Width
	if (LED_ON[11] == 1) led_value = led_value | 0x0800;		// BF Trigger Event
	if (LED_ON[12] == 1) led_value = led_value | 0x1000;		// Run/Stop Green
	if (LED_ON[13] == 1) led_value = led_value | 0x2000;		// Run/Stop Red
	if (LED_ON[14] == 1) led_value = led_value | 0x4000;		// Single Red					
	if (LED_ON[15] == 1) led_value = led_value | 0x8000;		// Single Green	

	
	buf[0] = (unsigned long) 0xAACC5523;		// Configuration Start Sign;
	//buf[1] = (unsigned long) 
	//buf[2] = (unsigned long) 0x22334455;
	//buf[3] = (unsigned long) 0x00;		
	//buf[4] = (unsigned long) tc_version;		// Version
	//buf[5] = (unsigned long) tc_subversion;	// SubVersion
	//buf[6] = (unsigned long) tc_dev_version;	// Revision
	//buf[7] = (unsigned long) tc_dev_subversion;	// Compile Status
	//buf[8] = (unsigned long) 0x00;   	

	buf[9] = (unsigned long) FlashPreset;		//BF preset flag

	// Timebase
	buf[10] = (unsigned long) MenuStatus[MENU_TIMEBASE][0];		//Main
	buf[11] = (unsigned long) MenuStatus[MENU_TIMEBASE][1];		//Delayed
	buf[12] = (unsigned long) MenuStatus[MENU_TIMEBASE][2];		//XY mode
	buf[13] = (unsigned long) MenuStatus[MENU_TIMEBASE][3];		
	buf[14] = (unsigned long) MenuStatus[MENU_TIMEBASE][4];		
	buf[15] = (unsigned long) MenuStatus[MENU_DISPLAY][2];	
	buf[16] = (unsigned long) XY_Mode;
	buf[17] = (unsigned long) MainTimebase;	
	buf[18] = (unsigned long) USTB_Mode;
	buf[19] = (unsigned long) USTB_Mode_bak;
	buf[20] = (unsigned long) USTB_Dir;	
	buf[21] = (unsigned long) USTB_Disp;	
	buf[22] = (unsigned long) DelayedTimebase;

	buf[23] = (unsigned long) VirtualTimebase;
	buf[24] = (unsigned long) dmode_Window_Offset_Pos;
	buf[25] = (unsigned long) MenuStatus[MENU_USTB][2];				// USTB buffer size

	buf[26] = (unsigned long) New_Menu;						//BF added
	buf[27] = (unsigned long) Active_Menu;						//BF added
	buf[28] = (unsigned long) MenuAktive;						//BF added

	buf[29] = (unsigned long) MenuStatus[MENU_USTB][0];				//USTB mode			
	buf[30] = (unsigned long) MenuStatus[MENU_USTB][1];				//USTB display
	buf[31] = (unsigned long) MenuStatus[MENU_TRIGGERSUB][0];				// LED 1
	buf[32] = (unsigned long) MenuStatus[MENU_TRIGGERSUB][1];				// LED 2
    	//buf[33] = (unsigned long) free;
	// buf[34] = (unsigned long) free;	
	
	// Trigger
	buf[35] = (unsigned long) MenuStatus[MENU_TRIGGERMODE][0];				// Triggering - Auto / Normal / Combi
	buf[36] = (unsigned long) MenuStatus[MENU_TRIGGERMODE][1];				// Triggering - Coupling DC/AC
	buf[37] = (unsigned long) MenuStatus[MENU_TRIGGERSUB][2];				// Trigger sub menu LED setting
	buf[38] = (unsigned long) MenuStatus[MENU_TRIGGERMODE][4];				// Triggering - External Probe
	buf[39] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][3];				// Triggering - TV

	buf[40] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][0];    				// Triggering - positive edge
	buf[41] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][1]; 				// Triggering - Edge Source 
	buf[42] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][2];				// Triggering - externel triggering
	buf[43] = (unsigned long) MenuStatus[MENU_PULSEWIDTH][0];				// Triggering - Source Channel 1
	buf[44] = (unsigned long) MenuStatus[MENU_PULSEWIDTH][1];				// Triggering - Negative Pulse
	buf[45] = (unsigned long) MenuStatus[MENU_PULSEWIDTH][2];				// Triggering - bigger - smaller - then
	buf[46] = (unsigned long) Hardware::_TriggerSource;					// Trigger - Source 
	buf[47] = (unsigned long) MenuStatus[MENU_TRIGGERSUB][3];				// Trigger sub menu Single Trigger setting;
	buf[48] = (unsigned long) MenuBtnFocus[MENU_PULSEWIDTH]; //PulseToggle;			// Pulse width button focus
	//buf[49] = (unsigned long) free;
	buf[50] = (unsigned long) FFT_TriggerMode_bak;					//BF backup the trigger mode
	buf[51] = (unsigned long) Trigger_Pos_CH1;                                          // Trigger Level Channel 1
	buf[52] = (unsigned long) Trigger_Pos_CH2;                                          // Trigger Level Channel 2
	buf[53] = (unsigned long) Trigger_Pos_CH3;                                          // Trigger Level Channel 3
	buf[54] = (unsigned long) Trigger_Pos_CH4;                                          // Trigger Level Channel 4
	//buf[55] = (unsigned long) trg_val_reg;                                              // Trigger Value
	//buf[56] = (unsigned long) trg_hyst_reg;                                             // Trigger Hysterese
	//buf[57] = (unsigned long) trig_range_reg;                                           // Trigger Range
	//buf[58] = (unsigned long) trig_holdoff_reg;                                         // Trigger Holdoff
	//buf[59] = (unsigned long) trig_width_reg;                                           // Trigger Width

	//buf[60] = (unsigned long) 				                                              
	//buf[61] = (unsigned long) adc_ctrl_reg;                                             	// ADC Control register
	//buf[62] = (unsigned long) pre_reg;                                                  	// PreTrigger Value
	//buf[63] = (unsigned long) Hardware::_ctrl_reg12;					// control register FPGA1                                         
	buf[64] = (unsigned long) MenuStatus[MENU_TRIGGERSUB][4];				// Trigger sub menu Auto Pre Trigger setting;
	//buf[65] = (unsigned long) free
	buf[66] = (unsigned long) TriggerWay;                                               // Triggerway
	buf[67] = (unsigned long) Trigger_Pos_CHE;                                          // Trigger extern value
	
	//---------------------------------------------------
	// 		Channel 1
	//---------------------------------------------------
	buf[68] = (unsigned long) MenuStatus[1][0];                                         // Channel - Coupling
	buf[69] = (unsigned long) MenuStatus[1][1];                                         // Channel - BW Linit
	buf[70] = (unsigned long) MenuStatus[1][2];                                         // Channel - Invert
	buf[71] = (unsigned long) MenuStatus[1][3];                                         // Channel - Probe
	
	buf[72] = (unsigned long) SwitchesCH1;                                              // Switches for Channel 1 and for the other the AC bit
	buf[73] = (unsigned long) Selected_Voltage_CH1;                                     // Voltage 5V
	buf[74] = (unsigned long) Trigger_Pos_CH1_Old;
	buf[75] = (unsigned long) NoDisplay[0];
	buf[76] = (unsigned long) thres_low[0];						// QM thresholds
	buf[77] = (unsigned long) thres_mdl[0];						// QM thresholds
	buf[78] = (unsigned long) thres_up[0];						// QM thresholds
	buf[79] = (unsigned long) thres_type[0];					// QM threshold type
	buf[80] = (unsigned long) Selected_Voltage_CH1_bak;
	buf[81] = (unsigned long) Trigger_Pos_CH1_bak; 
	buf[82] = (unsigned long) CH1_DAC_Offset;                                           // Channel DAC
	buf[83] = (unsigned long) ZeroLevelCH1;                                             // ZeroLevel
	buf[84] = (unsigned long) Virtual_ZeroLevelCH1;                                     // Virtual ZeroLevel
        //buf[85] = (unsigned long)
    	//buf[85] = (unsigned long) 

	//---------------------------------------------------
	// 		Channel 2
	//---------------------------------------------------
	buf[86] = (unsigned long) MenuStatus[2][0];                                         // Channel - Coupling
	buf[87] = (unsigned long) MenuStatus[2][1];                                         // Channel - BW Linit
	buf[88] = (unsigned long) MenuStatus[2][2];                                         // Channel - Invert
	buf[89] = (unsigned long) MenuStatus[2][3];                                         // Channel - Probe

	buf[90] = (unsigned long) SwitchesCH2;                                              // Switches for Channel
	buf[91] = (unsigned long) Selected_Voltage_CH2;                                     // Voltage 5V
	buf[92] = (unsigned long) Trigger_Pos_CH2_Old;
	buf[93] = (unsigned long) NoDisplay[1];
	buf[94] = (unsigned long) thres_low[1];						// QM thresholds
	buf[95] = (unsigned long) thres_mdl[1];						// QM thresholds
	buf[96] = (unsigned long) thres_up[1];						// QM thresholds
	buf[97] = (unsigned long) thres_type[1];					// QM threshold type
	buf[98] = (unsigned long) Selected_Voltage_CH2_bak;
	buf[99] = (unsigned long) Trigger_Pos_CH2_bak; 
	buf[100] = (unsigned long) CH2_DAC_Offset;                                           // Channel DAC
	buf[101] = (unsigned long) ZeroLevelCH2;                                             // ZeroLevel
	buf[102] = (unsigned long) Virtual_ZeroLevelCH2;                                     // Virtual ZeroLevel
        //buf[103] = (unsigned long) 
	//buf[103] = (unsigned long) 

	//---------------------------------------------------
	// 		Channel 3
	//---------------------------------------------------
	buf[104] = (unsigned long) MenuStatus[3][0];                                         // Channel - Coupling
	buf[105] = (unsigned long) MenuStatus[3][1];                                         // Channel - BW Linit
	buf[106] = (unsigned long) MenuStatus[3][2];                                         // Channel - Invert
	buf[107] = (unsigned long) MenuStatus[3][3];                                         // Channel - Probe
	buf[108] = (unsigned long) SwitchesCH3;                                              // Switches for Channel
	buf[109] = (unsigned long) Selected_Voltage_CH3;                                     // Voltage 5V
	buf[110] = (unsigned long) Trigger_Pos_CH3_Old;
	buf[111] = (unsigned long) NoDisplay[2];;
	buf[112] = (unsigned long) thres_low[2];					// QM thresholds
	buf[113] = (unsigned long) thres_mdl[2];					// QM thresholds
	buf[114] = (unsigned long) thres_up[2];						// QM thresholds
	buf[115] = (unsigned long) thres_type[2];					// QM threshold type
	buf[116] = (unsigned long) Selected_Voltage_CH3_bak;
	buf[117] = (unsigned long) Trigger_Pos_CH3_bak; 
	buf[118] = (unsigned long) CH3_DAC_Offset;                                           // Channel DAC
	buf[119] = (unsigned long) ZeroLevelCH3;                                             // ZeroLevel
	buf[120] = (unsigned long) Virtual_ZeroLevelCH3;                                     // Virtual ZeroLevel
       // buf[121] = (unsigned long)
	//buf[121] = (unsigned long)

	//---------------------------------------------------
	// 		Channel 4
	//---------------------------------------------------
	buf[122] = (unsigned long) MenuStatus[4][0];                                         // Channel - Coupling
	buf[123] = (unsigned long) MenuStatus[4][1];                                         // Channel - BW Linit

	buf[124] = (unsigned long) MenuStatus[4][2];                                         // Channel - Invert
	buf[125] = (unsigned long) MenuStatus[4][3];                                         // Channel - Probe
	buf[126] = (unsigned long) SwitchesCH4;                                              // Switches for Channel
	buf[127] = (unsigned long) Selected_Voltage_CH4;                                     // Voltage 5V
	buf[128] = (unsigned long) Trigger_Pos_CH4_Old;
	buf[129] = (unsigned long) NoDisplay[3];;
	buf[130] = (unsigned long) thres_low[3];					// QM thresholds;
	buf[131] = (unsigned long) thres_mdl[3];					// QM thresholds;
	buf[132] = (unsigned long) thres_up[3];						// QM thresholds;
	buf[133] = (unsigned long) thres_type[3];					// QM threshold type
	buf[134] = (unsigned long) Selected_Voltage_CH4_bak;
	buf[135] = (unsigned long) Trigger_Pos_CH4_bak; 
	buf[136] = (unsigned long) CH4_DAC_Offset;                                           // Channel DAC
	buf[137] = (unsigned long) ZeroLevelCH4;                                             // ZeroLevel
	buf[138] = (unsigned long) Virtual_ZeroLevelCH4;                                     // Virtual ZeroLevel
       
	buf[139] = (unsigned long) FFT_Layout;                                          // Layout in FFT function

	//Math Settings
	buf[140] = (unsigned long) Channel[eChM].active;                                      // Math channel
	//buf[141] = (unsigned long) MenuStatus[MENU_MATH][0];                                 // scale
	buf[142] = (unsigned long) MenuStatus[MENU_MATH][2];                                 // math 1*2 
	buf[143] = (unsigned long) MenuStatus[MENU_MATH][3];                                 // math 1-2 
	buf[144] = (unsigned long) MenuStatus[MENU_MATH][4];                                 // math 1+2 
	
	//FFT Settings
	buf[145] = (unsigned long) Signal::_FFT_Mode;
	buf[146] = (unsigned long) Signal::_FFT_Length;
	buf[147] = (unsigned long) Signal::_FFT_Level;
	buf[148] = (unsigned long) Signal::_FFT_Windex;
//BF not used	buf[149] = (unsigned long) FFT_Scale;
	
	buf[150] = (unsigned long) MenuStatus[MENU_FFT][0]; 
	buf[151] = (unsigned long) MenuStatus[MENU_FFT][1];
	buf[152] = (unsigned long) MenuStatus[MENU_FFT][2];
	buf[153] = (unsigned long) MenuStatus[MENU_FFT][3];
	
	// Cursors    	
	buf[154] = (unsigned long) MenuStatus[MENU_CURSOR][0];                                        // Cursor soruce
	buf[155] = (unsigned long) Cursor_Enabled;
	buf[156] = (unsigned long) Cursor_Both_Selected;
	buf[157] = (unsigned long) SelectedCursor;

	buf[158] = (unsigned long) Cursor_Horizontal_Active;
	buf[159] = (unsigned long) Cursor_Horizontal_Position_Real1;
	buf[160] = (unsigned long) Cursor_Horizontal_Position_Real2;

	buf[161] = (unsigned long) Cursor_Vertical_Active;
	buf[162] = (unsigned long) Cursor_Vertical_Position_TY_1;
	buf[163] = (unsigned long) Cursor_Vertical_Position_TY_2;

	// Quick Measure
	buf[164] = (unsigned long) QM_Enabled;
	buf[165] = (unsigned long) QM_Type[0];
	buf[166] = (unsigned long) QM_Type[1];
	buf[167] = (unsigned long) QM_Type[2];
	buf[168] = (unsigned long) QM_Channel[0];
	buf[169] = (unsigned long) QM_Channel[1];
	buf[170] = (unsigned long) QM_Channel[2];
	buf[171] = (unsigned long) QM_Horizontal_Active;
	buf[172] = (unsigned long) QM_Vertical_Active;
	buf[173] = (unsigned long) Quick_Measure_Threshold_btn_select;
	
	// Display
	buf[174] = (unsigned long) GridColorArrayPos; 					// Grid color array index
	buf[175] = (unsigned long) Hardware::_regADC[eFPGA1].grid_color;
	buf[176] = (unsigned long) MenuStatus[MENU_DISPLAY][3];				// no display
	buf[177] = (unsigned long) DottedGrid;
	buf[178] = (unsigned long) MenuStatus[MENU_DISPLAY][0];				// Display - Persist
	buf[179] = (unsigned long) MenuStatus[MENU_DISPLAY][4];                                         // Display - Draw Mode
	
	buf[180] = (unsigned long) MenuStatus[MENU_ACQUIRE][0];                                         // Acquire - Logic
	buf[181] = (unsigned long) MenuStatus[MENU_ACQUIRE][1];                                         // Acquire - Averaging
	buf[183] = (unsigned long) MenuStatus[MENU_ACQUIRE][2];                                         // Acquire - Filter Setting
	buf[184] = (unsigned long) MenuStatus[MENU_ACQUIRE][3];                                         // Acquire - Interpolation
	
	buf[185] = (unsigned long) MenuStatus[MENU_AUTOSCALE][3];                                       // Autoscale - Setup TB search
	//buf[186] = (unsigned long) 
	
	buf[187] = (unsigned long) MenuPopupStatus[8][0];
	buf[188] = (unsigned long) MenuPopupStatus[8][1];
	buf[189] = (unsigned long) MenuPopupStatus[8][2];  				//BF popup external trigger
	buf[190] = (unsigned long) MenuPopupStatus[11][0]; 				//BF popup TV trigger
	buf[191] = (unsigned long) MenuPopupStatus[11][1];

	buf[192] = (unsigned long)led_value;						// BF reserved for LED setting -> see at top

	buf[193] = (unsigned long) Channel[eCh1].active;
	buf[194] = (unsigned long) Channel[eCh2].active;
	buf[195] = (unsigned long) Channel[eCh3].active;
	buf[196] = (unsigned long) Channel[eCh4].active;

	buf[197] = (unsigned long) FFT_grid;				//FFT Grid division

 	buf[198] = (unsigned long) ColorPalette;
	
	
	buf[199] = (unsigned long) HoldOff_Value;
	buf[200] = (unsigned long) HoldOff_Expo;
	buf[201] = (unsigned long) HoldOff_Steps;
	
	buf[202] = (unsigned long) Pulse11_Value;
	buf[203] = (unsigned long) Pulse11_Expo;
	buf[204] = (unsigned long) Pulse11_Steps;
	
	buf[205] = (unsigned long) Pulse12_Value;
	buf[206] = (unsigned long) Pulse12_Expo;
	buf[207] = (unsigned long) Pulse12_Steps;
	
	buf[208] = (unsigned long) Pulse21_Value;
	buf[209] = (unsigned long) Pulse21_Expo;
	buf[210] = (unsigned long) Pulse21_Steps;
	
	buf[211] = (unsigned long) Pulse22_Value;
	buf[212] = (unsigned long) Pulse22_Expo;
	buf[213] = (unsigned long) Pulse22_Steps;

//BF new Math values
	buf[214] = (unsigned int) Math_Offset;
	buf[215] = (unsigned int) Math_Mul_Offset;
	buf[216] = (unsigned int) Math_Sub_Offset;
	buf[217] = (unsigned int) Math_Add_Offset;
	buf[218] = (unsigned int) math_mul_scale;
	buf[219] = (unsigned int) math_mul_scale;
	buf[220] = (unsigned int) math_sub_scale;
	buf[221] = (unsigned int) math_add_scale;
//BF end

	//buf[222] = (unsigned long)
	//buf[223] = (unsigned long)
	//buf[224] = (unsigned long)
	//buf[225] = (unsigned long)
	//buf[226] = (unsigned long)


	buf[229] = (unsigned long) Trig_Pos_Mem;
	buf[230] = (unsigned long) Trig_Pos_Display;
	//buf[231] = (unsigned long)
	buf[232] = (unsigned long) MemWinStart;
	
	buf[233] = (unsigned long) Trig_Pos_Mem_old;
	buf[234] = (unsigned long) Trig_Pos_Display_old;
	buf[235] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][4];
	buf[236] = (unsigned long) MemWinStart_old;
	
	buf[237] = (unsigned long) ZoomFactor;
	buf[238] = (unsigned long) ZoomFactor_old;
	buf[239] = (unsigned long) ZoomFactorDel;

	buf[240] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][0];
	buf[241] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][1];
	buf[242] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][2];

	buf[243] = (unsigned long) QM_Source;
	//buf[244] = (unsigned long)
	//buf[245] = (unsigned long)
	//buf[246] = (unsigned long)
	
	buf[247] = (unsigned long) QP_mode;		//BF #006

// BF new zerolevels for XY
	buf[248] = (unsigned long) Virtual_ZeroLevelXYCH1;
	buf[249] = (unsigned long) Virtual_ZeroLevelXYCH2;
	buf[250] = (unsigned long) Virtual_ZeroLevelXYCH3;
	buf[251] = (unsigned long) Virtual_ZeroLevelXYCH4;
// BF end

	//BF backup values must be saved too
	buf[252] = (unsigned long) ZeroLevelCH1_bak;
	buf[253] = (unsigned long) ZeroLevelCH2_bak;
	buf[254] = (unsigned long) ZeroLevelCH3_bak;
	buf[255] = (unsigned long) ZeroLevelCH4_bak;

	buf[256] = (unsigned long) Virtual_ZeroLevelCH1_bak;
	buf[257] = (unsigned long) Virtual_ZeroLevelCH2_bak;
	buf[258] = (unsigned long) Virtual_ZeroLevelCH3_bak;
	buf[259] = (unsigned long) Virtual_ZeroLevelCH4_bak;

	//buf[260] = (unsigned long) free;


	buf[261] = (unsigned long) Cursor_Vertical_Position_XY_1;
	buf[262] = (unsigned long) Cursor_Vertical_Position_XY_2;
	buf[263] = (unsigned long) Cursor_Vertical_Position_FFT_1;
	buf[264] = (unsigned long) Cursor_Vertical_Position_FFT_2;
	buf[265] = (unsigned long) SavePointer;


	buf[266] = (unsigned long) ChannelStatusOld[eCh1];
	buf[267] = (unsigned long) ChannelStatusOld[eCh2];
	buf[268] = (unsigned long) ChannelStatusOld[eCh3];
	buf[269] = (unsigned long) ChannelStatusOld[eCh4];
	buf[270] = (unsigned long) ChannelStatusOld[eChM];
	
	buf[271] = (unsigned long) DrawMono;
	buf[272] = (unsigned long) MainTimebase_bak;
	buf[273] = (unsigned long) MenuStatus[MENU_QMTHRESHOLDS][0];
	buf[274] = (unsigned long) MenuStatus[MENU_QMTHRESHOLDS][1];

	buf[275] = (unsigned long) ZeroLevelCH1_Main_bak;
	buf[276] = (unsigned long) ZeroLevelCH2_Main_bak;
	buf[277] = (unsigned long) ZeroLevelCH3_Main_bak;
	buf[278] = (unsigned long) ZeroLevelCH4_Main_bak;

	buf[279] = (unsigned long) Virtual_ZeroLevelCH1_Main_bak;
	buf[280] = (unsigned long) Virtual_ZeroLevelCH2_Main_bak;
	buf[281] = (unsigned long) Virtual_ZeroLevelCH3_Main_bak;
	buf[282] = (unsigned long) Virtual_ZeroLevelCH4_Main_bak;

	//buf[283] = (unsigned long) 
	//buf[284] = (unsigned long)
	//buf[285] = (unsigned long)
	//buf[286] = (unsigned long) 
	//buf[287] = (unsigned long) 
	//buf[288] = (unsigned long) 

	buf[289] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][0];			// grid color
	buf[290] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][1];			// grid layout
	buf[291] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][2];			// status layout
	buf[292] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][3];			// QM-cursors on/off
	buf[293] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][4];			// OSS on/off
	//buf[294] = (unsigned long)MenuStatus[MENU_DISPL_SETUP][5];			// not used

	buf[295] = (unsigned long) thres_low[4];					// QM thresholds;
	buf[296] = (unsigned long) thres_mdl[4];					// QM thresholds;
	buf[297] = (unsigned long) thres_up[4];						// QM thresholds;
	buf[298] = (unsigned long) thres_type[4];					// QM threshold type

	buf[299] = 0x23232323;

	if (!nr_flash_write_buffer(CONFIG_FLASH + _ConfigSlot, buf, CONFIG_SLOT_SIZE_INT))
	{ write_error = true; }

	sect_addr = (unsigned long) nasys_main_flash + CONFIG_FLASH + _ConfigSlot;		// get config slot address

	if (sect_addr[0] == 0xAACC5523 && sect_addr[299] == 0x23232323)
	{
		//printf("Configuration written to flash config sector\r\n");
	}
	else
	{ printf("Flash checksum error\r\n"); write_error = true;}


	if(_BackupConfig == true || write_error == true)
	{	
		//write backup
		if(!nr_flash_erase_sector(CONFIG_BACKUP)) { return; }				// erase config backup area 
		nr_flash_write_buffer(CONFIG_BACKUP, buf, 300);
	
		//Check values
		sect_addr = (unsigned long) nasys_main_flash + CONFIG_BACKUP;			// get config backup start address
		if (sect_addr[0] == 0xAACC5523 && sect_addr[299] == 0x23232323)
		{
			//printf("Configuration written to flash backup sector\r\n");
		}
		else
		printf("Flash checksum error\r\n");
	}


	ConfigChanged = 0;
	

}
//#####################################################################################################################################################
// load configuration from flash
void AMDFlash::ReadConfig(void)
{

	for(int lbackup=0;lbackup<2;lbackup++)								// try only two times - 1. normal config, 2. backup config 
	{
		long unsigned *sect_addr;
		long unsigned buf[CONFIG_SLOT_SIZE_INT];

		if (lbackup)
		{ sect_addr = (long unsigned) nasys_main_flash + CONFIG_BACKUP; }			// get config backup start address
		else
		{ sect_addr = (long unsigned) nasys_main_flash + CONFIG_FLASH +_ConfigSlot; }		// get config start address
		
		for (int x = 0; x < CONFIG_SLOT_SIZE_INT; x++) buf[x] = sect_addr[x];			// read data from flash into buffer
		
		if ((buf[0] == 0xAACC5523) && (buf[299] == 0x23232323))
		{
	
			//printf("AMDFlash::Read_Config_Flash\r\n");
	
			FlashPreset = (unsigned char) buf[9];
	
			// Timebase      	
			MenuStatus[MENU_TIMEBASE][0] = (short) buf[10];
			MenuStatus[MENU_TIMEBASE][1] = (short) buf[11];
			MenuStatus[MENU_TIMEBASE][2] = (short) buf[12];
			MenuStatus[MENU_TIMEBASE][3] = (short) buf[13];
			MenuStatus[MENU_TIMEBASE][4] = (short) buf[14];
	
			MenuStatus[MENU_DISPLAY][2] = (short) buf[15];
	
			XY_Mode = (char) buf[16];
			MainTimebase = (int32_t) buf[17];
			USTB_Mode = (short) buf[18];
			USTB_Mode_bak = (short) buf[19];
			USTB_Dir  = (short) buf[20];
			USTB_Disp = (short) buf[21];
		
			DelayedTimebase = (int32_t) buf[22];
			
			VirtualTimebase = (int32_t) buf[23];				

			dmode_Window_Offset_Pos = (int) buf[24];
			MenuStatus[MENU_USTB][2] = (short) buf[25];					//USTB buffer size
	
			New_Menu    = (char) buf[26];
			Active_Menu = (char) buf[27];
			MenuAktive  = (char) buf[28];
	
			MenuStatus[MENU_USTB][0] = (short) buf[29];					//ultra slow timebase mode
			MenuStatus[MENU_USTB][1] = (short) buf[30];
	      		MenuStatus[MENU_TRIGGERSUB][0] = (short) buf[31];				// Trigger LED 1
	      		MenuStatus[MENU_TRIGGERSUB][1] = (short) buf[32];				// Trigger LED 2
			//free = (int) buf[33];
			// free = (unsigned short) buf[34];
		
			// Trigger	
			MenuStatus[MENU_TRIGGERMODE][0] = (short) buf[35];				// Triggering - Auto
			MenuStatus[MENU_TRIGGERMODE][1] = (short) buf[36];				// Triggering - Coupling
			MenuStatus[MENU_TRIGGERSUB][2]  = (short) buf[37];				// Trigger sub menu LED setting Ch 3+4
			MenuStatus[MENU_TRIGGERMODE][4] = (short) buf[38];				// Triggering - External Probe = 1 : 1 
			MenuStatus[MENU_TRIGGEREDGE][3] = (short) buf[39];				// Triggering - TV
			MenuStatus[MENU_TRIGGEREDGE][0] = (short) buf[40];				// Triggering - positive edge
			MenuStatus[MENU_TRIGGEREDGE][1] = (short) buf[41];				// Triggering - Source = channel 1
			MenuStatus[MENU_TRIGGEREDGE][2] = (short) buf[42];				// Triggering - no externel triggering
			MenuStatus[MENU_PULSEWIDTH][0] = (short) buf[43];				// Triggering - Source Channel 1
			MenuStatus[MENU_PULSEWIDTH][1] = (short) buf[44];				// Triggering - Negative Pulse
			MenuStatus[MENU_PULSEWIDTH][2] = (short) buf[45];				// Triggering - bigger - smaller - then
			Hardware::_TriggerSource = (trigSrc_t)  buf[46];				// Trigger - Source
			MenuStatus[MENU_TRIGGERSUB][3] = (short) buf[47];  				// Trigger sub menu Single Trigger setting;
			MenuBtnFocus[MENU_PULSEWIDTH] = (unsigned char) buf[48];			// Pulse width button selection
			//free = (long) buf[49];
			
			FFT_TriggerMode_bak = (short) buf[50]; 						//BF backup the trigger mode
			Trigger_Pos_CH1 = (int) buf[51];                                          	// Trigger Level Channel 1
			Trigger_Pos_CH2 = (int) buf[52];                                          	// Trigger Level Channel 2
			Trigger_Pos_CH3 = (int) buf[53];                                          	// Trigger Level Channel 3
			Trigger_Pos_CH4 = (int) buf[54];                                          	// Trigger Level Channel 4
			//trg_val_reg = (unsigned int) buf[55];                               		// Trigger Value
			//trg_hyst_reg = (unsigned int) buf[56];                                          // Trigger Hysterese
			//trig_range_reg = (unsigned int) buf[57];                                        // Trigger Range
			//trig_holdoff_reg = (unsigned long) buf[58];                                     // Trigger Holdoff
			//trig_width_reg = (unsigned int) buf[59];                                        // Trigger Width
			
			// = (short) buf[60];                               
			//adc_ctrl_reg = (unsigned int) buf[61];                                          // ADC Control register
			//pre_reg = (unsigned int) buf[62];                                               // PreTrigger Value
			//Hardware::_ctrl_reg12 = (unsigned int) buf[63];                                 // control register FPGA 1
			MenuStatus[MENU_TRIGGERSUB][4] = (short) buf[64];			// Trigger sub menu auto pre trigger setting
			// free = (unsigned int) buf[65];                                            
			TriggerWay = (char) buf[66];                                               	// Triggerway
			Trigger_Pos_CHE = (int) buf[67];                                                // Trigger extern value
			
			//---------------------------------------------------
			// 		Channel 1
			//---------------------------------------------------
			MenuStatus[1][0] = (short) buf[68];                                         // Channel - Coupling
			MenuStatus[1][1] = (short) buf[69];                                         // Channel - BW Linit
			MenuStatus[1][2] = (short) buf[70];                                         // Channel - Invert
			MenuStatus[1][3] = (short) buf[71];                                         // Channel - Probe
			SwitchesCH1 = (unsigned short) buf[72];                                              // Switches for Channel 1 and for the other the AC bit
			Selected_Voltage_CH1 = (volatile char) buf[73];                                     // Voltage
			Trigger_Pos_CH1_Old = (int) buf[74];
			NoDisplay[0] = (unsigned char) buf[75];
			thres_low[0] = (short) buf[76];					// QM thresholds;
			thres_mdl[0] = (short) buf[77];					// QM thresholds;
			thres_up[0]  = (short) buf[78];					// QM thresholds;
			thres_type[0] = (short) buf[79];				// QM threshold type
			Selected_Voltage_CH1_bak  = (char) buf[80];
			Trigger_Pos_CH1_bak = (int) buf[81]; 
			CH1_DAC_Offset 	     = (int) buf[82];                                           // Channel DAC
			ZeroLevelCH1         = (int) buf[83];                                             // ZeroLevel
			Virtual_ZeroLevelCH1 = (int) buf[84];                                     // Virtual ZeroLevel
			// free            = (int) buf[85];                                         
			// free = (int) buf[85];                                         
			
			//---------------------------------------------------
			// 		Channel 2
			//---------------------------------------------------
			MenuStatus[2][0] = (short) buf[86];                                         // Channel - Coupling
			MenuStatus[2][1] = (short) buf[87];                                         // Channel - BW Linit
			MenuStatus[2][2] = (short) buf[88];                                         // Channel - Invert
			MenuStatus[2][3] = (short) buf[89];                                         // Channel - Probe
		
			SwitchesCH2 = (unsigned short) buf[90];                                              // Switches for Channel 2
			Selected_Voltage_CH2 = (volatile char) buf[91];                                     // Voltage
			Trigger_Pos_CH2_Old = (int) buf[92];
			NoDisplay[1] = (unsigned char) buf[93];
			thres_low[1] = (short) buf[94];					// QM thresholds;
			thres_mdl[1] = (short) buf[95];					// QM thresholds;
			thres_up[1]  = (short) buf[96];					// QM thresholds;
			thres_type[1] = (short) buf[97];				// QM threshold type
			Selected_Voltage_CH2_bak  = (char) buf[98];
			Trigger_Pos_CH2_bak = (int) buf[99]; 
			CH2_DAC_Offset       = (int) buf[100];                                           // Channel DAC
			ZeroLevelCH2         = (int) buf[101];                                             // ZeroLevel
			Virtual_ZeroLevelCH2 = (int) buf[102];                                     // Virtual ZeroLevel
			// free             = (int) buf[103];                                 
			// free = (int) buf[103];                                         
		
			//---------------------------------------------------
			// 		Channel 3
			//---------------------------------------------------
			MenuStatus[3][0] = (short) buf[104];                                         // Channel - Coupling
			MenuStatus[3][1] = (short) buf[105];                                         // Channel - BW Linit
			MenuStatus[3][2] = (short) buf[106];                                         // Channel - Invert
			MenuStatus[3][3] = (short) buf[107];                                         // Channel - Probe
			SwitchesCH3 = (unsigned short) buf[108];                                              // Switches for Channel 3
			Selected_Voltage_CH3 = (volatile char) buf[109];                                     // Voltage
			Trigger_Pos_CH3_Old = (int) buf[110];
			NoDisplay[2] = (unsigned char)buf[111];
			thres_low[2] = (short) buf[112];					// QM thresholds;
			thres_mdl[2] = (short) buf[113];					// QM thresholds;
			thres_up[2]  = (short) buf[114];					// QM thresholds;
			thres_type[2] = (short) buf[115];					// QM threshold type
			Selected_Voltage_CH3_bak  = (char) buf[116];
			Trigger_Pos_CH3_bak = (int) buf[117];
			CH3_DAC_Offset       = (int) buf[118];                                           // Channel DAC
			ZeroLevelCH3         = (int) buf[119];                                             // ZeroLevel
			Virtual_ZeroLevelCH3 = (int) buf[120];                                     // Virtual ZeroLevel
			// free   = (int) buf[121];                                         
			// free = (int) buf[121];                                         
		
			//---------------------------------------------------
			// 		Channel 4
			//---------------------------------------------------
			MenuStatus[4][0] = (short) buf[122];                                         // Channel - Coupling
			MenuStatus[4][1] = (short) buf[123];                                         // Channel - BW Linit
			MenuStatus[4][2] = (short) buf[124];                                         // Channel - Invert
			MenuStatus[4][3] = (short) buf[125];                                         // Channel - Probe
			SwitchesCH4 = (unsigned short) buf[126];                                              // Switches for Channel 4
			Selected_Voltage_CH4 = (volatile char) buf[127];                                     // Voltage
			Trigger_Pos_CH4_Old = (int) buf[128];
			NoDisplay[3] = (unsigned char) buf[129]; 
			thres_low[3] = (short) buf[130];					// QM thresholds;
			thres_mdl[3] = (short) buf[131];					// QM thresholds;
			thres_up[3]  = (short) buf[132];					// QM thresholds;
			thres_type[3] = (short) buf[133];					// QM threshold type
			Selected_Voltage_CH4_bak  = (char) buf[134];
			Trigger_Pos_CH4_bak = (int) buf[135]; 
			CH4_DAC_Offset       = (int) buf[136];                                           // Channel DAC
			ZeroLevelCH4         = (int) buf[137];                                             // ZeroLevel
			Virtual_ZeroLevelCH4 = (int) buf[138];                                     // Virtual ZeroLevel
	
	
			FFT_Layout = (int) buf[139];                                          // FFT layout
	
			Channel[eChM].active = (uint32_t) buf[140];                                      	// Math channel
			//MenuStatus[MENU_MATH][0] = (short) buf[141];                                        // scale
			MenuStatus[MENU_MATH][2] = (short) buf[142];                                        // math 1*2
			MenuStatus[MENU_MATH][3] = (short) buf[143];                                        // math 1-2
			MenuStatus[MENU_MATH][4] = (short) buf[144];                                        // math 1+2
	
			//FFT Settings
			Signal::_FFT_Mode   = (int)buf[145];
			Signal::_FFT_Length = (int)buf[146];
			Signal::_FFT_Level  = (int)buf[147];
			Signal::_FFT_Windex = (int)buf[148];
//BF not used			FFT_Scale  = (int)buf[149];
	
			MenuStatus[MENU_FFT][0] = (short)buf[150]; 
			MenuStatus[MENU_FFT][1] = (short)buf[151];
			MenuStatus[MENU_FFT][2] = (short)buf[152];
			MenuStatus[MENU_FFT][3] = (short)buf[153];
	
			// Cursors    	
			MenuStatus[MENU_CURSOR][0] = (short) buf[154];    			// Cursor source
			Cursor_Enabled = (char) buf[155];
		
			if (Cursor_Enabled) Cursor_Data_First_Draw = 1;
		
			Cursor_Both_Selected = (char) buf[156];
			SelectedCursor = (char) buf[157];

			Cursor_Horizontal_Active = (char) buf[158];
			Cursor_Horizontal_Position_Real1 = (long) buf[159];
			Cursor_Horizontal_Position_Real2 = (long) buf[160];

			Cursor_Vertical_Active = (char) buf[161];
			Cursor_Vertical_Position_TY_1 = (long) buf[162];
			Cursor_Vertical_Position_TY_2 = (long) buf[163];
		
			// Quick Measure
			QM_Enabled = (char) buf[164];
		
			if (QM_Enabled) Quick_Measure_First_Draw = 1;
		
			QM_Type[0] = (char) buf[165];
			QM_Type[1] = (char) buf[166];
			QM_Type[2] = (char) buf[167];
			QM_Channel[0] = (char) buf[168];
			QM_Channel[1] = (char) buf[169];
			QM_Channel[2] = (char) buf[170];
			QM_Horizontal_Active = (char) buf[171];
			QM_Vertical_Active = (char) buf[172];
			Quick_Measure_Threshold_btn_select = (char) buf[173];
		
			// Display
			GridColorArrayPos = (unsigned char) buf[174];
			Hardware::_regADC[eFPGA1].grid_color = (unsigned char) buf[175];
			Hardware::_regADC[eFPGA2].grid_color = (unsigned char) buf[175];
			MenuStatus[MENU_DISPLAY][3] = (short) buf[176];				// no display
			DottedGrid = (uint32_t) buf[177];
			MenuStatus[MENU_DISPLAY][0] = (short) buf[178];                              // Display - Persist
			MenuStatus[MENU_DISPLAY][4] = (short) buf[179];                              // Display - Draw Mode
		
			MenuStatus[MENU_ACQUIRE][0] = (short) buf[180];                        	// Acquire - Logic
			MenuStatus[MENU_ACQUIRE][1] = (short) buf[181];                     	// Acquire - Averaging
			MenuStatus[MENU_ACQUIRE][2] = (short) buf[183];                   	// Acquire - Filter Setting
			MenuStatus[MENU_ACQUIRE][3] = (short) buf[184];                   	// Acquire - Interpolation
		
			MenuStatus[MENU_AUTOSCALE][3] = (short) buf[185];			// Autoscale - Setup TB search
			 //   free = (short) buf[186]; 			
		
			MenuPopupStatus[8][0] = (short) buf[187];				//BF popup for external trigger
			MenuPopupStatus[8][1] = (short) buf[188];
			MenuPopupStatus[8][2] = (short) buf[189];
	
			MenuPopupStatus[11][0] = (short) buf[190];				//BF popup for TV trigger
			MenuPopupStatus[11][1] = (short) buf[191];
	
			//(unsigned char) buf[192];						//BF reserved for LED settings -> see below
	
			Channel[eCh1].active = (uint32_t) buf[193];
			Channel[eCh2].active = (uint32_t) buf[194];
			Channel[eCh3].active = (uint32_t) buf[195];
			Channel[eCh4].active = (uint32_t) buf[196];
	
			FFT_grid = (unsigned char) buf[197];
	
			ColorPalette = (volatile short) buf[198];
		
			HoldOff_Value = (int) buf[199];
			HoldOff_Expo = (int) buf[200];
			HoldOff_Steps = (int) buf[201];
		
			Pulse11_Value = (int) buf[202];
			Pulse11_Expo = (int) buf[203];
			Pulse11_Steps = (int) buf[204];
		
			Pulse12_Value = (int) buf[205];
			Pulse12_Expo = (int) buf[206];
			Pulse12_Steps = (int) buf[207];
		
			Pulse21_Value = (int) buf[208];
			Pulse21_Expo = (int) buf[209];
			Pulse21_Steps = (int) buf[210];
		
			Pulse22_Value = (int) buf[211];
			Pulse22_Expo = (int) buf[212];
			Pulse22_Steps = (int) buf[213];
	
	//BF new Math values
			Math_Offset     = (int) buf[214]; 
			Math_Mul_Offset = (int) buf[215]; 
			Math_Sub_Offset = (int) buf[216];
			Math_Add_Offset = (int) buf[217];
			math_scale      = (int) buf[218];
			math_mul_scale  = (int) buf[219];
			math_sub_scale  = (int) buf[220];
			math_add_scale  = (int) buf[221];
	//BF end
	
	
			// free = (short) buf[222];	
			// free = (short) buf[223];	
			// free = (short) buf[224];	
			// free = (short) buf[225];	
			// free = (short) buf[226];	
	
			Trig_Pos_Mem = (int) buf[229];
			Trig_Pos_Display = (int) buf[230];
			// free = (short) buf[231];	
			MemWinStart = (int) buf[232];
		
			Trig_Pos_Mem_old = (int) buf[233];
			Trig_Pos_Display_old = (int) buf[234];
			MenuStatus[MENU_QUICKMEASURE][4] = (short) buf[235];	
			MemWinStart_old = (int) buf[236];
		
			ZoomFactor        = (float) buf[237];
			ZoomFactor_old    = (float) buf[238];
			ZoomFactorDel     = (float) buf[239];

			MenuStatus[MENU_QUICKMEASURE][0] = (short) buf[240];	
			MenuStatus[MENU_QUICKMEASURE][1] = (short) buf[241];
			MenuStatus[MENU_QUICKMEASURE][2] = (short) buf[242];
		
			QM_Source = (short) buf[243];
			// free = (char) buf[244];	
			// free = (char) buf[245];	
			// free = (char) buf[246];
		
			QP_mode = (unsigned char) buf[247];	//BF #006
	
			Virtual_ZeroLevelXYCH1 = (int)buf[248];
			Virtual_ZeroLevelXYCH2 = (int)buf[249];
			Virtual_ZeroLevelXYCH3 = (int)buf[250];
			Virtual_ZeroLevelXYCH4 = (int)buf[251];
	
	
			//BF backup values must be restored too
			ZeroLevelCH1_bak = (int)buf[252];
			ZeroLevelCH2_bak = (int)buf[253];
			ZeroLevelCH3_bak = (int)buf[254];
			ZeroLevelCH4_bak = (int)buf[255];
	
			Virtual_ZeroLevelCH1_bak = (int)buf[256];
			Virtual_ZeroLevelCH2_bak = (int)buf[257];
			Virtual_ZeroLevelCH3_bak = (int)buf[258];
			Virtual_ZeroLevelCH4_bak = (int)buf[259];
	
			//free = (short)buf[260];
	
			Cursor_Vertical_Position_XY_1 = (long)buf[261];
			Cursor_Vertical_Position_XY_2 = (long)buf[262];
			Cursor_Vertical_Position_FFT_1 = (long)buf[263];
			Cursor_Vertical_Position_FFT_2 = (long)buf[264];
			SavePointer = (char)buf[265];

			ChannelStatusOld[eCh1] = (uint32_t)buf[266];
			ChannelStatusOld[eCh2] = (uint32_t)buf[267];
			ChannelStatusOld[eCh3] = (uint32_t)buf[268];
			ChannelStatusOld[eCh4] = (uint32_t)buf[269];
			ChannelStatusOld[eChM] = (uint32_t)buf[270];
	
			DrawMono = (unsigned char)buf[271];
			MainTimebase_bak   = (int)buf[272];
			MenuStatus[MENU_QMTHRESHOLDS][0] = (short)buf[273];
			MenuStatus[MENU_QMTHRESHOLDS][1] = (short)buf[274];
	
			ZeroLevelCH1_Main_bak = (int)buf[275];
			ZeroLevelCH2_Main_bak = (int)buf[276];
			ZeroLevelCH3_Main_bak = (int)buf[277];
			ZeroLevelCH4_Main_bak = (int)buf[278];
	
			Virtual_ZeroLevelCH1_Main_bak = (int)buf[279];
			Virtual_ZeroLevelCH2_Main_bak = (int)buf[280];
			Virtual_ZeroLevelCH3_Main_bak = (int)buf[281];
			Virtual_ZeroLevelCH4_Main_bak = (int)buf[282];
	
			// free = (unsigned short)buf[283];
			// free = (unsigned short)buf[284];
			// free = buf[285];
			// free = buf[286];
			// free   = buf[287];		
			// free    = buf[288];		

			MenuStatus[MENU_DISPL_SETUP][0] = (short) buf[289];			// grid color
			MenuStatus[MENU_DISPL_SETUP][1] = (short) buf[290];			// grid layout
			MenuStatus[MENU_DISPL_SETUP][2] = (short) buf[291];			// status layout
			MenuStatus[MENU_DISPL_SETUP][3] = (short) buf[292];			// QM-Cursors on/off
			MenuStatus[MENU_DISPL_SETUP][4] = (short) buf[293];			// OSS on/off
			//MenuStatus[MENU_DISPL_SETUP][5] = (short) buf[294];			// not used

			thres_low[4] = (short) buf[295];					// QM thresholds;
			thres_mdl[4] = (short) buf[296];					// QM thresholds;
			thres_up[4]  = (short) buf[297];					// QM thresholds;
			thres_type[4] = (short) buf[298];					// QM thresholds;
		
			LED_ON[0] = 0;
			LED_ON[1] = 0;
			LED_ON[2] = 0;
			LED_ON[3] = 0;
			LED_ON[4] = 0;
			LED_ON[5] = 0;
			LED_ON[6] = 0;
			LED_ON[7] = 0;
			LED_ON[8] = 0;
			LED_ON[9] = 0;
			LED_ON[10] = 0;
			LED_ON[11] = 0;
			LED_ON[12] = 1;		// Run/Stop Green
			LED_ON[13] = 0;
			LED_ON[14] = 0;
			LED_ON[15] = 0;
		
			if (((int) buf[192] & 0x0001) == 0x0001) LED_ON[0] = 1;
			if (((int) buf[192] & 0x0002) == 0x0002) LED_ON[1] = 1;
			if (((int) buf[192] & 0x0004) == 0x0004) LED_ON[2] = 1;
			if (((int) buf[192] & 0x0008) == 0x0008) LED_ON[3] = 1;
			if (((int) buf[192] & 0x0010) == 0x0010) LED_ON[4] = 1;  	// Math
			if (((int) buf[192] & 0x0020) == 0x0020) LED_ON[5] = 1; 	// Main Wheel
			if (((int) buf[192] & 0x0040) == 0x0040) LED_ON[6] = 1;
			if (((int) buf[192] & 0x0080) == 0x0080) LED_ON[7] = 1; 	// Quick Measure
			if (((int) buf[192] & 0x0100) == 0x0100) LED_ON[8] = 1;
			if (((int) buf[192] & 0x0200) == 0x0200) LED_ON[9] = 1;
			if (((int) buf[192] & 0x0400) == 0x0400) LED_ON[10] = 1;
			if (((int) buf[192] & 0x0800) == 0x0800) LED_ON[11] = 1;
			//if (((int) buf[192] & 0x1000) == 0x1000) LED_ON[12] = 1; 	// Run/Stop Green
			//if (((int) buf[192] & 0x2000) == 0x2000) LED_ON[13] = 1; 	// Run/Stop Red
			//if (((int) buf[192] & 0x4000) == 0x4000) LED_ON[14] = 1;
			//if (((int) buf[192] & 0x8000) == 0x8000) LED_ON[15] = 1;
		
	//BF changed		
			//Math_Offset_mul.Write_StepSize(1);
			Math_Offset_mul.Write_Value(float((float)Math_Mul_Offset /  (float)50000));
			Math_Offset_mul.RenderText();
	
			//Math_Offset_mul.Write_StepSize(1);
			Math_Offset_sub.Write_Value(float((float)Math_Sub_Offset /  (float)50000));
			Math_Offset_sub.RenderText();
	
			//Math_Offset_mul.Write_StepSize(1);
			Math_Offset_add.Write_Value(float((float)Math_Add_Offset /  (float)50000));
			Math_Offset_add.RenderText();
	//BF end
			ACQ_Average = MenuStatus[MENU_ACQUIRE][1] - 254;	

			config_loaded = true;
/*
			if (lbackup)
			printf("Config loaded from backup sector\r\n");
			else
			printf("Config loaded successfully\r\n");
*/
			break;
	
		}
		else
		{
			if (lbackup)
			{
				config_loaded = false;
				printf("Error - loading config from Flash failed\r\n");
			}
		}
	
	}

	/* del -> is done in startup!!
	
	 test del -> Signal::BuildScaleLookupTable();
	if (Signal::_FFT_Mode != FFT_OFF)
	Signal::FFT_buildScaleLookupTable();
	del */

}
//#####################################################################################################################################################

void AMDFlash::WriteConfigBackup(void)
{
	ConfigChanged = 1;
	_BackupConfig = true;
	
	WriteConfig();
	
	_BackupConfig = false;
}

//#####################################################################################################################################################
/*BF only needed for test purposes
void AMDFlash::Read_TestData_Flash(void)
{
	int x, cnt;
//printf("AMDFlash::Read_TestData_Flash\n");	
	unsigned long *flash_address;
	unsigned long buf[256];
	unsigned char buf2[1024];
    	
	flash_address = (unsigned long) nasys_main_flash + TestData_Flash;
	
	for (x = 0; x < 256; x++) buf[x] = flash_address[x];
	
	for (x = 0; x < 256; x++)
    {
        buf2[(x * 4) + 0] = (unsigned char) buf[x] & 0x000000FF;
        buf2[(x * 4) + 1] = (unsigned char) ((buf[x] & 0x0000FF00) >> 8);
        buf2[(x * 4) + 2] = (unsigned char) ((buf[x] & 0x00FF0000) >> 16);
        buf2[(x * 4) + 3] = (unsigned char) ((buf[x] & 0xFF000000) >> 24);
    }                                                       	
	
#ifdef _Debug_
    cnt = 0;
    for (x = 0; x < 48; x++)
    {
        printf("%3d %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x \n", cnt, buf2[(cnt * 21) + 0], buf2[(cnt * 21) + 1], buf2[(cnt * 21) + 2], buf2[(cnt * 21) + 3], buf2[(cnt * 21) + 4], buf2[(cnt * 21) + 5], buf2[(cnt * 21) + 6], buf2[(cnt * 21) + 7], buf2[(cnt * 21) + 8], buf2[(cnt * 21) + 9], buf2[(cnt * 21) + 10], buf2[(cnt * 21) + 11], buf2[(cnt * 21) + 12], buf2[(cnt * 21) + 13], buf2[(cnt * 21) + 14], buf2[(cnt * 21) + 15], buf2[(cnt * 21) + 16], buf2[(cnt * 21) + 17], buf2[(cnt * 21) + 18], buf2[(cnt * 21) + 19], buf2[(cnt * 21) + 20]);

        cnt++;
        nr_delay(10);
    }      	
#endif
}

void AMDFlash::Write_TestData_Flash(void)
{
    int x, cnt;
	unsigned long *flash_address;
	unsigned long buf[300];
//printf("AMDFlash::Write_TestData_Flash\n");	
	flash_address = (unsigned long) nasys_main_flash + TestData_Flash;
#ifdef _Debug_	
	printf("Flashadr : %x = %x\n", flash_address, flash_address[0]);
#endif	
	for (x = 0; x < 300; x++) buf[x] = flash_address[x];
	
	nr_flash_erase_sector((unsigned long *) nasys_main_flash, flash_address);

	buf[32] = 0x12345678;
	
	nr_flash_write_buffer((unsigned long *) nasys_main_flash, flash_address, buf, 34);
#ifdef _Debug_	
	if (Debug_Mode) printf("Flash TestData written\n");
#endif	
}
*/

//#####################################################################################################################################################
// BF used in CommIF::InterpretUART() -> USB programming interface
// Return value 1 on success, 0 on failure
int AMDFlash::WriteData_Byte_Flash(unsigned long *flash_address, unsigned char add, unsigned char data)
{
    unsigned char* fb = (unsigned char *) nasys_main_flash;
    unsigned char* a  = (unsigned char *) flash_address;

   unlock_bypass_mode();

   if (!nr_flash_write_bypass(a + add, data)) { reset_bypass_mode(); return 0; }

   reset_bypass_mode();
	
   return 1;	
}

//#####################################################################################################################################################
// BF -> used in CommIF::InterpretUART()
void AMDFlash::ReadData_Prepare_Flash(unsigned long *flash_address)
{
	int x, cnt;
	
	unsigned long *flash_adr;
	unsigned char result = 0;
	//printf("AMDFlash::ReadData_Prepare_Flash\n");    	
	flash_adr = (unsigned long) nasys_main_flash + flash_address;
	
	for (x = 0; x < 256; x++)
	{
		Flash_Read_Buffer[(x * 4) + 0] = (unsigned char) flash_adr[x] & 0x000000FF;
		Flash_Read_Buffer[(x * 4) + 1] = (unsigned char) ((flash_adr[x] & 0x0000FF00) >> 8);
		Flash_Read_Buffer[(x * 4) + 2] = (unsigned char) ((flash_adr[x] & 0x00FF0000) >> 16);
		Flash_Read_Buffer[(x * 4) + 3] = (unsigned char) ((flash_adr[x] & 0xFF000000) >> 24);
	//        nr_delay(5);
	}
}

// BF -> used in CommIF::Interpret UART()
unsigned char AMDFlash::ReadData_Byte_Flash(unsigned int add)
{
	unsigned char result = 0;
//printf("AMDFlash::ReadData_Byte_Flash\n");
    result = Flash_Read_Buffer[add];

    return result;
}
//#####################################################################################################################################################
void AMDFlash::ReadProtectedSector(void)
{
	int x;
	
	unsigned long *SectorAddress = (unsigned long) nasys_main_flash + PROTECTED_CONFIG;
 	//printf("Read protected config from flash\r");
	//printf("AMDFlash::Read_Protected_Flash\r\n");   	

		
		// read flash to global buffer
		for (x = 0; x < 128; x++)
		{ Flash_Protect_Buffer[x] = SectorAddress[x]; }
		
		if (Flash_Protect_Buffer[0] == 0xFF2332FF)
		{
			tc_model           = Flash_Protect_Buffer[1];
			tc_serial          = Flash_Protect_Buffer[2];
	
			tc_production_lot1 = Flash_Protect_Buffer[3];
			tc_production_lot2 = Flash_Protect_Buffer[4];
			tc_shipment_date   = Flash_Protect_Buffer[5];
			tc_hw_modif        = Flash_Protect_Buffer[6];		   // hardware modification

// 73 - 77 is reserved memory for factory register settings don't use!!!
		
	//BF del		adc_change12_reg = Flash_Protect_Buffer[73];			//BF Value is 0x02020800
	//BF del		adc_ctr12_reg    = 0;	//Flash_Protect_Buffer[74];			//BF Value is 0x00000000
	//BF del		adc_change34_reg = Flash_Protect_Buffer[75];			//BF Value is 0x00000000	
	//BF del		channel_Adr_add  = 0x5F0A;	//Flash_Protect_Buffer[76];		//BF Value is 0x5F0A
	//BF del		channel_Adr_add2 = 0;		//Flash_Protect_Buffer[77];		//BF Value is 0x0000
	
	/*
			printf("\r\n");
			printf("----------------------------------------------------------------\r\n");
			printf("- protected register settings -\r\n");
			printf("adc_change12_reg : %x \r\n", adc_change12_reg);
			printf("adc_change34_reg : %x \r\n", adc_change34_reg);
			printf("adc_ctr12_reg    : %x \r\n", adc_ctr12_reg);
			printf("channel_Adr_add12  : %x \r\n", channel_Adr_add12);
			printf("channel_Adr_add34 : %x \r\n", channel_Adr_add34);
			printf("-----------------------------------------------------------------\r\n");
			printf("\r\n");
	*/
	
			//printf("Read protected config from flash	- done\r\n");
		}
		else
		{
			printf("Error - loading protected flash failed\r\n"); 
		}


}
//#####################################################################################################################################################
void AMDFlash::WriteProtectedSector(void)
{
   	unsigned long *SectorAddress = (unsigned long) nasys_main_flash + PROTECTED_CONFIG;
 	
	//printf("AMDFlash::Write_Protected_Flash()\r\n");	

	if (!nr_flash_erase_sector(SectorAddress)) { return; }		// erase protected config area 

	Flash_Protect_Buffer[0] = 0xFF2332FF;              // Kennung
	Flash_Protect_Buffer[1] = tc_model;                // model 			BF del
	//Flash_Protect_Buffer[2] = tc_serial;               // serial nr		BF del
	Flash_Protect_Buffer[3] = tc_production_lot1;      // production lot	
	Flash_Protect_Buffer[4] = tc_production_lot2;      // production lot	
	Flash_Protect_Buffer[5] = tc_shipment_date;        // shipment date
	Flash_Protect_Buffer[6] = tc_hw_modif;		   // hardware modification

// 73 - 77 is reserved memory for factory register settings - don't use!!!

/* BF buffer must be written unchanged to flash to prevent loosing factory setting!
	Flash_Protect_Buffer[73] = (unsigned long) 0x02020F00;	//0x02020800;	//adc_change12_reg;  	
	Flash_Protect_Buffer[74] = (unsigned long) 0x0000;			//adc_ctr12_reg;
	Flash_Protect_Buffer[75] = (unsigned long) 0x0200000A;	//0x02020800;	//adc_change34_reg;
	Flash_Protect_Buffer[76] = (unsigned long) 0x5F0A;			//channel_Adr_add;
	Flash_Protect_Buffer[77] = (unsigned long) 0x5F5F			//channel_Adr_add2;
*/

	nr_flash_write_buffer(SectorAddress, Flash_Protect_Buffer, 80);

	// check values
	if (SectorAddress[0] == 0xFF2332FF)
	printf("protected config written to flash\r\n");
	else
	printf("flash checksum error\r\n");

	
}

//#####################################################################################################################################################
//BF #002 read protected register values to get factory setting
void AMDFlash::ReadProtReg(void)
{
	//Check if protected buffer is already loaded	
	if (Flash_Protect_Buffer[0] == 0xFF2332FF)
	{
		Hardware::_regADC[eFPGA1].adc_change = Flash_Protect_Buffer[73];
		//    				     = Flash_Protect_Buffer[74];		-> unused 
		Hardware::_regADC[eFPGA2].adc_change = Flash_Protect_Buffer[75];	
		Hardware::_regADC[eFPGA1].channel_Adr = (unsigned short)Flash_Protect_Buffer[76];
		Hardware::_regADC[eFPGA2].channel_Adr = (unsigned short)Flash_Protect_Buffer[77];
	}
	else
	{
		//if not - get values from flash
		unsigned long *SectorAddress = (unsigned long) nasys_main_flash + PROTECTED_CONFIG;
	
		if (SectorAddress[0] == 0xFF2332FF)
		{
			Hardware::_regADC[eFPGA1].adc_change = SectorAddress[73];
			//	                             = SectorAddress[74];	-> unused 
			Hardware::_regADC[eFPGA2].adc_change = SectorAddress[75];	
			Hardware::_regADC[eFPGA1].channel_Adr = (unsigned short)SectorAddress[76];
			Hardware::_regADC[eFPGA2].channel_Adr = (unsigned short)SectorAddress[77];
		}
		else
		{ printf("Loading Protected Registers failed\r\n");}


	}

/*
	printf("\r\n");
	printf("----------------------------------------------------------------\r\n");
	printf("- protected register settings -\r\n");
	printf("adc_change12_reg  : %x \r\n", adc_change12_reg);
	printf("adc_change34_reg  : %x \r\n", adc_change34_reg);
	printf("channel_Adr_add12 : %8x \r\n", channel_Adr_add12);
	printf("channel_Adr_add34 : %8x \r\n", channel_Adr_add34);
	//printf("adc_ctr12_reg     : %x \r\n", adc_ctr12_reg);
	//printf("adc_ctr34_reg     : %x \r\n", adc_ctr34_reg);
	printf("-----------------------------------------------------------------\r\n");
	printf("\r\n");
*/


	
}

//#####################################################################################################################################################
//BF #016 load selected trace from flash
void AMDFlash::RecallTrace(void)
{

	if (RC_overlay) return;	

	if (!RC_request)
	{
		// save actual configuration
		WriteConfig();
	}
	RC_request = 1;
	UI_request = 0;

	Run = 1;	//initialize operating mode
	
//printf("Sector address of Trace# %d is %X", SavePointer, FLASH_SECTOR(SavePointer+SAVE_SECTOR_OFFSET));

	// read trace data
	ReadTraceFromFlash(FLASH_SECTOR(SavePointer+SAVE_SECTOR_OFFSET));

	if (Run)
	{
		RC_request = 0;
	 	return; 	//no data found -> leave
	}

	Display::MenuInit();
	Display::RefreshScreen();
	Display::DRAW_ALL_CURSOR();

	// check if logic processor has to be started
	LA_active = 0;

	if (MenuStatus[MENU_ACQUIRE][0] != LOGA_OFF)
	{
		// restore voltage ranges
		Selected_Voltage_CH1 = Selected_Voltage_CH1_bak;
		Selected_Voltage_CH2 = Selected_Voltage_CH2_bak;
		Selected_Voltage_CH3 = Selected_Voltage_CH3_bak;
		Selected_Voltage_CH4 = Selected_Voltage_CH4_bak;

		// restore trigger settings
		Trigger_Pos_CH1 = Trigger_Pos_CH1_bak;
		Trigger_Pos_CH2 = Trigger_Pos_CH2_bak;
		Trigger_Pos_CH3 = Trigger_Pos_CH3_bak;
		Trigger_Pos_CH4 = Trigger_Pos_CH4_bak;

		Signal::LOGA_LogicProcessor();
	}


}


//#####################################################################################################################################################
//BF #016 save selected trace to flash
void AMDFlash::SaveTrace(void)
{
	if (RC_request)			//don't save signal which is recalled from Memory
	{ RC_request = 0; return; }

	if (RC_overlay)			//don't save signal which is recalled from Memory
	{ 
		//printf("Terminating Overlay Mode\r\n");
		RC_overlay = 0; 
		return;		
	}


	//&~np_timercontrol_cont_mask;	// Stop Timer
	timer_rotary_busy->np_timercontrol = ((timer_rotary_busy->np_timercontrol & 3) + np_timercontrol_stop_mask);
	timer_rotary_busy->np_timerstatus = 0;

	Display::DRAWROUNDBUTTON(250, 200, 140, 40, 0, 0);
	Display::TextOut12x9("Saving Signals", 260, 212, 1, UI_Plane2);

	// write trace data
	WriteTraceToFlash(FLASH_SECTOR(SavePointer+SAVE_SECTOR_OFFSET));

	Display::DRAWROUNDBUTTON(250, 200, 140, 40, 0, 1);
	Display::TextOut12x9("Saving Signals", 260, 212, 0, UI_Plane2); 



}


//##########################################################################################################################################################
//BF Recall Overlay function #013
void AMDFlash::RecallOverlay(void)
{
	if (Signal::_FFT_Mode != FFT_OFF) return;
	if (USTB_Mode != USTB_OFF) return;

	if (RC_overlay) return;		//prevent recursive call and return to normal 
	if (RC_request) return;		//prevent recursive call and return to normal 

	unsigned long *lp_TracePointer = NULL;	
	unsigned long *lp_PlanePointer = UI_Plane3;               //UI_Plane1;	

	unsigned long *lp_BackupPointer1 = Buffer_Plane1;
	unsigned long *lp_BackupPointer2 = Buffer_Plane2;
	unsigned long *lp_BackupPointer3 = Buffer_Plane3;
	unsigned long *lp_BackupPointer4 = Buffer_Plane4;



/*
unsigned long *Buffer_PlaneM = (unsigned long *) 0x00930C00;
unsigned long *Buffer_Plane1 = (unsigned long *) 0x0093A200;
unsigned long *Buffer_Plane2 = (unsigned long *) 0x00943800;
unsigned long *Buffer_Plane3 = (unsigned long *) 0x0094CE00;
unsigned long *Buffer_Plane4 = (unsigned long *) 0x00956400;



unsigned long *UI_Plane1 = (unsigned long *) 0x0095FA00;		// white/grey
unsigned long *UI_Plane2 = (unsigned long *) 0x00969000;		// black
unsigned long *UI_Plane3 = (unsigned long *) 0x00972600;		// light yellow
unsigned long *UI_Plane4 = (unsigned long *) 0x0097BC00;		// yellow
unsigned long *UI_Plane5 = (unsigned long *) 0x00985200;		// dark grey

unsigned long *Channel_Plane1 = (unsigned long *) 0x0098E800;		// neon yellow
unsigned long *Channel_Plane2 = (unsigned long *) 0x00997E00;		// neon green
unsigned long *Channel_Plane3 = (unsigned long *) 0x009A1400;		// neon blue
unsigned long *Channel_Plane4 = (unsigned long *) 0x009AAA00;		// neon red
unsigned long *Channel_Math_Plane = (unsigned long *) 0x009B4000;	// neon pink

unsigned long *Marker_Plane1 = (unsigned long *) 0x009D9800;		// light red
unsigned long *Marker_Plane2 = (unsigned long *) 0x009E2E00;		// light red

unsigned long *Grid_Plane = (unsigned long *) 0x009EC400;		// grey
*/


	
	//printf("Starting Overlay Mode\r\n");

	// save actual configuration
	WriteConfigBackup();

	Run = 1;	//initialize operating mode

	// let's point the trace pointer to the choosen trace in the flash memory
	lp_TracePointer = FLASH_SECTOR(SavePointer + SAVE_SECTOR_OFFSET);
	//printf("Trace Pointer points to %x \n\r",lp_TracePointer);	

	//Load trace settings from flash
	if (!LoadTraceSettings(lp_TracePointer)) return;

	if (USTB_Mode != USTB_OFF)			// stored trace was USTB -> leave overlay
	{ Hardware::Restore_From_Flash(); return; }

	Hardware::Start_Record();

	Display::RefreshScreen();

	// wait for ADC
	for(uint32_t timeout = 0;timeout < 100 && !ADC_DataAvailable;timeout++)
	nr_delay(20);

	// get live trace with loaded settings
	Hardware::Handle_ADC();

	// copy signals to buffer
	for (int i=0;i<0x4000;i++)
	{
		SIGNAL1_Buffer[i] = SIGNAL1[i];
		SIGNAL2_Buffer[i] = SIGNAL2[i];
		SIGNAL3_Buffer[i] = SIGNAL3[i];
		SIGNAL4_Buffer[i] = SIGNAL4[i];
	}	

	Signal::processSignal();

	//bend plane pointers to change signal colour 
	Buffer_Plane1 = lp_PlanePointer;
	Buffer_Plane2 = lp_PlanePointer;
	Buffer_Plane3 = lp_PlanePointer;
	Buffer_Plane4 = lp_PlanePointer;

	//clear grid area of plane 
	for (int i = GRID_PLANE_MIN; i < GRID_PLANE_MAX; i++) { lp_PlanePointer[i] = 0; }

	//draw actual signal with loaded settings
	//printf("Draw live trace....");
	Display::DrawSignals();		


	//restore plane pointers
	Buffer_Plane1 = lp_BackupPointer1;
	Buffer_Plane2 = lp_BackupPointer2;
	Buffer_Plane3 = lp_BackupPointer3;
	Buffer_Plane4 = lp_BackupPointer4;

	//load the choosen trace from flash
	//printf("Load trace from flash....");
	Run = 0;
	LoadTracePoints(lp_TracePointer);

	//???Signal::ProcessSignalData();

	//draw loaded trace
	//printf("Draw loaded trace\r\n");
	Display::DrawSignals();		

	Hardware::TransferPlanes();
	//CommIF::Transfer_USB_Data();

	OpSys::InitQueue();	// avoid recursive handle call -> clear simple queue
	RC_request = 1;
	UI_request = 0;
	RC_user_if = 0;

	
	// overlay loop
	for (RC_overlay=1;RC_overlay;)
	{
		//handle user interface
		UserIF::Rotary_Interface();
		UserIF::Keyboard_Interface();

		//redraw the traces	
		if(RC_user_if && RC_overlay)
		{
			Hardware::ClearPlanes();

			//Run  = 1;
			RC_request  = 0;
			ACQ_NewData = 1;
			UI_request  = 0;

			// restore live signals from buffer
			for (int i=0;i<0x4000;i++)
			{
				SIGNAL1[i] = SIGNAL1_Buffer[i];
				SIGNAL2[i] = SIGNAL2_Buffer[i];
				SIGNAL3[i] = SIGNAL3_Buffer[i];
				SIGNAL4[i] = SIGNAL4_Buffer[i];
			}	
			
			Signal::processSignal();
		
			//change signal colour
			Buffer_Plane1 = lp_PlanePointer;
			Buffer_Plane2 = lp_PlanePointer;
			Buffer_Plane3 = lp_PlanePointer;
			Buffer_Plane4 = lp_PlanePointer;
		
			//clear grid area of plane 
			for (int i = GRID_PLANE_MIN; i < GRID_PLANE_MAX; i++) { lp_PlanePointer[i] = 0; }


			//draw actual signal with loaded settings
			//printf("Draw live trace....");
			Display::DrawSignals();		

			//UpdateMenuTextPlane = true;
			//Hardware::TransferPlanes();
		

			//restore plane pointers
			Buffer_Plane1 = lp_BackupPointer1;
			Buffer_Plane2 = lp_BackupPointer2;
			Buffer_Plane3 = lp_BackupPointer3;
			Buffer_Plane4 = lp_BackupPointer4;


			//load the choosen trace from flash
			//printf("Load trace....");
			Run=0;
			LoadTracePoints(lp_TracePointer);
			//Run = 0;
			RC_request  = 0;
			ACQ_NewData = 1;
			UI_request  = 0;
			Signal::processSignal();
			
			//draw loaded trace
			//printf("Draw loaded trace\r\n");
			Display::DrawSignals();

			UpdateMenuTextPlane = true;
			
			RC_request = 1;
			RC_user_if = 0;
		}

		Hardware::TransferPlanes();

		inlExecSingle(0);
		inlExecSingle(1);
		inlExecSingle(2);
		inlExecSingle(3);
		inlExecSingle(4);
	}

	//clean up and leave the overlay routine 
	for (int i = GRID_PLANE_MIN; i < GRID_PLANE_MAX; i++) { lp_PlanePointer[i] = 0; }
	RC_user_if = 0;

}

//################################################################################################################
// Get actual config slot. Slot address is relative to sector address
void AMDFlash::GetConfigSlot(void)
{
	unsigned long *sector_start = (unsigned long) nasys_main_flash + CONFIG_FLASH ;		// get config start address

	_ConfigSlot = 0;
	//_SlotNumber = 0;

	//printf("\n\rsector_start %lX   \n\r", sector_start);

	for (unsigned long *sa = 0; (_ConfigSlot + CONFIG_SLOT_SIZE_INT) < FLASH_SECTOR_SIZE_INT; _ConfigSlot += CONFIG_SLOT_SIZE_INT/*, _SlotNumber++*/)
	{
		sa = sector_start + _ConfigSlot;
		//printf("\n\r(_ConfigSlot + CONFIG_SLOT_SIZE) %lX   FLASH_SECTOR_SIZE %lX \n\r", (_ConfigSlot + CONFIG_SLOT_SIZE_INT), FLASH_SECTOR_SIZE_INT);
		//printf("Checking for free slot @ %lX ... \n\r",sa);
		if ((sa[1] == 0xFFFFFFFF) && (sa[CONFIG_SLOT_SIZE_INT - 1] == 0xFFFFFFFF))
		{
			_ConfigSlot -= CONFIG_SLOT_SIZE_INT;				// found free Slot -> last slot was active slot
			//_SlotNumber--;
			break;
		}

	}

	//printf("active slot nunber is %d \n\r", _SlotNumber);

}


//#####################################################################################################################################################

unsigned long *AMDFlash::GetConfigStartAddr(void)
{
	unsigned long *sect_addr;
	return sect_addr = (unsigned long) nasys_main_flash + CONFIG_FLASH +_ConfigSlot; // get config start address
}


//################################################################################################################
// write all trigonometric tables and window tables to the corresponding sector address.
// All tables are sized for a 4096 points FFT, so we can store 4 tables in one sector - except the first one
// because we need some additional memory for the checksum
void AMDFlash::WriteTrigoTables(void)
{

	//test
	printf("Writing trigonometric tables to flash...");
	//return;
	
	int buf[TABSIZE];

	// Clear All Planes
	for (int i = 0; i < 153600; i++)
	{ Planes[i] = 0; }
	
	// show popup with message and progress indicator
	//if (MenuPopupActive > -1)
	//{ Display::Rect(190, 180, 260, 80, 0, UI_Plane2); }	// remove text under the message popup
	Display::DRAWROUNDBUTTON(180, 180, 280, 80, 0, 0);	// draw popup
	Display::TextOut12x9("Writing trigonometric tables to flash!", 190, 192, 1, UI_Plane2);
	Display::Rect(200, 220, 240, 20, 1, UI_Plane1);	// draw white box
	
	
	// first erase sector 
	if (!nr_flash_erase_sector(TRIGOTAB01))
	{ printf("Error on erasing flash sector -> terminate operation\r\n"); return; }
	
	/* memory from 0x0000 - 0x0FFF is reserved for checksum and other data */ 
	
	// generate sin table
	Signal::FFT_generateSin(TABSIZE, buf);
	
	// write table to flash
	if (!nr_flash_write_buffer(TRIGOTAB01 + 0x1000, (unsigned long*)buf, (int)TABSIZE))
	{ printf("Error on writing table to flash -> terminate operation\r\n"); return; }

	Display::Rect(200, 220, 20, 20, 1, UI_Plane2);	// draw progress indicator
	
	// generate cos table
	Signal::FFT_generateCos(TABSIZE, buf);
	
	// write table to flash
	if (!nr_flash_write_buffer(TRIGOTAB01 + 0x2000, (unsigned long*)buf, (int)TABSIZE))
	{ printf("Error on writing table to flash -> terminate operation\n\r"); return; }

	Display::Rect(220, 220, 20, 20, 1, UI_Plane2);	// draw progress indicator
	
	// generate first window function
	Signal::FFT_generateWindows(TABSIZE, 0, buf);
	
	// write table to flash
	if (!nr_flash_write_buffer(TRIGOTAB01 + 0x3000, (unsigned long*)buf, (int)TABSIZE))
	{ printf("Error on writing table to flash -> terminate operation\n\r"); return; }
	
	Display::Rect(240, 220, 20, 20, 1, UI_Plane2);	// draw progress indicator
	
	/* prepare next sector */
	
	// first erase sector 
	if (!nr_flash_erase_sector(TRIGOTAB02))
	{ printf("Error on erasing flash sector -> terminate operation\n\r"); return; }

	Display::Rect(260, 220, 20, 20, 1, UI_Plane2);	// draw progress indicator
	
	// generate next window function
	Signal::FFT_generateWindows(TABSIZE, 1, buf);
	
	// write table to flash
	if (!nr_flash_write_buffer(TRIGOTAB02 + 0x0000, (unsigned long*)buf, (int)TABSIZE))
	{ printf("Error on writing table to flash -> terminate operation\r\n"); return; }

	Display::Rect(280, 220, 20, 20, 1, UI_Plane2);	// draw progress indicator	
	
	// generate next window function
	Signal::FFT_generateWindows(TABSIZE, 2, buf);
	
	// write table to flash
	if (!nr_flash_write_buffer(TRIGOTAB02 + 0x1000, (unsigned long*)buf, (int)TABSIZE))
	{ printf("Error on writing table to flash -> terminate operation\r\n"); return; }	
	
	Display::Rect(300, 220, 20, 20, 1, UI_Plane2);	// draw progress indicator	
	
	// generate next window function
	Signal::FFT_generateWindows(TABSIZE, 3, buf);
	
	// write table to flash
	if (!nr_flash_write_buffer(TRIGOTAB02 + 0x2000, (unsigned long*)buf, (int)TABSIZE))
	{ printf("Error on writing table to flash -> terminate operation\n\r"); return; }	
	
	Display::Rect(320, 220, 20, 20, 1, UI_Plane2);	// draw progress indicator	
	
	// generate next window function
	Signal::FFT_generateWindows(TABSIZE, 4, buf);
	
	// write table to flash
	if (!nr_flash_write_buffer(TRIGOTAB02 + 0x3000, (unsigned long*)buf, (int)TABSIZE))
	{ printf("Error on writing table to flash -> terminate operation\r\n"); return; }	

	Display::Rect(340, 220, 20, 20, 1, UI_Plane2);	// draw progress indicator
	
	
	/* prepare next sector */
	
	// first erase sector 
	if (!nr_flash_erase_sector(TRIGOTAB03))
	{ printf("Error on erasing flash sector -> terminate operation\r\n"); return; }

	// generate next window function
	Signal::FFT_generateWindows(TABSIZE, 5, buf);
	
	// write table to flash
	if (!nr_flash_write_buffer(TRIGOTAB03 + 0x0000, (unsigned long*)buf, (int)TABSIZE))
	{ printf("Error on writing table to flash -> terminate operation\r\n"); return; }
	
	Display::Rect(360, 220, 20, 20, 1, UI_Plane2);	// draw progress indicator
	
	// generate next window function
	Signal::FFT_generateWindows(TABSIZE, 6, buf);
	
	// write table to flash
	if (!nr_flash_write_buffer(TRIGOTAB03 + 0x1000, (unsigned long*)buf, (int)TABSIZE))
	{ printf("Error on writing table to flash -> terminate operation\r\n"); return; }	
	
	Display::Rect(380, 220, 20, 20, 1, UI_Plane2);	// draw progress indicator
	
	// generate next window function
	Signal::FFT_generateWindows(TABSIZE, 7, buf);
	
	// write table to flash
	if (!nr_flash_write_buffer(TRIGOTAB03 + 0x2000, (unsigned long*)buf, (int)TABSIZE))
	{ printf("Error on writing table to flash -> terminate operation\r\n"); return; }	
	
	Display::Rect(400, 220, 20, 20, 1, UI_Plane2);	// draw progress indicator
	
	// generate next window function
	Signal::FFT_generateWindows(TABSIZE, 8, buf);
	
	// write table to flash
	if (!nr_flash_write_buffer(TRIGOTAB03 + 0x3000, (unsigned long*)buf, (int)TABSIZE))
	{ printf("Error on writing table to flash -> terminate operation\r\n"); return; }	
	
	Display::Rect(420, 220, 20, 20, 1, UI_Plane2);	// draw progress indicator
	
	// if all tables are writte successfully the checksum can be written
	for (int i=0; i<TABSIZE; i++)
	{ buf[i] = 0; } //delete
	
	buf[0] = TRIGO_CHECKSUM;
	
	
	// write table with checksum to flash
	if (!nr_flash_write_buffer(TRIGOTAB01, (unsigned long*)buf, (int)1))
	{ printf("Error on writing table to flash -> terminate operation\r\n"); return; }	
	
	//Display::DRAWROUNDBUTTON(190, 180, 260, 80, 0, 1);					// delete popup
	//Display::Rect(200, 220, 240, 20, 0, UI_Plane1);					// remove white box
	//Display::Rect(200, 190, 240, 52, 0, UI_Plane2);					// remove text
	
	// Clear All Planes
	for (int i = 0; i < 153600; i++)
	{ Planes[i] = 0; }
	
	printf("Writing trigonometric tables to flash                   - done\r\n\r\n");
	
}

//################################################################################################################
// read sin and cos tables from flash.
// Return value 1 on success, 0 on failure
int AMDFlash::ReadSinCos(unsigned int points, int *SinTab, int *CosTab)
{
	unsigned long *flash_addr;
        int offset;
	
	
	flash_addr = (unsigned long) nasys_main_flash + TRIGOTAB01;
	
	// checksum ok? Else leave with error.
	if (flash_addr[0] != TRIGO_CHECKSUM)
	{ return 0; }

	// offset depends on number of points stored in flash (4096) and 
	// number of points requested in actual FFT
	if (points == FFT_512) offset = 8;		// FFT 512 
	else if (points == FFT_1K) offset = 4;		// FFT 1024
	else if (points == FFT_2K) offset = 2;		// FFT 2048
	else offset = 1;				// FFT 4096

	// read sin table
	flash_addr = (unsigned long) nasys_main_flash + TRIGOTAB01 + 0x1000;
	
	for(int i=0, r=0; i < points; i++, r+=offset)
	{ SinTab[i] = (int)flash_addr[r]; }
	
	// read cos table
	flash_addr = (unsigned long) nasys_main_flash + TRIGOTAB01 + 0x2000;
	
	for(int i=0, r=0; i < points; i++, r+=offset)
	{ CosTab[i] = (int)flash_addr[r]; }
	
	return 1;
	
}

//################################################################################################################
// read window table from flash.
// Return value 1 on success, 0 on failure
int AMDFlash::ReadWindow(unsigned int points, int WinType, int *WinTab)
{
	unsigned long *flash_addr;
        int offset;

	// offset depends on number of points stored in flash (4096) and 
	// number of points requested in actual FFT
	if (points == FFT_512) offset = 8;		// FFT 512 
	else if (points == FFT_1K) offset = 4;		// FFT 1024
	else if (points == FFT_2K) offset = 2;		// FFT 2048
	else offset = 1;				// FFT 4096
	
	
	switch(WinType)
	{
		case -1: { return 1; }
		
		case 0: { flash_addr = (unsigned long) nasys_main_flash + TRIGOTAB01 + 0x3000; break;}	
		case 1: { flash_addr = (unsigned long) nasys_main_flash + TRIGOTAB02 + 0x0000; break;}	
		case 2: { flash_addr = (unsigned long) nasys_main_flash + TRIGOTAB02 + 0x1000; break;}	
		case 3: { flash_addr = (unsigned long) nasys_main_flash + TRIGOTAB02 + 0x2000; break;}	
		case 4: { flash_addr = (unsigned long) nasys_main_flash + TRIGOTAB02 + 0x3000; break;}	
		case 5: { flash_addr = (unsigned long) nasys_main_flash + TRIGOTAB03 + 0x0000; break;}	
		case 6: { flash_addr = (unsigned long) nasys_main_flash + TRIGOTAB03 + 0x1000; break;}	
		case 7: { flash_addr = (unsigned long) nasys_main_flash + TRIGOTAB03 + 0x2000; break;}	
		case 8: { flash_addr = (unsigned long) nasys_main_flash + TRIGOTAB03 + 0x3000; break;}	
		
		default: return 0;
	}
	

	for(int i=0, r=0; i < points; i++, r+=offset)
	{ WinTab[i] = (int)flash_addr[r]; }

	return 1;
	
}


//#####################################################################################################################################################
void AMDFlash::ReadCalibration(void)
{
	unsigned long *SectorAddress = (unsigned long) nasys_main_flash + CALIBRATION;
 	
	//printf("AMDFlash::ReadCalibration()\r\n");   	

	if (SectorAddress[0] == 0xAAFF55FF)
	{
		
		CalSet = (int)SectorAddress[6];
		GainIdx = (unsigned int)SectorAddress[7];
		Hardware::_Beta = (float)SectorAddress[8];		// ADC beta correction
		Hardware::_Beta /= 1000;		 		// -> convert back to float

		//Standard Set
		DAC_Correction[0][0][0] = (int)SectorAddress[9]; 	//channel 1
		DAC_Correction[0][0][1] = (int)SectorAddress[10]; 
		DAC_Correction[0][0][2] = (int)SectorAddress[11]; 

		DAC_Correction[0][1][0] = (int)SectorAddress[12]; 	//channel 2
		DAC_Correction[0][1][1] = (int)SectorAddress[13]; 
		DAC_Correction[0][1][2] = (int)SectorAddress[14]; 

		DAC_Correction[0][2][0] = (int)SectorAddress[15]; 	//channel 3
		DAC_Correction[0][2][1] = (int)SectorAddress[16]; 
		DAC_Correction[0][2][2] = (int)SectorAddress[17]; 

		DAC_Correction[0][3][0] = (int)SectorAddress[18]; 	//channel 4
		DAC_Correction[0][3][1] = (int)SectorAddress[19]; 
		DAC_Correction[0][3][2] = (int)SectorAddress[20]; 

		//Active Probe Set
		DAC_Correction[1][0][0] = (int)SectorAddress[21]; 	//channel 1
		DAC_Correction[1][0][1] = (int)SectorAddress[22]; 
		DAC_Correction[1][0][2] = (int)SectorAddress[23]; 

		DAC_Correction[1][1][0] = (int)SectorAddress[24]; 	//channel 2
		DAC_Correction[1][1][1] = (int)SectorAddress[25]; 
		DAC_Correction[1][1][2] = (int)SectorAddress[26]; 

		DAC_Correction[1][2][0] = (int)SectorAddress[27]; 	//channel 3
		DAC_Correction[1][2][1] = (int)SectorAddress[28]; 
		DAC_Correction[1][2][2] = (int)SectorAddress[29]; 

		DAC_Correction[1][3][0] = (int)SectorAddress[30]; 	//channel 4
		DAC_Correction[1][3][1] = (int)SectorAddress[31]; 
		DAC_Correction[1][3][2] = (int)SectorAddress[32]; 

		//Set3
		DAC_Correction[2][0][0] = (int)SectorAddress[33]; 	//channel 1
		DAC_Correction[2][0][1] = (int)SectorAddress[34]; 
		DAC_Correction[2][0][2] = (int)SectorAddress[35]; 

		DAC_Correction[2][1][0] = (int)SectorAddress[36]; 	//channel 2
		DAC_Correction[2][1][1] = (int)SectorAddress[37]; 
		DAC_Correction[2][1][2] = (int)SectorAddress[38]; 

		DAC_Correction[2][2][0] = (int)SectorAddress[39]; 	//channel 3
		DAC_Correction[2][2][1] = (int)SectorAddress[40]; 
		DAC_Correction[2][2][2] = (int)SectorAddress[41]; 

		DAC_Correction[2][3][0] = (int)SectorAddress[42]; 	//channel 4
		DAC_Correction[2][3][1] = (int)SectorAddress[43]; 
		DAC_Correction[2][3][2] = (int)SectorAddress[44]; 

		//Set4
		DAC_Correction[3][0][0] = (int)SectorAddress[45]; 	//channel 1
		DAC_Correction[3][0][1] = (int)SectorAddress[46]; 
		DAC_Correction[3][0][2] = (int)SectorAddress[47]; 

		DAC_Correction[3][1][0] = (int)SectorAddress[48]; 	//channel 2
		DAC_Correction[3][1][1] = (int)SectorAddress[49]; 
		DAC_Correction[3][1][2] = (int)SectorAddress[50]; 

		DAC_Correction[3][2][0] = (int)SectorAddress[51]; 	//channel 3
		DAC_Correction[3][2][1] = (int)SectorAddress[52]; 
		DAC_Correction[3][2][2] = (int)SectorAddress[53]; 

		DAC_Correction[3][3][0] = (int)SectorAddress[54]; 	//channel 4
		DAC_Correction[3][3][1] = (int)SectorAddress[55]; 
		DAC_Correction[3][3][2] = (int)SectorAddress[56]; 


		Hardware::_ADC_Offset[0][0] = (int) SectorAddress[57];
		Hardware::_ADC_Offset[0][1] = (int) SectorAddress[58];
		Hardware::_ADC_Offset[0][2] = (int) SectorAddress[59];
		Hardware::_ADC_Offset[0][3] = (int) SectorAddress[60];	
		
		Hardware::_ADC_Offset[1][0] = (int) SectorAddress[61];
		Hardware::_ADC_Offset[1][1] = (int) SectorAddress[62];
		Hardware::_ADC_Offset[1][2] = (int) SectorAddress[63];
		Hardware::_ADC_Offset[1][3] = (int) SectorAddress[64];
	
		Hardware::_ADC_Offset[2][0] = (int) SectorAddress[65];
		Hardware::_ADC_Offset[2][1] = (int) SectorAddress[66];
		Hardware::_ADC_Offset[2][2] = (int) SectorAddress[67];
		Hardware::_ADC_Offset[2][3] = (int) SectorAddress[68];
	
		Hardware::_ADC_Offset[3][0] = (int) SectorAddress[69];
		Hardware::_ADC_Offset[3][1] = (int) SectorAddress[70];
		Hardware::_ADC_Offset[3][2] = (int) SectorAddress[71];
		Hardware::_ADC_Offset[3][3] = (int) SectorAddress[72];

		Hardware::_regADC[eFPGA1].adc_change = SectorAddress[73];
		//	                             = SectorAddress[74];	-> unused 
		Hardware::_regADC[eFPGA2].adc_change = SectorAddress[75];	
		Hardware::_regADC[eFPGA1].channel_Adr = (unsigned short)SectorAddress[76];
		Hardware::_regADC[eFPGA2].channel_Adr = (unsigned short)SectorAddress[77];
		
		Hardware::_ChannelDelay[0] = (int) SectorAddress[78];
		Hardware::_ChannelDelay[1] = (int) SectorAddress[79];
		Hardware::_ChannelDelay[2] = (int) SectorAddress[80];
		Hardware::_ChannelDelay[3] = (int) SectorAddress[81];
		
		MenuStatus[MENU_HARDWARE][0] = (short) SectorAddress[82];		//ADC register setting
		MenuStatus[MENU_HARDWARE][1] = (short) SectorAddress[83];		//Pre Gain 
		MenuStatus[MENU_HARDWARE][3] = (short) SectorAddress[84];		//ADC driver 
	
		MenuStatus[MENU_EXTUTILS][0] = (short) SectorAddress[85];		// delay ch1
		MenuStatus[MENU_EXTUTILS][1] = (short) SectorAddress[86];		// delay ch2
		MenuStatus[MENU_EXTUTILS][2] = (short) SectorAddress[87];		// delay ch3
		MenuStatus[MENU_EXTUTILS][3] = (short) SectorAddress[88];		// delay ch4
	
		MenuStatus[MENU_UTILITY][3] = (short) SectorAddress[89];		// calibration set
		
	}

	//printf("\n\r Beta correction = %d / ", SectorAddress[8]);floatprintf(Hardware::_Beta); printf("\n\r");

	//printf("\n\r read calibration set = %d / ",CalSet);
}



//#####################################################################################################################################################
void AMDFlash::WriteCalibration(void)
{
   	unsigned long *sectoraddress = (unsigned long) nasys_main_flash + CALIBRATION;;
	unsigned long calibration[100];
	
	printf("AMDFlash::WriteCalibration()\r\n");	

	if (!nr_flash_erase_sector(sectoraddress)) { printf("Calibration erase error occured \r\n"); return; }		// erase protected config area 

	calibration[0] = 0xAAFF55FF;             			// sector checksum


	calibration[6] = (unsigned long) CalSet;
	calibration[7] = (unsigned long) GainIdx;
	calibration[8] = (unsigned long) (Hardware::_Beta * 1000);	// ADC beta correction -> convert to integer
	
	//Standard Set
	calibration[9] = (unsigned long) DAC_Correction[0][0][0]; 	//channel 1
	calibration[10] = (unsigned long) DAC_Correction[0][0][1]; 
	calibration[11] = (unsigned long) DAC_Correction[0][0][2]; 

	calibration[12] = (unsigned long) DAC_Correction[0][1][0]; 	//channel 2
	calibration[13] = (unsigned long) DAC_Correction[0][1][1]; 
	calibration[14] = (unsigned long) DAC_Correction[0][1][2]; 

	calibration[15] = (unsigned long) DAC_Correction[0][2][0]; 	//channel 3
	calibration[16] = (unsigned long) DAC_Correction[0][2][1]; 
	calibration[17] = (unsigned long) DAC_Correction[0][2][2]; 

	calibration[18] = (unsigned long) DAC_Correction[0][3][0]; 	//channel 4
	calibration[19] = (unsigned long) DAC_Correction[0][3][1]; 
	calibration[20] = (unsigned long) DAC_Correction[0][3][2]; 

	//Active Probe Set
	calibration[21] = (unsigned long) DAC_Correction[1][0][0]; 	//channel 1
	calibration[22] = (unsigned long) DAC_Correction[1][0][1]; 
	calibration[23] = (unsigned long) DAC_Correction[1][0][2]; 

	calibration[24] = (unsigned long) DAC_Correction[1][1][0]; 	//channel 2
	calibration[25] = (unsigned long) DAC_Correction[1][1][1]; 
	calibration[26] = (unsigned long) DAC_Correction[1][1][2]; 

	calibration[27] = (unsigned long) DAC_Correction[1][2][0]; 	//channel 3
	calibration[28] = (unsigned long) DAC_Correction[1][2][1]; 
	calibration[29] = (unsigned long) DAC_Correction[1][2][2]; 

	calibration[30] = (unsigned long) DAC_Correction[1][3][0]; 	//channel 4
	calibration[31] = (unsigned long) DAC_Correction[1][3][1]; 
	calibration[32] = (unsigned long) DAC_Correction[1][3][2]; 

	//Set 3
	calibration[33] = (unsigned long) DAC_Correction[2][0][0]; 	//channel 1
	calibration[34] = (unsigned long) DAC_Correction[2][0][1]; 
	calibration[35] = (unsigned long) DAC_Correction[2][0][2]; 

	calibration[36] = (unsigned long) DAC_Correction[2][1][0]; 	//channel 2
	calibration[37] = (unsigned long) DAC_Correction[2][1][1]; 
	calibration[38] = (unsigned long) DAC_Correction[2][1][2]; 

	calibration[39] = (unsigned long) DAC_Correction[2][2][0]; 	//channel 3
	calibration[40] = (unsigned long) DAC_Correction[2][2][1]; 
	calibration[41] = (unsigned long) DAC_Correction[2][2][2]; 

	calibration[42] = (unsigned long) DAC_Correction[2][3][0]; 	//channel 4
	calibration[43] = (unsigned long) DAC_Correction[2][3][1]; 
	calibration[44] = (unsigned long) DAC_Correction[2][3][2]; 

	//Set 4
	calibration[45] = (unsigned long) DAC_Correction[3][0][0]; 	//channel 1
	calibration[46] = (unsigned long) DAC_Correction[3][0][1]; 
	calibration[47] = (unsigned long) DAC_Correction[3][0][2]; 

	calibration[48] = (unsigned long) DAC_Correction[3][1][0]; 	//channel 2
	calibration[49] = (unsigned long) DAC_Correction[3][1][1]; 
	calibration[50] = (unsigned long) DAC_Correction[3][1][2]; 

	calibration[51] = (unsigned long) DAC_Correction[3][2][0]; 	//channel 3
	calibration[52] = (unsigned long) DAC_Correction[3][2][1]; 
	calibration[53] = (unsigned long) DAC_Correction[3][2][2]; 

	calibration[54] = (unsigned long) DAC_Correction[3][3][0]; 	//channel 4
	calibration[55] = (unsigned long) DAC_Correction[3][3][1]; 
	calibration[56] = (unsigned long) DAC_Correction[3][3][2]; 



	calibration[57] = (unsigned long) Hardware::_ADC_Offset[0][0];		//channel 1
	calibration[58] = (unsigned long) Hardware::_ADC_Offset[0][1];
	calibration[59] = (unsigned long) Hardware::_ADC_Offset[0][2];
	calibration[60] = (unsigned long) Hardware::_ADC_Offset[0][3];	
	
	calibration[61] = (unsigned long) Hardware::_ADC_Offset[1][0];		//channel 2
	calibration[62] = (unsigned long) Hardware::_ADC_Offset[1][1];
	calibration[63] = (unsigned long) Hardware::_ADC_Offset[1][2];
	calibration[64] = (unsigned long) Hardware::_ADC_Offset[1][3];	

	calibration[65] = (unsigned long) Hardware::_ADC_Offset[2][0];		//channel 3
	calibration[66] = (unsigned long) Hardware::_ADC_Offset[2][1];
	calibration[67] = (unsigned long) Hardware::_ADC_Offset[2][2];
	calibration[68] = (unsigned long) Hardware::_ADC_Offset[2][3];	

	calibration[69] = (unsigned long) Hardware::_ADC_Offset[3][0];		//channel 4
	calibration[70] = (unsigned long) Hardware::_ADC_Offset[3][1];
	calibration[71] = (unsigned long) Hardware::_ADC_Offset[3][2];
	calibration[72] = (unsigned long) Hardware::_ADC_Offset[3][3];	

	calibration[73] = (unsigned long) Hardware::_regADC[eFPGA1].adc_change;
	calibration[74] = 0;
	calibration[75] = (unsigned long) Hardware::_regADC[eFPGA2].adc_change;
	calibration[76] = (unsigned long) Hardware::_regADC[eFPGA1].channel_Adr;
	calibration[77] = (unsigned long) Hardware::_regADC[eFPGA2].channel_Adr;

	calibration[78] = (unsigned long)Hardware::_ChannelDelay[0];
	calibration[79] = (unsigned long)Hardware::_ChannelDelay[1];
	calibration[80] = (unsigned long)Hardware::_ChannelDelay[2];
	calibration[81] = (unsigned long)Hardware::_ChannelDelay[3];
	
	calibration[82] = (unsigned long) MenuStatus[MENU_HARDWARE][0];		//ADC register setting
	calibration[83]	= (unsigned long) MenuStatus[MENU_HARDWARE][1];		//Pre Gain 
	calibration[84]	= (unsigned long) MenuStatus[MENU_HARDWARE][3];		//ADC driver 
	
	calibration[85] = (unsigned long) MenuStatus[MENU_EXTUTILS][0];		// delay ch1
	calibration[86] = (unsigned long) MenuStatus[MENU_EXTUTILS][1];		// delay ch2
	calibration[87] = (unsigned long) MenuStatus[MENU_EXTUTILS][2];		// delay ch3
	calibration[88] = (unsigned long) MenuStatus[MENU_EXTUTILS][3];		// delay ch4
	
	calibration[89] = (unsigned long) MenuStatus[MENU_UTILITY][3];		// calibration set
	

	nr_flash_write_buffer(sectoraddress, calibration, 100);

	// check values
	if (sectoraddress[0] == 0xAAFF55FF)
	printf("\n\rcalibration data written to flash\r\n");
	else
	printf("\n\rflash checksum error\r\n");

	//printf("\n\r Beta correction = %d / ", calibration[8]);floatprintf(Hardware::_Beta); printf("\n\r");

	printf("\n\r write calibration set = %d\n\r",CalSet);
}


