
#ifndef hardware_t_h
#define hardware_t_h
#include <stdint.h>
#include "nios.h"

#define RELAY_SETTLE_TIME		5			// relay settle time is 4ms + 1ms safety margin
#define RELAY_SETTLE nr_delay(5)
#define DAC_MIDSCALE			32768			//Midscale of 16 Bit DAC
#define ADC_ZERO			128			//ADC zero position

typedef enum
{
	eFPGA1, //!< FPGA 1 always existing 
	eFPGA2, //!< FPGA 2 only in 4 channel models
} fpga_t;


//! trigger type 
typedef enum
{
	eTrigEdge, //!< level triggered by input value, configure with edge, level, channel
	eTrigPulse, //!< triggered by pulse width, configure with level, channel, timing
} trigType_t;

//! trigger source 
typedef enum
{
	eTrigCh1, //!< source is channel 1
	eTrigCh2, //!< source is channel 2
	eTrigCh3, //!< source is channel 3
	eTrigCh4, //!< source is channel 4
	eTrigExt, //!< source is external 
	eTrigTV,  //!< source is internaly generated trigger event with TV row timing

} trigSrc_t;

//------------------------------------------------------------------------------------------

/* old, don't use! */
//! external trigger subsource 
typedef enum
{
	eTrigExtLFR, //!< source is external trigger jack with low frequncy rejection
	eTrigExtHFR, //!< source is external trigger jack with high frequncy rejection
	eTrigExtLin, //!< source is power line signal
} trigExtSrc_t;

/* old, don't use! */
//! external trigger coupling 
typedef enum
{
	eTrigExtAC, //!< external trigger is AC coupled
	eTrigExtDC, //!< external trigger is DC coupled
} trigExtCpl_t;

// IDs (mostly) as defined by hardware
//! possible sources for external trigger
//! \todo check assignment
typedef enum
{
	eTrigSrcLF    = 2, //3, //!< trigger on unfiltered external signal
	eTrigSrcHF    = 3, //2, //!< trigger on highpass filtered external signal
	eTrigSrcMains = 1, //!< trigger on mains AC (50/60 Hz), not on input signal
	eTrigSrcVSync = 4, //!< trigger on channel 1 video sync seperator, vertical sync
	eTrigSrcCSync = 5, //!< trigger on channel 1 video sync seperator, composite sync
	eTrigSrcNone  = 0, //!< test, none of the syncs active
} trigExtSel_t;

//! external trigger coupling
typedef enum
{
	eTrigCoupleDC, //!< DC coupling
	eTrigCoupleAC, //!< AC coupling
} trigExtCoupling_t;

//------------------------------------------------------------------------------------------

//! trigger mode 
typedef enum
{
	eTrigFreerun,	//!< free running, continuous capture without trigger event
	eTrigAuto,	//!< waiting for trigger event but with auto timeout if nothing happens -> signal does not freeze
	eTrigNorm,    	//!< waiting for trigger event, capturing is completely stopped until event appears -> signal freezes  
} trigMode_t;

//! pulse trigger qualifier
typedef enum
{
	ePulseSmaller, //!< smaller than minimum
	ePulseLarger,  //!< larger than maximum
	ePulseBetween,  //!< between min and max
} trigPulseSel_t;


//! trigger edge selection
typedef enum
{
	eEdgeRise = 0, //!< falling edge, transition from higher to lower level
	eEdgeFall = 1, //!< rising edge, transition from lower to higher level
} trigEdgeSel_t;

// names for the FPGA register block
typedef union
{
	struct // naming similar to old firmware
	{	
		unsigned short ctrl;
		unsigned short adc_ctrl;
		unsigned int timebase;
		unsigned short pre;
		unsigned short channel_Adr;
		unsigned char trg_val[2];		// trigger level channel 1/2 + 3/4
		unsigned short trig_range;
		unsigned int trig_holdoff;
		unsigned short trig_width;
		unsigned char unused;
		unsigned char grid_color;
		unsigned int adc_change;
		unsigned int dummy;
	};
	unsigned int regword[8]; // access as words
} adc_t;

/*
typedef	struct	 //!< type of delayed queue with delay time
{	
	void (* hdl)(void);	//!< execution handle
	uint32_t prio;	//!< execution priority
	
} delay_t;
*/

class Hardware
{
	protected:
	//static unsigned long _ReadoutBuffer[5030];			// memory to store the signal temporarily
	
	static void (* _DelayHdl[5])(void);				// delay handle
	static void (* _PeriodicHdl)(void);				// periodic handle
	static uint32_t _VSyncCounter;					// VSync counter
	static uint32_t _VSyncBuffer;					// VSync counter buffer
	static uint32_t _DelayTimer[5];					// delay timer
	static uint32_t _PeriodicTimer;					// periodic timer
	static uint32_t _PTimerValue;					// buffer for the periodic timer value
		
	public:
	
	/* attribute definitions */
	
	static int _ChannelDelay[4];					// channel dependent signal delay
	static int _ADC_VirtualZero[4];					// Virtual Zero on ADC level (no scaling)
	static unsigned char _ADC_Offset[4][4];				// ADC correction for every single ADC of the quad-ADC -> will be loaded from flash
	
	static adc_t _regADC[2];					// register block sets for FPGA 1 + 2

	static unsigned int _regExtTrgLvl;
	static unsigned int _regExtTrgSetup;
	
	static trigSrc_t _TriggerSource;				// active trigger source

	
	static float _Beta;						// beta correction of ADC characteristic	

	
	/* method definitions */
	
	static void Init(void);	
									// Init the WatchDog
	static void ResetWatchdog(void);				// Resets the WatchDog
//	static void Restart_Record(void);				// Restart Recording
	static void Start_Record(void);                                 // Restart Recording	
	static void Stop_Record(void);				        // Stop Recording	
	static void Read_Version(void);                                 // Read out the hardware version
	static void Setup_Vars(void);					// Setup Variables with standart Values
	static void Setup_Hardware(void);				// Setup the Hardware
	static void Setup_Interrupts(void);                             // Setup the Interrupts
	static void Disable_All_Interrupts(void);		// Disable the Interrupts BF #010
	static void Set_Vars_Default(void);                             // Set Vars to default values
	//static char SearchZero(char channel, char level, int *DAC_Offset, int *ZeroSign);
	//static char SearchZeros(void);                                	// Search Signals for correct difference
//	static void GenerateFLASH(void);				// Generate FLASH content

//	Keyboard functions	
	static void DoEnableKeyInterrupt(void);				// Declare Enable PIO ISR
	static void DoDisableKeyInterrupt(void);			// Declare Disable PIO ISR
	static void ISR_KEY(int context);				// Declare Button Interrupt subroutine
	
	static void SetupTrigger(void);					// setup trigger modus
	
	static void Set_LED(int LED);					// Set LED On
	static void Reset_LED(int LED);					// Set LED Off
	//BF not needed static void Clear_LED(void);					// Clear all LED�s
	static void Send_LED(void);					// Send Shift Value for LED�s to Keyboard

	static void DoEnableRotInterrupt(void);				// Declare Enable PIO ISR
	static void DoDisableRotInterrupt(void);			// Declare Disable PIO ISR
	static void ISR_ROT(int32_t context);				// Declare Button Interrupt subroutine

	static void Zero_Levels_Center(void);				// 
	static void Zero_Levels_Restore(void);				// 

	static void DoEnableUARTInterrupt(void);			// Declare Enable UART2 ISR
	static void DoDisableUARTInterrupt(void);			// Declare Disable UART2 ISR
	static void ISR_UART(int context);				// Declare UART2 Interrupt subroutine
	
//**** Noch zu verbessern
	static void DoEnableUART2Interrupt(void);			// Declare Enable UART2 ISR
	static void DoDisableUART2Interrupt(void);			// Declare Disable UART2 ISR
	static void ISR_UART2(int context);				// Declare UART2 Interrupt subroutine
//****
	static void DoEnableTimer1Interrupt(void);			// Enable service routine
	static void DoDisableTimer1Interrupt(void);			// Disable service routine
	static void Start_Timer1(void);					// Start Timer1
	static void Stop_Timer1(void);					// Stop Timer1
	static void ISR_TIMER1(int32_t context);			// Timer1 ISR for combi trigger

	static void DoEnableTimer2Interrupt(void);			// Enable service routine Timer2
	static void DoDisableTimer2Interrupt(void);			// Disable service routine Timer2
	static void Start_Timer2(void);					//BF new start function
	static void Stop_Timer2(void);					//BF new stop function
	static void Reset_Timer2(void);					//BF new reset function
	static void ISR_TIMER2(int32_t context);			//Timer2 ISR
	
	static void DoEnableTimer3Interrupt(void);						// Enable service routine Timer3
	static void DoDisableTimer3Interrupt(void);						// Disable service routine Timer3
	static void ISR_TIMER3(int32_t context);						// Timer3 ISR for menu popups
	static void ResetTimer3(uint32_t period);						// Reset Timer3

	static void SetSwitches(uint8_t channel, uint32_t voltage);				// Set Switches for voltage devider
	static void ADDON_SetSwitches(uint8_t channel, uint32_t voltage);				// new input stage
	static void Sleep_us(uint32_t us);   		                     			// busy wait with microsecond resolution
//BF new
	static void SetDacOffset(uint8_t channel);							// Set the DAC Offset of a Channel
//BF test	static void SetCHDacOffset(char channel);					// Set the DAC Offset of a Channel
	
/********************* capture routines ********************************************************************************************************************/

	static void WriteADC(uint32_t channel, uint32_t value);					// write a value to the FGPA capture unit
	static uint32_t ReadADC(uint32_t channel);						// read a value from the FGPA capture unit
	
	static void ADC_ReadData(uint32_t channel, unsigned char *signal, unsigned int byte_count, bool invert);

	static void DoEnableADCInterrupt(void);							// Declare Enable ADC ISR
	static void DoDisableADCInterrupt(void);						// Declare Disable ADC ISR
	static void ISR_ADC_INT(int context);							// Declare ADC Interrupt subroutine
	static int Read_IRQ(void);                                      			// red the status of the irq
	static void Handle_ADC(void);							        // ADC Handler	
	static void Handle_Peak(void);
	
/***********************************************************************************************************************************************************/	

	static int TriggerCorrection(trigSrc_t src, unsigned char negativ, int TriggerPos);	// Search the Trigger
	static int AutoScale(uint32_t slow_tb);                                       				// Make Autoscale
	static int FindTimebase(uint8_t channel, uint32_t slow_tb);
	static int FindVoltage(uint8_t channel); 
	//BF del static void FindSlope(void);
	
	
	static void EnableVSyncInterrupt(void);						// Enable ReadVSync ISR
	static void DisableVSyncInterrupt(void);					// Disable ReadVSync ISR
	static void ISR_VSync(int context);						// ReadVSync Interrupt subroutine
	static void SetDelayTimer1(uint32_t timer, void (* hdl)(void));
	static void SetDelayTimer2(uint32_t timer, void (* hdl)(void));
	static void SetDelayTimer3(uint32_t timer, void (* hdl)(void));
	static void SetPeriodicTimer(uint32_t timer, void (* hdl)(void));		// set periodic timer value
	static void ClearPeriodicTimer(void);						// clear periodic timer	
	static uint32_t ReadVSync(void);						// read VSync timer value
	static void ResetStopWatch(void);						// load timer value to buffer
	static uint32_t ReadStopWatch(void);						// calculate elapsed time
	
	
	static void TransferPlanes(void);                               // Transfer Backbuffer to Planes
	static void ClearPlanes(void);                               // Clears and Transfer Backbuffer to Planes
	static void TransferDataPlane_asm_persistant(unsigned long Buffer_Adr1, unsigned long Plane_Adr1);
	static void TransferPlane_Set_Buffer_Adr(unsigned long Buffer_Adr1, unsigned long Plane_Adr1, unsigned long Buffer_Adr2, unsigned long Plane_Adr2);
	static void TransferPlanes_asm(unsigned long Buffer_Adr1, unsigned long Plane_Adr1, unsigned long Buffer_Adr2, unsigned long Plane_Adr2, unsigned long Buffer_Adr3, unsigned long Plane_Adr3, unsigned char count);
	static void TransferPlanes_asm_persistant(unsigned long Buffer_Adr1, unsigned long Plane_Adr1, unsigned long Buffer_Adr2, unsigned long Plane_Adr2, unsigned long Buffer_Adr3, unsigned long Plane_Adr3, unsigned char count);
	static void TransferPlanes_clear_asm(unsigned long Buffer_Adr1, unsigned long Plane_Adr1, unsigned long Buffer_Adr2, unsigned long Plane_Adr2, unsigned long Buffer_Adr3, unsigned long Plane_Adr3, unsigned char count);
	static void Planes_clear_asm(unsigned long Buffer_Adr1);
	static void Planes_clear_asm_all(unsigned char count, unsigned long Buffer_Adr1, unsigned long Buffer_Adr2, unsigned long Buffer_Adr3, unsigned long Buffer_Adr4, unsigned long Buffer_Adr5);
//BF del not used    	static void DoEnableLogicAnalyserInterrupt(void);	            // Declare Enable LogicAnalyser ISR
//BF del not used	static void DoDisableLogicAnalyserInterrupt(void);	            // Declare Disable LogicAnalyser ISR
//BF del not used	static void ISR_LogicAnalyser(int context);	                    // Declare LogicAnalyser Interrupt subroutine
	
	static void Calc_ADC_Offset(uint32_t channel);				//BF added
	static int Calc_DAC_Correction(uint32_t channel, int32_t *Correction);	//BF added
	static void Calibrate_ADC_ZeroOffsets(void);				//BF added
	static void Calibrate_DAC_ZeroOffsets(void);				//BF added

	static void FFT_Set_Channel(unsigned char backup);
	static void FFT_Restore_Channels(void);
	static void Reset_To_Default(void);
	static void Restore_From_Flash(void);
	static void Start_Up(void);
	static void LED_FunctionTest(char tmode);

	static void CenterTrace(uint8_t channel);
	static void DispatchTraces(void);
	static void TRIG_AutoLevel(void);
	static void TRIG_AutoPretrigger(void);

/**********************************************************************************************************************************************************/
				/* new capture trigger API from OSOZ */

//! write shadow registers to hardware 
static void CaptureSetupADC(void);
				
//! set the type of trigger event (edge, pulse)
static void CaptureTrigSetType( trigType_t ttype ); //!< trigger type

//! set the source for signal trigger events (channel 1 - 4, extern, TV)
static void CaptureTrigSetSource( trigSrc_t tsrc ); //!< trigger source

//! set the subsource for the external trigger
//static void CaptureTrigSetExtSource( trigExtSrc_t tsrc ); //!< external subsource

//! set the coupling for external trigger source (AC, DC)
//static void CaptureTrigSetExtCoupling( trigExtCpl_t tcpl ); //!< external coupling

//! set the trigger mode (auto, normal)
static void CaptureTrigSetMode( trigMode_t tmode ); //!< trigger mode

//! set the trigger level, relevant for internal level and pulse trigger
static void CaptureTrigSetLevel( trigSrc_t tsrc, unsigned char level );         //!< channel 0....3, external = 4, TV = 5 / trigger level -> range 0...255

//! set the trigger edge, relevant for internal level (and pulse?) trigger, external trigger
static void CaptureTrigSetEdge( trigEdgeSel_t edge ); //!< falling or rising edge

//! set the pulse trigger qualifier and timing
static void CaptureTrigSetPulse( trigPulseSel_t tselect, unsigned short min, unsigned short max ); //!< smaller, larger or within / minimum width in samples / maximum width in samples

//! set the trigger holdoff time, intentional re-arming delay time
static void CaptureTrigSetHoldoff( unsigned int holdoff ); //!< trigger holdoff time in samples, 0=disabled, no holdoff

static void CaptureSetTimebase(uint32_t subsampling);

//----------------------------------------------------------------------------------------------------------------------------------------------
				/* external trigger API */
//! how should the external trigger be used
static void TriggerSetExtSrc( trigExtSel_t select); //!< source of the trigger

//! level of external trigger -> realized as inline function
//static void TriggerSetExtLevel(unsigned int level); //!< comparison threshold, valid range 0...255, useful unsaturated range ca. 60...190

//! set AC/DC coupling of external trigger
static void TriggerSetExtCoupling( trigExtCoupling_t coupling); //!< DC or AC coupling

	
/**********************************************************************************************************************************************************/	


//static void SetReadoutLength(void);  // not used

//static void GetFullSignal(void);	// not used


};




#endif //tc_classes_h

