/***********************************************************************************
Sensirion SHTxx Sensor Library 0v2

***********************************************************************************/
/* Copyright Notice
 	This library for the SHT temperature and humidity sensors is based on the 
	application datasheet "Sample Code humidity sensor SHTxx" from Sensirion. 
	(c) Timo Dittmar
		(For now. If I get enough feedback and a positive answer from
		Sensirion I plan to release the code under GPL)
	Use without any warranty
*/


/* History
	Date, Version, Comment 
	2006-07-14	NA		- Initial conversion from Sensirion application note
						- inserted test code
	2006-07-17	NA		- Adjusted timing in measurement
						- changed data transfer in calc_sht; 
	2006-07-20	0v1rc	- Included heating element code 
	2006-08-05	0v1		- Some additional cleanup
						- homogenized function names;
	2006-08-13	0v2rc	- included functions for measuring with recurring interrupt
						- fewer orthographic errors :-)
						- first implementation of crc check
	2006-08-27	0v2		- second release
*/

#include <avr/io.h> //Microcontroller specific library, e.g. port definitions
#include <math.h> //
#include <stdlib.h> //
#include <inttypes.h> // 
#include <util/delay.h>
#include "libsht.h"
#include <avr/pgmspace.h>

const uint8_t CRC_TABLE[] PROGMEM = {0, 49, 98, 83, 196, 245, 166, 151,
 185, 136, 219, 234, 125, 76, 31, 46, 67, 114, 33, 16, 135, 182, 229,
 212, 250, 203, 152, 169, 62, 15, 92, 109, 134, 183, 228, 213, 66, 115,
 32, 17, 63, 14, 93, 108, 251, 202, 153, 168, 197, 244, 167, 150, 1,
 48, 99, 82, 124, 77, 30, 47, 184, 137, 218, 235, 61, 12, 95, 110, 249,
 200, 155, 170, 132, 181, 230, 215, 64, 113, 34, 19, 126, 79, 28, 45,
 186, 139, 216, 233, 199, 246, 165, 148, 3, 50, 97, 80, 187, 138, 217,
 232, 127, 78, 29, 44, 2, 51, 96, 81, 198, 247, 164, 149, 248, 201, 
 154, 171, 60, 13, 94, 111, 65, 112, 35, 18, 133, 180, 231, 214, 122, 
 75, 24, 41, 190, 143, 220, 237, 195, 242, 161, 144, 7, 54, 101, 84, 
 57, 8, 91, 106, 253, 204, 159, 174, 128, 177, 226, 211, 68, 117, 38, 
 23, 252, 205, 158, 175, 56, 9, 90, 107, 69, 116, 39, 22, 129, 176, 227, 
 210, 191, 142, 221, 236, 123, 74, 25, 40, 6, 55, 100, 85, 194, 243, 
 160, 145, 71, 118, 37, 20, 131, 178, 225, 208, 254, 207, 156, 173, 58, 
 11, 88, 105, 4, 53, 102, 87, 192, 241, 162, 147, 189, 140, 223, 238, 
 121, 72, 27, 42, 193, 240, 163, 146, 5, 52, 103, 86, 120, 73, 26, 43, 
 188, 141, 222, 239, 130, 179, 224, 209, 70, 119, 36, 21, 59, 10, 89, 
 104, 255, 206, 157, 172};
 
 const uint8_t REVERSED_BIT_ORDER_LIST[] PROGMEM = { 0, 128, 64, 192, 32, 160, 96, 224, 16, 144, 80, 208, 48, 176, 112, 
 240, 8, 136, 72, 200, 40, 168, 104, 232, 24, 152, 88, 216, 56, 184, 
 120, 248, 4, 132, 68, 196, 36, 164, 100, 228, 20, 148, 84, 212, 52, 
 180, 116, 244, 12, 140, 76, 204, 44, 172, 108, 236, 28, 156, 92, 220, 
 60, 188, 124, 252, 2, 130, 66, 194, 34, 162, 98, 226, 18, 146, 82, 
 210, 50, 178, 114, 242, 10, 138, 74, 202, 42, 170, 106, 234, 26, 
 154, 90, 218, 58, 186, 122, 250, 6, 134, 70, 198, 38, 166, 102, 230, 
 22, 150, 86, 214, 54, 182, 118, 246, 14, 142, 78, 206, 46, 174, 110, 
 238, 30, 158, 94, 222, 62, 190, 126, 254, 1, 129, 65, 193, 33, 161, 
 97, 225, 17, 145, 81, 209, 49, 177, 113, 241, 9, 137, 73, 201, 41, 
 169, 105, 233, 25, 153, 89, 217, 57, 185, 121, 249, 5, 133, 69, 197, 
 37, 165, 101, 229, 21, 149, 85, 213, 53, 181, 117, 245, 13, 141, 77, 
 205, 45, 173, 109, 237, 29, 157, 93, 221, 61, 189, 125, 253, 3, 131, 
 67, 195, 35, 163, 99, 227, 19, 147, 83, 211, 51, 179, 115, 243, 11, 
 139, 75, 203, 43, 171, 107, 235, 27, 155, 91, 219, 59, 187, 123, 251, 
 7, 135, 71, 199, 39, 167, 103, 231, 23, 151, 87, 215, 55, 183, 119, 
 247, 15, 143, 79, 207, 47, 175, 111, 239, 31, 159, 95, 223, 63, 191, 127, 255};

//-for ICC
/*
//----------------------------------------------------------------------------------
void _delay_us(unsigned short time_us) 
//----------------------------------------------------------------------------------
{
	unsigned short delay_loops;
	register unsigned short i;

	delay_loops = (time_us+3)/5*CYCLES_PER_US; // +3 for rounding up (dirty) 

	// one loop takes 5 cpu cycles 
	for (i=0; i < delay_loops; i++) {};
}
*/
//----------------------------------------------------------------------------------
unsigned char sht_write_byte(unsigned char sht_value)
//----------------------------------------------------------------------------------
// writes a byte on the Sensibus and checks the acknowledge
{
unsigned char i;
unsigned char error=0;
MAKE_SHT_DATA_PIN_OUTPUT;
asm volatile ("nop"::); // necessary because of the sync circuitry 
for (i=0x80;i>0;i/=2) //shift bit for masking
	{if (i & sht_value) SET_SHT_DATA; //masking value with i , write to SENSI-BUS
	 else CLEAR_SHT_DATA;
	 SET_SHT_SCK; //clk for SENSI-BUS
	 asm volatile ("nop"::);
	 asm volatile ("nop"::);
	 CLEAR_SHT_SCK;
	 asm volatile ("nop"::);
	 asm volatile ("nop"::);
	}
SET_SHT_DATA; //release DATA-line
MAKE_SHT_DATA_PIN_INPUT;
asm volatile ("nop"::);
SET_SHT_SCK; //clk #9 for ack
asm volatile ("nop"::);
asm volatile ("nop"::);
if (SHT_DATA) error =1; //check ack (DATA will be pulled down by SHT11)
CLEAR_SHT_SCK;
return error; //error=1 in case of no acknowledge
}



//----------------------------------------------------------------------------------
unsigned char sht_read_byte(unsigned char ack)
//----------------------------------------------------------------------------------
// reads a byte form the Sensibus and gives an acknowledge in case of "ack=1"
{
unsigned char i;
unsigned char val=0;
MAKE_SHT_DATA_PIN_OUTPUT;
asm volatile ("nop"::); // necessary because of the sync circuitry
SET_SHT_DATA; //release DATA-line
MAKE_SHT_DATA_PIN_INPUT;
asm volatile ("nop"::); // necessary because of the sync circuitry
for (i=0x80;i>0;i/=2) //shift bit for masking
	{ SET_SHT_SCK; //clk for SENSI-BUS
	 asm volatile ("nop"::); 
	 asm volatile ("nop"::);
	 if (SHT_DATA) val=(val | i); //read bit
	 CLEAR_SHT_SCK;
	 asm volatile ("nop"::); 
	 asm volatile ("nop"::);
	}
MAKE_SHT_DATA_PIN_OUTPUT;
asm volatile ("nop"::); // necessary because of the sync circuitry
if (ack) CLEAR_SHT_DATA; else SET_SHT_DATA; // Sent ack
SET_SHT_SCK; //clk #9 for ack
asm volatile ("nop"::);
asm volatile ("nop"::);
asm volatile ("nop"::);
CLEAR_SHT_SCK;
asm volatile ("nop"::);
asm volatile ("nop"::);
SET_SHT_DATA; //release DATA-line
return val;
}



//----------------------------------------------------------------------------------
void sht_transstart(void)
//----------------------------------------------------------------------------------
// generates a transmission start
{
MAKE_SHT_DATA_PIN_OUTPUT;
asm volatile ("nop"::);
SET_SHT_DATA; CLEAR_SHT_SCK; //Initial state
asm volatile ("nop"::);
asm volatile ("nop"::);
SET_SHT_SCK;
asm volatile ("nop"::);
asm volatile ("nop"::);
CLEAR_SHT_DATA;
asm volatile ("nop"::);
asm volatile ("nop"::);
CLEAR_SHT_SCK;
asm volatile ("nop"::);
asm volatile ("nop"::);
asm volatile ("nop"::);
asm volatile ("nop"::);
asm volatile ("nop"::);
asm volatile ("nop"::);
SET_SHT_SCK;
asm volatile ("nop"::);
asm volatile ("nop"::);
SET_SHT_DATA;
asm volatile ("nop"::);
asm volatile ("nop"::);
CLEAR_SHT_SCK;
}


//----------------------------------------------------------------------------------
void sht_connectionreset(void)
//----------------------------------------------------------------------------------
// communication reset: DATA-line=1 and at least 9 SCK cycles followed by transstart
{
unsigned char i;
MAKE_SHT_DATA_PIN_OUTPUT;
asm volatile ("nop"::);
SET_SHT_DATA; CLEAR_SHT_SCK; //Initial state
for(i=0;i<9;i++) //9 SCK cycles
{ SET_SHT_SCK;
asm volatile ("nop"::);
asm volatile ("nop"::);
CLEAR_SHT_SCK;
asm volatile ("nop"::);
asm volatile ("nop"::);
}
sht_transstart(); //transmission start
}



//----------------------------------------------------------------------------------
unsigned char sht_softreset(void)
//----------------------------------------------------------------------------------
// resets the sensor by a softreset
{
unsigned char error=0;
MAKE_SHT_SCK_PIN_OUTPUT;
sht_connectionreset(); //reset communication
error+=sht_write_byte(RESET); //send RESET-command to sensor
return error; //error=1 in case of no response form the sensor
}



//----------------------------------------------------------------------------------
unsigned char sht_read_statusreg(unsigned char *p_sht_value, unsigned char *p_checksum)
//----------------------------------------------------------------------------------
// reads the status register with checksum (8-bit)
{
unsigned char error=0;

sht_transstart(); //transmission start
error=sht_write_byte(STATUS_REG_R); //send command to sensor
*p_sht_value=sht_read_byte(ACK); //read status register (8-bit)
*p_checksum=sht_read_byte(noACK); //read checksum (8-bit)
return error; //error=1 in case of no response form the sensor
}



//----------------------------------------------------------------------------------
unsigned char sht_write_statusreg(unsigned char *p_sht_value)
//----------------------------------------------------------------------------------
// writes the status register with checksum (8-bit)
{
unsigned char error=0;
sht_transstart(); //transmission start
error+=sht_write_byte(STATUS_REG_W);//send command to sensor
error+=sht_write_byte(*p_sht_value); //send value of status register
return error; //error>=1 in case of no response form the sensor
}



//----------------------------------------------------------------------------------
unsigned char sht_measure(sht_value *p_sht_value, unsigned char *p_checksum, unsigned char mode)
//----------------------------------------------------------------------------------
// makes a measurement (humidity/temperature) with checksum
{
unsigned char error=0;
unsigned int i;
sht_value sht_value_temp;
/* sht_transstart(); //transmission start */
sht_connectionreset();
switch(mode){ //send command to sensor
case TEMP : error+=sht_write_byte(MEASURE_TEMP); break;
case HUMI : error+=sht_write_byte(MEASURE_HUMI); break;
default : break;
}
for (i=0;i<65535;i++) {
	if(SHT_DATA==0) break; //wait until sensor has finished the measurement
	_delay_us(6);  	// neccesary since the AVR is so much faster than the old 8051
					// and the 210ms for 14bit measuremnt are only a rough estimate
	}
if(SHT_DATA) error+=1; // or timeout is reached 
sht_value_temp.i = 256*sht_read_byte(ACK); //read the first byte (MSB)
sht_value_temp.i +=sht_read_byte(ACK); //read the second byte (LSB)
*p_checksum =sht_read_byte(noACK); //read checksum
*(p_sht_value)= sht_value_temp;
return error;
}




//----------------------------------------------------------------------------------------
void sht_raw_to_physical(sht_value *p_humidity ,sht_value *p_temperature)
//----------------------------------------------------------------------------------------
// calculates temperature [C] and humidity [%RH]
// input : humi [Ticks] (12 bit)
// temp [Ticks] (14 bit)
// output: humi [%RH]
// temp [C]
{ 
#ifdef FOST02
  const float C2= 0.648;    // for 8 Bit ( FOST02 )
  const float C3=-0.00072;  // for 8 Bit ( FOST02 )
#else
  const float C2= 0.0405;    // for 12 Bit
  const float C3=-0.0000028; // for 12 Bit
#endif
  const float C1=-4.0;       // for 12 Bit and 8 Bit
  const float T1=0.01; // for 14 Bit @ 5V
  const float T2=0.00008; // for 14 Bit @ 5V

float rh_lin; // rh_lin: Humidity linear
float rh_true; // rh_true: Temperature compensated humidity
float t_C; // t_C : Temperature [C]

#ifdef FOST02
t_C = 0.04*(*p_temperature).i +(SHT_TEMP_OFFSET); //calc. Temperature from ticks to [C]
#else
t_C = 0.01*(*p_temperature).i +(SHT_TEMP_OFFSET); //calc. Temperature from ticks to [C]
#endif

rh_lin=C3*(*p_humidity).i*(*p_humidity).i + C2*(*p_humidity).i + C1; //calc. Humidity from ticks to [%RH]
rh_true=(t_C-25)*(T1+T2*(*p_humidity).i)+rh_lin; //calc. Temperature compensated humidity [%RH]

if(rh_true>100)rh_true=100; //cut if the value is outside of
if(rh_true<0.1)rh_true=0.1; //the physical possible range

(*p_temperature).f=t_C;  //return temperature [C]
(*p_humidity).f=rh_true; //return humidity[%RH]
}



//--------------------------------------------------------------------
float calc_dewpoint(float h,float t)
//--------------------------------------------------------------------
// calculates dew point
// input: humidity [%RH], temperature [C]
// output: dew point [C]
{ float k,dew_point ;
k = (log10(h)-2)/0.4343 + (17.62*t)/(243.12+t);
dew_point = 243.12*k/(17.62-k);
return dew_point;
}

//--------------------------------------------------------------------
void sht_switch_heating_element(unsigned char onoff)
//--------------------------------------------------------------------
// shwitches the internal heating element
// input onoff 0:off 1:on 
{
unsigned char status;
unsigned char checksum;
sht_read_statusreg(&status, &checksum);
if (onoff==0) status &= ~(HEATER_BIT);
	else status |= (HEATER_BIT);
sht_write_statusreg(&status);
}


//--------------------------------------------------------------------
unsigned char sht_check_crc(sht_value *p_sht_value, unsigned char *p_checksum, unsigned char mode, unsigned char statusreg)
//--------------------------------------------------------------------
// Checks whether the data integrity is ok. 
// Returns 1 if the crc value calculated from the data and the crc value from sensor are the same, else 0
// See application note "CRC" from sensirion for further details
{

unsigned char crc;

crc = pgm_read_byte(&REVERSED_BIT_ORDER_LIST[statusreg & 15]);
if (mode ==TEMP) crc =pgm_read_byte(&CRC_TABLE[crc ^ MEASURE_TEMP]); 
	else if(mode == HUMI) crc = pgm_read_byte(&CRC_TABLE[ crc ^ MEASURE_HUMI]);  
		else crc =255;  // error: mode is not recognized
		
crc = pgm_read_byte(&CRC_TABLE[crc ^ ((*p_sht_value).i >> 8)]);

crc = pgm_read_byte(&CRC_TABLE[crc ^ ((*p_sht_value).i & 0x00ff)]);
return (crc==pgm_read_byte(&REVERSED_BIT_ORDER_LIST[*p_checksum]));

}

/* 	The following functions should be used together with an recuring interrupt.
	They allow the splitting of the measurement in three parts in order to avoid
	the dump burning of CPU time. Although the naming of the functions should make their 
	use obvios here some ideas how to use them:
	
	void recurring_interrupt_service_routine()
	{
	...
	if (flag== MEASURING) && (sht_is_measurement_finished() == 1) flag == FINISHED; 
	...
	}
	
	void main()
	{
	unsigned char error =0;
	unsigned char checksum;
	sht_value humidity;
	...
	error += _sht_begin_measurement(SHT_HUMI);
	flag = MEASURING;
	for{;;}
		{
		if (flag == FINISHED) {
			_sht_get_measurement_data(&humidity, (unsigned char*)&checksum);
			flag = DONE;
			}
		}
	}
*/
	

//--------------------------------------------------------------------
unsigned char _sht_begin_measurement(unsigned char mode)
//--------------------------------------------------------------------
{
unsigned char error=0;
sht_connectionreset();
switch(mode){ //send command to sensor
case TEMP : error+=sht_write_byte(MEASURE_TEMP); break;
case HUMI : error+=sht_write_byte(MEASURE_HUMI); break;
default : {error = 20;break;}
}
return error;
}


//--------------------------------------------------------------------
unsigned char _sht_is_measurement_finished(void)
//--------------------------------------------------------------------
{
if (SHT_DATA==0) return 1;
else return 0;
// return (unsigned char)(~SHT_DATA);
}


//--------------------------------------------------------------------
void _sht_get_measurement_data(sht_value *p_sht_value, unsigned char *p_checksum)
//--------------------------------------------------------------------
{
sht_value sht_value_temp;
sht_value_temp.i = 256*sht_read_byte(ACK); //read the first byte (MSB)
sht_value_temp.i +=sht_read_byte(ACK); //read the second byte (LSB)
*p_checksum =sht_read_byte(noACK); //read checksum
*(p_sht_value)= sht_value_temp;
}
#ifdef FOST02_MAIN
//----------------------------------------------------------------------------------
void main( void )
//----------------------------------------------------------------------------------
// sample program that shows how to use FOST02 functions
// 1. connection reset 
// 2. measure humidity [ticks](8 bit) and temperature [ticks](12 bit)
// 3. calculate humidity [%RH] and temperature [C]
// 4. calculate dew point [C]
// 5. print temperature, humidity, dew point  

{ sht_value humi_val,temp_val;
  float dew_point;
  float volatile temp, humi;     //- debug
  unsigned char error,checksum;
  unsigned int i;
  //MAKE_SHT_SCK_PIN_OUTPUT;
  //init_uart();
  //init_devices();
  error = sht_softreset();
  while(1)
  { 
    error=0;
    error+=sht_measure( &humi_val,&checksum,HUMI);  //measure humidity
    error+=sht_measure( &temp_val,&checksum,TEMP);  //measure temperature
    if(error!=0) sht_connectionreset();                 //in case of an error: connection reset
    else
    { 

      sht_raw_to_physical(&humi_val, &temp_val);
      humi = humi_val.f;
      temp = temp_val.f;
      dew_point=calc_dewpoint(humi_val.f,temp_val.f); //calculate dew point
      //printf("temp:%5.1fC humi:%5.1f%% dew point:%5.1fC\n",temp_val.f,humi_val.f,dew_point);
    }
    //----------wait approx. 0.8s to avoid heating up SHTxx------------------------------      
    for (i=0;i<40000;i++);     //(be sure that the compiler doesn't eliminate this line!)
    //-----------------------------------------------------------------------------------                       
  }
} 
#endif