// generated with AvrWiz v0.14.2771.34743 (by g@greschenz.de)
// -----------------------------------------
// cpu:     ATtiny2313
// speed:   8 mhz (max: 20 mhz)
// voltage: 5 V
// ram:     128 bytes (0x0060-0x00df)
// rom:     1024 bytes (0x0000-0x03ff)
// eeprom:  256 bytes (0x0000-0x00ff)
// -----------------------------------------
/*

Doorsequencer for Tigercat
==========================

Rev 0.0: 08.08.2007 p.b.  Initial Release
Rev 0.1: 30.08.2007 p.b.  Some #defines added, Frametime now 20ms, 
                          fixed error: "re-enable servo 1 while adjusting servo 2"


Thanks to Gnter Greschenz, Peter Dannegger, Tobias Tetzlaff and others

*/
// -----------------------------------------

// todo:
// - richtiges servo abgeschaltet?

 
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include "gearseq.h"

#if !defined (__AVR_ATtiny2313__)
#error __AVR_ATtiny2313__ not defined !
#endif

#if !defined (F_CPU)
#define F_CPU 8000000
#endif

#define RAMSTART 0x0060
#define RAMSIZE (RAMEND-RAMSTART)


typedef union {
  struct {
    int16_t i16l;     // 16bit access
    int16_t i16h;
    };
  int32_t i32;        // 32bit access
} int32_hl;


// Servo 1 at PB3 is for moving the gear (will be disabled after PWOFF_TOUT seconds)
// Servo 2 at PB4 is for moving the gear doors (stays enabled)

struct servo {
  uint16_t start;     // Servo start position (in usec)
  uint16_t end;       // Servo end position (in usec)
  int16_t time;       // Servo move time (in steps)
  int16_t step;       // Servo actual time position (in steps)
  int32_hl pos;       // Servo actual position (in usec)
  int32_hl diff;      // Servo way to move (in usec)
  uint8_t  port;      // port number of OCR1A/B
} s1, s2;

struct savedata {
  int16_t pause;      // Pause between Servo 1 and Servo 2 (in steps)
  uint8_t rx_down1;   // lower pulse limit of "gear down"
  uint8_t rx_down2;   // upper pulse limit of "gear down"
  uint8_t rx_up1;     // lower pulse limit of "gear up"
  uint8_t rx_up2;     // upper pulse limit of "gear up"
  uint8_t servo_state;// state machine
  int8_t  dir;        // direction flag
} sd;
  
int16_t p_step;                                // Pause counter
uint8_t s1_pwoff = PWOFF_TOUT*FRAMES_PER_SEC;  // countdown for servo 1 power off
uint8_t adjust;                                // references the value to be adjusted

volatile uint8_t tmr1_flag;     // set on Timer1 Compare A Match Int
volatile uint8_t int0_flag;     // set on falling edge of INT0

// pulse lengths from rc-receiver:
// Graupner MC22:
//   -150% = 27..28
//   -100% = 33..34
//      0% = 46..47
//   +100% = 58..59
//   +150% = 65..66
volatile uint8_t rx_pulse;  // pulse length
uint8_t          rx_cntup;  // counter of valid up-pulses
uint8_t          rx_cntdn;  // counter of valid down-pulses

struct ee_servo {
  uint16_t start;           // Servo start position (in usec)
  uint16_t end;             // Servo end position (in usec)
  int16_t time;             // Servo move time (in steps)
};

 
// servo data in EEPROM:
uint16_t dummy EEMEM = 0;
struct ee_servo ee_s1 EEMEM = {STARTDEFAULT, ENDDEFAULT, TIMEDEFAULT*FRAMES_PER_SEC};  // Servo 1 data in EEPROM
struct ee_servo ee_s2 EEMEM = {STARTDEFAULT, ENDDEFAULT, TIMEDEFAULT*FRAMES_PER_SEC};  // Servo 2 data in EEPROM
struct savedata ee_sd EEMEM = {PAUSEDEFAULT*FRAMES_PER_SEC, 30, 36, 55, 61, 
                               ST_DOWN, DIR_DOWN};                                     // misc data

const unsigned int tens[] PROGMEM = { 10, 100, 1000};

// -----------
// --- isr ---
// -----------

ISR(INT0_vect) // External Interrupt Request 0
{
  if(PIND & (1<<PD2))       // INT from rising edge?
    { 
    TCNT0  = 0;             // reset Timer
    TCCR0B = (1<<CS02);     // start Timer 1/256 MCU clock
    MCUCR  = FALL_EDGE;     // INT0 on falling edge
    }
  else                      // INT from falling edge
    {
    TCCR0B = 0x00;          // stop Timer
    rx_pulse = TCNT0;       // read pulse width
    MCUCR  = RISE_EDGE;     // INT0 on rising edge
    int0_flag = 1;
    } 
}


ISR(TIMER1_COMPA_vect)      // Timer/Counter1 Compare A Match
{
  tmr1_flag = 1;            // set flag for use in main()
}



// --------------
// --- EEPROM ---
// --------------

void servo_ee_read (struct servo *s, struct ee_servo *ee)
{
  eeprom_read_block(s, ee, sizeof(struct ee_servo));
}


void servo_ee_write (struct servo *s, struct ee_servo *ee)
{
  eeprom_write_block(s, ee, sizeof(struct ee_servo));
}


void data_ee_write (void)
{
  eeprom_write_block(&sd, &ee_sd, sizeof(struct savedata));
}


// --------------
// --- USART0 ---
// --------------

void USART0_write(char data)
{
  while (!(UCSRA & (1<<UDRE)));
  UDR = data;
}


void outs_P (const char *str ) 
{
  char c;

  while ((c = pgm_read_byte(str)))
    {
    USART0_write (c);
    str++;    
    }
}


// print an integer as decimal in two different ways:
// mode = 0: print sign, print 3 digits, drop the 4th digit (-> div by 10)
// mode = 1: no sign, print 3 digits
void outint(int16_t val, uint8_t mode)
{
  
  uint8_t  d, i;
  uint16_t uval = val;

  if (!mode)
    {
    if( val < 0 )
      {
      uval = -val;
      USART0_write( '-' );
      }
    else
      USART0_write( '+' );
    i = 3;
    }
  else
    {
    USART0_write( ' ' );
    i = 2;
    }

  do{
    i--;
    for( d = '0'; uval >= pgm_read_word(&tens[i]); uval -= pgm_read_word(&tens[i]) ){
      d++;
    }
  USART0_write( d );
  }while( i );

  if (mode)
    USART0_write( (uint8_t)uval + '0' );
}


// print servo position 1000...2000 usec as
// -100...+100:
void outint1(int16_t val)
{
  outint((val<<1)-3000, 0);         // outint drops the last digit
}


// print int as 0...999:
void outint2(int16_t val)
{
  outint(val, 1);                   // print 3 digits without sign
}


// --------------
// --- Servos ---
// --------------

void servo_init (struct servo *s)
{
  s->pos.i16l  = 0; 
  s->diff.i16l = 0;

  if (sd.servo_state == ST_DOWN)
    {
    s->step      = 0;               // set time position to start
    s->pos.i16h  = s->start;        // servo in start position
    }
  else
    {
    s->step      = s->time;         // set time position to end
    s->pos.i16h  = s->end;          // servo in end position
    }

  s->diff.i16h = s->end - s->start;
  s->diff.i32 /= s->time;           // increments for each step

  _SFR_IO16(s->port) = s->pos.i16h; // set Timer Compare register (Pos of Servo)
}
 

void servo_calc (struct servo *s)
{
  if (sd.dir == DIR_UP)                     // are we going up?
    {
    if (s->step < s->time)                  // still moving?
      {
      s->step++;
      s->pos.i32 += s->diff.i32;            // calculate new position
      }
    else
      {
      sd.servo_state++;                     // servo movment finished
      p_step = sd.pause;                    // goto next state
      s1_pwoff = PWOFF_TOUT*FRAMES_PER_SEC; // start countdown for servo 1 poweroff
      }
    }
  else                                      // we are going down
    {
    if (s->step > 0)                        // still moving?
      {
      s->step--;
      s->pos.i32 -= s->diff.i32;            // calculate new position
      }
    else
      {
      sd.servo_state--;                     // servo movment finished
      p_step = sd.pause;                    // goto previous state
      s1_pwoff = PWOFF_TOUT*FRAMES_PER_SEC; // start countdown for servo 1 poweroff
      }
    }

  _SFR_IO16(s->port) = s->pos.i16h;         // set Timer Compare register (Pos of Servo)
}


void print_spos (struct servo *s)
{
  if (sd.servo_state == ST_DOWN) outint1(s->start);   // print actual servo start or
  if (sd.servo_state == ST_UP)   outint1(s->end);     // end position
}


void print_menu (void)
{
  switch (adjust)
    {
    case A_NOTHING:
      break;    

    case A_SERVO1:
      outs_P(PSTR("S1"));
      print_spos (&s1);
      break;    
  
    case A_SERVO2:
      outs_P(PSTR("S2"));
      print_spos (&s2);
      break;    
  
    case A_TIME1:
      outs_P(PSTR("T1"));
      outint2(s1.time);
      break;    

    case A_TIME2:
      outs_P(PSTR("T2"));
      outint2(s2.time);
      break;    

    case A_PAUSE:
      outs_P(PSTR("Pa"));
      outint2(sd.pause);
      break;    

    case A_RXUP:
      outs_P(PSTR("Ru"));
      outint2((int)rx_pulse);
      break;    

    case A_RXDOWN:
      outs_P(PSTR("Rd"));
      outint2((int)rx_pulse);
      break;    
    }
  USART0_write('\r');
}


void adj_servo (struct servo *s, int16_t diff)
{
  uint16_t *val;
  
  if (sd.servo_state == ST_DOWN) val = &s->start;  // adjust start position 
  if (sd.servo_state == ST_UP)   val = &s->end;    // adjust end position   
    
  *val += diff;                          // add the difference
  if (*val < SERVOMIN) *val = SERVOMIN;  // and check boundaries
  if (*val > SERVOMAX) *val = SERVOMAX;

  servo_init(s);                         // re-calculate variables
}


void adj_time (int16_t *val, int16_t diff)
{
  *val += diff;                          // add the difference
  if (*val < TIMEMIN) *val = TIMEMIN;    // and check boundaries
  if (*val > TIMEMAX) *val = TIMEMAX;
}


void adj_value (int16_t diff)
{
  if (sd.servo_state == ST_DOWN ||       // adjusting is only possible
      sd.servo_state == ST_UP)           // in the end positions
    {
    switch (adjust)
      {
      case A_NOTHING:
        break;    

      case A_SERVO1:
        adj_servo (&s1, diff);           // adjust servo start or end position
        PORTD &= ~(1<<PD4);              // LED1 on
        TCCR1A = EN_S1S2;                // re-enable pulses for servo 1  
        break;    
    
      case A_SERVO2:
        adj_servo (&s2, diff);           // adjust servo start or end position
        PORTD &= ~(1<<PD5);              // LED2 on
        break;    
    
      case A_TIME1:
        adj_time (&s1.time, diff);       // inc or dec travel time
        servo_init(&s1);                 // re-calculate variables
        break;    

      case A_TIME2:
        adj_time (&s2.time, diff);       // inc or dec travel time
        servo_init(&s2);                 // re-calculate variables
        break;    

      case A_PAUSE:
        sd.pause += diff;                // inc or dec pause time
        if (sd.pause < 0) sd.pause = 0;
        if (sd.pause > 999) sd.pause = TIMEMAX;
        break;    

      case A_RXUP:
      case A_RXDOWN:
        break;    
      }
    print_menu ();
    }
}


// check and decrement poweroff-counter
// for servo 1:
void check_s1_off(void)
{
  if (s1_pwoff)                     // are we still counting down?
    {
    s1_pwoff--;                     // yes: dec counter
    if (!s1_pwoff)                  // is it now zero?
      {
      TCCR1A = DIS_S1;              // yes: disable pulses for servo 1
      data_ee_write ();             // save servo positions to EEPROM
      PORTD |= (1<< PD5 | 1<<PD4);  // switch LEDs off
      }
    }

  print_menu();
}


// --------------
// --- main() ---
// --------------

int main(void)
{
  // init ports:        // 7  6  5  4  3  2  1  0
  PORTA = 0;            // ---------------LowLowLow
  DDRA  = 0;            // ---------------In In In 
  PORTB = 0;            // LowLowLowLowLowLowLowLow
  DDRB  = 0x1E;         // In In In OutOutOutOutIn 
  PORTD = 0x04;         // ---LowLowLowLowPulLowLow
  DDRD  = 0x30;         // ---In OutOutIn In In In 

  // init interrupts:
  MCUCR |= RISE_EDGE;   // Rising edge of INT0 generates an interrupt request
  GIMSK |= (1<<INT0);   // External Interrupt Request 0 Enable

  // USART0 settings: 19200 baud 8-n-1
  // WARNING: real baud = 19230: err = 0,156249999999991%
  UBRRH = 0;
  UBRRL = 25;
  UCSRB = (1<<RXEN)  | (1<<TXEN);
  UCSRC = (1<<UCSZ1) | (1<<UCSZ0);

  // read data from EEPROM:
  servo_ee_read (&s1, &ee_s1);
  servo_ee_read (&s2, &ee_s2);
  eeprom_read_block(&sd, &ee_sd, sizeof(struct savedata));

  // init servo variables:
  s1.port = OCR1A_P;    // set port of OCR1A
  s2.port = OCR1B_P;    // set port of OCR1B
  servo_init (&s1);
  servo_init (&s2);

  // init timer 1:
  ICR1   = FRAME_TIME * 1000;                   // PWM cycle time in usec
  TCCR1A = EN_S1S2;                             // OC1A/B clr on match, set on TOP 
  TCCR1B = (1<<WGM13) | (1<<WGM12) | (1<<CS11); // TOP = ICR1, clk = sysclk/8 (->1us)
  TCNT1  = 0;                                   // reset Timer
  TIMSK  = (1<<OCIE1A);                         // Timer Compare A Match Int enable 

  sei();                // enable interrupts


  for (;;)
    {
    if (int0_flag)                      // synced with rx Frame
      {
      int0_flag = 0;

      if (rx_pulse >= sd.rx_up1 &&      // valid rx-"up" frame?
          rx_pulse <= sd.rx_up2)
        {
        if (rx_cntup <  RX_VALID) rx_cntup++;         // count valid frames
        if (rx_cntup == RX_VALID) sd.dir = DIR_UP;    // enough to switchover
        }
      else
        rx_cntup = 0;
        
      if (rx_pulse >= sd.rx_down1 &&    // valid rx-"down" signal?
          rx_pulse <= sd.rx_down2)
        {
        if (rx_cntdn <  RX_VALID) rx_cntdn++;         // count valid frames
        if (rx_cntdn == RX_VALID) sd.dir = DIR_DOWN;  // enough to switchover
        }
      else
        rx_cntdn = 0;
      }

    if (tmr1_flag)                      // synced with Servo Frame
      {
      tmr1_flag = 0;
      
      // Servo state machine:
      switch (sd.servo_state)       
        {
        case ST_DOWN:
          check_s1_off();               // poweroff servo 1 if timed out
          if (sd.dir == DIR_UP)         // we are down and received DIR_UP:
            {
            sd.servo_state = ST_GO_S1;  // start with servo 1
            TCCR1A = EN_S1S2;           // re-enable pulses for servo 1  
            PORTD &= ~(1<<PD4);         // LED1 on
            }
          break;
        
        case ST_GO_S1:                  // servo 1 is moving
          servo_calc(&s1);              // calculate new position
          break;
      
        case ST_WAIT:
          if (p_step)
             p_step--;                  // pause countdown running
          else
            {
            sd.servo_state += sd.dir;
            PORTD ^= (1<< PD5 | 1<<PD4);// toggle both LEDs
            }
          break;
         
        case ST_GO_S2:                  // servo 2 is moving
          servo_calc(&s2);              // calculate new position
          break;

        case ST_UP:    
          check_s1_off();               // poweroff servo 1 if timed out
          if (sd.dir == DIR_DOWN)       // we are up and received DIR_DOWN:
            {
            sd.servo_state = ST_GO_S2;  // start with servo 2
            TCCR1A = EN_S1S2;           // re-enable pulses for servo 1  
            PORTD &= ~(1<<PD5);         // Test: LED2 on
            }
          break;
        }
      }


    if (UCSRA & (1<<RXC))               // UART char available?
      {
      switch (UDR)                      // yes: get it
        {
        case 'u':
          if (adjust < 7) adjust++;     // next value to adjust
          break;

        case 'd':
          if (adjust > 1) adjust--;     // prev value to adjust
          break;

        case '+':
          adj_value(+ADJ_STEP);         // increment value
          break;

        case '-':
          adj_value(-ADJ_STEP);         // increment value
          break;

        case '\r':
          if (adjust == A_RXUP)
            {
            sd.rx_up1 = rx_pulse - RX_TOL;   // save Rx pulse lower limit
            sd.rx_up2 = rx_pulse + RX_TOL;   // save Rx pulse upper limit
            }
          if (adjust == A_RXDOWN)
            {        
            sd.rx_down1 = rx_pulse - RX_TOL; // save Rx pulse lower limit
            sd.rx_down2 = rx_pulse + RX_TOL; // save Rx pulse upper limit
            }
          servo_ee_write (&s1, &ee_s1);      // save adjusted values to eeprom
          servo_ee_write (&s2, &ee_s2);
          data_ee_write ();
        }
      }
    }
  return 0;
}
