// --- [ DALI_main.c ] ------------------------------------------------------
//
//       tab = 3
//       28.09.2015
//
//       Status: kein Fehler bekannt

#define EXTERN

#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdlib.h>
#include <string.h>
#include <util/delay.h>

#include "rs232_read_c.h"
#include "global.h"

#define DALI_PORT       PORTB
#define DALI_DDR        DDRB
#define DALI_PIN        PINB
#define DALI_OUTPUT     PB1
#define DALI_INPUT      PB0

#define DALI_ON         DALI_PORT |= (1<<DALI_OUTPUT)
#define DALI_OFF        DALI_PORT &= !(1<<DALI_OUTPUT)

#define DALI_WAIT       (_delay_us(416))

// Prescale 64 -> (64 * 256) /3.686.400 = 4.444ms
// ein Takt = 833us
#define TIMER0_START    TCCR0B = (1<<CS01 | 1<<CS00)
#define TIMER0_STOP     TCCR0B = 0

// die Zhlerstnde Timer0 fr halben/ganzen Takt
#define HTMIN           20
// Mitte = 24
#define HTMAX           29
#define GTMIN           45
// Mitte = 49
#define GTMAX           55

// Prescale 8
#define TIMER1_START    TCCR1B = (1<<CS11 | 0<<CS10)
#define TIMER1_STOP     TCCR1B = 0
// ByteDauer 9.17ms
// 0.00917 / ( 8 / 3.686.400) = 4226
#define PRELOAD         (65536 - 4226)
#define ARRAYSIZE       23

#define DATA_RECEIVED   0
#define NO_ANSWER       1
#define TIMER_0_OVL     2
#define TIMER_1_OVL     3
#define BYTECNT_ERROR   0x10
#define BYTE4_ERROR     0x11

void dali_send_data(uint8_t * data);
void dali_send_byte(uint8_t d);
void dali_send_bit0(void);
void dali_send_bit1(void);
uint8_t dali_read_byte(void);

EXTERN volatile uint8_t rx_flag;

// --- Lokale Variablen ------------------------------------------------------
volatile uint8_t timer_ovl;
uint8_t write2x = FALSE;
uint8_t time[ARRAYSIZE];                   // Zeiten der Flankenwechsel

// ----------------------------------------------------------------------------
int main( void )
{
    uint8_t data[4];                // Array zum Abholen der Daten
    uint8_t j;
    uint16_t jj;

    ACSR = (1<<ACD);                // Analog Comparator Disable
    DALI_DDR |= (1<<DALI_OUTPUT);   // Den Ausgang fr TX setzen

    RS232_init();
    asm volatile ("sei");

    while(TRUE)
    {
        if (rx_flag)                // Sind Daten eingegangen ??
        {
            rx_flag = FALSE;
            RX_Get_Data(data);      // die 4 empfangenen Bytes abholen

                                    // Das 4. Byte auf Gltigkeit testen
            if (!((data[3] == WRITE1x) || (data[3] == WRITE2x) || data[3]== READ_BITS))
            {
                put_c(START_BYTE);
                put_c(BYTE4_ERROR);
                put_c(BYTE4_ERROR);
            }
            else
            {
                // Den Datenteil (Byte1 und Byte 2) senden
                dali_send_data(data);
                // Eine Antwort abholen, sofern eine geliefert wird
                // timerovl zeigt an, welchen Erfolg das read hatte:
                // 0 = ein Byte empfangen
                // 1 = kein Antwort eingegangen
                // 2/3 = Fehler im Bit bzw. Byte-Timing
                j = dali_read_byte();

                if (data[3] == WRITE1x)
                {
                    put_c(START_BYTE);
                    put_c(timer_ovl);
                    put_c(j);
                    // Nach dem Einlesen einer Antwort sollte eine Wartezeit von
                    // 9.16ms eingelegt werden.
                    // Durch die Kommunikation vie RS232 bei 9800 Baud entsteht diese
                    // Verzgerung "automatisch"
                    // 3 Byte zurckliefern und dann 4 Byte empfangen,
                    // das dauert schon 7ms.
                }

                // falls die beiden Datenbytes ein weiteres Mal gesendet werden sollen
                else if (data[3] == WRITE2x)
                {
                    // Es macht eigentlich keinen Sinn, Daten 2x lesen zu wollen,
                    // aber falls es doch jemand macht ....
                    // Wenn Daten zurckgeliefert wurden, dann muss noch eine
                    // Wartezeit von 9.16ms eingelegt werden, bevor die nschste
                    // bertragung startet.
                    if (timer_ovl == 0) _delay_ms(10);

                    dali_send_data(data);
                    j = dali_read_byte();

                    put_c(START_BYTE);
                    put_c(timer_ovl);
                    put_c(j);
                }

                // Falls nur das Timing ausgelesen werden soll
                else if (data[3] == READ_BITS)
                {
                    // in time[0] und time[1] steht der Zhlerstand von Timer1
                    // bei der fallenden Flanke der Antwort
                    // - oder beim Timer1_ovl, wenn keine Antwort eingeht.
                    // Den Zhlerstand des Timer1 um den Preload reduzieren
                    // wenn timer_ovl == 0 (kein Fehler)
                    // damit die Wartezeit einfacher bestimmt werden kann.
                    if (!(timer_ovl))
                    {
                        jj = (time[0] << 8) + time[0];
                        jj -= PRELOAD;
                        time[0] = (uint8_t) (jj >> 8);
                        time[1] = (uint8_t) jj & 0xFF;
                    }

                    for (j = 0; j < ARRAYSIZE; j++)
                    {
                        put_c(time[j]);
                        time[j] = 0;    // Die Daten gleich wieder lschen
                    }
                }
            }
        }

        else        // wenn das Flag nach Ablauf von 4ms nicht gesetzt ist,
                    // dann sind keine 4 Byte empfangen worden,
                    // in diesem Fall den Counter auf 0 zurckstellen
        {
            put_c(START_BYTE);
            put_c(BYTECNT_ERROR);
            put_c(BYTECNT_ERROR);
            RX_Clear_Counter();
        }

// --- Hier wird ein SLEEP eingelegt ------------------------------------------
        while((UCSRA & 1<< TXC) == 0);

        MCUCR = (1<<SE);
        asm volatile ("sleep");		// das Sleep wird durch RXE unterbrochen
        MCUCR = 0;
        // Das erste Byte wurde empfangen, nun auf die nchsten drei Byte warten,
        // falls sie nicht innerhalb von 4ms eintreffen, das Warten beenden.
        // Timer0 = 256 * 64 / 3.686.400 = 4.44ms
        // Das gilt fr 9600 Baud
        timer_ovl = 0;
        TCNT0 = 0;
        TIMSK = (1<<TOIE0 | 0<<TOIE1);
        TIMER0_START;
        // Das rx_flag oder der timer_ovl beenden das Warten
        while ((rx_flag + timer_ovl) == 0);

        TIMER0_STOP;
        TIMSK = 0;
    }           // while(1)
}	            // main()

// -----------------------------------------------------------------------------
// Sendet die 16 Dali-Bits, zuerst das Startbit,
// dann die 16 Bit beginnend mit dem hchstwerigen Bit
// und anschlieend zwei Stopbits.
void dali_send_data(uint8_t *data)
{
    // Startbit senden
    dali_send_bit1();

    // die beiden Datenbytes senden
    dali_send_byte(data[1]);
    dali_send_byte(data[2]);
    DALI_OFF;

    // 2 Stopbits einlegen
    DALI_WAIT;
    DALI_WAIT;
    DALI_WAIT;
    DALI_WAIT;

    // Wartezeit bis zum nchsten Frame >= 9.17ms
}

// -----------------------------------------------------------------------------
// Sendet ein einzelnes Datenbyte
// und zwar beginnend mit dem hchstwertigen Bit
void dali_send_byte(uint8_t d)
{
    uint8_t i = 8;

    while(i--)
    {
        if (d & 0x80) dali_send_bit1();
        else dali_send_bit0();
        d <<= 1;
    }
}

// -----------------------------------------------------------------------------
// Sendet das Bit 0 -  __|--
void dali_send_bit0(void)
{
    DALI_OFF;
    DALI_WAIT;
    DALI_ON;
    DALI_WAIT;
}

// -----------------------------------------------------------------------------
// Sendet das Bit 1 -  --|__
void dali_send_bit1(void)
{
    DALI_ON;
    DALI_WAIT;
    DALI_OFF;
    DALI_WAIT;
}

// -----------------------------------------------------------------------------
// Liest die Antwort ein,
// die beginnt mit einem Startbit, 8 Datenbits sowie 2 Stopbits
uint8_t dali_read_byte(void)
{
    // Timer2 als Langzeittimer fr das gesamte Byte
    // Timer0 fr die Dauer der Halbbits
    // Warten auf ein steigende Flanke des Startbits

    // Timer 0 , Prescale 64, -> 256 * 64 /3.686400 = 4.44ms
    // Halber Takt = 416us -> Zhler bei 24 +- 10% (20-29)
    // ganzer Takt = 833us -> Zhler bei 50 +- 10% (45-56

    // der Timer1 setzt ein Flag, wenn die maximale Bytezeit um ist,
    // verhindert ein ewiges Warten. Begrenzung auf 20ms (22 * 883us)

    // Der Ruhelevel des DALI-Signalsist 1 !!!!

    uint8_t bitnr = 0;
    uint8_t timecnt = 2;    // Index 0 / 1 fr 16Bit-Timer-Wert
    uint8_t databyte = 0;
    uint8_t pinstate = (1<<DALI_INPUT);
    uint8_t t;
    timer_ovl = FALSE;

    // Den Timer1 nutzen und maximal 9.17
    // auf einen Flankenwechsel (= Beginn einer Antwort werten)
    TCNT1H = (uint8_t) (PRELOAD >> 8);
    TCNT1L = (uint8_t) (PRELOAD & 0xFF);

    TIMSK = (0<<TOIE0 | 1<<TOIE1);
    TIMER1_START;

    // warten, bis der DALI-Pin auf low gezogen wird
    // oder die Wartezeit auf eine Antwort abgelaufen ist.
    while ((DALI_PIN & (1<<DALI_INPUT)) == (1<<DALI_INPUT) && (!timer_ovl));

    TIMER1_STOP;

    // Zeiten protokollieren
    time[1] = TCNT1L;
    time[0] = TCNT1H;

    // Die Wartezeit ist berschritten, es ist keine Antwort eingegangen
    if (timer_ovl)
    {
        timer_ovl = NO_ANSWER;   // signalisiert die ausbleibende Antwort
        TIMSK = 0;
        return(0);
    }

    // Der Counter1 bentigt 0.142 sec bis zum berlauf (beim Start bei 0)
    // Die Bytedauer betrgt 9.17 ms
    // Daher wird ein Preload eingesetzt
    TCNT1H = (uint8_t)((PRELOAD) >> 8);
    TCNT1L = (uint8_t)((PRELOAD) & 0xFF);

    TIMER1_START;
    TCNT1H = (uint8_t)((65536 - 4213) >> 8);
    TCNT1L = (uint8_t)((65536 - 4213) & 0xFF);
    TCNT0 = 0;
    TIMER0_START;
    TIMSK = (1<<TOIE0 | 1<<TOIE1);

     // _delay_us(50);
     // warten, bis der DALI-Pin auf low gezogen wird
    while ((DALI_PIN & (1<<DALI_INPUT)) == 0);

    time[timecnt++] = TCNT0;
    TCNT0 = 0;

    // die zweite Hlfte des Startbits luft ab
    // auf die steigende Flanke im ersten Datenbit warten
    pinstate = (1<<DALI_INPUT);

    while(!(timer_ovl))
    {
        // _delay_us(50)
        // auf die nchste Flanke warten
        while (((DALI_PIN & 1<<DALI_INPUT) == pinstate) && (!(timer_ovl)));

        t = TCNT0;
        time[timecnt] = t;
        if (timecnt < (ARRAYSIZE - 1)) timecnt++;

        // Beenden, wenn sich en Timerovl ereignet hat
        // das aufrufende Programm muss das Flag ebenfalls testen
        if (timer_ovl) break;

        if (pinstate) pinstate = 0;
        else pinstate = (1<<DALI_INPUT);

        // Halber Takt, ignorieren
        if ((t >= HTMIN) && (t <= HTMAX))
        {
            continue;
        }

        // wenn ganze Bitzeit, dann den Wert einlesen / addieren
        // die Bits nach links schieben,
        // den Bitcounter incrementiern und
        // den Timer neu starten.
        else if ((t >= GTMIN) && (t <= GTMAX))
        {
            TCNT0 = 0;
            databyte <<= 1;
            // Das empfangene Bit addieren
            if (pinstate) databyte++;
            bitnr++;
        }
    }
    // Prfen, ob eventuell der Timer0 die Schleife beendet hat
    if (timer_ovl == TIMER_1_OVL)
    {
        // dann ist ein Byte eingegangen
        if (bitnr == 8) timer_ovl = DATA_RECEIVED;

    }
    // else erbrigt sich, dann hat der Timer0 ausgelst und timer_ovl
    // zeigt auf TIMER_0_OVL, was einen Fehler signalisiert

    TIMER0_STOP;
    TIMER1_STOP;
    TIMSK = 0;

    return(databyte);
}

// -----------------------------------------------------------------------------
// dient zur Messung der Bit-Taktdauer
// Ein overflow ist ein fataler Fehler
ISR(TIMER0_OVF_vect)
{
    timer_ovl = TIMER_0_OVL;
}

// -----------------------------------------------------------------------------
// Die Bytedauer ist berschritt
// kann bei ausbleibender Antwort (= 'n') der Fall sein
// dann wird aber oben auf 1 korrigiert
// oder aber das Timing stimmt nicht, dann bleibt die 2
ISR(TIMER1_OVF_vect)
{
    timer_ovl = TIMER_1_OVL;
}
// --- [ eof ] ----------------------------------------------------------------
