// --- [ rs232_read_c.c ] -------------------------------------------------------
//
// Erwartet im Hintergrund (interruptgesteuert) Eingaben ber die serielle Schnittstelle.
//
// Als erstes Byte wird das Startbyte 0xA5 erwartet, nur dann startet die
// Aufzeichnung der Daten
// Byte 1 und 2 sind Datenbytes
// Byte 3 ist entweder das Startbyte 0xA5 oder 0x5A
// 0x5A sagt an, dass eine Antwort auf die Sendung erwartet wird
// Die empfangenen Daten werden in einem lokalen Puffer abgelegt.
// Sobald eine vereinbarte Anzahl von Bytes ist, wird das lokale Flag rx_flag gesetzt.
//
// Von aussen kann mit RX_Received_Bytes() geprft werden, ob bzw. wieviele Bytes
// empfangen worden sind.
//
// Mit RX_Get_Data() werden dann die Daten abgeholt.
//
// Wenn der Puffer voll ist, dann werden keine Daten mehr bernommen.
// Das INT_Activity_Flag zeigt an, ob sich ein RX-Interrupt ereignet hat.

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>

#include "global.h"

#define RX_BUFFER_SIZE	4
#define TX_BUFFER_SIZE	4

#define BAUD  	9600
#define TEILER ((F_CPU /(BAUD * 16l)) -1)

// Register USCRB konfigurieren (ATTINY2313)
#define ENABLE_RX() 	(UCSRB = (1<<RXCIE | 0<<TXCIE | 0<<UDRIE | 1<<RXEN | 1<<TXEN))
#define DISABLE_RX()	(UCSRB = (0<<RXCIE | 0<<TXCIE | 0<<UDRIE | 0<<RXEN | 1<<TXEN))

// --- lokale Variablen -------------------------------------------------------

uint8_t rx_buffer[RX_BUFFER_SIZE];
volatile uint8_t rx_head;
volatile uint8_t rx_flag;

// ----------------------------------------------------------------------------
// initialisiert die RS232-Schnittstelle
void RS232_init(void)
{
    // ATTINY2313
    UBRRL = TEILER;
    UBRRH = 0;
    UCSRC = (0<<UMSEL) | (1<<UCSZ1) | (1<<UCSZ0);
    rx_flag = FALSE;
    rx_head = 0;
    ENABLE_RX();
}

// ATTINY2313
// ----------------------------------------------------------------------------
// Es werden 4 Byte empfangen, danach wird das Flag rx_flag gesetzt.

ISR (USART_RX_vect)
{
	uint8_t i;
	i = UDR;
											// den Index incrementieren,
											// sofern der Puffer noch nicht voll
	if (rx_head < RX_BUFFER_SIZE)
	{
		// Das erste Byte muss das Startbyte sein, wenn nicht,
		//  dann wird nichts gespeichert.
		if (rx_head == 0)
		{
			if (!(i == START_BYTE)) return;
		}

		rx_buffer[rx_head++] = i;			// die Daten speichern
	}

	if (rx_head >= RX_BUFFER_SIZE)			// wenn die Daten komplett
	{
        rx_flag = TRUE;						// das Flag setzen
		rx_head = 0;
	}
}

// ---------------------------------------------------------------------------
// liefert die empfangenen Bytes via Array zurck und lscht das Flag rx_flag
// damit diese Daten nicht noch ein weiteres Mal ausgelesen werden.
// Die Anzahl der Bytes steht in rx_head

void RX_Get_Data(uint8_t * data)
{
    uint8_t i = 0;
    do	{
        data[i] = rx_buffer[i];
        } while ((i++) < 4);

    rx_head = 0;
}

// ---------------------------------------------------------------------------
// Wenn weniger als 4 Byte empfangen wurden, dann den Counter
// zurckstellen
void RX_Clear_Counter(void)
{
    rx_head = 0;
}
// --- Standard-RS232-Routinen --- ATTINY2313 ---------------------------------
// ----------------------------------------------------------------------------
// ein einzelnes Zeichen auf COM ausgeben
void put_c(uint8_t Zeichen)
{
    while(!(UCSRA & (1<<UDRE)));
    UCSRA |= (1<<TXC);	// Flag TXC lschen fr die Auswertung vor dem Sleep
    UDR = Zeichen;
}

/*
// ----------------------------------------------------------------------------
// einen String auf COM ausgeben
void put_s(char *s)
{
while(*s)
	{
	put_c(*s);
	s++;
	}
}
*/
// --- [ eof ] ----------------------------------------------------------------

