/*
 * Demo for the Densitron PC-6749 LC Display (Pollin 120818)
 * LCD connected to an Arduino nano D2..D4, see ports.h for details
 */

#include <avr/io.h>
#include <util/delay.h>

#include "ports.h"
#include "seg16font.h"
#include "pc_6749.h"


/*
 * initilize uC IO resources
 */
void init_io(void)
{
        DDRB = 0b00100000; /* PB5=D13 is output */
        DDRC = 0;          /* inputs */
        DDRD = 0b00011100; /* PD2..4 are outputs */
        /* turn on pullups on all inputs */
        PORTB = 0xFF;
        PORTC = 0xFF;
        PORTD = 0xFF;
}

/*
 * Send a command to the LCD controller
 * ! we send the command code MSB first !
 *  so it matches the datasheet notation
 */
void lcd_cmd(uint8_t cmd) {
    CS_low();
    DATA(1); WR_pulse();
    DATA(0); WR_pulse();
    DATA(0); WR_pulse();
    uint8_t mask= 0x80;
    while (mask) {
        DATA(cmd&mask);
        WR_pulse();
        mask>>=1;
    }
    DATA(1); WR_pulse();
    CS_high();
}

/*
 * initialize and enable the LCD controller
 */
void lcd_init(void) {
    lcd_cmd(0b00000001); // system enable
    lcd_cmd(0b00101011); // 1/3 Bias, 4 commons
    lcd_cmd(0b00000011); // turn on LCD
}

/*
 * the LCD buffer
 */
#define LCD_BUF_SIZE 7
uint8_t lcd_buf[LCD_BUF_SIZE];

/*
 * clear LCD buffer
 */
void lcd_clear_buf(void)
{
    for (uint8_t i=0; i<LCD_BUF_SIZE; i++) {
        lcd_buf[i]= 0;
    }
}

/*
 * set segment bits from 'seg' for digit at position 'pos'
 */
void lcd_update_buf(uint8_t pos, uint16_t seg)
{
    uint8_t segno= 0;
    uint16_t mask= 0x8000;
    for ( ; segno<16; segno++, mask>>=1) {
        if (seg & mask) {
            uint8_t code= segmap[pos][segno];
            lcd_buf[code>>4] |= 1<<(code&7);
        }
    }
}

/*
 * flush LCD buffer to the LCD controller
 */
void lcd_flush_buf(void) {
    CS_low();
    DATA(1); WR_pulse();
    DATA(0); WR_pulse();
    DATA(1); WR_pulse();

    /* address is always 0 (start of buffer) */
    DATA(0); WR_pulse();
    DATA(0); WR_pulse();
    DATA(0); WR_pulse();
    DATA(0); WR_pulse();
    DATA(0); WR_pulse();
    DATA(0); WR_pulse();

    /* bulk write the complete buffer */
    for (uint8_t i=0; i<LCD_BUF_SIZE; i++) {
        uint8_t mask= 0x01;
        while (mask) {
            DATA(lcd_buf[i] & mask);
            WR_pulse();
            mask<<=1;
        }
    }
    DATA(1);
    CS_high();
}


int main(void)
{
    init_io();
    lcd_init();

    while (1) {
        /* cycle through the font, displaying 3 characters at a time */
        for (uint8_t i=0; seg16font[i+1].c; i++) {
            LED_on();
            lcd_clear_buf();
            lcd_update_buf(0, seg16font[i].s);
            lcd_update_buf(1, seg16font[i+1].s);
            lcd_update_buf(2, seg16font[i+2].s);
            lcd_flush_buf();
            LED_off();
            _delay_ms(500);
        }
    }

    return 0;
}

