/********************************************************************************
 * Time : C-module for calculation of local time and date from a unix timestamp *
 ********************************************************************************
 * File: time.c                                                                 *
 * Created : 2007-10-20, Patrick Wieland <patrick-wieland@t-online.de>          *
 * Modified:                                                                    *
 ********************************************************************************
 * Copyright (C) 2001-2003 by egnite Software GmbH. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY EGNITE SOFTWARE GMBH AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL EGNITE
 * SOFTWARE GMBH OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * For additional information see http://www.ethernut.de/
 *
 */

#include "time.h"

#define _DAY_SEC           (24UL * 60UL * 60UL) /* secs in a day */
#define _YEAR_SEC          (365L * _DAY_SEC)    /* secs in a year */
#define _FOUR_YEAR_SEC     (1461L * _DAY_SEC)   /* secs in a 4 year interval */
#define _DEC_SEC           315532800UL  /* secs in 1970-1979 */
#define _BASE_YEAR         70L  /* 1970 is the base year */
#define _BASE_DOW          4    /* 01-01-70 was a Thursday */
#define _LEAP_YEAR_ADJUST  17L  /* Leap years 1900 - 1970 */
#define _MAX_YEAR          138L /* 2038 is the max year */

#define MEZ 0
#define MESZ 1

#define LONG_MAX   2147483647L

unsigned char _daylight = 1;
long _timezone = -3600; // Germany:  UTC + 1 hour
long _dstbias = -3600;  // CET -> CEST

int _lpdays[] = {
     -1, 30, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365
};

int _days[] = {
   -1, 30, 58, 89, 119, 150, 180, 211, 242, 272, 303, 333, 364
};


int _isindst(tm * tb) { // FIXME: this function is only valid for UTC+1 with european CET/CEST
  if ( (tb->tm_mon>2) && (tb->tm_mon<9) ) return MESZ;
  if ( (tb->tm_mon<2) || (tb->tm_mon>9) ) return MEZ;

  if ( tb->tm_mon==2 ) {
    if (tb->tm_mday<25) return MEZ;
    if (tb->tm_wday==0) {
      if (tb->tm_hour>1) {
        return MESZ;
      } else {
        return MEZ;
      }
    }
    if (tb->tm_wday > tb->tm_mday-25) {
      return MEZ;
    } else {
      return MESZ;
    }
  }

  if ( tb->tm_mon==9 ) {
    if (tb->tm_mday<25) return MESZ;
    if (tb->tm_wday==0) {
      if (tb->tm_hour>1) {
        return MEZ;
      } else {
        return MESZ;
      }
    }
    if (tb->tm_wday > tb->tm_mday-25) {
      return MESZ;
    } else {
      return MEZ;
    }
  }

  return 0; // should never be reached
}


int gmtime_r(time_t * timer, tm * ptm) {

  time_t ctimer = *timer;     /* var to calculate with */
  unsigned char isleapyear = 0;      /* current year is leap year */
  unsigned long tmptimer;
  int *mdays;                 /* pointer to _numdayslp or _numdays */

  tmptimer = (unsigned long) (ctimer / _FOUR_YEAR_SEC);
  ctimer -= ((time_t) tmptimer * _FOUR_YEAR_SEC);

  /* Determine the correct year within the interval */
  tmptimer = (tmptimer * 4) + 70;     /* 1970, 1974, 1978,... */
  if (ctimer >= (time_t)_YEAR_SEC) {
    tmptimer++;             /* 1971, 1975, 1979,... */
    ctimer -= _YEAR_SEC;
    if (ctimer >= (time_t)_YEAR_SEC) {
      tmptimer++;         /* 1972, 1976, 1980,... (all leap years!) */
      ctimer -= _YEAR_SEC;
      /* A leap year has 366 days, so compare to _YEAR_SEC + _DAY_SEC */
      if (ctimer >= (time_t)(_YEAR_SEC + _DAY_SEC)) {
        tmptimer++;     /* 1973, 1977, 1981,... */
        ctimer -= (_YEAR_SEC + _DAY_SEC);
      } else 
        isleapyear = 1; /*If leap year, set the flag */
    }
  }


  /*
     tmptimer now has the value for tm_year. ctimer now holds the
     number of elapsed seconds since the beginning of that year.
  */
  ptm->tm_year = tmptimer;
 
  /*
     Calculate days since January 1st and store it to tm_yday.
     Leave ctimer with number of elapsed seconds in that day.
   */
  ptm->tm_yday = (int) (ctimer / _DAY_SEC);
  ctimer -= (time_t) (ptm->tm_yday) * _DAY_SEC;
 
  /*
     Determine months since January (Note, range is 0 - 11)
     and day of month (range: 1 - 31)
   */
  if (isleapyear)
    mdays = _lpdays;
  else
    mdays = _days;

  for (tmptimer = 1; mdays[tmptimer] < ptm->tm_yday; tmptimer++);
 
  ptm->tm_mon = --tmptimer;
 
  ptm->tm_mday = ptm->tm_yday - mdays[tmptimer];
 
  /* Calculate day of week. Sunday is 0 */
  ptm->tm_wday = ((int) (*timer / _DAY_SEC) + _BASE_DOW) % 7;
 
  /* Calculate the time of day from the remaining seconds */
  ptm->tm_hour = (int) (ctimer / 3600);
  ctimer -= (time_t) ptm->tm_hour * 3600L;
 
  ptm->tm_min = (int) (ctimer / 60);
  ptm->tm_sec = (int) (ctimer - (ptm->tm_min) * 60);
 
  ptm->tm_isdst = 0;

  return 0;
}


int localtime_r(time_t * timer, tm * ptm) {
  long ltime;
  if ((*timer > (time_t)(3 * _DAY_SEC)) && (*timer < (time_t)(LONG_MAX - 3 * _DAY_SEC))) {
    /*
     * The date does not fall within the first three, or last
     * three, representable days of the Epoch. Therefore, there
     * is no possibility of overflowing or underflowing the
     * time_t representation as we compensate for timezone and
     * Daylight Savings Time.
     */
 
    ltime = (long) *timer - _timezone;
    gmtime_r((time_t *) & ltime, ptm);
 
    /*
     * Check and adjust for Daylight Saving Time.
     */
    if (_daylight && _isindst(ptm)) {
      ltime -= _dstbias;
      gmtime_r((time_t *) & ltime, ptm);
      ptm->tm_isdst = 1;
    }
  } else {
    gmtime_r(timer, ptm);
 
    /*
     * The date falls with the first three, or last three days
     * of the Epoch. It is possible the time_t representation
     * would overflow or underflow while compensating for
     * timezone and Daylight Savings Time. Therefore, make the
     * timezone and Daylight Savings Time adjustments directly
     * in the tm structure. The beginning of the Epoch is
     * 00:00:00, 01-01-70 (UCT) and the last representable second
     * in the Epoch is 03:14:07, 01-19-2038 (UCT). This will be
     * used in the calculations below.
     *
     * First, adjust for the timezone.
     */
    if (_isindst(ptm))
      ltime = (long) ptm->tm_sec - (_timezone + _dstbias);
    else
      ltime = (long) ptm->tm_sec - _timezone;
      ptm->tm_sec = (int) (ltime % 60);
      if (ptm->tm_sec < 0) {
        ptm->tm_sec += 60;
        ltime -= 60;
      }
 
      ltime = (long) ptm->tm_min + ltime / 60;
      ptm->tm_min = (int) (ltime % 60);
      if (ptm->tm_min < 0) {
        ptm->tm_min += 60;
        ltime -= 60;
      }
 
      ltime = (long) ptm->tm_hour + ltime / 60;
      ptm->tm_hour = (int) (ltime % 24);
      if (ptm->tm_hour < 0) {
        ptm->tm_hour += 24;
        ltime -= 24;
      }
 
      ltime /= 24;
 
      if (ltime > 0L) {
        /*
         * There is no possibility of overflowing the tm_mday
         * and tm_yday fields since the date can be no later
         * than January 19.
         */
        ptm->tm_wday = (ptm->tm_wday + ltime) % 7;
        ptm->tm_mday += ltime;
        ptm->tm_yday += ltime;
      } else if (ltime < 0L) {
        /*
         * It is possible to underflow the tm_mday and tm_yday
         * fields. If this happens, then adjusted date must
         * lie in December 1969.
         */
        ptm->tm_wday = (ptm->tm_wday + 7 + ltime) % 7;
        if ((ptm->tm_mday += ltime) <= 0) {
          ptm->tm_mday += 31;
          ptm->tm_yday = 364;
          ptm->tm_mon = 11;
          ptm->tm_year--;
        } else {
          ptm->tm_yday += ltime;
        }
      }
    }
  return 0;
}




