// ***************************************************************
//  TransferPrinter FW Version:  1.0     date: 08.11.2007
//  -------------------------------------------------------------
//  File Name : SegmentDriver.c
//  -------------------------------------------------------------
//  Copyright (C) 2007 - All Rights Reserved
//	Author : Markus Stopper
// ***************************************************************
// Description: Shows a Character on a single 7 Segment Block
// ***************************************************************
// Remarks: Need Standard Data Types defined in Globals.h
// ***************************************************************

#include "SegmentDriver.h"


// ***************************************************************
// Character Set for 7Segment Driver resides in FLASH
// Remarks: '0' has an ASCII offset of 0x30 
//			'a' has an ASCII offset of 0x41
//			all other chars are not indexed
// ***************************************************************
const prog_uchar SegmentChars[] PROGMEM=
{
	0x40,0xF9,0x24,0x30,	// 0, 1, 2, 3
	0x19,0x12,0x02,0xf8,	// 4, 5, 6, 7
	0x00,0x18,0x08,0x03,	// 8, 9, A, b
	0x27,0x21,0x06,0x0e,	// c, d, E, F
	0x10,0x0b,0x47,0x2b,	// g, h, L, n 
	0x23,0x0c,0x2f,0x63,	// o, p, r, u
	0x11,0x37,0x3f,0xff,	// y, =, -, Blank
	0x46,0xf0,0xf7,0xfe,    // [, ], _, Overscore(`)
	0x5c,0x36,0x0f,0xfb		// over n (m), character not interpretable, t, i
};

// ***************************************************************
// Single segment sequence if asymmetrical stored in FLASH table
// ***************************************************************
const prog_uchar AssymetricBlocks[SEGMENTBLOCKSIZE] PROGMEM=
{
	SEGMENTBLOCK_LEDS,SEGMENTBLOCK_DIGIT3,SEGMENTBLOCK_DIGIT2,SEGMENTBLOCK_DIGIT1,SEGMENTBLOCK_DIGIT0
};

// ***************************************************************
// Heap Variables - 2 Bytes + 5 Bytes Displaymemory
// ***************************************************************
static BYTE scrollindex;
static BYTE segmentindex;
static BYTE segmentlocation[SEGMENTBLOCKSIZE];
static chardisplaybuffer displaybuffer;
static BYTE repeatcounter;

// ***************************************************************
// Internal only used function Prototypes
// ***************************************************************
void InternalRotateRightByteArray(char* bytearray);
void InternalRotateLeftByteArray(char* bytearray);
static int DisplayPutChar(char c, FILE* stream);

// ***************************************************************
// Standard IO Redirector
// ***************************************************************
// static FILE displayout = FDEV_SETUP_STREAM(DisplayPutChar,NULL,_FDEV_SETUP_WRITE);
// // Initializes Segments and Buffer
// // Standard Stream Redirector for Display
// static int DisplayPutChar(char c, FILE* stream)
// {
// 	static BYTE index=0;
// 	if(c=='\n')
// 	{
// 		index=0;
// 		displaybuffer.bufferlock=0;
// 	}
// 	else
// 	{
// 		if(!displaybuffer.bufferlock)
// 			displaybuffer.bufferlock=1;
// 		displaybuffer.buffer[index]=c;
// 	}
// 	return 0 ;
// }

// p_segmentlocation points to Buffer of Characters to display
void InitDisplay(void)
{
	segmentindex=0;
	memset(segmentlocation+1,' ',SEGMENTBLOCKSIZE-1);
	segmentlocation[0]=0;
}
//returns DisplayBuffer for manipulation purposes
void GetDisplayBuffer(chardisplaybuffer** p_displaybuffer)
{
	*p_displaybuffer= &displaybuffer;
}
// Draws single character to current (segmentindex) 7Segment Block
void SegmentShow(void)
{
	BYTE index = 0;
	BYTE block = 0;
	//read current Location of Digit
	block = pgm_read_byte(&AssymetricBlocks[segmentindex]);
	//fetch current Byte from SRAM
	index = segmentlocation[segmentindex];
	//Lookup in Table for 7 Segment Blocks
	if(segmentindex)
	{		
		//translate index
		if(index >= 0x30 && index <= 0x39) // 0-9
			index -= 0x30;
		else if(index >= 0x61 && index <= 0x68) //a-h
			index -= 0x61-10; //a-h offset in table
		else
		{
			switch(index)
			{
				case ' ':
					index = 27;
					break;
				case '=':
					index = 25;
					break;
				case '-':
					index = 26;
					break;
				case 'y':
					index = 24;
					break;
				case 'u':
					index = 23;
					break;
				case 'r':
					index = 22;
					break;
				case 'p':
					index = 21;
					break;
				case 'o':
					index = 20;
					break;
				case 'n':
					index = 19;
					break;
				case 'l':
					index = 18;
					break;
				case '[':
					index = 28;
					break;
				case ']':
					index = 29;
					break;
				case '_':
					index = 30;
					break;
				case '`':
					index = 31;
					break;
				case 'm':
					index = 32;
					break;
				case 't':
					index = 34;
					break;
				case 'i':
					index = 35;
					break;
				default:
					index = 33;
			}
		}
		index = pgm_read_byte(&SegmentChars[index]);
	}
	else
	{
		//otherwise index stays the same and it will be the led bar displayable
		index=~index;
		segmentindex=SEGMENTBLOCKSIZE;
	}	
	// Darken all 7 Segment Blocks 
	SEGMENTBLOCK_ACTOR |= (SEGMENTBLOCK_LEDS | SEGMENTBLOCK_DIGIT0 | SEGMENTBLOCK_DIGIT1| SEGMENTBLOCK_DIGIT2 | SEGMENTBLOCK_DIGIT3);
	// Setup new Displayable Value
	SEGMENTBLOCK_DRIVE&=SEGMENT_DRIVE_NODOT;	//Erase Port exept PIN which is not used
	SEGMENTBLOCK_DRIVE|=(index&(~SEGMENT_DRIVE_NODOT));					//Write to Segment a-f Port
	// Illuminate current Block
	SEGMENTBLOCK_ACTOR &= ~block;				//invert because PNP control of blocks
	segmentindex--;
}


// Rotates entire String one character to left or right or just displays it
// Input: p_string -> String in Flash with PROGMEM
//	      p_ppos   -> Current Position
//		  p_dir	   -> Direction (LEFT,RIGHT,NONE)
/*
void SegmentScrollStringFlash(PGM_P p_string,BYTE p_dir)
{
	static BYTE segmentposition;
	BYTE tempstringlen;
	char* tempstring;
	BYTE pos;
	tempstringlen=strlen_P(p_string);
	if(++scrollindex==tempstringlen)
		scrollindex=0;
	pos = scrollindex;	
	tempstring=(BYTE*)malloc(tempstringlen);
	strcpy_P(tempstring,p_string);
	while(pos)
	{
		switch(p_dir)
		{
		case LEFT:
			InternalRotateLeftByteArray(tempstring);
			break;
		case RIGHT:
			InternalRotateRightByteArray(tempstring);
			break;
		}
		pos--;
	}
	tempstring[4]=0; //WindowEnd
	memcpy(segmentlocation+1,tempstring,SEGMENTBLOCKSIZE-1);
	free(tempstring);
}
*/

// Rotates entire String one character to left or right or just displays it
// Input: p_string -> String in SRAM 
//		  p_dir	   -> Direction (LEFT,RIGHT,NONE)
void SegmentScrollString(void)
{
	BYTE tempstringlen;
	char* tempstring;
	BYTE pos;
	//check if already initialized
	if(displaybuffer.buffer==0)
		return;
	// Check RepeatCounter
	displaybuffer.bufferlock=1;					//lock buffer
	if(displaybuffer.scroll!=0)
	{
		repeatcounter++;
		if(repeatcounter==displaybuffer.scroll)
			return;
	}
	tempstringlen=strlen(displaybuffer.buffer);
	if(++scrollindex==tempstringlen)
		scrollindex=0;
	pos = scrollindex;	
	tempstring=(char*)malloc(tempstringlen);
	strcpy(tempstring,displaybuffer.buffer);
	displaybuffer.bufferlock=0;					//unlock buffer
	//Scroll here
	while(pos)
	{
		switch(displaybuffer.direction)
		{
		case LEFT:
			InternalRotateLeftByteArray(tempstring);
			break;
		case RIGHT:
			InternalRotateRightByteArray(tempstring);
			break;
		}
		pos--;
	}
	tempstring[4]=0; //WindowEnd
	memcpy(segmentlocation+1,tempstring,SEGMENTBLOCKSIZE-1);
	free(tempstring);
	
}
//Byte Array Rotating Function (Round Robin Principle)
void InternalRotateRightByteArray(char* bytearray)
{
	BYTE i;
	BYTE temp;
	BYTE arlen=strlen(bytearray);
	temp=bytearray[0];
	for(i=1;i<arlen;i++)
	{
		BYTE swp=bytearray[i];
		bytearray[i]=temp;
		temp=swp;
	}
	bytearray[0]=temp;
}

//Byte Array Rotating Function (Round Robin Principle)
void InternalRotateLeftByteArray(char* bytearray)
{
	BYTE i;
	BYTE temp;
	BYTE arlen=strlen(bytearray);
	temp=bytearray[arlen-1];
	for(i=arlen-2;i!=0;i--)
	{
		BYTE swp=bytearray[i];
		bytearray[i]=temp;
		temp=swp;
	}
	bytearray[arlen-1]=bytearray[0];
	bytearray[0]=temp;
}

//Sets the starting point to display string
void SetScrollIndex(BYTE p_scrollindex)
{
	scrollindex=p_scrollindex;
}
//BYte Value Translator for LEDS
void SetLEDS(BYTE p_LEDS)
{
	BYTE relocator=p_LEDS;
	BYTE destination=0;
	destination |= ((relocator&0b00000001)<<4); //LED 0
	destination |= ((relocator&0b00000010)<<4); //LED 1
	destination |= ((relocator&0b00000100));	//LED 2 stays the same
	destination |= ((relocator&0b00001000)<<3); //LED 3
	destination |= ((relocator&0b00010000)>>3); //LED 4
	destination |= ((relocator&0b00100000)>>5); //LED 5
	destination |= ((relocator&0b01000000)>>3); //LED 6
	segmentlocation[0]=destination;
}

//Display String out of Flash with variable Arguments
void DisplayStringFlash(PGM_P p_flashstring, ...)
{
	va_list ap;	
	BYTE len= strlen_P(p_flashstring);
	char* tempstring;
	len+=8;	//safty for word
	//Allocate new memory if memory already exist
	if(displaybuffer.buffer)
	{
		//buffer needs to be deleted and newly allocated
		while(displaybuffer.bufferlock){};
		free(displaybuffer.buffer);
		displaybuffer.buffer=(char*)malloc(len);
		tempstring=(char*)malloc(len);
	}
	else
	{
		//buffer does not exist an need to be allocated
		displaybuffer.buffer = (char*)malloc(len);		
		tempstring=(char*)malloc(len);
	}
	strcpy_P(tempstring,p_flashstring);
	va_start(ap,tempstring);
	while(displaybuffer.bufferlock){};
	vsprintf_P(displaybuffer.buffer,p_flashstring,ap);
	va_end(ap);
	free(tempstring);
	displaybuffer.direction=LEFT;
	displaybuffer.scroll=1;
	displaybuffer.startindex=0;
}
