/*! \file enc28j60.h \brief Microchip ENC28J60 Ethernet Interface Driver. */
//*****************************************************************************
//
// File Name	: 'enc28j60.h'
// Title		: Microchip ENC28J60 Ethernet Interface Driver
// Author		: Pascal Stang (c)2005
// Created		: 9/22/2005
// Revised		: 9/22/2005
// Version		: 0.1
// Target MCU	: Atmel AVR series
// Editor Tabs	: 4
//
///	\ingroup network
///	\defgroup enc28j60 Microchip ENC28J60 Ethernet Interface Driver (enc28j60.c)
///	\code #include "net/enc28j60.h" \endcode
///	\par Overview
///		This driver provides initialization and transmit/receive
///	functions for the Microchip ENC28J60 10Mb Ethernet Controller and PHY.
/// This chip is novel in that it is a full MAC+PHY interface all in a 28-pin
/// chip, using an SPI interface to the host processor.
///
//
//*****************************************************************************

#include "enc28j60.h"

#if USE_ENC28J60
unsigned char mymac[6];
unsigned char Enc28j60Bank;
unsigned int NextPacketPtr;

//----------------------------------------------------------------------------
//
unsigned char enc28j60ReadOp(unsigned char op, unsigned char address)
{
	unsigned char data;
   
	// assert CS
	ENC28J60_CONTROL_PORT &= ~(1<<ENC28J60_CONTROL_CS);
	// issue read command
	SPDR = op | (address & ADDR_MASK);
	while(!(SPSR & (1<<SPIF)));
	// read data
	SPDR = 0x00;
	while(!(SPSR & (1<<SPIF)));
	// do dummy read if needed
	if(address & 0x80)
	{
		SPDR = 0x00;
		while(!(SPSR & (1<<SPIF)));
	}
	data = SPDR;
	
	// release CS
	ENC28J60_CONTROL_PORT |= (1<<ENC28J60_CONTROL_CS);

	return data;
}

//----------------------------------------------------------------------------
//
void enc28j60WriteOp(unsigned char op, unsigned char address, unsigned char data)
{
	// assert CS
	ENC28J60_CONTROL_PORT &= ~(1<<ENC28J60_CONTROL_CS);
	// issue write command
	SPDR = op | (address & ADDR_MASK);
	while(!(SPSR & (1<<SPIF)));
	// write data
	SPDR = data;
	while(!(SPSR & (1<<SPIF)));

	// release CS
	ENC28J60_CONTROL_PORT |= (1<<ENC28J60_CONTROL_CS);
}

//----------------------------------------------------------------------------
//
void enc28j60ReadBuffer(unsigned int len, unsigned char* data)
{
	// assert CS
	ENC28J60_CONTROL_PORT &= ~(1<<ENC28J60_CONTROL_CS);
	// issue read command
	SPDR = ENC28J60_READ_BUF_MEM;
	while(!(SPSR & (1<<SPIF)));
	while(len--)
	{
		// read data
		SPDR = 0x00;
		while(!(SPSR & (1<<SPIF)));
		*data++ = SPDR;
	}	
	// release CS
	ENC28J60_CONTROL_PORT |= (1<<ENC28J60_CONTROL_CS);
}

//----------------------------------------------------------------------------
//
void enc28j60WriteBuffer(unsigned int len, unsigned char* data)
{
	// assert CS
	ENC28J60_CONTROL_PORT &= ~(1<<ENC28J60_CONTROL_CS);
	// issue write command
	SPDR = ENC28J60_WRITE_BUF_MEM;
	while(!(SPSR & (1<<SPIF)));
	while(len--)
	{
		// write data
		SPDR = *data++;
		while(!(SPSR & (1<<SPIF)));
	}	
	// release CS
	ENC28J60_CONTROL_PORT |= (1<<ENC28J60_CONTROL_CS);
}

//----------------------------------------------------------------------------
//
void enc28j60SetBank(unsigned char address)
{
	// set the bank (if needed)
	if((address & BANK_MASK) != Enc28j60Bank)
	{
		// set the bank
		enc28j60WriteOp(ENC28J60_BIT_FIELD_CLR, ECON1, (ECON1_BSEL1|ECON1_BSEL0));
		enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, ECON1, (address & BANK_MASK)>>5);
		Enc28j60Bank = (address & BANK_MASK);
	}
}

//----------------------------------------------------------------------------
//
unsigned char enc28j60Read(unsigned char address)
{
	// set the bank
	enc28j60SetBank(address);
	// do the read
	return enc28j60ReadOp(ENC28J60_READ_CTRL_REG, address);
}

//----------------------------------------------------------------------------
//
void enc28j60Write(unsigned char address, unsigned char data)
{
	// set the bank
	enc28j60SetBank(address);
	// do the write
	enc28j60WriteOp(ENC28J60_WRITE_CTRL_REG, address, data);
}

//----------------------------------------------------------------------------
//
unsigned int enc28j60PhyRead(unsigned char address)
{
	unsigned int data;

	// Set the right address and start the register read operation
	enc28j60Write(MIREGADR, address);
	enc28j60Write(MICMD, MICMD_MIIRD);

	// wait until the PHY read completes
	while(enc28j60Read(MISTAT) & MISTAT_BUSY);

	// quit reading
	enc28j60Write(MICMD, 0x00);
	
	// get data value
	data  = enc28j60Read(MIRDL);
	data |= enc28j60Read(MIRDH);
	// return the data
	return data;
}

//----------------------------------------------------------------------------
//
void enc28j60PhyWrite(unsigned char address, unsigned int data)
{
	// set the PHY register address
	enc28j60Write(MIREGADR, address);
	
	// write the PHY data
	enc28j60Write(MIWRL, data);	
	enc28j60Write(MIWRH, data>>8);

	// wait until the PHY write completes
	while(enc28j60Read(MISTAT) & MISTAT_BUSY);
}

//----------------------------------------------------------------------------
//
void enc28j60Init(void)
{
	// initialize I/O
	ENC28J60_CONTROL_DDR |= (1<<ENC28J60_CONTROL_CS);
	ENC28J60_CONTROL_PORT |= (1<<ENC28J60_CONTROL_CS);

	// setup SPI I/O pins
	ENC28J60_SPI_PORT |= (1<<ENC28J60_SPI_SCK);	// set SCK hi
	ENC28J60_SPI_DDR |= (1<<ENC28J60_SPI_SCK);	// set SCK as output
	ENC28J60_SPI_DDR &= ~(1<<ENC28J60_SPI_MISO);	// set MISO as input
	ENC28J60_SPI_DDR |= (1<<ENC28J60_SPI_MOSI);	// set MOSI as output
	ENC28J60_SPI_DDR |= (1<<ENC28J60_SPI_SS);		// SS must be output for Master mode to work
	// initialize SPI interface
	// master mode
	SPCR |= (1<<MSTR);

	// switch to f/4 2X = f/2 bitrate
	//SPCR|= (1<<SPR0);
	//SPCR |= (1<<SPR1);
	SPSR |= (1<<SPI2X);
	// enable SPI
	SPCR |= (1<<SPE);
	delay_us(500000);
	// perform system reset
	enc28j60WriteOp(ENC28J60_SOFT_RESET, 0, ENC28J60_SOFT_RESET);
	// check CLKRDY bit to see if reset is complete
	delay_us(100000);
	while(!(enc28j60Read(ESTAT) & ESTAT_CLKRDY));

	if(ENCTAKT==1)
	{
		enc28j60Write(ECOCON,0x02);
		delay_us(500000);
	}
	
	// do bank 0 stuff
	// initialize receive buffer
	// 16-bit transfers, must write low byte first
	// set receive buffer start address
	NextPacketPtr = RXSTART_INIT;
	enc28j60Write(ERXSTL, RXSTART_INIT&0xFF);
	enc28j60Write(ERXSTH, RXSTART_INIT>>8);
	// set receive pointer address
	enc28j60Write(ERXRDPTL, RXSTART_INIT&0xFF);
	enc28j60Write(ERXRDPTH, RXSTART_INIT>>8);
	// set receive buffer end
	// ERXND defaults to 0x1FFF (end of ram)
	enc28j60Write(ERXNDL, RXSTOP_INIT&0xFF);
	enc28j60Write(ERXNDH, RXSTOP_INIT>>8);
	// set transmit buffer start
	// ETXST defaults to 0x0000 (beginnging of ram)
	enc28j60Write(ETXSTL, TXSTART_INIT&0xFF);
	enc28j60Write(ETXSTH, TXSTART_INIT>>8);

	//Enter Bank 2 and configure the MAC
	enc28j60SetBank(MACON1);
	//Enable the receive portion of the MAC
	enc28j60Write(MACON1, MACON1_MARXEN|MACON1_TXPAUS|MACON1_RXPAUS);
	
	//Pad packets to 60 bytes, add CRC, and check Type/Length field.
	enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, MACON3, MACON3_PADCFG0|MACON3_TXCRCEN|MACON3_FRMLNEN);
	
	// Allow infinite deferals if the medium is continuously busy 
    // (do not time out a transmission if the half duplex medium is 
    // completely saturated with other people's data)
    //enc28j60Write(MACON4, MACON4_DEFER);
	
	//Late collisions occur beyond 63 bytes (default for 802.3 spec)
	enc28j60Write(MACON2, 63);

	// Set non-back-to-back inter-packet gap to 9.6us.  The back-to-back 
	// inter-packet gap (MABBIPG) is set by MACSetDuplex() which is called 
	// later.
	enc28j60Write(MAIPGL, 0x12);
	enc28j60Write(MAIPGH, 0x0C);
	// set inter-frame gap (back-to-back)
	//enc28j60Write(MABBIPG, 0x12);
	// Set the maximum packet size which the controller will accept
	enc28j60Write(MAMXFLL, MAX_FRAMELEN&0xFF);	
	enc28j60Write(MAMXFLH, MAX_FRAMELEN>>8);

    //Enter Bank 3 and initialize physical MAC address registers
	enc28j60Write(MAADR5, mymac[0]);
	enc28j60Write(MAADR4, mymac[1]);
	enc28j60Write(MAADR3, mymac[2]);
	enc28j60Write(MAADR2, mymac[3]);
	enc28j60Write(MAADR1, mymac[4]);
	enc28j60Write(MAADR0, mymac[5]);

	//Disable half duplex loopback in PHY.  Bank bits changed to Bank 2 as a 
	//side effect.
	enc28j60PhyWrite(PHCON2, PHCON2_HDLDIS);

	//config NIC LEDs
	enc28j60PhyWrite(0x14,0x047A);

	// switch to bank 0
	enc28j60SetBank(ECON1);
	// enable interrutps
	enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, EIE, EIE_INTIE|EIE_PKTIE);
	// enable packet reception
	enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, ECON1, ECON1_RXEN);
}

//----------------------------------------------------------------------------
//
void enc28j60PacketSend(unsigned int len, unsigned char* packet)
{
	unsigned int tmp;
	//still sending ?
	for(tmp=0; tmp<50; tmp++){
		if (!(enc28j60Read(ECON1) & (1<<3))){
			//send finished -> tx packet
			break;
		}
		//tx not finished yet, wait 10ms
		delay_us(10000);
	}

	if (tmp == 50)
	{
		enc28j60Init();
	}

	// Set the write pointer to start of transmit buffer area
	enc28j60Write(EWRPTL, TXSTART_INIT);
	enc28j60Write(EWRPTH, TXSTART_INIT>>8);
	// Set the TXND pointer to correspond to the packet size given
	enc28j60Write(ETXNDL, (TXSTART_INIT+len));
	enc28j60Write(ETXNDH, (TXSTART_INIT+len)>>8);

	// write per-packet control byte
	enc28j60WriteOp(ENC28J60_WRITE_BUF_MEM, 0, 0x00);

	// copy the packet into the transmit buffer
	enc28j60WriteBuffer(len, packet);
	
	// send the contents of the transmit buffer onto the network
	enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, ECON1, ECON1_TXRTS);
}

//----------------------------------------------------------------------------
//
unsigned int enc28j60PacketReceive(unsigned int maxlen, unsigned char* packet)
{
	unsigned int rxstat, rs, re;
	unsigned int len;

	// check if a packet has been received and buffered
//	if( !(enc28j60Read(EIR) & EIR_PKTIF) )
	if( !enc28j60Read(EPKTCNT) )
		return 0;
	
	// Make absolutely certain that any previous packet was discarded	
	//if( WasDiscarded == FALSE)
	//	MACDiscardRx();

	// Set the read pointer to the start of the received packet
	enc28j60Write(ERDPTL, (NextPacketPtr));
	enc28j60Write(ERDPTH, (NextPacketPtr)>>8);
	// read the next packet pointer
	NextPacketPtr  = enc28j60ReadOp(ENC28J60_READ_BUF_MEM, 0);
	NextPacketPtr |= enc28j60ReadOp(ENC28J60_READ_BUF_MEM, 0)<<8;
	// read the packet length
	len  = enc28j60ReadOp(ENC28J60_READ_BUF_MEM, 0);
	len |= enc28j60ReadOp(ENC28J60_READ_BUF_MEM, 0)<<8;
	// read the receive status
	rxstat  = enc28j60ReadOp(ENC28J60_READ_BUF_MEM, 0);
	rxstat |= enc28j60ReadOp(ENC28J60_READ_BUF_MEM, 0)<<8;

	// limit retrieve length
	// (we reduce the MAC-reported length by 4 to remove the CRC)
	len = MIN(len, maxlen);

	// copy the packet from the receive buffer
	enc28j60ReadBuffer(len, packet);
	
	// an implementation of Errata B1 Section #13
    rs = enc28j60Read(ERXSTH);
    rs <<= 8;
    rs |= enc28j60Read(ERXSTL);
    re = enc28j60Read(ERXNDH);
    re <<= 8;
    re |= enc28j60Read(ERXNDL);
    if (NextPacketPtr - 1 < rs || NextPacketPtr - 1 > re)
    {
        enc28j60Write(ERXRDPTL, (re));
        enc28j60Write(ERXRDPTH, (re)>>8);
    }
    else
    {
        enc28j60Write(ERXRDPTL, (NextPacketPtr-1));
        enc28j60Write(ERXRDPTH, (NextPacketPtr-1)>>8);
    }


	// Move the RX read pointer to the start of the next received packet
	// This frees the memory we just read out
	//enc28j60Write(ERXRDPTL, (NextPacketPtr));
	//enc28j60Write(ERXRDPTH, (NextPacketPtr)>>8);

	// decrement the packet counter indicate we are done with this packet
	enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, ECON2, ECON2_PKTDEC);
	
	// enable interrutps
	enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, EIE, EIE_INTIE|EIE_PKTIE);
	
	return len;
}

#endif //USE_ENC28J60


