#include <Arduino.h>
#include "itwControl.h"

// states of reception
#define S_WAIT      1     // wait for the >10ms low at the beginning of a transmission
#define S_STARTHIGH 2     // high pulse after start
#define S_STARTLOW  3     // low pulse after start
#define S_DATHIGH   4     // high pulse of data part
#define S_DATLOW    5     // low after data pulse
#define S_READY     6     // all data collected, information can be read

// for interal timing function micros is used
// to reduce the number of long calculations the
// result of micros is divided by T_US for further
// calculations
#define T_US      32

// long low before transmission starts
#define T_10MS    (8000/T_US)

// longer low after start pulse
#define T_2MS5    (2500/T_US)
#define T_3MS0    (3000/T_US)

// min/max high pulse
#define T_0MS15   (100/T_US)
#define T_0MS3    (350/T_US)

// min/max short low 
#define T_0MS25   (200/T_US)
#define T_0MS4    (450/T_US)

// min/max long low 
#define T_1MS1    (1100/T_US)
#define T_1MS6    (1600/T_US)



void itwControl::rxInterrupt () {
  uint16_t now = micros()/T_US;
  uint8_t i;
  uint16_t t;

  t = now - _last;
  _last = now;

  switch ( _state ) {
    case S_WAIT:
      if ( ( t > T_10MS ) && digitalRead ( _pin ) ) {
        _state = S_STARTHIGH;
      }
      break;
    case S_STARTHIGH:
      if ( (t > T_0MS15) && (t < T_0MS3) )
        _state = S_STARTLOW;
      else
        _state = S_WAIT;
      break;
    case S_STARTLOW:
      if ( (t > T_2MS5) && (t < T_3MS0) ) {
        for ( i = 0; i < ITW_NUM; i++ )
          _dat[i] = '0';
        _dat[ITW_NUM] = (char)0;
        _idx = 0;
        _state = S_DATHIGH;
      } else
        _state = S_WAIT;
      break;
    case S_DATHIGH: 
      if ( (t > T_0MS15) && (t < T_0MS3) ) {
        _state = S_DATLOW;
      } else
        _state = S_WAIT;
      break;
    case S_DATLOW:
      _dat[_idx]++;
      if ( (t > T_0MS25) && (t < T_0MS4) ) {
        _state = S_DATHIGH;
      } else if ( (t > T_1MS1) && ((t < T_1MS6) || (_idx == ITW_NUM-1)) ) {
        _idx++;
        if ( _idx > 32 /*ITW_NUM*/ ) {
          _state = S_READY;
        } else  
          _state = S_DATHIGH;
      } else
        _state = S_WAIT;  
      break;
    case S_READY:  // this state shows the availability of valid date
      break;       // will be left by release()
    default:
      _state = S_WAIT;      
  }
}

itwControl::itwControl( uint8_t pin ) {
  _pin = pin;
  _state = S_WAIT;
}

itwControl::~itwControl() {
}

// not implemented
uint8_t itwControl::getKey() {
  return _key;
}

// not implemented
uint16_t itwControl::getAddress() {
  return _addr;
}

// debug, length of last pulse
uint16_t itwControl::getLast() {
  return _last;
}

// debug, state of reception state machine
uint8_t itwControl::getState() {
  return _state;
}

// normally debug, but the only output in the moment
volatile char *itwControl::getData() {
  if ( !available() )
    return NULL;
  return _dat;
}

// valid reception available
uint8_t itwControl::available() {
  return (_state == S_READY);
}

// release data of last reception by restarting the state machine
void itwControl::release() {
  _state = S_WAIT;
}

// not working
const char* itwControl::getKeyText ( uint8_t key ) {
  switch ( key ) {
    case KEY_1ON: return "1 ON";
    case KEY_1OFF: return "1 OFF";
    case KEY_2ON: return "2 ON";
    case KEY_2OFF: return "2 OFF";
    default: return "unknown";
  }
}

