/*
    Dieses Programm sucht in einer .hex Datei nach den Hexwerten dieser Folge:

        cli
        out  SP_H, r?
        out  SREG, r?
        out  SP_L, r?

    und ersetzt sie durch Hexwerte dieser Folge:

        cli
        out  SP_H, r?
        out  SP_L, r?
        out  SREG, r?

    Das Vertauschen von SREG und SP_L (Stackpointer low) ist ein Workaround fuer
    einen Bug in VMLAB, bei dem der Interrupt um einen Befehl zu frueh freigegeben
    wird, was zu falschen Werten des Stackpointers fuehren kann.

    Damit VMLAB diese Korrektur bei jedem Build durchfuehrt, fuegt man sie in das
    Makefile Template 
        VMLAB\bin\templa33.mak 
    (bzw. in das unter Options->CodeMaker->GCC angegebene Template) ein:

        # Create final output files (.hex, .eep) from ELF output file.
        %.hex: %.elf
            @echo
            @echo $(MSG_FLASH) $@
            $(OBJCOPY) -O $(FORMAT) -R .eeprom $< $@
            swap_sreg_spl.exe $@ -quite

    Wer mag, kann die Option -quite weglassen, dann werden die ersetzten Bereiche 
    angezeigt.

    Erste Tests zeigen, dass es funktioniert, aber eine Garantie, dass es unter
    allen moeglichen Umstaenden fehlerfrei funktioniert, kann ich nicht geben.
    Rueckmeldungen, Kritik, Verbesserungsvorschlaege nehme ich gerne an
    (retep.enieh@gmx.de)

    Muenchen, 1.12.2007  Version 1.0   Peter

*/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>


#define STR_SIZE        256
#define LASTLINE        ":00000001FF"


// globale Variablen //
unsigned short  *gpMemFeld;
int             gQuite      = 0;



//===========================================================================//
int Ersetzen (unsigned short maxAdr)
{
    /*

    CLI             1011 0100 1111 1000
    OUT A,Rr        1011 1AAr rrrr AAAA


    -       f8 94   cli             1111 1000 1011 0100   mask=0xffff   cmp=0xf894
    SP_H    de bf   out 0x3e, r?    ???? 1110 1011 111?   mask=0x0ffe   cmp=0x0ebe
    SREG    0f be   out 0x3f, r?    ???? 1111 1011 111?   mask=0x0ffe   cmp=0x0fbe  <--+  tauschen
    SP_L    cd bf   out 0x3d, r?    ???? 1101 1011 111?   mask=0x0ffe   cmp=0x0dbe  <--+  

        +0   +1   +2   +3   +4   +5   +6   +7       x
        01F8 0194 01DE 01BF 010F 01BE 01CD 01BF     gMemFeld[adresse+x]
    &   ffff ffff ff0f fffe ff0f fffe ff0f fffe
    =   01F8 0194 010E 01BE 010F 01BE 010D 01BE

    */

    unsigned short  adresse;

    for (adresse = 0; adresse < maxAdr+1 -7; ++adresse)
    {
        if (
                   ((gpMemFeld[adresse+0] & 0xffff) == 0x01f8)      // cli
                && ((gpMemFeld[adresse+1] & 0xffff) == 0x0194)      
                && ((gpMemFeld[adresse+2] & 0xff0f) == 0x010e)      // out 0x3e     SP_H
                && ((gpMemFeld[adresse+3] & 0xfffe) == 0x01be)
                && ((gpMemFeld[adresse+4] & 0xff0f) == 0x010f)      // out 0x3f     SREG
                && ((gpMemFeld[adresse+5] & 0xfffe) == 0x01be)
                && ((gpMemFeld[adresse+6] & 0xff0f) == 0x010d)      // out 0x3d     SP_L
                && ((gpMemFeld[adresse+7] & 0xfffe) == 0x01be)
           )
        {
            // Folge gefunden, teilweise ersetzen //
            gpMemFeld[adresse+4] = 0x01CD;                          // out 0x3d     SP_L
            gpMemFeld[adresse+5] = 0x01BF;  
            gpMemFeld[adresse+6] = 0x010F;                          // out 0x3f     SREG
            gpMemFeld[adresse+7] = 0x01BE;
            if (!gQuite) printf ("Folge ersetzt ab Adresse 0x%04X\n", adresse);
        }
    }

    return 0;
}



//===========================================================================//
unsigned char getAnzahlBytes (char zeile[], unsigned char *chksum)
{
    char tmpBuf[STR_SIZE];
    unsigned char result;

    memset     (tmpBuf, 0, sizeof(tmpBuf));
    strncpy    (tmpBuf, &zeile[1], 2);                  // "10"
    result   = (unsigned char) strtol (tmpBuf, 0, 16);  // 0x10
    *chksum += result;
    return result;
}


//===========================================================================//
unsigned int getStartAdresse (char zeile[], unsigned char *chksum)
{
    char tmpBuf[STR_SIZE];
    unsigned int result;

    memset     (tmpBuf, 0, sizeof(tmpBuf));
    strncpy    (tmpBuf, &zeile[3], 4);                  // "1234"
    result   = (unsigned int)  strtol (tmpBuf, 0, 16);  // 0x1234
    *chksum += (unsigned char) result;
    *chksum += (unsigned char) (result/256);
    return result;
}


//===========================================================================//
unsigned char getRecordType (char zeile[], unsigned char *chksum)
{
    char tmpBuf[STR_SIZE];
    unsigned char result;

    memset     (tmpBuf, 0, sizeof(tmpBuf));
    strncpy    (tmpBuf, &zeile[7], 2);                  // "00"
    result   = (unsigned char) strtol (tmpBuf, 0, 16);  // 0x00
    *chksum += result;
    return result;
}


//===========================================================================//
unsigned char getDataByte (char *cp, unsigned char *chksum)
{
    char tmpBuf[STR_SIZE];
    unsigned char result;

    memset     (tmpBuf, 0, sizeof(tmpBuf));
    strncpy    (tmpBuf, cp, 2);                         // "ab"
    result   = (unsigned char) strtol (tmpBuf, 0, 16);  // 0xab
    *chksum += result;
    return result;
}


//===========================================================================//
int isCheckSummeOk (char *cp, unsigned char *chksum)
{
    char tmpBuf[STR_SIZE];
    unsigned char result;

    memset     (tmpBuf, 0, sizeof(tmpBuf));
    strncpy    (tmpBuf, cp, 2);                         // "45"
    result   = (unsigned char) strtol (tmpBuf, 0, 16);  // 0x45
    *chksum += result;
    return *chksum==0 ? 1 : 0;
}


//===========================================================================//
int isEndLine (char zeile[])
{
    return (!strncmp (zeile, LASTLINE, strlen(LASTLINE)));
}


//===========================================================================//
void createZeile (char zeile[], unsigned char anzahlBytes, 
                  unsigned short startAdresse, unsigned short *us)
{
    char            tmpBuf[STR_SIZE];
    unsigned char   n, c;
    unsigned char   chksum;

    chksum =   anzahlBytes
             + (unsigned char)  startAdresse 
             + (unsigned char) (startAdresse/256);

    sprintf (zeile, ":%02X%04X00", anzahlBytes, startAdresse);

    for (n=0; n < anzahlBytes; ++n) {
        c =  (unsigned char)*us;
        sprintf (tmpBuf, "%02X", c);
        strcat (zeile, tmpBuf);
        chksum += c;
        ++us;
    }
    sprintf (tmpBuf, "%02X\n", 256-chksum);
    strcat (zeile, tmpBuf);
}


//===========================================================================//
int ReadInputFile (char dateiname[], char mode)
{
    FILE            *datei;
    char            zeile[STR_SIZE];
    char            *cp;
    unsigned int    n;
    unsigned int    anzahlBytes;
    unsigned char   recordType;
    unsigned char   dataByte;
    unsigned char   checkSumme;
    unsigned short  startAdresse;
    unsigned short  adresse;
    unsigned short  maxAdresse      = 0;
    unsigned int    lineCount       = 0;


    // Eingabedatei oeffnen //
    if ( (datei = fopen(dateiname, "rt")) == NULL ) {
        fprintf (stderr, "Error: can not open input file %s\n", dateiname);
        return (1);
    }


    // ganze Eingabedatei zeilenweise lesen //
    while (!feof(datei))
    {
        ++lineCount;

        // eine Zeile lesen //
        if ( fgets(zeile, STR_SIZE-1, datei) == NULL ) {
            fprintf (stderr, "Error: can not read line %d from input file %s\n", lineCount, dateiname);
            return (1);
        }

        // zeile[] = 
        //
        // 0 1  3    7  9
        // : 10 0840 00 FA89F80FF053FD87F7FF02C02FE72D87 D5  (original ohne Leerzeichen, alles hex)
        //   |  |    |  |                                |
        //   |  |    |  |                                +- Checksumme
        //   |  |    |  +- Datenbytes (2 Zeichen fuer 1 Byte)
        //   |  |    +- Recordtype
        //   |  +- Startadresse (Byteadresse) fuer diese Zeile
        //   +- Anzahl der Datenbytes in dieser Zeile


        checkSumme    = 0;
        anzahlBytes   = getAnzahlBytes  (zeile, &checkSumme);
        startAdresse  = getStartAdresse (zeile, &checkSumme);
        recordType    = getRecordType   (zeile, &checkSumme);

        // letzte Zeile ? //
        if ((anzahlBytes == 0) && (isEndLine (zeile)))
            break;          // fertig, exit while 

        // groesste Adresse ermitteln //
        if (maxAdresse < (startAdresse + anzahlBytes))
            maxAdresse = startAdresse + anzahlBytes;

        if (mode == 0)      
            continue;       // mode=0 : nur maximale Adresse ermitteln


        // unterstuetzter Recordtype ? //
        if (recordType != 0) {
            fprintf (stderr, "Error: unsupported record type %d in line %d in input file %s\n", recordType, lineCount, dateiname);
            return (1);
        }


        // Datenbytes in das Memoryfeld schreiben //
        cp = &zeile[9];
        adresse = startAdresse;
        for (n = 0; n < anzahlBytes; ++n) {
            dataByte = getDataByte (cp, &checkSumme);
            cp += 2;
            if (gpMemFeld[adresse]/256 != 0) {
                fprintf (stderr, "Error: double entry at address 0x%04X, line %d in input file %s\n", adresse, lineCount, dateiname);
                return (1);
            }
            gpMemFeld[adresse] = 256 + dataByte;            // hi= 0x01, lo= dataByte
            ++adresse;
        }// for n


        // Zeilenchecksumme ok ? //
        if (!isCheckSummeOk (cp, &checkSumme)) {
            fprintf (stderr, "Error: wrong checksum %u in line %d in input file %s\n", checkSumme, lineCount, dateiname);
            return (1);
        }

    }// while not eof


    // Eingabedatei schliessen //
    fclose (datei);

    if (mode == 0)
        return (- (int)maxAdresse);
    else
        return 0;
}



//===========================================================================//
int WriteOutputFile (char dateiname[], unsigned short maxAdr)
{
    FILE            *datei;
    char            zeile[STR_SIZE];
    unsigned short  startAdresse;
    unsigned short  adresse;
    unsigned char   k               = 0;
    unsigned int    lineCount       = 0;


    // Ausgabedatei oeffnen //
    if ( (datei = fopen (dateiname, "wt")) == NULL )
    {
        fprintf(stderr, "Error: can not open output file %s\n", dateiname);
        return (1);
    }


    // alle belegten Eintraege des gesamten 
    // Memoryfeldes zeilenweise in Datei schreiben
    for (adresse = 0; adresse < maxAdr+1; ++adresse)
    {
        // neue Zeile beginnen //
        if (k == 0)
            startAdresse = adresse;


        // belegter Eintrag ? //
        if (gpMemFeld[adresse]/256 != 0) 
        {
            ++k;
            if (k == 16) {
                createZeile (zeile, k, startAdresse, &(gpMemFeld[startAdresse]));
                ++lineCount;
                if (0 > fprintf (datei, "%s", zeile)) {
                    fprintf(stderr, "Error: can not write line %d to output file %s\n", lineCount, dateiname);
                    return (1);
                }
                k = 0;
            }
        }
        else 
        {
            // unbelegter Eintrag, Zeile fertigstellen und schreiben //
            if (k > 0) {
                createZeile (zeile, k, startAdresse, &(gpMemFeld[startAdresse]));
                ++lineCount;
                if (0 > fprintf (datei, "%s", zeile)) {
                    fprintf(stderr, "Error: can not write line %d to output file %s\n", lineCount, dateiname);
                    return (1);
                }
            }
            k = 0;
        }

    }// for adresse


    // Schluss-Satz schreiben //
    fprintf (datei, "%s\n", LASTLINE);

    // Klappe zu //
    fclose (datei);

    return 0;
}



//===========================================================================//
//===========================================================================//
int main (int argc, char* argv[])
{
    int             r;
    unsigned short  maxAdr;


    // Kommanozeilen Parameter //
    if (argc < 2) {
        fprintf (stderr, "Usage: %s  filename.hex [-quite]\n", argv[0]);
        return 1;
    }
    if (argc == 3)
        gQuite = strcmp (argv[2], "-quite") ? 0 : 1;


    // Memoryfeld anlegen //
    r = - ReadInputFile (argv[1], 0);
    if (r <= 0)
        return 1;
    maxAdr = r;

    gpMemFeld = malloc ((1+maxAdr)*sizeof(*gpMemFeld));
    if (gpMemFeld == 0) {
        fprintf(stderr, "Error: insufficient memory for %u values\n", maxAdr+1);
        return (1);
    }

    // Memoryfeld clear //
    memset (gpMemFeld, 0, (1+maxAdr)*sizeof(*gpMemFeld));
    

    // Datei lesen //
    r = ReadInputFile (argv[1], 1);
    if (r) {
        free (gpMemFeld);
        return (r);
    }


    // Teile ersetzen //
    Ersetzen (maxAdr);


    // Datei schreiben //
    r = WriteOutputFile (argv[1], maxAdr);
    if (r) {
        free (gpMemFeld);
        return (r);
    }


    free (gpMemFeld);
    return 0;
}
