/*

DMX512 Receiver controls 20 open collector switches, used to contol LEDs, relays etc.

uses ATtiny2313 @8MHz, 3x 74HC595 and 3x ULN2803

Martiniman@gmx.li 02/2008

*/

#define F_CPU 8000000

#define spi_port PORTD		// Soft SPI
#define spi_ddr  DDRD
#define sclock   PD5
#define load     PD4
#define sdata    PD1

#define dmx_size 20			// number of used DMX channels

#define sr_size ((dmx_size+7)/8)			// number of shift registers, rounded up

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <util/delay.h>

#define BAUD 250000L		// DMX512          
 
#define UBRR_VAL  ((F_CPU+BAUD*8)/(BAUD*16)-1)  
#define BAUD_REAL (F_CPU/(16*(UBRR_VAL+1)))     
#define BAUD_ERROR ((BAUD_REAL*1000)/BAUD-1000) 
 
#if ((BAUD_ERROR>10) || (BAUD_ERROR<-10))
  #error Systematischer Fehler der Baudrate grsser 1% und damit zu hoch! 
#endif
 
// global variables

uint8_t dmx_array[dmx_size];		// DMX data
volatile uint8_t flag;				// new data available
uint8_t relais_array[sr_size];		// relay data
	
// write plain data to shift registers, Soft SPI

void write_relais(uint8_t *data) {
	int8_t i, j;
	uint8_t tmp;

	spi_port &= ~(1<<load);
	for (i=sr_size-1; i>=0; i--) {
		tmp=data[i];
		for (j=0; j<8; j++) {
			if (tmp & 0x80)				// copy MSB to SDATA
				spi_port |= (1<<sdata);
			else
				spi_port &= ~(1<<sdata);
			spi_port |= (1<<sclock);	// shift pulse
			spi_port &= ~(1<<sclock);
			tmp <<=1;					// get next bit
		}
	}
	spi_port |= (1<<load);				// load pulse
}

// convert DMX data to plain data for shift registers

void convert_data(uint8_t *data) {

	uint8_t i, j, tmp;
	uint8_t *ptr = dmx_array;

	for (i=0; i<sr_size; i++) {
		tmp=0;
		for (j=0; j<8; j++) {
			tmp >>=1;
			if (*ptr>127) tmp |= 0x80;
			ptr++;
		}
		data[i]=tmp;
	}
}

// main

int main(void) {

// init

	PORTB = 0xFF;		// all pull-ups ON
	DDRD  = 0x3F;		// PD0..5 outputs
	PORTD = 0xC0;		// pull-ups ON for PD6 and PD7

// UART config
 
    UBRRH = UBRR_VAL >> 8;
    UBRRL = UBRR_VAL & 0xFF;
    UCSRB = (1<<RXCIE) | (1<<RXEN);

// Relay reset

	write_relais(relais_array);

	sei();	 

// main loop

	while (1) {
		if (!flag) {
			set_sleep_mode(SLEEP_MODE_IDLE);
			sleep_mode();
		}
		else {
			flag=0;
			convert_data(relais_array);
			write_relais(relais_array);
		}
	}
}

// UART RX complete interrupt

ISR(USART_RX_vect) {
    static uint16_t dmx_channel, dmx_base;
    uint8_t status, data;
	int16_t index;
 
    // read UART data, also clears interrupt flag            
    status = UCSRA;
	data = UDR;

	if (status & (1<<FE)) {					// frame error
		if (data==0) {						// break -> DMX Reset
			dmx_channel=0;
			dmx_base = (~PINB & 0xFF)+1;	// read dip switches which define DMX base address
			if (~PIND & (1<<PD6)) dmx_base +=256;
			flag=1;							// trigger update
		}
		else								// rx error
			dmx_channel++;
	}
	else {
		index = dmx_channel-dmx_base;
		if (index>=0 && index<dmx_size)
			dmx_array[index]=data;
		dmx_channel++;
	}
}
