library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;

entity TEM_uni_5_adc_stream is Port(
	-- Trenz Board
	clk: in std_logic;
	sysled: out std_logic;
	-- UM232H
	D: out std_logic_vector(7 downto 0); --inout
	RXF: in std_logic;
	TXE: in std_logic;
	RD: out std_logic;
	WR: out std_logic;
	SIWU: out std_logic;
	CLKOUT: in std_logic; --60MHz FTDI
	OE: out std_logic;
	PWRSAV: out std_logic;
	--UM232H_RX: in std_logic;
	--UM232H_TX: out std_logic;
	-- Board LEDs
	LED_r: out std_logic;
	LED_o: out std_logic;
	-- FT232RL UART
	FT232RL_RX: in std_logic;
	FT232RL_TX: out std_logic;
	-- ADCs
	AD7356_12_nCS: out std_logic;
	AD7356_12_SCK: out std_logic;
	AD7356_12_SDO_A: in std_logic;
	AD7356_12_SDO_B: in std_logic;
	AD7356_34_nCS: out std_logic;
	AD7356_34_SCK: out std_logic;
	AD7356_34_SDO_A: in std_logic;
	AD7356_34_SDO_B: in std_logic;
	AD7356_56_nCS: out std_logic;
	AD7356_56_SCK: out std_logic;
	AD7356_56_SDO_A: in std_logic;
	AD7356_56_SDO_B: in std_logic);
end TEM_uni_5_adc_stream;

architecture Behavioral of TEM_uni_5_adc_stream is

component AD7356 is Port(
	clk80 : in std_logic;
	data_ready : out std_logic;
	data_a : out std_logic_vector(11 downto 0);
	data_b : out std_logic_vector(11 downto 0);
	AD7356_nCS : out std_logic;
	AD7356_SCK : out std_logic;
	AD7356_SDO_A : in std_logic;
	AD7356_SDO_B : in std_logic);
end component;

component ADUM_AD7356 is Port(
	clk80 : in std_logic;
	data_ready : out std_logic;
	data_a : out std_logic_vector(11 downto 0);
	data_b : out std_logic_vector(11 downto 0);
	AD7356_nCS : out std_logic;
	AD7356_SCK : out std_logic;
	AD7356_SDO_A : in std_logic;
	AD7356_SDO_B : in std_logic);
end component;

component clk_wiz_0 is Port(
	clk80: out std_logic;
	clkin: in std_logic);
end component;

component workshop_uart is
	generic(
	clk_freq: integer;
	baudrate: integer);
	Port (
	clk : in std_logic;
	start_snd: in std_logic;
	start_rcv: out std_logic;
	byte_snd : in std_logic_vector(7 downto 0);
	byte_rcv : out std_logic_vector(7 downto 0);
	tx : out std_logic;
	rx : in std_logic;
	ready : out std_logic);
end component;

component fifo_generator_0 IS PORT(
    wr_clk : IN STD_LOGIC;
    rd_clk : IN STD_LOGIC;
    din : IN STD_LOGIC_VECTOR(7 DOWNTO 0);
    wr_en : IN STD_LOGIC;
    rd_en : IN STD_LOGIC;
    dout : OUT STD_LOGIC_VECTOR(7 DOWNTO 0);
    full : OUT STD_LOGIC;
    empty : OUT STD_LOGIC);
end component;

signal counter: unsigned(26 downto 0):=(others => '0');
--------------------------------------- PLL ----------------------
signal clk80: std_logic:='0';
--------------------------------------- UART ---------------------
signal FT232RL_start_snd: std_logic:='0';
signal FT232RL_start_rcv: std_logic:='0';
signal FT232RL_byte_snd: std_logic_vector(7 downto 0):="00000000";
signal FT232RL_byte_rcv: std_logic_vector(7 downto 0):="00000000";
signal FT232RL_ready: std_logic:='0';
signal UM232H_start_snd: std_logic:='0';
signal UM232H_start_rcv: std_logic:='0';
signal UM232H_byte_snd: std_logic_vector(7 downto 0):="00000000";
signal UM232H_byte_rcv: std_logic_vector(7 downto 0):="00000000";
signal UM232H_ready: std_logic:='0';
-------------------------------------- AD7356_x ------------------
signal AD7356_12_data_ready: std_logic:='0';
signal AD7356_1_data, AD7356_2_data: std_logic_vector(11 downto 0):=(others => '0');
signal AD7356_34_data_ready: std_logic:='0';
signal AD7356_3_data, AD7356_4_data: std_logic_vector(11 downto 0):=(others => '0');
signal AD7356_56_data_ready: std_logic:='0';
signal AD7356_5_data, AD7356_6_data: std_logic_vector(11 downto 0):=(others => '0');
-------------------------------------- ADC Select & DATA ----------
signal ADC_select: std_logic_vector(2 downto 0):="110";
signal ADC_data: std_logic_vector(11 downto 0):=(others => '0');
signal ADC_data_ready: std_logic:='0';
-------------------------------------- ADC Samples & FSM FT232RL ----------
constant samplenummer: integer:= 18; --18 = 256kBytes
type   ram_xk_12Bit is array (0 to 2**samplenummer-1) of std_logic_vector(11 downto 0);
signal SampleRAM: ram_xk_12Bit;
signal RW_Adress_max: integer range 0 to 2**samplenummer-1:=0;
signal RW_Adress_max_5bits: integer range 0 to 31:=0;
signal RW_Adress: integer range 0 to 2**samplenummer-1:=0;
type fsm_type is(s_record, s_wait, s_send_h, s_send_l, s_wait_send);
signal fsm: fsm_type:=s_wait;
signal ADC_send_buffer: std_logic_vector(11 downto 0):=(others => '0');
--------------------- FIFO & FT232H-------------------------------------------------
type fsm_fifo_type is(fifo_write_high, fifo_write_low, fifo_wait);
signal fsm_fifo: fsm_fifo_type:=fifo_wait;
signal fifo_byte_h, fifo_byte_l: std_logic_vector(7 downto 0):=(others => '0');
signal fifo_full, fifo_empty, fifo_empty_buff, fifo_rd, fifo_rd_buff, fifo_wr, ft232h_fail, txe_buff: std_logic:='0';
signal fifo_din, fifo_dout: std_logic_vector(7 downto 0):=(others => '0');
signal blinkcounter_60: unsigned(25 downto 0):=(others => '0');
signal fifo_counter: std_logic_vector(11 downto 0):=(others => '0');
signal fifo_counter_5MHz: unsigned(3 downto 0):=(others => '0');
signal fifo_counter_ready: std_logic:='0';

begin

pll0: clk_wiz_0 port map(
	clk80 => clk80,
	clkin => clk);
	
fifo0: fifo_generator_0 PORT MAP(
    wr_clk => CLK80, --80MHz
    rd_clk => CLKOUT,--60MHz
    din => fifo_din,
    wr_en => fifo_wr,
    rd_en => fifo_rd,
    dout => D,
    full => fifo_full,
    empty => fifo_empty);

FT232RL_uart: workshop_uart 
	generic map(
	clk_freq => 80000000,
	baudrate => 921600)
	port map(
	clk => clk80,
	start_snd => FT232RL_start_snd,
	start_rcv => FT232RL_start_rcv,
	byte_snd => FT232RL_byte_snd,
	byte_rcv => FT232RL_byte_rcv,
	tx => FT232RL_TX,
	rx => FT232RL_RX,
	ready => FT232RL_ready);
	
AD7356_12: ADUM_AD7356 port map(
	clk80 => clk80,
	data_ready => AD7356_12_data_ready,
	data_a => AD7356_1_data,
	data_b => AD7356_2_data,
	AD7356_nCS => AD7356_12_nCS,
	AD7356_SCK => AD7356_12_SCK,
	AD7356_SDO_A => AD7356_12_SDO_A,
	AD7356_SDO_B => AD7356_12_SDO_B);
	
AD7356_34: AD7356 port map(
	clk80 => clk80,
	data_ready => AD7356_34_data_ready,
	data_a => AD7356_3_data,
	data_b => AD7356_4_data,
	AD7356_nCS => AD7356_34_nCS,
	AD7356_SCK => AD7356_34_SCK,
	AD7356_SDO_A => AD7356_34_SDO_A,
	AD7356_SDO_B => AD7356_34_SDO_B);
	
AD7356_56: AD7356 port map(
	clk80 => clk80,
	data_ready => AD7356_56_data_ready,
	data_a => AD7356_5_data,
	data_b => AD7356_6_data,
	AD7356_nCS => AD7356_56_nCS,
	AD7356_SCK => AD7356_56_SCK,
	AD7356_SDO_A => AD7356_56_SDO_A,
	AD7356_SDO_B => AD7356_56_SDO_B);

--------- select ADC ---------------------------------
RW_Adress_max_5bits <= to_integer(unsigned(FT232RL_byte_rcv(7 downto 3)));
process begin
	wait until rising_edge(clk80);
	if FT232RL_start_rcv = '1' then
		if RW_Adress_max_5bits > 18 then
			RW_Adress_max <= 2**18-1;
		else
			RW_Adress_max <= 2**RW_Adress_max_5bits-1;
		end if;
		ADC_select <= FT232RL_byte_rcv(2 downto 0);
	end if;
	
	case ADC_select is
		when "001"  => ADC_data <= AD7356_1_data; ADC_data_ready <= AD7356_12_data_ready;
		when "010"  => ADC_data <= AD7356_2_data; ADC_data_ready <= AD7356_12_data_ready;
		when "011"  => ADC_data <= AD7356_3_data; ADC_data_ready <= AD7356_34_data_ready;
		when "100"  => ADC_data <= AD7356_4_data; ADC_data_ready <= AD7356_34_data_ready;
		when "101"  => ADC_data <= AD7356_5_data; ADC_data_ready <= AD7356_56_data_ready;
		when "110"  => ADC_data <= AD7356_6_data; ADC_data_ready <= AD7356_56_data_ready;
		when others => ADC_data <= fifo_counter;  ADC_data_ready <= fifo_counter_ready;
	end case;
	
	--ADC_data <= fifo_counter;  ADC_data_ready <= fifo_counter_ready;
end process;

--------- write to FIFO ------------------------------------
process begin
	wait until rising_edge(clk80);
	fifo_wr <= '0';
	if	  fsm_fifo = fifo_wait then
		if ADC_data_ready = '1' then
			fsm_fifo <= fifo_write_high;
		end if;
	elsif fsm_fifo = fifo_write_high then
		if fifo_full = '0' then
			fifo_wr <= '1';
			fifo_din <= "00" & ADC_data(11 downto 7) & '1';
			fsm_fifo <= fifo_write_low;
		end if;
	elsif fsm_fifo = fifo_write_low then
		if fifo_full = '0' then
			fifo_wr <= '1';
			fifo_din <= ADC_data(6 downto 0) & '0';
			fsm_fifo <= fifo_wait;
		end if;
	end if;
end process;

--------- sägezahn generator für "ADC7" ------------------------------------
process begin
	wait until rising_edge(clk80);
	fifo_counter_ready <= '0';
	fifo_counter_5MHz <= fifo_counter_5MHz +1;
	if fifo_counter_5MHz = 0 then
		fifo_counter <= std_logic_vector(unsigned(fifo_counter) +1);
		fifo_counter_ready <= '1';
	end if;
end process;

----------------- Store & Send Samples UART FT232RL ------------------------------
process begin
	wait until rising_edge(clk80);
	FT232RL_start_snd <= '0';
	ADC_send_buffer <= SampleRAM(RW_Adress);
	
	if fsm = s_wait then
		RW_Adress <= 0;
		if FT232RL_start_rcv = '1' then
			fsm <= s_record;
		end if;
	elsif fsm = s_record then
		if ADC_data_ready = '1' then
			SampleRAM(RW_Adress) <= ADC_data;
			if RW_Adress < RW_Adress_max then --2**samplenummer-1 then
				RW_Adress <= RW_Adress +1;
			else
				RW_Adress <= 0;
				fsm <= s_wait_send;
			end if;
		end if;
	elsif fsm = s_wait_send then
		fsm <= s_send_h;
	elsif fsm = s_send_h then
		if FT232RL_ready = '1' then
			FT232RL_byte_snd <= "0000" & ADC_send_buffer(11 downto 8);
			FT232RL_start_snd <= '1';
			fsm <= s_send_l;
		end if;
	elsif fsm = s_send_l then
		if FT232RL_ready = '1' then
			FT232RL_byte_snd <= ADC_send_buffer(7 downto 0);
			FT232RL_start_snd <= '1';
			if RW_Adress < RW_Adress_max then --2**samplenummer-1 then
				RW_Adress <= RW_Adress +1;
				fsm <= s_wait_send;
			else
				RW_Adress <= 0;
				fsm <= s_wait;
			end if;
		end if;
	end if;
end process;

------------ FT232H sync FIFO---------------------------------------------
RD <= '1';
SIWU <= '1';
OE <= '1';
PWRSAV <= 'Z';
WR <= '0' when (TXE = '0' and fifo_rd_buff = '1') or (TXE = '0' and ft232h_fail = '1') else '1';
fifo_rd <= '1' when TXE = '0' and fifo_empty = '0' else '0';
process 
	begin   
	wait until rising_edge(CLKOUT);--CLK60);
	fifo_rd_buff <= fifo_rd;
	TXE_buff <= TXE;
	fifo_empty_buff <= fifo_empty;
	if TXE = '0' then
		ft232h_fail <= '0';
	end if;
	if fifo_empty_buff = '0' and TXE = '1' and TXE_buff = '0' then
		ft232h_fail <= '1';
	end if;
	blinkcounter_60 <= blinkcounter_60 +1;
end process;

----------------- LEDs -----------------------------------------------
process begin
	wait until rising_edge(clk80);
	counter <= counter +1;
	if counter = 79999999 then
		counter <= (others => '0');
	end if;
end process;

sysled <= blinkcounter_60(25);
LED_r <= '1' when fsm = s_record else '0';
LED_o <= counter(26);

end Behavioral;