#ifndef DEFINES_H_
#define DEFINES_H_

#include "LPC23xx.h"
#include <stdint.h>
#include <stdbool.h>

#include "target.h"

typedef char               INT8;
typedef unsigned char     UINT8;
typedef          short    INT16;
typedef unsigned short   UINT16;
typedef          int      INT32;
typedef unsigned int     UINT32;
typedef long long         INT64;

/*****************************************************************************/
/* Alternate port access macros from avr-gcc list
 * http://lists.gnu.org/archive/html/avr-gcc-list/2004-12/msg00097.html
 * Dave Hylands
 *
 * LPC2000 port by Michael Geselbracht (let)
 * 
 * Example defines:
#define DATA1         (0, 7)  // PORT0.7
#define DATA2         (0, 3)  // PORT0.3
#define DATA3         (1, 3)  // PORT1.3
*/

#define PORT0 0
#define PORT1 1
#define PORT2 2
#define PORT3 3
#define PORT4 4

/* Code to manipulate PORTx, PINx, and DDRx using definitions like those above */
#ifndef USE_FAST_IO
#error "IO access mode not specified."
#endif
#if USE_FAST_IO
#define IOSET(port,pin)      (FIO##port##SET)
#define IOCLR(port,pin)      (FIO##port##CLR)
#define DDR(port,pin)        (FIO##port##DIR)
#define DDR_OPT(port,pin)    (FIO##port##DIR_OPT)
#define PIN(port,pin)        (FIO##port##PIN)
#else
/* Use traditional GPIO access */
#define IOSET(port,pin)      (IOSET##port)
#define IOCLR(port,pin)      (IOCLR##port)
#define DDR(port,pin)        (IODIR##port)
#define DDR_OPT(port,pin)    (IODIR##port##_OPT)
#define PIN(port,pin)        (IOPIN##port)
#endif
#define PIN_MASK(port,pin) (1<<(pin))

/* Set a pin */
#define SETPIN(pindef)       IOSET pindef = PIN_MASK pindef

/* Clear a pin */
#define CLRPIN(pindef)       IOCLR pindef = PIN_MASK pindef

/* Set a IO to output */
#define SET_DDR_OUT(pindef) DDR pindef |= PIN_MASK pindef
#define SET_DDR_OPT_OUT(pindef) DDR_OPT pindef |= PIN_MASK pindef

/* Set a IO to input */
#define SET_DDR_IN(pindef)  DDR pindef &= ~PIN_MASK pindef
#define SET_DDR_OPT_IN(pindef)  DDR_OPT pindef &= ~PIN_MASK pindef

/* Read the value of a pin - a byte with the correspondig bit set or cleard is returned.
   Example:

   #define PIN7 (0, 7)
   ...
   SETPIN(PIN7);
   x = GETPIN(PIN7);
   // x = 128
  
  The _OPT versions perform a non-volatile access the the memory locations. Therefore the
  compiler is able to group serveral expressions into a single command.
  Example:
  #define LED0 (0, 10)
  #define LED0 (0, 11)
  #define LED0 (0, 11)
  SET_DDR_OPT_OUT(LED0);
  SET_DDR_OPT_OUT(LED1);
  SET_DDR_OPT_OUT(LED2);
  
  In this case a single read-write-modify cycle will be performed instead of three.
  
  In order to use these macros a modified device header file is required.
  Macros to be defined are:
  IOSET_OPT, IOCLR_OPT, IODIR_OPT for traditional GPIO access or
  FIOSET, FIOCLR, FIODIR for fast GPIO access.  
*/

#define GETPIN(pindef)      (PIN pindef & PIN_MASK pindef)
#define GETPIN_BOOL(pindef) (GETPIN(pindef) ? 1 : 0)
#define TOGGLEPIN(pindef)   PIN pindef ^= PIN_MASK pindef

/*
Example usage:

	SET_DDR_OUT(DATA1);
	SET_DDR_IN(DATA2);
	SETPIN(DATA1);
	CLRPIN(DATA1);
	x = GETPIN(DATA2);
*/

/*********************************************************************************/

/* Binary constant generator macro
  By Tom Torfs - donated to the public domain
*/
 
/* All macro's evaluate to compile-time constants */

 
/* *** helper macros ***/

 
/* turn a numeric literal into a hex constant
  (avoids problems with leading zeroes)
  8-bit constants max value 0x11111111, always fits in unsigned long
*/
#define HEX__(n) 0x##n##LU

 
/* 8-bit conversion function */
#define B8__(x) ((x&0x0000000FLU)?1:0)      \
              +((x&0x000000F0LU)?2:0)       \
              +((x&0x00000F00LU)?4:0)       \
              +((x&0x0000F000LU)?8:0)       \
              +((x&0x000F0000LU)?16:0)      \
              +((x&0x00F00000LU)?32:0)      \
              +((x&0x0F000000LU)?64:0)      \
              +((x&0xF0000000LU)?128:0)

 
/* *** user macros ***/

 
/* for upto 8-bit binary constants */
#define B8(d) ((uint8_t)B8__(HEX__(d)))

 
/* for upto 16-bit binary constants, MSB first */
#define B16(dmsb,dlsb) (((uint16_t)B8(dmsb)<<8)     \
                        + B8(dlsb))

 
/* for upto 32-bit binary constants, MSB first */
#define B32(dmsb,db2,db3,dlsb) (((uint32_t)B8(dmsb)<<24)      \
                                  + ((uint32_t)B8(db2)<<16) \
                                  + ((uint32_t)B8(db3)<<8)    \
                                  + B8(dlsb))

 
/* Sample usage:
     B8(01010101) = 85
     B16(10101010,01010101) = 43605
     B32(10000000,11111111,10101010,01010101) = 2164238933
*/

/*************************************************************/

/* let: Some additional definitions
 */
 
/*
 * Configure the pin connect block
 * Example:
 * 
 * SET_PCB(PINSEL0, 4, PCB_ALT1); // set bits 5:4 of PINSEL0 to 0b01
 */
#define PCB_PRIMARY 0
#define PCB_ALT1    1
#define PCB_ALT2    2
#define PCB_ALT3    3
#define SET_PCB(name, offset, value) name = (name & ~(3 << offset)) | (value << offset)

#ifndef TRUE
#define TRUE (1)
#endif
#ifndef FALSE
#define FALSE (0)
#endif

#ifndef _BV
#define _BV(_x) (1<<(_x))
#endif

#ifndef NULL
#define NULL ((void*)0)
#endif

#endif /*DEFINES_H_*/
