#include "../inc/uart.h"
#include "../inc/defines.h"
#include "../inc/irq.h"

#include <stdint.h>

static int (*saGlblReadCallback[4])(int);
static void int_rda0(void);
static void int_rda1(void);
static void int_rda2(void);
static void int_rda3(void);

static struct {
	unsigned char divaddval;
	unsigned char mulval;
	unsigned short divider;
} __attribute__((packed)) saGlblBaudrate[12] =
{
		{  0, 15, 0x1d4c }, //    600
		{  0, 15, 0x0ea6 }, //   1200
		{  0, 15, 0x0753 }, //   2400
		{  1, 14, 0x036b }, //   4800
		{  3, 12, 0x0177 }, //   9600
		{  3, 12, 0x00fa }, //  14400
		{  7,  8, 0x007d }, //  19200
		{  3, 12, 0x007d }, //  28800
		{  7, 12, 0x004a }, //  38400
		{  1, 14, 0x004b }, //  56000
		{ 10,  9, 0x0025 }, //  57600
		{  7, 10, 0x0017 }  // 115200
};

void uart_enable(enum uart_e uart, enum uart_layout_e layout, bool bUseHandshaking)
{
	switch (uart) {
		case UART0:
			switch (layout) {
				case LAYOUT_FIRST:
					SET_PCB(PINSEL0,  4, PCB_ALT1); // TXD: P0.2
					SET_PCB(PINSEL0,  6, PCB_ALT1); // RXD: P0.3
					break;
				default:
					break;
			}
			U0FCR = _BV(0) | _BV(1) | _BV(2) | (B8(10) << 6);
			break;
		case UART1:
			switch (layout) {
				case LAYOUT_FIRST:
					SET_PCB(PINSEL0, 30, PCB_ALT1); // TXD: P0.15
					SET_PCB(PINSEL1,  0, PCB_ALT1); // RXD: P0.16
					break;
				case LAYOUT_SECOND:
					SET_PCB(PINSEL4,  0, PCB_ALT2); // TXD: P2.0
					SET_PCB(PINSEL4,  2, PCB_ALT2); // RXD: P2.1
					break;
				default:
					break;
			}
			U1FCR = _BV(0) | _BV(1) | _BV(2) | (B8(10) << 6);
			break;
		case UART2:
			switch (layout) {
				case LAYOUT_FIRST:
					SET_PCB(PINSEL0, 20, PCB_ALT1); // TXD: P0.10
					SET_PCB(PINSEL0, 22, PCB_ALT1); // RXD: P0.11
					break;
				case LAYOUT_SECOND:
					SET_PCB(PINSEL4, 16, PCB_ALT2); // TXD: P2.8
					SET_PCB(PINSEL4, 18, PCB_ALT2); // RXD: P2.9
					break;
				default:
					break;
			}
			U2FCR = _BV(0) | _BV(1) | _BV(2) | (B8(10) << 6);
			break;
		case UART3:
			switch (layout) {
				case LAYOUT_FIRST:
					SET_PCB(PINSEL0,  0, PCB_ALT2); // TXD: P0.0
					SET_PCB(PINSEL0,  2, PCB_ALT2); // RXD: P0.1
					break;
				case LAYOUT_SECOND:
					SET_PCB(PINSEL1, 18, PCB_ALT3); // TXD: P0.25
					SET_PCB(PINSEL1, 20, PCB_ALT3); // RXD: P0.26
					break;
				case LAYOUT_THIRD:
					SET_PCB(PINSEL9, 24, PCB_ALT3); // TXD: P4.28
					SET_PCB(PINSEL9, 26, PCB_ALT3); // RXD: P4.29
					break;
			}
			U3FCR = _BV(0) | _BV(1) | _BV(2) | (B8(10) << 6);
			break;
	}
	
	if (bUseHandshaking) {
		switch (uart) {
			case UART1:
				switch (layout) {
					case LAYOUT_FIRST:
						SET_PCB(PINSEL1,  2, PCB_ALT1); // CTS1
						SET_PCB(PINSEL1, 12, PCB_ALT1); // RTS1
						break;
					case LAYOUT_SECOND:
						SET_PCB(PINSEL4,  4, PCB_ALT2); // CTS1
						SET_PCB(PINSEL4, 14, PCB_ALT2); // RTS1
						break;
					default:
						break;
				}
				break;
			default:
				break;
		}
	}
}

void uart_setMode(enum uart_e uart, enum uart_bits_e bits, enum uart_parity_e parity, enum uart_stop_e stopbits)
{
	unsigned long value = 0;
	volatile unsigned long *lcr=0;
	
	switch (uart) {
		case UART0:
			lcr = &U0LCR;
			break;
		case UART1:
			lcr = &U1LCR;
			break;
		case UART2:
			lcr = &U2LCR;
			break;
		case UART3:
			lcr = &U3LCR;
			break;
	}
	
	switch (bits) {
		case DATABIT_5:
			value = B8(00);
			break;
		case DATABIT_6:
			value = B8(01);
			break;
		case DATABIT_7:
			value = B8(10); 
			break;
		case DATABIT_8:
			value = B8(11);
			break;
	}
		
	switch (parity) {
		case P_NONE:
			break;
		case P_ODD:
			value |= _BV(3) | (B8(00) << 4);
			break;
		case P_EVEN:
			value |= _BV(3) | (B8(01) << 4);
			break;
	}
	
	switch (stopbits) {
		case STOPBIT_1:
			break;
		case STOPBIT_2:
			value |= _BV(2);
			break;
	}
	
	*lcr = value;
}

void uart_setBaudrate(enum uart_e uart, enum uart_baudrate_e baudrate)
{
	uart_setDivider(uart, saGlblBaudrate[baudrate].divider, saGlblBaudrate[baudrate].mulval, saGlblBaudrate[baudrate].divaddval);
}

void uart_setDivider(enum uart_e uart, unsigned divider, unsigned mulval, unsigned divaddval)
{
	switch (uart) {
		case UART0:
			U0LCR |= _BV(7);
			U0DLL = divider;
			U0DLM = divider >> 8;
			U0LCR &= ~_BV(7);
			U0FDR = divaddval | (mulval << 4);
			break;
		case UART1:
			U1LCR |= _BV(7);
			U1DLL = divider;
			U1DLM = divider >> 8;
			U1LCR &= ~_BV(7);
			U1FDR = divaddval | (mulval << 4);
			break;
		case UART2:
			U2LCR |= _BV(7);
			U2DLL = divider;
			U2DLM = divider >> 8;
			U2LCR &= ~_BV(7);
			U2FDR = divaddval | (mulval << 4);
			break;
		case UART3:
			U3LCR |= _BV(7);
			U3DLL = divider;
			U2DLM = divider >> 8;
			U3LCR &= ~_BV(7);
			U3FDR = divaddval | (mulval << 4);
			break;
	}
}

void uart_writeData(enum uart_e uart, unsigned char data)
{
	switch (uart) {
		case UART0:
			while (!(U0LSR & _BV(5)));
			U0THR = data;
			break;
		case UART1:
			while (!(U1LSR & _BV(5)));
			U1THR = data;
			break;
		case UART2:
			while (!(U2LSR & _BV(5)));
			U2THR = data;
			break;
		case UART3:
			while (!(U3LSR & _BV(5)));
			U3THR = data;
			break;
	}
}

void uart_setReadCallback(enum uart_e uart, int (*cb)(int))
{
	saGlblReadCallback[uart] = cb;
	
	switch (uart) {
		case UART0:
			install_irq(UART0_INT, int_rda0, HIGHEST_PRIORITY);
			U0IER = _BV(0);
			break;
		case UART1:
			install_irq(UART1_INT, int_rda1, HIGHEST_PRIORITY);
			U1IER = _BV(0);
			break;
		case UART2:
			install_irq(UART2_INT, int_rda2, HIGHEST_PRIORITY);
			U2IER = _BV(0);
			break;
		case UART3:
			install_irq(UART3_INT, int_rda3, HIGHEST_PRIORITY);
			U3IER = _BV(0);
			break;
	}
}

static void int_rda0() __irq
{
	while (U0LSR & _BV(0)) {
		saGlblReadCallback[0](U0RBR);
	}
	
	VICVectAddr = 0;
}

static void int_rda1() __irq
{
	while (U1LSR & _BV(0)) {
		saGlblReadCallback[1](U1RBR);
	}
	
	VICVectAddr = 0;
}

static void int_rda2() __irq
{
	while (U2LSR & _BV(0)) {
		saGlblReadCallback[2](U2RBR);
	}
	
	VICVectAddr = 0;
}

static void int_rda3() __irq
{
	while (U3LSR & _BV(0)) {
		saGlblReadCallback[3](U3RBR);
	}
	
	VICVectAddr = 0;
}
