/*
 * Copyright (c) 2001-2004 Swedish Institute of Computer Science.
 * All rights reserved. 
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
 * OF SUCH DAMAGE.
 *
 * This file is part of the lwIP TCP/IP stack.
 * 
 * Author: Adam Dunkels <adam@sics.se>
 *
 */

/*
 * This file is a skeleton for developing Ethernet network interface
 * drivers for lwIP. Add code to the low_level functions and do a
 * search-and-replace for the word "ethernetif" to replace it with
 * something that better describes your network interface.
 */

#include "string.h"

#include "lwip/opt.h"
#include "lwip/def.h"
#include "lwip/mem.h"
#include "lwip/pbuf.h"
#include "lwip/sys.h"
#include "lwip/stats.h"

#include "netif/etharp.h"

#include "arch\ethernetif.h"

#include "emac.h"
#include "timer.h"

//! debug
//#include "trace.h"
//#include "LPC23xx.h"

/* Define those to better describe your network interface. */
#define IFNAME0 'e'
#define IFNAME1 'n'

/* Forward declarations. */
static err_t ethernetif_output(struct netif *netif, struct pbuf *p,
             struct ip_addr *ipaddr);

static void low_level_init(struct netif *netif)
{
  /* set MAC hardware address length */
  netif->hwaddr_len = 6;

  /* set MAC hardware address */
  netif->hwaddr[0] = MYMAC_1;
  netif->hwaddr[1] = MYMAC_2;
  netif->hwaddr[2] = MYMAC_3;
  netif->hwaddr[3] = MYMAC_4;
  netif->hwaddr[4] = MYMAC_5;
  netif->hwaddr[5] = MYMAC_6;

  /* maximum transfer unit */
  netif->mtu = ETH_MAX_FLEN;
  
  /* broadcast capability */
  netif->flags = NETIF_FLAG_BROADCAST;
 
  /* Do whatever else is needed to initialize interface. */  

	Init_EMAC();
}

/*
 * low_level_output():
 *
 * Should do the actual transmission of the packet. The packet is
 * contained in the pbuf that is passed to the function. This pbuf
 * might be chained.
 *
 */
static err_t low_level_output(struct netif *netif, struct pbuf *p)
{
  struct pbuf*       q;
	unsigned int       len;
	unsigned int       error;

	// Initiate transfer();
	RequestSend();

#if ETH_PAD_SIZE
  pbuf_header(p, -ETH_PAD_SIZE);			/* drop the padding word */
#endif

	len   = 0;
	error = 0;

  for(q = p; q != NULL; q = q->next)
	{
    /* Send the data from the pbuf to the interface, one pbuf at a
       time. The size of the data in each pbuf is kept in the ->len
       variable. */

		if (((unsigned int)len + q->len) >= (unsigned int) ETH_FRAG_SIZE)
		{
			error = 1;
			break;
		}

		CopyToFrame_EMAC(q->payload, q->len);
		len = len + q->len;
  }

	// Sent packet
	if (!error)
		DoSend_EMAC(len);

#if ETH_PAD_SIZE
  pbuf_header(p, ETH_PAD_SIZE);			/* reclaim the padding word */
#endif
  
#if LINK_STATS
	if (!error)
		lwip_stats.link.xmit++;
#endif /* LINK_STATS */      

  return ERR_OK;
}

/*
 * low_level_input():
 *
 * Should allocate a pbuf and transfer the bytes of the incoming
 * packet from the interface into the pbuf.
 *
 */
static struct pbuf* low_level_input(struct netif *netif)
{
  struct pbuf       *p, *q;
	unsigned int      len;
	unsigned int      cpyix;
	unsigned int      cpylen;

  /* Obtain the packet and the size of the packet */
	if (CheckFrameReceived() == 0)
		return NULL;

//	tracePrintf("Received packet. (%x)\n", MAC_RSV);
	
	len = StartReadFrame();

#if ETH_PAD_SIZE
  len += ETH_PAD_SIZE;						/* allow room for Ethernet padding */
#endif

  /* We allocate a pbuf chain of pbufs from the pool. */
  p = pbuf_alloc(PBUF_RAW, len, PBUF_POOL);
  
  if (p != NULL)
	{

#if ETH_PAD_SIZE
    pbuf_header(p, -ETH_PAD_SIZE);			/* drop the padding word */
#endif

    /* We iterate over the pbuf chain until we have read the entire
     * packet into the pbuf. */
		cpyix  = 0;

    for(q = p; q != NULL; q = q->next)
	{
      /* Read enough bytes to fill this pbuf in the chain. The
       * available data in the pbuf is given by the q->len
       * variable. */

			if (q->len >= len)
				cpylen = len;
			else
				cpylen = q->len;

			CopyFromFrame_EMAC(q->payload, cpylen);

			len   = len - cpylen;
			cpyix = cpyix + cpylen;
    }
	EndReadFrame();

#if ETH_PAD_SIZE
    pbuf_header(p, ETH_PAD_SIZE);			/* reclaim the padding word */
#endif

#if LINK_STATS
    lwip_stats.link.recv++;
#endif /* LINK_STATS */      
  }
	else 
	{
		// Drop packet because no free pbuf available
		EndReadFrame();

#if LINK_STATS
    lwip_stats.link.memerr++;
    lwip_stats.link.drop++;
#endif /* LINK_STATS */
  }

  return p;
}

/*
 * ethernetif_output():
 *
 * This function is called by the TCP/IP stack when an IP packet
 * should be sent. It calls the function called low_level_output() to
 * do the actual transmission of the packet.
 *
 */

static err_t ethernetif_output(struct netif *netif, struct pbuf *p,
      struct ip_addr *ipaddr)
{
 /* resolve hardware address, then send (or queue) packet */
  return etharp_output(netif, ipaddr, p);
}

/*
 * ethernetif_handlepackets
 *
 */
void ethernetif_handlepackets(struct netif *netif)
{
  struct eth_hdr*    ethhdr;
  struct pbuf*       p;

  /* move received packet into a new pbuf */
  p = low_level_input(netif);
  /* no packet could be read */
  if (p == NULL) return;

  /* points to packet payload, which starts with an Ethernet header */
  ethhdr = p->payload;

#if LINK_STATS
  lwip_stats.link.recv++;
#endif /* LINK_STATS */

  ethhdr = p->payload;
    
  switch (htons(ethhdr->type))
	{
		/* IP packet? */
		case ETHTYPE_IP:
			/* CSi disabled ARP table update on ingress IP packets.
				This seems to work but needs thorough testing. */
			/* update ARP table */
			etharp_ip_input(netif, p);
			
			/* skip Ethernet header */
			pbuf_header(p, -((s16_t) sizeof(struct eth_hdr)));
			
			/* pass to network layer */
			netif->input(p, netif);
		break;
      
		case ETHTYPE_ARP:
			/* pass p to ARP module  */
			etharp_arp_input(netif, (struct eth_addr *)&(netif->hwaddr[0]), p);
			break;
			
		default:
			pbuf_free(p);
			p = NULL;
			break;
  }
}

/*
 * ethernetif_init():
 *
 * Should be called at the beginning of the program to set up the
 * network interface. It calls the function low_level_init() to do the
 * actual setup of the hardware.
 *
 */
err_t ethernetif_init(struct netif *netif)
{
#if LWIP_SNMP
  /* ifType ethernetCsmacd(6) @see RFC1213 */
  netif->link_type = 6;
  /* your link speed here */
  netif->link_speed = 10;
  netif->ts = 0;
  netif->ifinoctets = 0;
  netif->ifinucastpkts = 0;
  netif->ifinnucastpkts = 0;
  netif->ifindiscards = 0;
  netif->ifoutoctets = 0;
  netif->ifoutucastpkts = 0;
  netif->ifoutnucastpkts = 0;
  netif->ifoutdiscards = 0;
#endif
  
  netif->name[0]      = IFNAME0;
  netif->name[1]      = IFNAME1;
  netif->output       = ethernetif_output;
  netif->linkoutput   = low_level_output;
  
  low_level_init(netif);

  etharp_init();

  return ERR_OK;
}

/* TODO: Function to change hardware address
          When linked
           - calls ETHDRIVER_Start
           - calls etharp_init
*/
