/* 
   "Epsilon"
   LPC2378 Demo by Martin Thomas, Kaiserslautern, Germany
   http://www.siwawi.arubi.uni-kl.de/avr_projects/arm_projects

   Target:   LPC2378
   Board:    Olimex LPC-2378-STK
   Compiler: GNU arm-elf or arm-eabi toolchain 
             (tested with WinARM 5/07-beta)
   
   Partly based on drivers from the NXP LPC23xx & 24xx 
   example-collection available from www.nxp.com.
*/

#include <string.h>

#include <LPC23xx.h>
#include <irq.h>
#include <target.h>
#include <timer.h>
#include <defines.h>

#include <swi.h>

#include <emac.h>

#include "lwip/opt.h"
#include "lwip/stats.h"
#include "lwip/sys.h"
#include "lwip/pbuf.h"
#include "lwip/udp.h"

#if LWIP_TCP
#include "lwip/tcp.h"
#endif

#include "lwip/ip_addr.h"

#include "netif/etharp.h"
#include "netif/loopif.h"

#include "arch/ethernetif.h"
#include "arch/clock.h"

#define P_LED_STATUS (2, 7)
#define P_LED_CMD    (2, 6)

static struct netif  netif_eth0_data;
struct netif* pGlblNetIF_eth0 = &netif_eth0_data;
static struct udp_pcb *conn;

static struct ip_addr my_ipaddr_data;
static struct ip_addr my_netmask_data;
static struct ip_addr my_gw_data;

static timer_t sGlblLastArpTime;

static void processUdp(void *arg, struct udp_pcb *udp, struct pbuf *p, struct ip_addr *addr, uint16_t port);

static void lwip_init(void)
{
	struct ip_addr rip;
	
	stats_init();
	sys_init();
	mem_init();
	memp_init();
	pbuf_init();
	etharp_init();
	ip_init();
	udp_init();
	
#if LWIP_TCP
	tcp_init();
#endif

	IP4_ADDR(&my_ipaddr_data, 192, 168, 0, 100);
	IP4_ADDR(&my_netmask_data, 255, 255, 255, 0);
	IP4_ADDR(&my_gw_data, 192, 168, 0, 1);

	netif_add(pGlblNetIF_eth0, &my_ipaddr_data, &my_netmask_data, &my_gw_data, NULL, ethernetif_init, ip_input);
	netif_set_default(pGlblNetIF_eth0);
	netif_set_up(pGlblNetIF_eth0);
	
	IP4_ADDR(&rip, 192,168,0,20);
	conn = udp_new();
	udp_connect(conn, &rip, 333);
	udp_bind(conn, NULL, 9999);
	udp_recv(conn, processUdp, NULL);
}

static void sendUDP(void)
{
	static int counter;
	struct pbuf *buf;

	buf = pbuf_alloc(PBUF_TRANSPORT, sizeof(counter), PBUF_RAM);
	memcpy(buf->payload, (void*) &counter, sizeof(counter));
	counter++;
	
	udp_send(conn, buf);
	pbuf_free(buf);
}

static void processUdp(void *arg, struct udp_pcb *udp, struct pbuf *p, struct ip_addr *addr, uint16_t port)
{
	pbuf_free(p);
}

static void hardwareInit(void)
{	
   SET_DDR_OUT(P_LED_STATUS);
   SET_DDR_OUT(P_LED_CMD);
}

static void flashLED(void)
{
	int i;
	
	for (i = 0; i < 4; i++)
	{
		SETPIN(P_LED_STATUS);
		SETPIN(P_LED_CMD);
		delay_ms(100);
		CLRPIN(P_LED_STATUS);
		CLRPIN(P_LED_CMD);
		delay_ms(100);
	}
}

int main (void)
{
	timer_t time;
	
	hardwareInit();
	IntEnable();
	timer_init();
	
	flashLED();
	
	lwip_init();
		
	#if LWIP_TCP
		http_init();
	#endif

	time = currentTimer();
	
	for (;;)
	{
		ethernetif_handlepackets(pGlblNetIF_eth0);
			
		if (sGlblLastArpTime == 0 || elapsedTime(sGlblLastArpTime) > 5000)
		{
			sGlblLastArpTime = currentTimer();
			etharp_tmr();
		}
		
		if (elapsedTime(time) > 500)
		{
			time = currentTimer();
			sendUDP();
		}
	}
		
	return 0;
}
