#include <stdlib.h>
#include <avr/io.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include "lcd.h"
#include "mmc.h"
#include "spi.h"
#include "delay.h"
#include "fat16.h"
#include "types.h"

#ifdef __AVR_ATmega168__
#include "rc5.h"

//mega168
#define UDR UDR0
#define UCSRA UCSR0A
#define UCSRB UCSR0B
#define UCSRC UCSR0C
#define UBRRL UBRR0L
#define UBRRH UBRR0H
#define UDRE UDRE0
#define RXEN RXEN0
#define TXEN TXEN0
#define RXCIE RXCIE0
#define OCR2 OCR2A
#define TCCR0 TCCR0B
#define TCCR2 TCCR2B
#endif

#ifndef F_CPU
	#define F_CPU 8000000
#endif

// Buttons
#define KEY_PORT PIND
#define KEY_PORT2 PINB

#define KEY_MASK 0xe8
#define KEY_PLAY 0x80

#define KEY_MASK2 0x03

#define KEY_STOP 0x08
#define KEY_REC 0x10
#define KEY_NEXT 0x20
#define KEY_LAST 0x40
#define KEY_LEFT 0x02
#define KEY_RIGHT 0x01

#define RC5_CODE_PLAY 0x35
#define RC5_CODE_STOP 0x36
#define RC5_CODE_REC 0x37
#define RC5_CODE_LEFT 0x32
#define RC5_CODE_RIGHT 0x34
#define RC5_CODE_NEXT 0x20
#define RC5_CODE_LAST 0x21
#define RC5_CODE_DELETE 0x31
#define RC5_CODE_MENU 0x1d
#define RC5_CODE_TEST 0x3c

// Pins
#define PIN_LIGHT 0x40
#define PIN_ADCCS 0x80

// States
#define STOP 0
#define PLAY 1
#define REC 2
#define ABORT_REC 3
#define DELETE 4
#define MENU 5

// Menu
#define MAX_MENU 5
#define M_SMPLR 0
#define M_CHNR 1
#define M_SSECT 2
#define M_REP 3
#define M_LIGHT 4

// Flags
#define KEY_FLAG 0x01
#define DT_FLAG 0x02
#define OCCHNG_FLAG 0x04
#define IR_FLAG 0x08
#define IR_TGL 0x10
#define FAT_FLAG 0x20
#define DIRNUM_FLAG 0x40
#define MMCOK_FLAG 0x80

// MMC
#define START_SECTOR 528	// This is the sector number where the search for midi files begins (no FAT32 readings!)

#define TX_BUF_SIZE 256	// lasts for at least 80ms on full MIDI data load (for *really* slow MMC write sector delays)

// Variables
u08 sharedmem[512];
u08 tx_buf[TX_BUF_SIZE], tx_wrp=0;//, tx_rdp=0, tx_wrp=0;	// TX buffer
u08 flags=KEY_FLAG, state=STOP;	// Flags
u08 file_num=0, file_cnt=0;	// File
u16 *file_pos;
u08 ee_chnr=1, ee_rep=2, ee_light=1, menu_cnt;//, smpl_state=0;
u16 ee_ssect=START_SECTOR, ee_smplr=32000;
u32 last;

extern struct fat_filedata_st fat_filedata;
extern u32 sect;


/*
Offset 	Lnge (in bytes) 	Inhalt
0 	4 	'RIFF'
4 	4 	<Dateigre - 8>
8 	4 	'WAVE'

12 	4 	'fmt ' 	Header Signatur
16 	4 	<fmt length> 	Lnge des restlichen fmt-Headers (16 Byte)
20 	2 	<format tag> 	sample Datenformat (siehe separate Tabelle weiter unten)
22 	2 	<channels> 	Anzahl der Kanle: 1 = mono, 2 = stereo
24 	4 	<sample rate> 	Abtastrate pro Sekunde (z.B. 44100)
28 	4 	<bytes/second> 	Sample-Rate * Block-Align
32 	2 	<block align> 	Kanle * bits/sample / 8
34 	2 	<bits/sample> 	8, 16 oder 24

36 	4 	'data' 	Header Signatur
40 	4 	<length> 	Lnge des Datenblocks
44 	<bits/sample>/8 		Sample Data
*/

prog_uchar WAVE_HEADER[44]=
{ //                Filesize-8
    'R','I','F','F', 0,0,0,0,'W','A','V','E',
  //                fmtsize  PCM #ch Smplrate  b/sec     algnbits
    'f','m','t',' ',16,0,0,0,1,0,1,0,0,125,0,0,0,250,0,0,2,0,16,0,
  //                Length
    'd','a','t','a', 0,0,0,0
};


// Local Functions
static void hw_init(void);
static void menu_save(void);
/*static u08 fetchbyte(void);
static void writebyte(u08 b);
static void writestring(u08 *buf, u08 len);
static void checkbyte(u08 b);
static u08 checkstring(u08 *buf, u08 len);
*/
static void key_detect(void);
static void debounce(u08 mode);
static void print_main(void);
static void print_menu(void);


//---------------------------------------------------


static u16 getReload(u16 smplr) {
	return F_CPU/smplr;
}


// Init MCU
static void hw_init(void) {
//	u08 k;

	// Oscillator
	// Ports
	DDRC = 0x3F;	// LCD
	PORTC = 0x00;
	lcd_init();
	lcd_string(DISP_MAIN, LINE_MAIN);
	lcd_string(DISP_BOOT, LINE_BOOT);

	DDRB = 0x2C;
	PORTB = 0xFF;

	DDRD = 0x12;	// Input-Port, auer TxD
	PORTD = 0xEF;	// Pull-Ups aktivieren

	// Read parameters from EEPROM
	if (eeprom_read_byte(0) != 0x27) {	// eeprom empty
		eeprom_write_byte(0, 0x27);
		menu_save();
	}
	else {	// load parameters
		ee_smplr = eeprom_read_word(1);
		if (ee_smplr > 50000)	// invalid sample rate
			ee_smplr = 40000;
		ee_chnr = eeprom_read_byte(3);
		ee_ssect = eeprom_read_word(4);
		ee_rep = eeprom_read_byte(6);
		if (ee_rep > 4)
			ee_rep = 2;
		ee_light = eeprom_read_byte(7);
	}
	if (ee_light)
		PORTB &= ~PIN_LIGHT;

	//Initialisierung der MMC/SD-Karte
	MMC_hw_init();
	spi_init();
	rc5_init(RC5_ALL);
	sei();
	delay_ms(100);
	TCCR1B = 1;
	if (MMC_init()) {
		lcd_string(DISP_OK, LINE_OK);
		flags |= MMCOK_FLAG;
	}
	else
		lcd_string(DISP_CARDERR, LINE_CARDERR);
	TCCR1B = 0;
	srand(TCNT1);
	TIFR1 |= 7;
	TCNT1 = 0;

	TIMSK1 = 2;
	TIMSK2 = 2;

	OCR1A = getReload(ee_smplr);
}


// Save data
static void menu_save(void) {
	eeprom_write_word(1, ee_smplr);	
	eeprom_write_byte(3, ee_chnr);
	eeprom_write_word(4, ee_ssect);
	eeprom_write_byte(6, ee_rep);
	eeprom_write_byte(7, ee_light);
}


//----------------------------------------------------------
// MMC IO
/*
// Ein Byte von MMC lesen
static u08 fetchbyte(void) {
	file_len--;
	return mmc_fetch_byte();
}


// Write byte to MMC
static void writebyte(u08 b) {
	file_len++;
	mmc_write_byte(b);
}


// Write string to MMC
static void writestring(u08 *buf, u08 len) {
	u08 cnt=0;
	for (cnt=0; cnt<len; cnt++)
		mmc_write_byte(buf[cnt]);
}


// Read byte from MMC and verify value
static void checkbyte(u08 b) {
	u08 temp = fetchbyte();
	if (b != temp)
	{
		lcd_string(DISP_FILEERR, LINE_FILEERR);
		state = STOP;
		delay_ms(1000);
	}
}


// Read string from MMC and verify content
static u08 checkstring(u08 *buf, u08 len) {
	u08 b, cnt=0;

	for (cnt=0; cnt<len; cnt++) {
		b = fetchbyte();
		if (b != (u08)buf[cnt])
			return 0;
	}
	return 1;
}
*/

//----------------------------------------------------------
// Buttons

// Entprellen
static void debounce(u08 mode) {
	flags &= ~KEY_FLAG;
	cli();
	WDTCSR = 0x18;	// Setup WDT-Init
	WDTCSR = 0x40|mode;	// WDIE=1->Interrupt mode; 32ms Timeout
	wdt_reset();
	sei();
}


// Check buttons
static void key_detect(void) {
	static u08 ok=0, zif1=0xff;
	u08 k = ((u08)(~KEY_PORT)&KEY_MASK) | ((u08)(~KEY_PORT2)&KEY_MASK2);
	if ((k&KEY_NEXT) && (k&KEY_LAST))
		k = RC5_CODE_MENU;
	else if ((k&KEY_LEFT) && (k&KEY_RIGHT))
		k = RC5_CODE_DELETE;
	else if (k&KEY_STOP)
		k = RC5_CODE_STOP;
	else if (k&KEY_REC)
		k = RC5_CODE_REC;
	else if (k&KEY_LEFT)
		k = RC5_CODE_LEFT;
	else if (k&KEY_RIGHT)
		k = RC5_CODE_RIGHT;
	else if (k&KEY_NEXT)
		k = RC5_CODE_NEXT;
	else if (k&KEY_LAST)
		k = RC5_CODE_LAST;
	else if (k&KEY_PLAY)
		k = RC5_CODE_PLAY;

	if (!(flags & KEY_FLAG))
		return;
		if (rc5.flip != -1) {
			k = rc5.code|0x80;
			if (k != ok || rc5.flip != ((flags>>4)&1) || (flags&IR_FLAG) == 0) {
				ok = k;
				k = 0;
			}
			flags |= IR_FLAG;
			if (rc5.flip)
				flags |= IR_TGL;
			else
				flags &= ~IR_TGL;
			rc5.flip = -1;
			debounce(3);
		}
	if (k != ok) {
		if (k != ok && !(flags&IR_FLAG)) {
			debounce(1);
			ok = k;
		}
		if (ok != 0) {
			k = ok & 0x7f;
//			lcd_setcur(20,0);
//			lcd_hex_u08(k);
			if (k <= 9 && file_cnt && (state == STOP)) {	// Zifferntaste
				if (file_cnt < 11) {
					if (k == 0)
						file_num=10;
					file_num = k-1;
					lcd_string(DISP_PLAY, LINE_PLAY);
					lcd_setcur(PLAYFNUM_X, PLAYFNUM_Y);
					lcd_number(file_num+1, 1);
					//start_time();	// Start Time
					state = PLAY;
					flags |= DIRNUM_FLAG;
				}
				else {
					if (zif1 == 0xff) {
						zif1 = k;
						lcd_setcur(DWAHL_X, DWAHL_Y);
						lcd_data('-');
						lcd_data('>');
						lcd_data(k+'0');
					}
					else {
						file_num = k + zif1*10;
						if (file_num <= file_cnt && file_num > 0) {
							file_num--;
							lcd_string(DISP_PLAY, LINE_PLAY);
							lcd_setcur(PLAYFNUM_X, PLAYFNUM_Y);
							lcd_number(file_num+1, 1);
							//start_time();	// Start Time
							state = PLAY;
							flags |= DIRNUM_FLAG;
						}
						lcd_setcur(DWAHL_X, DWAHL_Y);
						lcd_data(' ');
						lcd_data(' ');
						lcd_data(' ');
						zif1=0xff;
					}
				}
			}
			else if ((k == RC5_CODE_DELETE) && file_cnt && state == STOP) {	// Left&Right together->Delete last file
				state = DELETE;
			}
			else if ((k == RC5_CODE_MENU) && state == STOP) {	// Next&Last together->Enter Menu
				menu_cnt = 0;
				print_menu();
				state = MENU;
			}
			else if ((k == RC5_CODE_LAST)) {	// One file back
				if (state == STOP && file_cnt) {
					if (file_num == 0)
						file_num = file_cnt;
					file_num--;
					lcd_setcur(STOPFNUM_X, STOPFNUM_Y);
					lcd_number(file_num+1, 1);
					if (flags & FAT_FLAG) {
						fat_read_filedata(file_num);
						print_filename(NAME_LINE);
						lcd_setcur(STOPSECT_X, STOPSECT_Y);
						lcd_hex_u16(fat_filedata.startsect);
						print_filesize();
					}
					else {
						lcd_setcur(STOPSECT_X, STOPSECT_Y);
						lcd_hex_u16(file_pos[file_num]);
					}
				}
				else if (state == MENU) {	// Menu select
					menu_cnt++;
					if (menu_cnt == MAX_MENU)
						menu_cnt = 0;
					print_menu();
				}
			}
			else if ((k == RC5_CODE_NEXT)) {	// Next file
				if (state == STOP && file_cnt) {
					file_num++;
					if (file_num == file_cnt)
						file_num = 0;
					lcd_setcur(STOPFNUM_X, STOPFNUM_Y);
					lcd_number(file_num+1, 1);
					if (flags & FAT_FLAG) {
						fat_read_filedata(file_num);
						print_filename(NAME_LINE);
						lcd_setcur(STOPSECT_X, STOPSECT_Y);
						lcd_hex_u16(fat_filedata.startsect);
						print_filesize();
					}
					else {
						lcd_setcur(STOPSECT_X, STOPSECT_Y);
						lcd_hex_u16(file_pos[file_num]);
					}
				}
				else if (state == MENU) {	// Menu select
					menu_cnt--;
					if (menu_cnt >= MAX_MENU)
						menu_cnt = MAX_MENU-1;
					print_menu();
				}
			}
			else if (k == RC5_CODE_STOP) {	// Stop
				if (state == MENU) {	// Menu exit
					print_main();
					menu_save();
				}
				state = STOP;
			}
			else if ((k == RC5_CODE_PLAY) && file_cnt) {	// Play/Pause
				if (state == PLAY) {
					if (TCCR1B) {
						lcd_string(DISP_PAUSE, LINE_PAUSE);
						lcd_setcur(PLAYFNUM_X, PLAYFNUM_Y);
						lcd_number(file_num+1, 1);
						//stop_time();	// Pause
					}
					else {
						lcd_string(DISP_PLAY, LINE_PLAY);
						lcd_setcur(PLAYFNUM_X, PLAYFNUM_Y);
						lcd_number(file_num+1, 1);
						//start_time();	// Start Time
					}
				}
				else if (state == STOP && file_cnt) {
					state = PLAY;
				}
			}
			else if (k == RC5_CODE_REC) {	// Record
				if (state == STOP && flags & MMCOK_FLAG)
					state = REC;
				else if (state == REC)
					state = ABORT_REC;
			}
			else if (k == RC5_CODE_LEFT) {	// Left
				if (state == MENU) {	// Menu edit
					switch (menu_cnt) {
						case M_SMPLR:
							if (ee_smplr > 8000)
								ee_smplr-=100;
							break;
						case M_CHNR:
							if (ee_chnr > 1)
								ee_chnr--;
							break;
						case M_SSECT:
							if (ee_ssect > 0)
								ee_ssect--;
							break;
						case M_REP:
							if (ee_rep > 0)
								ee_rep--;
							break;
						case M_LIGHT:
							if (ee_light != 0)
								ee_light = 0;
							PORTB |= PIN_LIGHT;
							break;
					}
					print_menu();
				}
			}
			else if (k == RC5_CODE_RIGHT) {	// Right -> Faster
				if (state == MENU) {	// Menu edit
					switch (menu_cnt) {
						case M_SMPLR:
							if (ee_smplr < 50000)
								ee_smplr+=100;
							break;
						case M_CHNR:
							if (ee_chnr < 2)
								ee_chnr++;
							break;
						case M_SSECT:
							ee_ssect++;
							break;
						case M_REP:
							if (ee_rep < 4)
								ee_rep++;
							break;
						case M_LIGHT:
							if (ee_light == 0)
								ee_light = 1;
							PORTB &= ~PIN_LIGHT;
							break;
					}
					print_menu();
				}
			}
			else if (k == RC5_CODE_TEST) {	// File test
				if (state == STOP) {
					lcd_setcur(0,0);
					lcd_hex_u16(0);

					/*u16 i=1000;
					volatile u08 dummy;
					PORTB &= ~PIN_ADCCS;
					SPDR=0;
					while (i) {
						if (SPSR&0x80) {
							dummy = SPDR;
							SPDR=i--;
						}
					}
					PORTB |= PIN_ADCCS;
					*/
					if (flags & FAT_FLAG) {
						fat_openfile(FILEMODE_WRITE, file_cnt);
						u16 i;
						sect=fat_filedata.startsect;
						lcd_hex_u16(sect);
						mmc_speedtest_start();
						for (i=0; i<512; i++)
							sharedmem[i] = i;
						i=3520;
						while (i--)
							mmc_speedtest();
						mmc_speedtest_end();
						fat_filedata.len = 1802240;
						fat_closefile();
						file_cnt++;
					}
					
					lcd_setcur(0,0);
					lcd_hex_u16(1);
				}
			}
		}
	}
}


// Anzeige Hauptbildschirm
static void print_main(void) {
	lcd_string(DISP_MAIN, LINE_MAIN);
	if (file_cnt) {
		lcd_string(DISP_SELECT, LINE_SELECT);
		lcd_setcur(STOPFNUM_X, STOPFNUM_Y);
		lcd_number(file_num+1, 1);
		if (flags & FAT_FLAG) {
			fat_read_filedata(file_num);
			print_filename(NAME_LINE);
			lcd_setcur(STOPSECT_X, STOPSECT_Y);
			lcd_hex_u16(fat_filedata.startsect);
			print_filesize();
		}
		else {
			lcd_setcur(STOPSECT_X, STOPSECT_Y);
			lcd_hex_u16(file_pos[file_num]);
		}
		lcd_setcur(FCNT_X, FCNT_Y);
		lcd_number(file_cnt, 1);
	}
	else if (flags & MMCOK_FLAG)
		lcd_string(DISP_NOFILE, LINE_NOFILE);
	else
		lcd_string(DISP_CARDERR, LINE_CARDERR);
}


// Anzeige Men
static void print_menu(void) {
	lcd_string(DISP_MENU, LINE_MENU);
	lcd_string(DISP_SMPLR+menu_cnt, LINE_OPTION);
	switch (menu_cnt) {
		case M_SMPLR:
			lcd_setcur(SMPLR_X, LINE_OPTION);
			lcd_number_10k(ee_smplr);
			break;
		case M_CHNR:
			lcd_setcur(CHNR_X, LINE_OPTION);
			lcd_data(ee_chnr+'0');
			break;
		case M_SSECT:
			lcd_setcur(SSECT_X, LINE_OPTION);
			lcd_hex_u16(ee_ssect);
			break;
		case M_REP:
			lcd_setcur(REP_X, LINE_OPTION);
			lcd_data(ee_rep+'0');
			break;
		case M_LIGHT:
			lcd_setcur(LIGHT_X, LINE_OPTION);
			if (ee_light == 0)
				lcd_data('0');
			else
				lcd_data('1');
			break;
	}
}


//----------------------------------------------------------
// Time
/*
// Zeit am Display anzeigen
static void print_time(void) {
	lcd_setcur(TIME_X, TIME_Y);
	lcd_number(time_min, 0);
	lcd_data(':');
	lcd_number(time_sec, 0);
}


// Zeit zurcksetzen
static void clear_time(void) {
	time_sec = 0;
	time_min = 0;
	real_time = 0;
	limit_count = RTC_SOFT_COUNT;
	print_time();
}
*/


// Uhr starten
inline static void start_time(void) {
	TCNT1 = 0;
	TCCR1B = 8+1;
}


// Uhr stoppen
inline static void stop_time(void) {
	TCCR1B = 0;
	PORTB |= PIN_ADCCS;
}

//----------------------------------------------------------
// Main

static u08 random_song(u08 avoid, u08 top) {
	u08 temp;
	if (top < 2)
		return 0;
	do
		temp = rand();
	while (temp == avoid || temp >= top);
	return temp;
}



void main(void) {
	u08 i;

	file_pos = (u16 *)sharedmem;
	hw_init();

	if (flags & MMCOK_FLAG) {
		if (!fat_init()) {
			mmc_count_files(START_SECTOR);
		}
		else {
			flags |= FAT_FLAG;
			fat_count_files();
		}
		file_num = 0;
		print_main();
	}

	while (1) {
		switch (state) {
			case STOP:
				key_detect();
				break;

			case MENU:
				key_detect();
				break;

			case PLAY:
				// Wiedergabe starten
				if (ee_rep == 4 && !(flags & DIRNUM_FLAG)) {
					file_num = random_song(file_num, file_cnt);
				}
				flags &= ~DIRNUM_FLAG;
				lcd_string(DISP_PLAY, LINE_PLAY);	// Anzeige Wiedergabebildschirm
				lcd_string(DISP_TEMPO, LINE_TEMPO);
				lcd_setcur(PLAYFNUM_X, PLAYFNUM_Y);
				lcd_number(file_num+1, 1);
				//clear_time();

				if (flags & FAT_FLAG) {
					fat_openfile(FILEMODE_READ, file_num);
				}
				else {
					mmc_load_start(file_pos[file_num]);
				}
				//stop_time();
				if (flags & FAT_FLAG)
					fat_closefile();
				else
					mmc_complete_read();
				// repeat modes:
				// 0=single song
				// 1=repeat single song
				// 2=all songs
				// 3=repeat all songs
				// 4=random song
					if (ee_rep > 1) {
						if (file_num < file_cnt-1) {
							file_num++;
						}
						else if (ee_rep == 3) {
							file_num = 0;
						}
						if (ee_rep == 4) {
							file_num = random_song(file_num, file_cnt);
						}
						state = PLAY;
					}
					else if (ee_rep == 1) {
						state = PLAY;
					}
				print_main();
				break;

			case REC:
				// Rekorder-Task
				if (flags & FAT_FLAG)
					fat_openfile(FILEMODE_WRITE, file_cnt);
				sect=fat_filedata.startsect;
				mmc_speedtest_start();
			
				OCR1A = getReload(ee_smplr);	// Reload festlegen
				u16 sect_cnt = 0;

				spi_io(0xfc);	// start token
				for (i=0; i<44; i++)
					spi_io((u08)pgm_read_byte( &(WAVE_HEADER[i]) ));

				lcd_string(DISP_REC, LINE_REC);	// Bildschirm anzeigen
				lcd_setcur(RECFNUM_X, RECFNUM_Y);
				lcd_number(file_cnt+1, 1);
				lcd_string(DISP_BLANK, LINE_REC_BLANK);

				volatile u08 dummy;
				if (SPSR&0x80)
					dummy = SPDR;

				TCCR0B = 0; // RC5 off
				EIMSK = 0;
				start_time();

				u16 bufcnt=44;

				do {
					volatile register u08 xl asm("r26")=0;
					volatile register u08 xh asm("r27")=0;
					asm("out 0x2a, r1");
					asm("out 0x2b, r1");
					volatile register u08 r16 asm("r16");
				do {
					while (bufcnt<512) {
						asm("in r26, 0x2a");
						asm("in r27, 0x2b");
						if (xh != xl) {
							//while (!(UCSR0A&(1<<UDRE0)));
							asm("in r26, 0x2a");
							asm("ldi r27, 2");
							asm("ld r16, X+");
							//UDR0 = r16;
							spi_io(r16);
							//while (!(UCSR0A&(1<<TXC0)));
							//dummy = UDR0;
							//while (!(UCSR0A&(1<<UDRE0)));
							asm("ld r16, X+");
							//UDR0 = r16;
							spi_io(r16);
							asm("out 0x2a, r26");
							bufcnt+=2;
							//while (!(UCSR0A&(1<<TXC0)));
							//dummy = UDR0;
						}
						else if ((u08)(~KEY_PORT) & KEY_STOP) {
							state = STOP;
							stop_time();
							while (bufcnt<512) {
								//while (!(UCSR0A&(1<<UDRE0)));
								asm("in r26, 0x2a");
								asm("ldi r27, 2");
								asm("ld r16, X+");
								//UDR0 = r16;
								spi_io(r16);
								asm("out 0x2a, r26");
								bufcnt++;
								//while (!(UCSR0A&(1<<TXC0)));
								//dummy = UDR0;
							}
						}
					}
					/*while (!(UCSR0A&(1<<UDRE0)));
					UDR0 = 0;
					while (!(UCSR0A&(1<<TXC0)));
					dummy = UDR0;*/
					spi_io(0);

					/*while (!(UCSR0A&(1<<UDRE0)));
					UDR0 = 0;
					while (!(UCSR0A&(1<<TXC0)));
					dummy = UDR0;*/
					spi_io(0);

					/*while (!(UCSR0A&(1<<RXC0)));
					dummy = UDR0;*/

					//dummy = spiio(0xff);
					/*UDR0 = 0xff;
					while (!(UCSR0A&(1<<RXC0)));
					dummy = UDR0;*/
					spi_io(0xff);

					// wait while card is busy
					/*do {
						UDR0 = 0;
						while (!(UCSR0A&(1<<RXC0)));
					} while (UDR0 != 0xff);*/
					while (spi_io(0xff) != 0xff);

					//spiio(0xff);
					/*UDR0 = 0xff;
					while (!(UCSR0A&(1<<RXC0)));
					dummy = UDR0;*/
					spi_io(0xff);

					sect_cnt++;
					if (state == STOP)
						break;
					//spiio(0xfc);	// start token
					/*UDR0 = 0xfc;
					while (!(UCSR0A&(1<<RXC0)));
					dummy = UDR0;*/
					spi_io(0xfc);
					bufcnt = 0;
				} while (1);
				} while (0);

				mmc_speedtest_end();
				mmc_read_sector(fat_filedata.startsect, sharedmem);
				u32 temp = sect_cnt*512-8;
				sharedmem[4] = temp;
				sharedmem[5] = temp>>8;
				sharedmem[6] = temp>>16;
				sharedmem[7] = temp>>24;
				temp -= (44+8);
				sharedmem[40] = temp;
				sharedmem[41] = temp>>8;
				sharedmem[42] = temp>>16;
				sharedmem[43] = temp>>24;
				mmc_write_sector(fat_filedata.startsect, sharedmem);
				TCCR0B = (1 << CS02); // RC5 on
				EIFR = 1;
				EIMSK = 1;

				if (flags & FAT_FLAG) {
					fat_filedata.len = sect_cnt*512;
					fat_closefile();
				}
				else {
					file_pos[file_cnt] = last;
					last += (sect_cnt*512-1)/512+1;
				}
				file_num = file_cnt;
				file_cnt++;

				print_main();
				break;

			case DELETE:
				state = STOP;
				if (file_cnt == 0)
					break;
				lcd_string(DISP_DELETE, LINE_DELETE);	// Bildschirm anzeigen
				file_cnt--;
				if (flags & FAT_FLAG) {
					fat_delete_file();
				}
				else {
					mmc_write_start(file_pos[file_cnt]);
					mmc_write_byte(0);
					mmc_complete_write();
				}
				if (file_num == file_cnt)
					file_num--;
				delay_ms(1000);
				print_main();
				break;
		}
	}
}


//----------------------------------------------------------
// IRQ

// Timer for delta times
ISR(SIG_OUTPUT_COMPARE1A) {
/*	u08 dbyte;
	u16 data;
	PORTB &= ~PIN_ADCCS;
	SPDR=0x00;
	while (!(SPSR & (1<<SPIF)));
	dbyte = SPDR;
	SPDR = 0;
	data = (dbyte&7) << 13;
	while (!(SPSR & (1<<SPIF)));
	dbyte = SPDR;
	SPDR = 0;
	data |= dbyte << 5;
	while (!(SPSR & (1<<SPIF)));
	dbyte = SPDR;
	PORTB |= PIN_ADCCS;
	data |= dbyte >> 3;
	data -= 0x8000;
	tx_buf[tx_wrp++] = (u08)data;
	tx_buf[tx_wrp++] = (u08)(data>>8);*/
	volatile register u08 r16 asm("r16");
	PORTB &= ~PIN_ADCCS;
	SPDR=0x00;
	while (!(SPSR & (1<<SPIF)));
	r16 = SPDR;
	SPDR=0x00;
	asm("in r26, 0x2b");
	asm("ldi r27, 2");
	asm("st X+, r16");
	while (!(SPSR & (1<<SPIF)));
	r16 = SPDR;
	asm("st X+, r16");
	asm("out 0x2b, r26");
	PORTB |= PIN_ADCCS;
}


// Timer for real time clock
ISR(SIG_OUTPUT_COMPARE2A) {
}


ISR(SIG_WATCHDOG_TIMEOUT) {
	wdt_disable();
	flags |= KEY_FLAG;
	if (flags & IR_FLAG && rc5.flip == -1)
		flags &= ~IR_FLAG;
}
