#include <stdlib.h>
#include <inttypes.h>
#include <avr/io.h>
#include <avr/interrupt.h>

#define F_CPU 16000000UL

//ADC defines
#define AD_DRDY PA0
#define AD_RFS PA1
#define AD_TFS PA2 		//TFS mit A0 verbinden!
#define AD_MODE PA3

// Timer defines
#define MEASUREMENTFREQUENCY 100
#define TIMER0RELOAD 256 - ((F_CPU / 1024) / MEASUREMENTFREQUENCY)

#define UART_BAUD_CALC(UART_BAUD_RATE,F_OSC) ((F_CPU)/((UART_BAUD_RATE)*16L)-1)
#define UART_buffer_length 30

// UART vars
volatile uint8_t *UART_buffer = NULL;
volatile uint8_t UART_buffer_count = 0;
volatile uint8_t UART_command_length = 0;
// Messbox vars
//volatile static uint8_t ID;
volatile uint8_t MesswertNoH = 0;
volatile uint16_t MesswertNoL = 0;
volatile uint8_t AD_DATA[3];

// Methoden
void Timer0_start(void);
void Timer0_stop(void);
void AD_Init(void);
void AD_Read(void);
void UART_send_byte(uint8_t data);
void UART_send_string(const char *s);
void ProcessUARTbuffer(void);
void SendSensorData(void);
void StartMeasurement(void);
void StopMeasurement(void);


// Timer0 overflow
ISR(TIMER0_OVF_vect)			
{
	// Timer 0 nachladen
	TCNT0 = TIMER0RELOAD;
	// ADC wert einlesen
	AD_Read();
	// Sensordaten senden
	SendSensorData();
}

// UART receive interrupt handler
ISR(USART_RXC_vect)			
{
	uint8_t data = UDR;
	if (UART_buffer_count == 0)
	{
		switch (data)
		{
			case 0x01:
			{
				UART_command_length = 1;
				break;
			}
			case 0x02:
			{
				UART_command_length = 1;
				break;
			}
			case 0x03:
			{
				UART_command_length = 1;
				break;
			}
		}
	}
	UART_buffer_count++;
	if (UART_buffer_count <= UART_buffer_length)
	{
		UART_buffer[UART_buffer_count - 1] = data;
		if ((UART_command_length > 0) && (UART_buffer_count == UART_command_length))
		{
			ProcessUARTbuffer();
			UART_command_length = 0;
		}
	}	
	else UART_send_string("UART buffer full!!!");
}

// Timer 0 start
void Timer0_start(void)
{
	TCCR0 |= (1<<CS02) | (0<<CS01) | (1<<CS00); // Prescaler von 1024
	TCNT0 = TIMER0RELOAD;	// vorladen
	TIMSK |= (1<<TOIE0);	// Timer0 overflow interrupt aktivieren
}

// Timer 0 stoppen
void Timer0_stop(void)
{
	TCCR0 = 0;				// kein clocksource also ausgeschlatet
	TCNT0 = TIMER0RELOAD;	// vorladen
	TIMSK &= ~(1<<TOIE0);	// Timer0 overflow interrupt deaktivieren
}

void AD_Init(void)
{
	//AD_TFS, AD_RFS, AD_MODE = output, AD_DRDY = input
	DDRA |= (1<<DDA1) | (1<<DDA2) | (1<<DDA3);
	DDRA &= ~(1<<DDA0);
	//MOSI, SCK, SS = output, MISO = input	
	DDRB |= (1<<DDB4) | (1<<DDB5) | (1<<DDB7);	
	DDRB &= ~(1<<DDB6);
	//SPI Master, einschalten
	SPCR = (1<<SPE) | (1<<MSTR) ;
	//SPI CLK = FCPU / 16
	SPCR &= ~((1<<SPR1) | (1<<SPI2X));
	SPCR |= (1<<SPR0);
	//SPI Dataorder = MSB first
	SPCR &= ~(1<<DORD);
	//ADC in external clocking mode
	PORTA &= ~(1<<AD_MODE);
}

void AD_Read(void)
{
	//RFS, TFS = H
	PORTA |= (1<<AD_RFS) | (1<<AD_TFS);
	//warten bis DRDY = L
	while ( PINA & (1<<AD_DRDY) ) ;
	//RFS = L
	PORTA &= ~(1<<AD_RFS);
	uint8_t i = 0;
	for (i = 0; i << 3; i++)
	{
		//Dummy byte senden
		SPDR = 0xFF;
		//warten bis SPIF = H
		while (!(SPSR & (1<<SPIF)));
		//Datenbyte lesen
		AD_DATA[i] = SPDR;
	}
	PORTA |= (1<<AD_RFS);
	MesswertNoL++;
	if (MesswertNoL == 0) MesswertNoH++;
}

// UART initialisieren
void UART_init(uint16_t UART_BAUD_RATE)
{
	UBRRH=(uint8_t)(UART_BAUD_CALC(UART_BAUD_RATE,F_CPU)>>8); //baudrate (highbyte)
	UBRRL=(uint8_t)UART_BAUD_CALC(UART_BAUD_RATE,F_CPU);	//baudrate (lowbyte)
	UCSRB = (1<<RXEN) | (1<<TXEN) | (1<<RXCIE) | (0<<TXCIE); //aktiviere RxD & TxD & RxInt & deaktiviere TxInt
	UCSRC = (1<<URSEL)|(0<<UMSEL)|(0<<UPM1)|(0<<UPM0)|(0<<USBS)|(1<<UCSZ0)|(1<<UCSZ1) ;  // 8 Data Bits, 1 Stop Bit, No Parity
}

// Ein Byte per UART senden
void UART_send_byte(uint8_t data)
{
	while (!(UCSRA & (1<<UDRE)));  	//warten bis Senden mglich
    UDR = data;	
}

// Einen String per UART senden
void UART_send_string(const char *s)
{
	do
    {
        UART_send_byte (*s);
    }
    while (*s++);
}

// Inhalt des UART buffers verarbeiten
void ProcessUARTbuffer(void)	
{
	switch (UART_buffer[0])
	{
		case 0x01:
		{
			StartMeasurement();
			break;
		}
		case 0x02:
		{
			StopMeasurement();
			break;
		}
	}
	UART_buffer_count = 0;	
}

// Messung starten
void StartMeasurement(void)
{
	MesswertNoL = 0;
	MesswertNoH = 0;
	// Timer starten
	Timer0_start();
}

// Messung beeden
void StopMeasurement(void)
{
	// Timer stoppen
	Timer0_stop();
}

// Sensor daten senden
void SendSensorData(void)
{
	/*MesswertNoL++;
	if (MesswertNoL == 0) MesswertNoH++;*/
	UART_send_byte(MesswertNoH);
	uint8_t temp = (uint8_t)(MesswertNoL >>8);
	UART_send_byte(temp);
	temp = (uint8_t)(MesswertNoL);
	UART_send_byte(temp);
	// AD data
	UART_send_byte(AD_DATA[0]);
	UART_send_byte(AD_DATA[1]);
	UART_send_byte(AD_DATA[2]);
}


int main(void)
{
	AD_DATA[0] = 0;
	AD_DATA[1] = 0;
	AD_DATA[2] = 0;
	
	UART_buffer = (uint8_t*) malloc (UART_buffer_length);
	if (UART_buffer == NULL) return 0;
//	UART_init(38400);
	UART_init(57600);
	AD_Init();

	sei();

	while (1)
	{

	}

	return 1;

}
