/*******************************************************
 Author:					Manfred Langemann
 mailto:					Manfred.Langemann t t-online.de
 Begin of project:			05.09.2007
 Latest version generated:	12.04.2008
 Filename:					RFM12.c
 Description:    			Implementation of RFM12 Interface.
 							Transmit or Receive data via
							RFM12 transceiver module

 Description of code:
 
 This software package allows to transmit OR receive data
 via the RFM12 Transceiver module, which is designed for
 the 433 MHz ISM frequency band.

 It shall be noted, that simultaneous receive and transmit
 is not possible with this code.

 The data to be transmitted / received has to be provided
 by a pointer in an unit8_t array. The address of this array
 will be provided by calling RFM12_Init.

 The software is designed to transmit / receive up to
 512 data bytes in one stream.

 Transmitting the bytes shall be perfomred by calling RFM12_TX_Data.

 Receiving the data will be done by Interrupt 1.
 Use the receiving functions as described below.
 
 In order to ensure proper clock recovery of the RFM12 transceiver
 module, each data byte will be transmitted by 2 Manchester
 encoded byte. This of course increases the transmitted data amount
 by a factor of 2.

 To ensure transmission error checking, a cyclic redundancy check (CRC)
 is implemented, with a polynomial coefficient of 4129 = 0x1021.
 Manchester encoding will also be applied to this CRC word (2 bytes), which
 will be transmitted at the end of each data transmission.

 The software has been implemented on an ATMega32 with an 8 MHz crystal,
 but can easily be converted to other uControlers.

 The software can be configured with different RFM12 parameters.
 These are defined in this software as follows:

 	uint16_t RX_Bandwidth	= RFM12_RxBW200;	 200 kHz
	uint16_t LNA_Gain		= RFM12_LNA_6;		 -6 dB
	uint16_t Drssi			= RFM12_DRSSI_79;	 -79 dBm
	uint16_t Frequency		= RFM12FREQ(433.92); 433.92 MHz
	uint16_t Baudrate		= 9600; 			 9600 Bit/s
	uint16_t TX_Power		= RFM12_PWRdB_0;	 0 dBm
	uint16_t Frequency_Shift= RFM12_TxFS105;	 105 kHz
 
 For setting the parameters to other values, see the #define
 statements in the first part of the code.

 Code has been derived from www.mikrocontroller.net
  - Benedikt K.
  - Ulrich Radig

 The physical interface is as follows:

	RFM12 Module	AVR ATMega32
	----------------------------
	nSEL			Port D6
	SCK				Port D5
	SDI				Port D4
	SDO				Port D7
	nIRQ			INT1
	FSK is connected to +5V via a 1K resisant

 This configuration can be adapted to other configurations,
 see below under "Defines for the Port Configuration".

 Usage of this software is as follows:

 The Initiation:
 
 	uint8_t *Data; Pointer where RX / TX data will reside (max bytes = 512)
	Data = RFM12_Init (); Initiate the RFM12 software and set the AVR ports.
						  Has to be called only once and will return a pointer
						  to a RFM12 data buffer with size of 512 bytes.
						  Size is defined in RFM12.h with:
						  #define	RFM12_MAX_DATA_BYTES 512

 The Transmission chain:

 	i = RFM12_TX_Data (Data, Number);	Data is a pointer to an uint8_t array
									holding the data bytes to be transmitted.
									Number is the number of bytes to be transmitted
									(unit16_t value)
	if i == TRUE, then TX is OK
	if i == FALSE, then RFM module is in RX mode expecting data via INT1

 The Receiving chain:

	a) RFM12_Enable_IRQ_Receive_Data (void); Enables to receive data via INT1 interrupt.

	Check whether data has been received by:

	b) val = RFM12_RX_Status (....);

	if val == TRUE, then data has been received. Check the parameters in the 
	above function to get status of received bytes.

	For receiving the next data stream, restart at point a)

 *******************************************************/
#include <stdio.h>
#include <avr/io.h>
#include <avr/interrupt.h>

#include "General.h"
#include "Delay.h"
#include "RFM12.h"
/*
** Defines for the Port Configuration
*/
#define RFM12_CS_DDR	SBIT(DDRD,  6) // DDR
#define RFM12_CS		SBIT(PORTD, 6) // Output

#define RFM12_SCK_DDR	SBIT(DDRD,  5) // DDR
#define RFM12_SCK		SBIT(PORTD, 5) // Output

#define RFM12_SDI_DDR	SBIT(DDRD,  4) // DDR
#define RFM12_SDI		SBIT(PORTD, 4) // Output

#define RFM12_SDO_DDR	SBIT(DDRD,  7) // DDR
#define RFM12_SDO		SBIT(PIND,  7) // Input

#define RFM12_IRQ_DDR	SBIT(DDRD,  3) // DDR
#define RFM12_IRQ		SBIT(PIND,  3) // Input
/*
** Global private structure and variables
*/
struct RFM12_status_tag
	{
	uint16_t	Numbytes_Received;
	uint16_t	CRC_Word;
	uint16_t	CRC_Word_Received;
	uint8_t		RX_Completed;
	uint8_t		Max_Numbytes_Exceeded;
//	uint8_t		RX_Running;
	};
struct 	RFM12_status_tag RFM12_status;
volatile uint8_t	RFM_Data[RFM12_MAX_DATA_BYTES];
/*
** Defines for the RX baseband bandwdth
** in Receiver Control Command
** Default = (*)
*/
#define RFM12_RxBW400	1	// 400 kHz
#define RFM12_RxBW340	2	// 340 kHz
#define RFM12_RxBW270	3	// 270 kHz
#define RFM12_RxBW200	4	// 200 kHz (*)
#define RFM12_RxBW134	5	// 134 kHz
#define RFM12_RxBW67	6	// 67 kHz
/*
** Defines for the LNA Gain
** in Receiver Control Command
** Default = (*)
*/
#define RFM12_LNA_0		0	//  0 dB
#define RFM12_LNA_6		1	// -6 dB (*)
#define RFM12_LNA_14	2	// -14 dB
#define RFM12_LNA_20	3	// -20 dB
/*
** Defines for the DRSSI Threshold
** in Receiver Control Command
** Default = (*)
*/
#define RFM12_DRSSI_103	0	// -103 dBm
#define RFM12_DRSSI_97	1	// -97 dBm (*)
#define RFM12_DRSSI_91	2	// -91 dBm
#define RFM12_DRSSI_85	3	// -85 dBm
#define RFM12_DRSSI_79	4	// -79 dBm
#define RFM12_DRSSI_73	5	// -73 dBm
#define RFM12_DRSSI_67	6	// -67 dBm
#define	RFM12_DRSSI_61	7	// -61 dBm
/*
** Defines the RF output Power
** in AFC Command
** Default = (*)
*/
#define RFM12_PWRdB_0	0	//  0 dBm (*)
#define RFM12_PWRdB_3	1	// -3 dBm
#define RFM12_PWRdB_6	2	// -6 dBm
#define RFM12_PWRdB_9	3	// -9 dBm
#define RFM12_PWRdB_12	4	// -12 dBm
#define RFM12_PWRdB_15	5	// -15 dBm
#define RFM12_PWRdB_18	6	// -18 dBm
#define RFM12_PWRdB_21	7	// -21 dBm
/*
** Defines for the frequency shift
** in AFC Command
** Default = (*)
*/
#define RFM12_TxFS15	0	// 15 kHz
#define RFM12_TxFS30	1	// 30 kHz
#define RFM12_TxFS45	2	// 45 kHz
#define RFM12_TxFS60	3	// 60 kHz
#define RFM12_TxFS75	4	// 75 kHz
#define RFM12_TxFS90	5	// 90 kHz
#define RFM12_TxFS105	6	// 105 kHz (*)
#define RFM12_TxFS120	7	// 120 kHz
/*
** Public RFM12 Interface Functions Prototypes
** are defined in RFM12.h
**
**	volatile uint8_t *RFM12_Init (void);								// initialize module
**	uint8_t	RFM12_TX_Data (volatile uint8_t *Data, uint16_t Number);	// transmit number of bytes from array
**	void 	RFM12_Enable_IRQ_Receive_Data (void); 			// Enable to receive data via INT1 interrupt
**	uint8_t RFM12_RX_Status (uint16_t *Numbytes_Received,
**							 uint16_t *CRC_Word,
**							 uint16_t *CRC_Word_Received,
**							 uint8_t  *Max_Numbytes_Exceeded); // Return status of RX
**
** Define Private Function Prototypes
** Note: Public Function Prototypes are defined in file RFM12.h
*/
void 	 RFM12_Ready (void);
void 	 RFM12_Encode_Manchester (uint8_t byte, uint8_t *ms, uint8_t *ls);
uint8_t  RFM12_Decode_Manchester (uint8_t ms, uint8_t ls);
uint16_t RFM12_TXRX (uint16_t Wert);
uint16_t RFM12_CRC_Update (uint16_t crc, uint8_t byte);
uint8_t	 RFM12_RX_ManchesterBytes (void);
void 	 RFM12_TX_ManchesterBytes (uint8_t byte);
/*
** Define macro for calculating frequency value
** out of frequency in MHz.
** Physical frequency between:		430.09 to 439.7575 MHz
** Allowed ISM frequency between:	433.05 to 434.79 MHz
*/
#define RFM12FREQ(freq)	((freq-430.0)/0.0025)

/*******************************************************
 Public Function: RFM12_Init

 Purpose: Initialise the RFM12 RF link

 Input Parameter: None

 Return Value:
  	- Pointer to char address of RFM12 data buffer

*******************************************************/
volatile uint8_t *RFM12_Init (uint8_t RXTX_channel)
	{
	uint8_t		i;
	uint16_t	Frequency;
/*
** TX and RX Default (*) Parameters, see above
*/
	uint16_t RX_Bandwidth	= RFM12_RxBW200;	// 200 kHz
	uint16_t LNA_Gain		= RFM12_LNA_6;		// -6 dB
	uint16_t Drssi			= RFM12_DRSSI_79;	// -79 dBm
	uint16_t Baudrate		= 9600; 			// 9600 Bit/s
	uint16_t TX_Power		= RFM12_PWRdB_0;	// 0 dBm
	uint16_t Frequency_Shift= RFM12_TxFS105;	// 105 kHz
/*
** Set selected TXRX channel frequency
*/
	if (RXTX_channel == RXTX_CHANNEL_INDOOR_OUTDOOR) Frequency = RFM12FREQ(433.92);// 433.92 MHz
	if (RXTX_channel == RXTX_CHANNEL_INDOOR_DISPLAY) Frequency = RFM12FREQ(433.00);// 433.00 MHz
/*
** Set I/O Port directions
*/
	RFM12_CS_DDR  = 1;	// CS
	RFM12_SCK_DDR = 1;	// SCK
	RFM12_SDI_DDR = 1;	// SDI
	RFM12_SDO_DDR = 0;	// SDO
	RFM12_IRQ_DDR = 0;	// nINT
/*
** Set chip select to 1 and wait a while for POR
*/
	RFM12_CS = 1;
	for (i=0;i<50;i++)
		{
		Delay_ms (10);
		}
/*
** Set RFM12 internal TX and RX registers
*/
	RFM12_TXRX (0xC0E0); // AVR CLK: 10MHz, Low Battery = Off (2.2V)
	RFM12_TXRX (0x80D7); // Enable TX register and FIFO, crystal capacitor = 12p
	RFM12_TXRX (0xC2AB); // Data Filter: internal
	RFM12_TXRX (0xCA81); // Set FIFO mode
	RFM12_TXRX (0xE000); // disable wakeuptimer
	RFM12_TXRX (0xC800); // disable low duty cycle
	RFM12_TXRX (0xC4F7); // AFC settings: autotuning: -10kHz...+7,5kHz
/*
** Set TX/RX frequency
*/
	if (Frequency < 96)	Frequency = 96;		// Minimum = 430.2400 MHz
	if (Frequency > 3903) Frequency = 3903;	// Maximum = 439.7575 MHz
	RFM12_TXRX (0xA000 | Frequency);
/*
** Set Rx Bandwidth, LNA Gain and DRSSI threshold
*/
	RFM12_TXRX (0x9400|((RX_Bandwidth&7)<<5)|((LNA_Gain&3)<<3)|(Drssi&7));
/*
** Set Baudrate
** Baudrate = 344827.58621/(R+1)/(1+CS*7)
*/
	if (Baudrate < 664) Baudrate = 644;
	if (Baudrate < 5400)
		{
		RFM12_TXRX (0xC680|((43104/Baudrate)-1));
		}
	else
		{
		RFM12_TXRX (0xC600|((344828UL/Baudrate)-1));
		}
/*
** Set TX Power and Frequency Shift
*/
	RFM12_TXRX (0x9800|(TX_Power&7)|((Frequency_Shift&15)<<4));
/*
** Return pointer to RFM12 data buffer
*/
	return &RFM_Data[0];
	}
/*******************************************************
 Public Function: RFM12_Enable_IRQ_Receive_Data

 Purpose: Enable to receive RFM12 data
 			via interrupt at INT1

 Input Parameter: None

 Return Value: None

*******************************************************/
void RFM12_Enable_IRQ_Receive_Data (void)
	{
/*
** Reset global status parameters for data receiption
*/
	RFM12_status.Numbytes_Received 		= 0;
	RFM12_status.CRC_Word 				= 0;
	RFM12_status.CRC_Word_Received 		= 0;
	RFM12_status.RX_Completed 			= FALSE;
	RFM12_status.Max_Numbytes_Exceeded	= FALSE;
/*
** Set RX Parameters
*/
	RFM12_TXRX (0x82C8);	// RX on
	RFM12_TXRX (0xCA81);	// set FIFO mode
	RFM12_TXRX (0xCA83);	// enable FIFO
	RFM12_TXRX (0x0000);	// Read status / reset IRQ
/*
**	Set INT1 sense to low level interrupt
*/
	CLEAR_BIT (MCUCR, ISC10);
	CLEAR_BIT (MCUCR, ISC11);
/*
** Allow interrupts at INT1
*/
	SET_BIT (GICR, INT1);
/*
** Enable global interrupts
*/
	sei();

	return;
	}
/*******************************************************
 Public Function: RFM12_Disable_IRQ_Receive_Data

 Purpose: Disable to receive RFM12 data
 			via interrupt at INT1

 Input Parameter: None

 Return Value: None

*******************************************************/
void RFM12_Disable_IRQ_Receive_Data (void)
	{
/*
** Reset global status parameters for data receiption
*/
	RFM12_status.Numbytes_Received 		= 0;
	RFM12_status.CRC_Word 				= 0;
	RFM12_status.CRC_Word_Received 		= 0;
	RFM12_status.RX_Completed 			= FALSE;
	RFM12_status.Max_Numbytes_Exceeded	= FALSE;
/*
** Disable interrupts at INT1
*/
	CLEAR_BIT (GICR, INT1);
/*
** Enable global interrupts
*/
	return;
	}
/*******************************************************
 Public Function: RFM12_RX_Status

 Purpose: Get status of data receiption

 Input / Output Parameter:
 	uint16_t *Numbytes_Received:		The number of bytes received
 	uint16_t *CRC_Word:					The computed CRC word
 	uint16_t *CRC_Word_Received:		The received CRC word
 	uint8_t  *Max_Numbytes_Exceeded:	TRUE = More than RFM12_MAX_DATA_BYTES
										received
 Return Value:
 	TRUE: Receiption of data bytes is completed
 	FALSE: No data received

*******************************************************/
uint8_t RFM12_RX_Status    (uint16_t *Numbytes_Received,
							uint16_t *CRC_Word,
							uint16_t *CRC_Word_Received,
							uint8_t  *Max_Numbytes_Exceeded)
	{
	if (RFM12_status.RX_Completed)
		{
		*Numbytes_Received		= RFM12_status.Numbytes_Received;
		*CRC_Word				= RFM12_status.CRC_Word;
		*CRC_Word_Received		= RFM12_status.CRC_Word_Received;
		*Max_Numbytes_Exceeded	= RFM12_status.Max_Numbytes_Exceeded;
		}

	return RFM12_status.RX_Completed;
	}
/*******************************************************
 Private Function: SIGNAL (SIG_INTERRUPT1)

 Purpose: Interrupt function at INT1
 			for receiving RFM12 data

*******************************************************/
SIGNAL (SIG_INTERRUPT1)
	{
	uint8_t		k;
	uint16_t	i;
/*
** Initialise the CRC word
*/
	RFM12_status.CRC_Word = RFM12_CRC_Update (0,0);
/*
** Get the number of bytes to be received
*/
	k = RFM12_RX_ManchesterBytes ();
	RFM12_status.Numbytes_Received = (((uint16_t)RFM12_RX_ManchesterBytes ()) << 8) | (uint16_t) k;
	if (RFM12_status.Numbytes_Received > RFM12_MAX_DATA_BYTES)
		{
		RFM12_status.Max_Numbytes_Exceeded = TRUE;
		}
/*
** Receive the data:
** Note: All data bytes are transmitted as 2-byte manchester word
** - Receive the data byte
** - Then calculate/update the CRC word with this data byte
*/
	for (i=0;i<RFM12_status.Numbytes_Received;i++)
		{
		k = RFM12_RX_ManchesterBytes ();
		if (i<RFM12_MAX_DATA_BYTES) RFM_Data[i] = k;
		RFM12_status.CRC_Word = RFM12_CRC_Update (RFM12_status.CRC_Word, k);
		}
/*
** Receive the two CRC bytes, low byte first, then high byte
** Then convert to a CRC word.
*/
	k = RFM12_RX_ManchesterBytes ();
	RFM12_status.CRC_Word_Received = (((uint16_t)RFM12_RX_ManchesterBytes ()) << 8) | (uint16_t) k;
/*
** Disable FIFO
*/
	RFM12_TXRX (0xCA00);
/*
** Switch off receiver
*/
	RFM12_TXRX (0x8208);
/*
** Read status will reset nINT
*/
	RFM12_TXRX (0x0000);
/*
** Indicate, that RX is completed
*/
	RFM12_status.RX_Completed = TRUE;
/*
** Disable interrupts at INT1
*/
	CLEAR_BIT (GICR, INT1);

	}
/*******************************************************
 Public Function: RFM12_TX_Data

 Purpose: Transmit Data, disable any interrupt

 Input Parameter:
 	- uint8_t	*Data: Pointer to the data to transmit
	- uint16_t	Number: Number of bytes to transmit

 Return Value: uint8_t:
 	TRUE:  Data transmitted crrectly
 	FALSE: No data transmitted, because we are expecting to receive data
*******************************************************/
uint8_t RFM12_TX_Data (volatile uint8_t *Data, uint16_t Number)
	{
	uint8_t		temp;
	uint16_t	i;
	uint16_t	CRC_Word;
	uint8_t 	tmp_sreg;
/*
** Save status register and disable interrupts
*/
	tmp_sreg = SREG;
	cli();
/*
** If Number > RFM12_MAX_DATA_BYTES, reset to RFM12_MAX_DATA_BYTES
*/
	if (Number > RFM12_MAX_DATA_BYTES) Number = RFM12_MAX_DATA_BYTES;
/*
** Initialise the CRC word
*/
	CRC_Word = RFM12_CRC_Update (0,0);
/*
** Start the transmission
*/
	RFM12_TXRX (0x8238);	// TX on
	RFM12_Ready ();
	RFM12_TXRX (0xB8AA);	// Sync byte
	RFM12_Ready ();
	RFM12_TXRX (0xB8AA);	// Sync byte
	RFM12_Ready ();
	RFM12_TXRX (0xB8AA);	// Sync byte
	RFM12_Ready ();
	RFM12_TXRX (0xB82D);	// Prepare start of transmission
	RFM12_Ready ();
	RFM12_TXRX (0xB8D4);	// Start of transmission
/*
** Send the number of bytes to be tranbsmitted.
*/
	RFM12_TX_ManchesterBytes ((uint8_t) (Number & 0x00FF));
	RFM12_TX_ManchesterBytes ((uint8_t) (Number >> 8));
/*
** Send the data bytes
** Note: All data bytes will be transmitted as 2-byte manchester word
** - Calculate/update the CRC word
** - Encode data byte to two manchester bytes
** - Transmit the two bytes (ms first, ls second)
*/
	for (i=0;i<Number;i++)
		{
		temp = Data[i];
		CRC_Word = RFM12_CRC_Update (CRC_Word, temp);
		RFM12_TX_ManchesterBytes (temp);
		}
/*
** Send the CRC word, low byte first, then high byte.
** Note: also the two CRC bytes will be transmitted as
** 2-byte manchester word.
*/
	RFM12_TX_ManchesterBytes ((uint8_t) (CRC_Word & 0x00FF));
	RFM12_TX_ManchesterBytes ((uint8_t) (CRC_Word >> 8));
/*
** Send two dummy bytes, here sync bytes.
** These will not be considered during receive.
** Necessary to ensure proper function of RFM12 module
*/
	RFM12_Ready ();
	RFM12_TXRX (0xB8AA);
	RFM12_Ready ();
	RFM12_TXRX (0xB8AA);
/*
** Switch off Transmitter
*/
	RFM12_TXRX (0x8208);
/*
** Clear nIRQ
*/
	RFM12_TXRX (0x0000);
/*
** Function exit point, always with restoring the
** status register (also with global interrupt bit)
*/
	SREG = tmp_sreg;

	return TRUE;
	}
/*******************************************************
 Private Function: RFM12_TXRX

 Purpose: Set and get data from the RFM12 RF link

 Input Parameter:
 	- uint16_t	Wert: The value to be set

 Return Value:
 	- uint16_t	The returned value

*******************************************************/
uint16_t RFM12_TXRX (uint16_t Wert)
	{
	uint8_t		i;
/*
** Enable RFM12 chip by chip select = 0
*/
	RFM12_CS = 0;
/*
** Go through 16 bits
*/
	for (i=0;i<16;i++)
		{
		if (Wert&32768)
			{
			RFM12_SDI = 1;
			}
		else
			{
			RFM12_SDI = 0;
			}
		Wert = Wert << 1;
		if (RFM12_SDO == 1) Wert = Wert + 1;
/*
** Perform RFM12 clock pulse.
** Set clock pin to 1.
** Wait a while and then set to 0.
** Clock duration specified as > 25 ns
*/
		RFM12_SCK = 1;
		asm("nop");
		asm("nop");
		asm("nop");
		RFM12_SCK = 0;
		}
/*
** Disable RFM12 chip select
*/
	RFM12_CS = 1;
	return Wert;
	}
/*******************************************************
 Private Function: RFM12_Ready

 Purpose: Check, whether RFM12 is ready for next command

 Input Parameter: None

 Return Value: None

*******************************************************/
void RFM12_Ready (void)
	{
/*
** Set SDI and Chip select to 0
*/
	RFM12_CS = 0;
	RFM12_SDI = 0;
/*
** Wait for signal settings
*/
	asm("nop");
	asm("nop");
	asm("nop");
/*
** Wait until FIFO is ready
*/
	while (!RFM12_SDO);
/*
	//wait until FIFO ready or timeout
	while (!(RF_PIN & (1<<SDO)))
	{
		if ((timeout++)>100000){break;};
	};
*/
	}
/*******************************************************
 Private Function: RFM12_TX_ManchesterBytes

 Purpose: Encode data byte to two Manchester bytes.
 			Then send the two Manchester bytes via RFM12 module

 Input Parameter:
 	- unit8_t	byte: data byte

 Return Value: None

*******************************************************/
void RFM12_TX_ManchesterBytes (uint8_t byte)
	{
	uint8_t		ms_byte, ls_byte;

	RFM12_Encode_Manchester (byte, &ms_byte, &ls_byte);

	RFM12_Ready ();
	RFM12_TXRX (0xB800 | ms_byte);
	RFM12_Ready ();
	RFM12_TXRX (0xB800 | ls_byte);

	return;
	}
/*******************************************************
 Private Function: RFM12_RX_ManchesterBytes

 Purpose: Receive two manchester encoded bytes and 
 			decode these to one data byte.

 Input Parameter: None

 Return Value:
 	- uint8_t	The received data byte

*******************************************************/
uint8_t	RFM12_RX_ManchesterBytes (void)
	{
	uint8_t		ms_byte, ls_byte, byte;

	RFM12_Ready ();
	RFM12_TXRX (0x0000);
	ms_byte = (uint8_t) (RFM12_TXRX (0xB000) & 0x00FF);
	RFM12_Ready ();
	RFM12_TXRX (0x0000);
	ls_byte = (uint8_t) (RFM12_TXRX (0xB000) & 0x00FF);
	byte = RFM12_Decode_Manchester (ms_byte, ls_byte);

	return byte;
	}
/*******************************************************
 Private Function: RFM12_CRC_Update

 Purpose: Update the CRC word
 		  Polynomial coefficient = 4129 = 0x1021

 Input Parameter:
 	- uint16_t 	crc: The CRC word to be updated by the content of the byte
	- uint8_t	byte: The byte to be used to update the CRC word

 Return Value:
  	- uint16_t 	The updated CRC word

*******************************************************/
uint16_t RFM12_CRC_Update (uint16_t crc, uint8_t byte)
	{
	uint16_t	tmp;
	uint8_t		j;

	tmp = byte << 8;
	for (j=0;j<8;j++)
		{
		if((crc^tmp) & 0x8000)
			{
			crc = (crc<<1) ^ 0x1021;
			}
		else
			{
			crc = crc << 1;
			}
		tmp = tmp << 1;
        }

	return crc;
	}
/*******************************************************
 Private Function: RFM12_Encode_Manchester

 Purpose: Encode a byte two a two byte manchester code
 			This is necessary in order to have 
			enough level changes in a transmission
			to allow proper clock recovery.

 Input Parameter:
	- uint8_t	byte: The byte to be encoded
 	- uint8_t	*ms: Pointer to a uint8_t byte, where the most significant
				    bytes of the encoded Manchester word will be stored
 	- uint8_t	*ls: Pointer to a uint8_t byte, where the least significant
				 	bytes of the encoded Manchester word will be stored

 Return Value: None

	Encoding Algorithm in C:
	------------------------
	ms = (c & 0xAA) | ((~c & 0xAA)>>1); encode odd bits w/ their complements
	ls = (c & 0x55) | ((~c & 0x55)<<1); now even bits
	return ((ushort)ms << 8) | ls;

*******************************************************/
void RFM12_Encode_Manchester (uint8_t byte, uint8_t *ms, uint8_t *ls)
	{
	*ms = ((byte&0xAA) | (((~byte)&0xAA)>>1));
	*ls = ((byte&0x55)|(((~byte)&0x55)<<1));
	return;
	}
/*******************************************************
 Private Function: RFM12_Decode_Manchester

 Purpose: Decode a two byte encoded manchester code to a
 			one byte value
 			This is necessary in order to have 
			enough level chages in a transmission
			to allow proper clock recovery.

 Input Parameter:
 	- uint8_t	ms: Most significant byte of manchester word
 	- uint8_t	ls: Least significant byte of manchester word

 Return Value:
 	- uint8_t	The decoded data byte

	Decoding Algorithm in C:
	------------------------
	ms = (uchar)(mc >> 8);      get ms byte from short
	if ( (ms & 0xAA) ^ ((~ms & 0x55) << 1) ) then error
	ms &= 0xAA;         		otherwise, valid data, so just clear complements
	ls = (uchar)mc;         	now test even bits...
	if ( (ls & 0x55) ^ ((~ls & 0xAA)>>1) ) then error
	ls &= 0x55;         		valid, so clear complements
	return ms | ls;         	OR original odd and even bits back together

 Note: Error checking statements are not used

*******************************************************/
uint8_t RFM12_Decode_Manchester (uint8_t ms, uint8_t ls)
	{
	uint8_t		byte;

//	if ( (byte & 0xAA) ^ (((~byte)& 0x55) << 1) )
//		printf("Decode Error %d\n",byte);
	byte = ms & 0xAA;         	// otherwise, valid data, so just clear complements

//	if ( (byte & 0x55) ^ (((~byte)& 0xAA) >> 1) )
//		printf("Decode Error %d\n",byte);
	
	return byte | (((uint8_t)ls) & 0x55);         	// valid, so clear complements 
	}

