/*
 * file:    init.c
 * purpose: initialize device
 */
#include <inttypes.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include "Config.h"
#include "DDS.h"
#include "Data.h"
#include "SPI.h"
#include "UI.h"


void init() {
   __asm volatile ("clt");
   work.step        = 0;
   work.mode        = 0;
   work.level       = 0;
   work.wave        = 0;

   DDR_RIGHT_ENCODE &= (uint8_t) ~(1 << PIN_RIGHT_ENCODE);
   DDR_LEFT_ENCODE  &= (uint8_t) ~(1 << PIN_LEFT_ENCODE);
   DDR_PUSH_ENCODE  &= (uint8_t) ~(1 << PIN_PUSH_ENCODE);

   TCCR1B = 1 << CS12 | 1 << CS11 | 1 << CS10;             // T1 is clock source for 16bit timer (frequency counting)

   spiInit();
   listen2User(1);
}


/*
 * pin change interrupt is only used when user interface is switched off
 */
void listen2User(uint8_t really) {
   if (really) {
      TCCR0B = 1 << CS01;                                  // start timer ...
      EIMSK &= (uint8_t) ~(1 << INT1 | 1 << INT0);
   } else {
      TCCR0B = 0;                                          // stop timer ...
      EIMSK |= 1 << INT1 | 1 << INT0;
   }
}


/*
 * setup device to work in selected mode
 */
void setup() {
   listen2User(0);
   if (work.mode) {                                        // setup function generator
      DDR_DAC = 0xFF;
   } else {                                                // setup frequency counter
  
   } 
}


/*
 * Pin change interrupt - is activated when user interface is switched off,
 * so pressing any button or turning the wheel will reenable the user interface.
 */
ISR(wakeup) {
   listen2User(1);
}

void __attribute__((naked)) PCINT0_vect(void) {
   __asm volatile ("clt"            "\n\t"
                   "rjmp wakeup");
}


void __attribute__((naked)) PCINT1_vect(void) {
   __asm volatile ("clt"            "\n\t"
                   "rjmp wakeup");
}


void __attribute__((naked)) PCINT2_vect(void) {
   __asm volatile ("clt"            "\n\t"
                   "rjmp wakeup");
}


/*
 * gray-decoding by Peter Dannegger (mikrocontroller.net)
 */
ISR(TIMER0_OVF_vect) {
   static int8_t enc_last = 0x01;
   struct sm * pUI = ui + work.level;
   int8_t i=0;

    
   if (PORT_RIGHT_ENCODE & 1 << PIN_RIGHT_ENCODE) 
      i = 1;

   if (PORT_LEFT_ENCODE & 1 << PIN_LEFT_ENCODE) 
      i ^= 3;        // gray to binary


   i -= enc_last;

   if (i & 1) {
      enc_last += i;

      if (pUI->type == CHOICE_STATE || pUI->type == COUNTER_STATE) {
         *pUI->pVar += (i & 2) - 1;

         if (*pUI->pVar < 0)        *pUI->pVar = pUI->max;
         if (*pUI->pVar > pUI->max) *pUI->pVar = 0;
      }
   }

   if (PORT_PUSH_ENCODE & (1 << PIN_PUSH_ENCODE)) work.btPush++;
   else                                           work.btPush = 0;
}


