
#include <avr/io.h>
#include <avr/pgmspace.h>

#ifndef GLCD_H
#define GLCD_H

#include "glcd.inc"


#ifdef USE_HIGHCOLOR
typedef uint16_t glcdColor_t;         // 65535 colors
#else
typedef uint8_t glcdColor_t;          // 256 colors
#endif
typedef uint8_t glcdCoord_t;          // coordinates, ONLY byte coords supported without rewriting of lib

typedef uint8_t *glcdFontData_t;      // pointer to font datas
typedef uint8_t (*glcdFontFunc_t)(const glcdFontData_t index); // custom font data read callback, thus we can support ae. I2C EEPROM
typedef uint8_t (*glcdCharFunc_t)(char c);

typedef struct {
    glcdCoord_t X;
    glcdCoord_t Y;
} glcdPoint_t;

typedef struct {
    glcdCoord_t X1;
    glcdCoord_t Y1;
    glcdCoord_t X2;
    glcdCoord_t Y2;
} glcdRect_t;

typedef union {
    uint8_t All;
    struct {
    uint8_t AutoLineFeed:1,          // font routines should use implizit linefeed and wrap around glcd_Window
            FixedFont:1,             // == 0 fonts painted proportional, == 1 fonts painted in fixed pitch
            Clipping:1;              // activate clipping
    }        
} glcdFlags_t;


extern uint8_t         glcd_MemCtrl;  // display RAM access and degree of rotation
extern glcdFlags_t     glcd_Flags;    // glcd lib flags
extern glcdColor_t     glcd_Colors[1 << COLOR_TABLE_BITS];

#ifdef USE_CLIPPING
extern glcdRect_t      glcd_Clip;            // clipping coordinates, only paint operation in this rect are visible
#endif


extern glcdPoint_t     glcd_Cursor;          // current cursor for painting characters
extern glcdRect_t      glcd_Window;          // text output window, AutoLineFeed checks this
extern char            glcd_FontFirstChar;   // first defined chararcter in selected font
extern char            glcd_FontLastChar;    // last defined chararcter in selected font
extern uint8_t         glcd_FontWidth;       // character width for fixed pitch of selected font
extern uint8_t         glcd_FontHeight;      // character height of selected font
extern uint8_t         glcd_FontBitsPixel;   // color depth of selected font, if MSB is set the font is compressed
extern glcdFontFunc_t  glcd_FontRead;        // user definied read callback of font data, if NULL it's assumed font is stored in flash
extern glcdFontData_t  glcd_FontData;        // pointer to font data, a read callback can store own values here
                                             // the lib access trough this pointer ONLY if glcd_FontRead == NULL to flash
                                             // otherwise this value is only a paramter to the user callback


extern void glcdDisplayCommand(uint8_t cmd); // send a command to display
extern void glcdDisplayData(uint8_t data);   // send data to display
extern void glcdDisplayBuffer(uint8_t *buffer, uint8_t buffersize, uint8_t flashstored); // send a buffer of bytes as data to the display

extern void glcdDisplayInit(void);           // initialize the display, MUST be normaly NEVER called, it's automaticaly called by the initialization code
extern void glcdDisplayOn(void);             // switch display on, booster on, spleep out
extern void glcdDisplayOff(void);            // switch display off, booster off, sleep in
extern void glcdDisplaySetColors(uint8_t *red, uint8_t *green, uint8_t *blue, uint8_t flashstored); // setup colortable in 256 color mode

extern void glcdWait(uint16_t ms);           // wait some milliseconds, XTAL must be right defined

extern void glcdClearScreen(uint8_t ignoreclipping);
extern void glcdSetOrientation(uint8_t orientation); // degree 0,90,180,270 -> 0,1,2,3
extern void glcdSetAddr(glcdCoord_t x1, glcdCoord_t y1, glcdCoord_t x2, glcdCoord_t y2);   // set RAM access pointer of display
extern void glcdPutPixel(glcdColor_t color);                                               // put a pixel, glcdSetAddr() MUST be prior called
extern void glcdSetPixel(glcdCoord_t x, glcdCoord_t y, glcdColor_t color);                 // set a pixel
extern void glcdFillRect(glcdCoord_t x1, glcdCoord_t y1, glcdCoord_t x2, glcdCoord_t y2, glcdColor_t color);
extern void glcdRectangle(glcdCoord_t x1, glcdCoord_t y1, glcdCoord_t x2, glcdCoord_t y2); // paint border with glcdFgColor and fill background with glcdBkColor
extern void glcdFrame(glcdCoord_t x1, glcdCoord_t y1, glcdCoord_t x2, glcdCoord_t y2); // paint a frame, colors glcdFgColor, glcdBkColor, glcdFrColor, glcdShColor used
extern void glcdLine(glcdCoord_t x1, glcdCoord_t y1, glcdCoord_t x2, glcdCoord_t y2);
extern void glcdCircle(glcdCoord_t x, glcdCoord_t y, glcdCoord_t r);                   // circle at x,y with radius r
extern void glcdEllipse(glcdCoord_t x, glcdCoord_t y, glcdCoord_t a, glcdCoord_t b);   // ellipse at x,y with radius a,b
extern void glcdSwapClipRect(void); // swaps glcd_Clip.X1 <-> glcd_Clip.Y1, glcd_Clip_X2 <-> glcd_Clip.Y2

extern void glcdSelectFont(PGM_VOID_P data, glcdFontFunc_t callback); // select a font
extern void glcdNewLine(void);                                        // <CR><LF> in glcd_Window of glcd_Cursor
extern uint8_t glcdDrawChar(char c);                                  // draw a character, returns width of char, if any painted
extern uint8_t glcdCharWidth(char c);                                 // return with of char

extern uint16_t glcdCharsCallback(char *addr, uint8_t InFlashStored, glcdCharFunc_t callback);;
extern uint16_t glcdPrint(char *addr, uint8_t InFlashStored);         // print a null terminated string stored in flash or ram
extern uint16_t glcdCharsWidth(char *addr, uint8_t InFlashStored);    // compute with of null terminated string stored in flash or ram
extern int glcdPut(char c);                                           // handler for fdevopen()


#define glcdBkColor glcd_Colors[0]
#define glcdFgColor glcd_Colors[1]
#define glcdFrColor glcd_Colors[2]
#define glcdShColor glcd_Colors[3]
#define glcdSetColor(index, color) {glcd_Colors[index] = color;}
#define glcdSetBkColor(color) {glcdSetColor(0, color);}
#define glcdSetFgColor(color) {glcdSetColor(1, color);}
#define glcdSetFrColor(color) {glcdSetColor(2, color);}
#define glcdSetShColor(color) {glcdSetColor(3, color);}
#define glcdSetColors(fgColor, bkColor) {glcdSetFgColor(fgColor); glcdSetBkColor(bkColor);}

#define glcdSetRect(r,x1,y1,x2,y2) {r.X1 = x1; r.Y1 = y1; r.X2 = x2; r.Y2 = y2;}
#define glcdMoveTo(x,y) {glcd_Cursor.X = x; glcd_Cursor.Y = y;}

#endif
