/*
 * This file is part of lcd library for ssd1306/sh1106 oled-display.
 *
 * lcd library for ssd1306/sh1106 oled-display is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or any later version.
 *
 * lcd library for ssd1306/sh1106 oled-display is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Diese Datei ist Teil von lcd library for ssd1306/sh1106 oled-display.
 *
 * lcd library for ssd1306/sh1106 oled-display ist Freie Software: Sie können es unter den Bedingungen
 * der GNU General Public License, wie von der Free Software Foundation,
 * Version 3 der Lizenz oder jeder späteren
 * veröffentlichten Version, weiterverbreiten und/oder modifizieren.
 *
 * lcd library for ssd1306/sh1106 oled-display wird in der Hoffnung, dass es nützlich sein wird, aber
 * OHNE JEDE GEWÄHRLEISTUNG, bereitgestellt; sogar ohne die implizite
 * Gewährleistung der MARKTFÄHIGKEIT oder EIGNUNG FÜR EINEN BESTIMMTEN ZWECK.
 * Siehe die GNU General Public License für weitere Details.
 *
 * Sie sollten eine Kopie der GNU General Public License zusammen mit diesem
 * Programm erhalten haben. Wenn nicht, siehe <http://www.gnu.org/licenses/>.
 *
 *  lcd.h
 *
 *  Created by Michael Köhler on 22.12.16.
 *  Copyright 2016 Skie-Systems. All rights reserved.
 *
 *  lib for OLED-Display with ssd1306/sh1106-Controller
 *  first dev-version only for I2C-Connection
 *  at ATMega328P like Arduino Uno
 *
 *  at GRAPHICMODE lib needs SRAM for display
 *  DISPLAY-WIDTH * DISPLAY-HEIGHT + 2 bytes
 *
 *  14.08.2018 Harry L.
 *  added STM32/HAL Support
 *
 *  i2c.c and i2c.h renamed to i2c_master.c and i2c_master.h
 *  to avoid conflicts with foreign i2c-routines
 *
 *  replaced YES/NO with TRUE/FALSE
 *
 *  corrected NORMAL/INVERSE switching
 *
 *  corrected lcd_init()
 *  there is no way to write to PROGMEM
 *  LCD_DISP_ON moved to init_sequence
 *
 */

#ifdef __ARM_ARCH
#include "main.h"
#include "stm32f1xx_hal.h"
#include "i2c.h"
#endif

#include "lcd.h"
#include <string.h>

uint8_t cursorPosition = 0;

#if defined GRAPHICMODE
#include <stdlib.h>

static struct			// Framebuffer only needed or GRAPHICMODE
{
#ifdef __ARM_ARCH
  uint8_t prefix;
#endif
  uint8_t buf[DISPLAYSIZE];
} displayBuffer;
#endif

uint16_t actualIndex = 0;

#ifdef __ARM_ARCH		// some variables only needed on ARM

/*
 * the I²C-handle used y the driver
 */
static I2C_HandleTypeDef *oled_hi2c;

/*
 * signal the DMA-callback that a custom command is waiting for insertion
 */
static volatile uint8_t cmdPending;

/*
 * temporary buffer for inserting custom OLED-commands
 * in DMA-transfer-sequence
 */
static struct
{
  uint8_t cnt;
  uint8_t prefix;
  uint8_t cmd[2];
} cmdBuf;
#endif

typedef struct
{
  char c;		// char-code
  uint8_t idx;		// index in font-table
} fnt_map_t;

// font-map for extra-chars
const fnt_map_t fnt_map[] PROGMEM =
  {
    { 132, 97 },		// ä
    { 148, 99 },		// ö
    { 129, 95 },		// ü
    { 142, 98 },		// Ä
    { 153, 100 },		// Ö
    { 154, 96 },		// Ü
    { 248, 101 },		// °
    { 225, 102 },		// ß
//    { 230, 103 },		// µ		* not yet implemented *
//    { 234, 104 },		// Omega (Ohm)	* not yet implemented *
    { 0, 0xff }			// end of table
  };


/* Standard ASCII 6x8 font */
static const uint8_t oled_font6x8[][6] PROGMEM =
  {
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, // sp
	{ 0x00, 0x00, 0x00, 0x2f, 0x00, 0x00 }, // !
	{ 0x00, 0x00, 0x07, 0x00, 0x07, 0x00 }, // "
	{ 0x00, 0x14, 0x7f, 0x14, 0x7f, 0x14 }, // #
	{ 0x00, 0x24, 0x2a, 0x7f, 0x2a, 0x12 }, // $
	{ 0x00, 0x62, 0x64, 0x08, 0x13, 0x23 }, // %
	{ 0x00, 0x36, 0x49, 0x55, 0x22, 0x50 }, // &
	{ 0x00, 0x00, 0x05, 0x03, 0x00, 0x00 }, // '
	{ 0x00, 0x00, 0x1c, 0x22, 0x41, 0x00 }, // (
	{ 0x00, 0x00, 0x41, 0x22, 0x1c, 0x00 }, // )
	{ 0x00, 0x14, 0x08, 0x3E, 0x08, 0x14 }, // *
	{ 0x00, 0x08, 0x08, 0x3E, 0x08, 0x08 }, // +
	{ 0x00, 0x00, 0x00, 0xA0, 0x60, 0x00 }, // ,
	{ 0x00, 0x08, 0x08, 0x08, 0x08, 0x08 }, // -
	{ 0x00, 0x00, 0x60, 0x60, 0x00, 0x00 }, // .
	{ 0x00, 0x20, 0x10, 0x08, 0x04, 0x02 }, // /
	{ 0x00, 0x3E, 0x51, 0x49, 0x45, 0x3E }, // 0
	{ 0x00, 0x00, 0x42, 0x7F, 0x40, 0x00 }, // 1
	{ 0x00, 0x42, 0x61, 0x51, 0x49, 0x46 }, // 2
	{ 0x00, 0x21, 0x41, 0x45, 0x4B, 0x31 }, // 3
	{ 0x00, 0x18, 0x14, 0x12, 0x7F, 0x10 }, // 4
	{ 0x00, 0x27, 0x45, 0x45, 0x45, 0x39 }, // 5
	{ 0x00, 0x3C, 0x4A, 0x49, 0x49, 0x30 }, // 6
	{ 0x00, 0x01, 0x71, 0x09, 0x05, 0x03 }, // 7
	{ 0x00, 0x36, 0x49, 0x49, 0x49, 0x36 }, // 8
	{ 0x00, 0x06, 0x49, 0x49, 0x29, 0x1E }, // 9
	{ 0x00, 0x00, 0x36, 0x36, 0x00, 0x00 }, // :
	{ 0x00, 0x00, 0x56, 0x36, 0x00, 0x00 }, // ;
	{ 0x00, 0x08, 0x14, 0x22, 0x41, 0x00 }, // <
	{ 0x00, 0x14, 0x14, 0x14, 0x14, 0x14 }, // =
	{ 0x00, 0x00, 0x41, 0x22, 0x14, 0x08 }, // >
	{ 0x00, 0x02, 0x01, 0x51, 0x09, 0x06 }, // ?
	{ 0x00, 0x32, 0x49, 0x59, 0x51, 0x3E }, // @
	{ 0x00, 0x7C, 0x12, 0x11, 0x12, 0x7C }, // A
	{ 0x00, 0x7F, 0x49, 0x49, 0x49, 0x36 }, // B
	{ 0x00, 0x3E, 0x41, 0x41, 0x41, 0x22 }, // C
	{ 0x00, 0x7F, 0x41, 0x41, 0x22, 0x1C }, // D
	{ 0x00, 0x7F, 0x49, 0x49, 0x49, 0x41 }, // E
	{ 0x00, 0x7F, 0x09, 0x09, 0x09, 0x01 }, // F
	{ 0x00, 0x3E, 0x41, 0x49, 0x49, 0x7A }, // G
	{ 0x00, 0x7F, 0x08, 0x08, 0x08, 0x7F }, // H
	{ 0x00, 0x00, 0x41, 0x7F, 0x41, 0x00 }, // I
	{ 0x00, 0x20, 0x40, 0x41, 0x3F, 0x01 }, // J
	{ 0x00, 0x7F, 0x08, 0x14, 0x22, 0x41 }, // K
	{ 0x00, 0x7F, 0x40, 0x40, 0x40, 0x40 }, // L
	{ 0x00, 0x7F, 0x02, 0x0C, 0x02, 0x7F }, // M
	{ 0x00, 0x7F, 0x04, 0x08, 0x10, 0x7F }, // N
	{ 0x00, 0x3E, 0x41, 0x41, 0x41, 0x3E }, // O
	{ 0x00, 0x7F, 0x09, 0x09, 0x09, 0x06 }, // P
	{ 0x00, 0x3E, 0x41, 0x51, 0x21, 0x5E }, // Q
	{ 0x00, 0x7F, 0x09, 0x19, 0x29, 0x46 }, // R
	{ 0x00, 0x46, 0x49, 0x49, 0x49, 0x31 }, // S
	{ 0x00, 0x01, 0x01, 0x7F, 0x01, 0x01 }, // T
	{ 0x00, 0x3F, 0x40, 0x40, 0x40, 0x3F }, // U
	{ 0x00, 0x1F, 0x20, 0x40, 0x20, 0x1F }, // V
	{ 0x00, 0x3F, 0x40, 0x38, 0x40, 0x3F }, // W
	{ 0x00, 0x63, 0x14, 0x08, 0x14, 0x63 }, // X
	{ 0x00, 0x07, 0x08, 0x70, 0x08, 0x07 }, // Y
	{ 0x00, 0x61, 0x51, 0x49, 0x45, 0x43 }, // Z
	{ 0x00, 0x00, 0x7F, 0x41, 0x41, 0x00 }, // [
	{ 0x00, 0x55, 0x2A, 0x55, 0x2A, 0x55 }, // backslash
	{ 0x00, 0x00, 0x41, 0x41, 0x7F, 0x00 }, // ]
	{ 0x00, 0x04, 0x02, 0x01, 0x02, 0x04 }, // ^
	{ 0x00, 0x40, 0x40, 0x40, 0x40, 0x40 }, // _
	{ 0x00, 0x00, 0x01, 0x02, 0x04, 0x00 }, // '
	{ 0x00, 0x20, 0x54, 0x54, 0x54, 0x78 }, // a
	{ 0x00, 0x7F, 0x48, 0x44, 0x44, 0x38 }, // b
	{ 0x00, 0x38, 0x44, 0x44, 0x44, 0x20 }, // c
	{ 0x00, 0x38, 0x44, 0x44, 0x48, 0x7F }, // d
	{ 0x00, 0x38, 0x54, 0x54, 0x54, 0x18 }, // e
	{ 0x00, 0x08, 0x7E, 0x09, 0x01, 0x02 }, // f
	{ 0x00, 0x18, 0xA4, 0xA4, 0xA4, 0x7C }, // g
	{ 0x00, 0x7F, 0x08, 0x04, 0x04, 0x78 }, // h
	{ 0x00, 0x00, 0x44, 0x7D, 0x40, 0x00 }, // i
	{ 0x00, 0x40, 0x80, 0x84, 0x7D, 0x00 }, // j
	{ 0x00, 0x7F, 0x10, 0x28, 0x44, 0x00 }, // k
	{ 0x00, 0x00, 0x41, 0x7F, 0x40, 0x00 }, // l
	{ 0x00, 0x7C, 0x04, 0x18, 0x04, 0x78 }, // m
	{ 0x00, 0x7C, 0x08, 0x04, 0x04, 0x78 }, // n
	{ 0x00, 0x38, 0x44, 0x44, 0x44, 0x38 }, // o
	{ 0x00, 0xFC, 0x24, 0x24, 0x24, 0x18 }, // p
	{ 0x00, 0x18, 0x24, 0x24, 0x18, 0xFC }, // q
	{ 0x00, 0x7C, 0x08, 0x04, 0x04, 0x08 }, // r
	{ 0x00, 0x48, 0x54, 0x54, 0x54, 0x20 }, // s
	{ 0x00, 0x04, 0x3F, 0x44, 0x40, 0x20 }, // t
	{ 0x00, 0x3C, 0x40, 0x40, 0x20, 0x7C }, // u
	{ 0x00, 0x1C, 0x20, 0x40, 0x20, 0x1C }, // v
	{ 0x00, 0x3C, 0x40, 0x30, 0x40, 0x3C }, // w
	{ 0x00, 0x44, 0x28, 0x10, 0x28, 0x44 }, // x
	{ 0x00, 0x1C, 0xA0, 0xA0, 0xA0, 0x7C }, // y
	{ 0x00, 0x44, 0x64, 0x54, 0x4C, 0x44 }, // z
	{ 0x00, 0x00, 0x08, 0x77, 0x41, 0x00 }, // {
	{ 0x00, 0x00, 0x00, 0x63, 0x00, 0x00 }, // ¦
	{ 0x00, 0x00, 0x41, 0x77, 0x08, 0x00 }, // }
	{ 0x00, 0x08, 0x04, 0x08, 0x08, 0x04 }, // ~
	{ 0x00, 0x3D, 0x40, 0x40, 0x20, 0x7D }, // ü
	{ 0x00, 0x3D, 0x40, 0x40, 0x40, 0x3D }, // Ü
	{ 0x00, 0x21, 0x54, 0x54, 0x54, 0x79 }, // ä
	{ 0x00, 0x7D, 0x12, 0x11, 0x12, 0x7D }, // Ä
	{ 0x00, 0x39, 0x44, 0x44, 0x44, 0x39 }, // ö
	{ 0x00, 0x3D, 0x42, 0x42, 0x42, 0x3D }, // Ö
	{ 0x00, 0x02, 0x05, 0x02, 0x00, 0x00 }, // °
	{ 0x00, 0x7E, 0x01, 0x49, 0x55, 0x73 }  // ß
  };

/*
 *
 * Initialization Sequence
 *
 */
const uint8_t init_sequence[] PROGMEM =
  {
    OLED_DISPLAYOFF,		// Display OFF (sleep mode)
    OLED_MEMORYMODE, 0b00,	// Set Memory Addressing Mode
				// 00=Horizontal Addressing Mode; 01=Vertical Addressing Mode;
				// 10=Page Addressing Mode (RESET); 11=Invalid
    OLED_PAGESTART,		// Set Page Start Address for Page Addressing Mode, 0-7
    OLED_COMSCANDEC,		// Set COM Output Scan Direction
    OLED_SETLOWCOLUMN,		// --set low column address
    OLED_SETHIGHCOLUMN,		// --set high column address
    OLED_SETSTARTLINE,		// --set start line address
    OLED_SETCONTRAST, 0x3F,	// Set contrast control register
    (OLED_SEGREMAP | 1),	// Set Segment Re-map. A0=address mapped; A1=address 127 mapped.
    OLED_NORMALDISPLAY,		// Set display mode. A6=Normal; A7=Inverse
    OLED_SETMULTIPLEX, 0x3F,	// Set multiplex ratio(1 to 64)
    OLED_DISPLAYALLON_RESUME,	// Output RAM to Display
				// 0xA4=Output follows RAM content; 0xA5,Output ignores RAM content
    OLED_SETDISPLAYOFFSET, 0x00,// Set display offset. 00 = no offset
    OLED_SETDISPLAYCLOCKDIV,	// --set display clock divide ratio/oscillator frequency
    0xF0,			// --set divide ratio
    OLED_SETPRECHARGE, 0x22,	// Set pre-charge period
    OLED_SETCOMPINS, 0x12,	// Set com pins hardware configuration
    OLED_SETVCOMDETECT,		// --set vcomh
    OLED_MEMORYMODE,		// 0x20,0.77xVcc
    OLED_CHARGEPUMP, 0x14	// Set DC-DC enable

    };

/*
 * AVR specific I²C-functions
 */
#ifndef __ARM_ARCH
void lcd_command(uint8_t cmd[], uint8_t size)
  {
    i2c_start(OLED_I2C_ADR);
    i2c_byte(OLED_CMD_PREFIX);	// 0x80 for command, 0x40 for data
    for (uint8_t i=0; i<size; i++)
      {
	i2c_byte(cmd[i]);
      }
    i2c_stop();
  }

void lcd_data(uint8_t data[], uint16_t size)
  {
    i2c_start(OLED_I2C_ADR);
    i2c_byte(OLED_DTA_PREFIX);	// 0x00 for command, 0x40 for data
    for (uint16_t i = 0; i<size; i++)
      {
	i2c_byte(data[i]);
      }
    i2c_stop();
  }
#endif

/*
 * initialize the display
 * on STM32 you hav to put the I²C-handle inteh parameter
 */
#ifdef __ARM_ARCH
void
lcd_init (I2C_HandleTypeDef *hi2c)
{
  oled_hi2c = hi2c;
  displayBuffer.prefix = OLED_DTA_PREFIX;
  cmdBuf.prefix = OLED_CMD_PREFIX;
  cmdPending = FALSE;
  // enshure that I2C is ready
  while (HAL_I2C_GetState (oled_hi2c) != HAL_I2C_STATE_READY)
    ;
  // send init-sequence
  HAL_I2C_Master_Transmit_DMA (oled_hi2c, OLED_I2C_ADR,
			       (uint8_t *) &init_sequence,
			       sizeof(init_sequence));

#else
void lcd_init(uint8_t dispAttr)
    {
      i2c_init();
      uint8_t commandSequence[sizeof(init_sequence)+1];
      memcpy_P(commandSequence, init_sequence, sizeof(init_sequence));
      commandSequence[sizeof(init_sequence)]=(dispAttr);
      lcd_command(commandSequence, sizeof(commandSequence));
#endif

  // clear FB
  lcd_clrscr ();
}

#ifdef __ARM_ARCH
/*
 * Callback function is called on any successfull DMA-trasfer
 */
void
HAL_I2C_MasterTxCpltCallback (I2C_HandleTypeDef *hi2c)
{
  if (hi2c == oled_hi2c)
    {
      if (cmdPending)	//inject command
	{
	  HAL_I2C_Master_Transmit (hi2c, OLED_I2C_ADR,
				   (uint8_t *) &cmdBuf.prefix, cmdBuf.cnt, 3);
	  cmdPending = FALSE;
	}
#ifndef SH1106
      HAL_I2C_Master_Transmit_DMA (hi2c, OLED_I2C_ADR,
				   (uint8_t *) &displayBuffer,
				   sizeof(displayBuffer));
#else
// Todo: Code for SH1106 update
#endif
    }
}
#endif

/*
 * set txt-cursor to left upper corner
 */
void
lcd_home (void)
{
  lcd_gotoxy (0, 0);
}

/*
 * invert the whole display
 * swaps black and white
 */
void
lcd_invert (uint8_t invert)
{
#ifdef __ARM_ARCH
  while (cmdPending)
    ;
  cmdBuf.cnt = 2;
  if (invert == TRUE)
    cmdBuf.cmd[0] = OLED_INVERTDISPLAY;
  else
    cmdBuf.cmd[0] = OLED_NORMALDISPLAY;
  cmdPending = TRUE;
#else
  i2c_start(OLED_I2C_ADR);
  uint8_t commandSequence[1];
  if (invert == TRUE)
    {
      commandSequence[0] = OLED_INVERTDISPLAY;
    }
  else
    {
      commandSequence[0] = OLED_NORMALDISPLAY;
    }
  lcd_command(commandSequence, 1);
#endif
}

/*
 * control the display-contrast
 * te current-consumption of the OLED increases with conrast
 * valid range: 0-255
 */
void
lcd_set_contrast (uint8_t contrast)
{
#ifdef __ARM_ARCH
  while (cmdPending)
    ;
  cmdBuf.cnt = 3;
  cmdBuf.cmd[0] = OLED_SETCONTRAST;
  cmdBuf.cmd[1] = contrast;
  cmdPending = TRUE;

#else
  uint8_t commandSequence[2] =
    { OLED_SETCONTRAST, contrast};
  lcd_command(commandSequence, sizeof(commandSequence));
#endif
}

/*
 * maps char to index of font-table
 * if char not found, 0xff is returned
 */
static inline uint8_t
map_char2fnt (char c)
{
  uint8_t i, idx;
  if ((c >= 0x20) && (c <= 0x7f))
    idx = (uint8_t) c - 0x20;
  else
    {
      for (i = 0;
	  (pgm_read_byte(&fnt_map[i].idx) != 0Xff)
	      && (pgm_read_byte(&fnt_map[i].c) != c); i++)
	;
      idx = pgm_read_byte(&fnt_map[i].idx);
    }
  return idx;
}

void

/*
 * print a single character on display
 *
 */
lcd_putc (char c)
{
  uint8_t fnt_idx;
  fnt_idx = map_char2fnt (c);
  cursorPosition++;
  if (fnt_idx != 0xff)
    {
#ifndef GRAPHICMODE
      i2c_start(OLED_I2C_ADR);
      i2c_byte(OLED_DTA_PREFIX);	// 0x80 for command, 0x40 for data
#endif
      for (uint8_t i = 0; i < 6; i++)
	{
#ifdef GRAPHICMODE
	  displayBuffer.buf[actualIndex + i] = pgm_read_byte(
	      &oled_font6x8[fnt_idx][i]);			// print font to ram, print 6 columns
#else
	  i2c_byte(pgm_read_byte(&oled_font6x8[fnt_idx][i]));	// print font to ram, print 6 columns
#endif
	}
#ifndef GRAPHICMODE
      i2c_stop();
#endif
    }
#ifdef GRAPHICMODE
  actualIndex += 6;
#endif
}

void
lcd_puts (const char* s)
{
  while (*s)
    {
      lcd_putc (*s++);
    }
}

#ifndef __ARM_ARCH
void lcd_puts_p(const char* progmem_s)
  {
    register uint8_t c;
    while ((c = pgm_read_byte(progmem_s++)))
      {
    	lcd_putc(c);
      }
  }
#endif


void
lcd_clrscr (void)
{
#ifdef GRAPHICMODE
  memset (displayBuffer.buf, 0x00, sizeof(displayBuffer.buf));
#else

uint8_t tmpLine[OLED_WIDTH];
memset(tmpLine, 0, OLED_WIDTH);

#ifdef SH1106
  /*      for (uint8_t j=0; j< OLED_WIDTH; j++)
  	{
  	  actualLine[j]=displayBuffer.buf[i*OLED_WIDTH+j];

  	}*/

#else

  for (uint8_t i=0; i <= OLED_VLINES; i++)
    {
      lcd_gotoxy(0, i);
      lcd_data(tmpLine, OLED_WIDTH);
    }
#endif
#endif
  lcd_home ();
}

void
lcd_gotoxy (uint8_t x, uint8_t y)
{
  if ((x > (OLED_WIDTH / 6)) || (y > (OLED_VLINES - 1)))
    return;	// out of display
  cursorPosition = x;
//  x = x * 6;					// one char: 6 pixel width
  actualIndex = (x*6) + (y * OLED_WIDTH);
#ifndef __ARM_ARCH
#if defined SH1106
  uint8_t commandSequence[] =
  { OLED_PAGESTART + y, OLED_COLUMNADDR, 0x00+((2+x) & (0x0f)), 0x10+( ((2+x) & (0xf0)) >> 4 ), 0x7f};
#else
  uint8_t commandSequence[] =
  { OLED_PAGESTART + y, OLED_COLUMNADDR, OLED_SETLOWCOLUMN + (x & 0x0f),  OLED_SETHIGHCOLUMN + ((x & (0xf0)) >> 4 ), 0x7f};
#endif
  lcd_command(commandSequence, sizeof(commandSequence));
#endif
}

#ifdef GRAPHICMODE
void
lcd_drawPixel (uint8_t x, uint8_t y, uint8_t color)
{
  if (x > OLED_WIDTH - 1 || y > (OLED_HEIGHT - 1))
    return; // out of Display
  if (color == WHITE)
    {
      displayBuffer.buf[(uint8_t) (y / (OLED_HEIGHT / 8)) * OLED_WIDTH + x] |=
	  (1 << (y % (OLED_HEIGHT / 8)));
    }
  else
    {
      displayBuffer.buf[(uint8_t) (y / (OLED_HEIGHT / 8)) * OLED_WIDTH + x] &=
	  ~(1 << (y % (OLED_HEIGHT / 8)));
    }
}

void
lcd_drawLine (uint8_t x1, uint8_t y1, uint8_t x2, uint8_t y2, uint8_t color)
{
  if (x1 > OLED_WIDTH - 1 || x2 > OLED_WIDTH - 1 || y1 > OLED_HEIGHT - 1
      || y2 > OLED_HEIGHT - 1)
    return;
  int dx = abs (x2 - x1), sx = x1 < x2 ? 1 : -1;
  int dy = -abs (y2 - y1), sy = y1 < y2 ? 1 : -1;
  int err = dx + dy, e2; /* error value e_xy */

  while (1)
    {
      lcd_drawPixel (x1, y1, color);
      if (x1 == x2 && y1 == y2)
	break;
      e2 = 2 * err;
      if (e2 > dy)
	{
	  err += dy;
	  x1 += sx;
	} /* e_xy+e_x > 0 */
      if (e2 < dx)
	{
	  err += dx;
	  y1 += sy;
	} /* e_xy+e_y < 0 */
    }
}

void
lcd_drawRect (uint8_t px1, uint8_t py1, uint8_t px2, uint8_t py2, uint8_t color)
{
  if (px1 > OLED_WIDTH - 1 || px2 > OLED_WIDTH - 1 || py1 > OLED_HEIGHT - 1
      || py2 > OLED_HEIGHT - 1)
    return;
  lcd_drawLine (px1, py1, px2, py1, color);
  lcd_drawLine (px2, py1, px2, py2, color);
  lcd_drawLine (px2, py2, px1, py2, color);
  lcd_drawLine (px1, py2, px1, py1, color);
}

void
lcd_fillRect (uint8_t px1, uint8_t py1, uint8_t px2, uint8_t py2, uint8_t color)
{
  if (px1 > px2)
    {
      uint8_t temp = px1;
      px1 = px2;
      px2 = temp;
      temp = py1;
      py1 = py2;
      py2 = temp;
    }
  for (uint8_t i = 0; i <= (py2 - py1); i++)
    {
      lcd_drawLine (px1, py1 + i, px2, py1 + i, color);
    }
}

void
lcd_drawCircle (uint8_t center_x, uint8_t center_y, uint8_t radius,
		uint8_t color)
{
  if (((center_x + radius) > OLED_WIDTH - 1)
      || ((center_y + radius) > OLED_HEIGHT - 1) || center_x < radius
      || center_y < radius)
    return;
  int16_t f = 1 - radius;
  int16_t ddF_x = 1;
  int16_t ddF_y = -2 * radius;
  int16_t x = 0;
  int16_t y = radius;

  lcd_drawPixel (center_x, center_y + radius, color);
  lcd_drawPixel (center_x, center_y - radius, color);
  lcd_drawPixel (center_x + radius, center_y, color);
  lcd_drawPixel (center_x - radius, center_y, color);

  while (x < y)
    {
      if (f >= 0)
	{
	  y--;
	  ddF_y += 2;
	  f += ddF_y;
	}
      x++;
      ddF_x += 2;
      f += ddF_x;

      lcd_drawPixel (center_x + x, center_y + y, color);
      lcd_drawPixel (center_x - x, center_y + y, color);
      lcd_drawPixel (center_x + x, center_y - y, color);
      lcd_drawPixel (center_x - x, center_y - y, color);
      lcd_drawPixel (center_x + y, center_y + x, color);
      lcd_drawPixel (center_x - y, center_y + x, color);
      lcd_drawPixel (center_x + y, center_y - x, color);
      lcd_drawPixel (center_x - y, center_y - x, color);
    }
}

void
lcd_fillCircle (uint8_t center_x, uint8_t center_y, uint8_t radius,
		uint8_t color)
{
  for (uint8_t i = 0; i <= radius; i++)
    {
      lcd_drawCircle (center_x, center_y, i, color);
    }
}

void
lcd_drawBitmap (uint8_t x, uint8_t y, const uint8_t *picture, uint8_t width,
		uint8_t height, uint8_t color)
{
  uint8_t i, j, byteWidth = (width + 7) / 8;
  for (j = 0; j < height; j++)
    {
      for (i = 0; i < width; i++)
	{
	  if (pgm_read_byte(picture + j * byteWidth + i / 8) & (128 >> (i & 7)))
	    {
	      lcd_drawPixel (x + i, y + j, color);
	    }
	}
    }
}

void
lcd_display ()
{
#ifndef __ARM_ARCH
#ifdef SH1106
  for (uint8_t i=0; i < OLED_HEIGHT/8; i++)
    {
      lcd_gotoxy(0, i);
      uint8_t actualLine[OLED_WIDTH];
      for (uint8_t j=0; j < OLED_WIDTH; j++)
	{
	  actualLine[j]=displayBuffer.buf[i*OLED_WIDTH+j];
	}
      lcd_data(actualLine, sizeof(actualLine));
    }
#else
  lcd_gotoxy(0,0);
  lcd_data(displayBuffer.buf, sizeof(displayBuffer.buf));
#endif
#endif
}
#endif
