#include <stdint.h>
#include <stdbool.h>
#include "stm32f1xx.h"

// delay loop for 8 MHz CPU clock
void delay(uint16_t msec)
{
    for (uint32_t j=0; j<2000*msec; j++)
    {
        __NOP ();
    }
}

/**
 * Initialize the I²C interface.
 * The related I/O pins must be configured for alternate I/O open-drain AFTER this function.
 *
 * @param registerStruct May be either I2C1 or I2C2
 * @param fastMode false=100 kHz, true=400 kHz
 * @param apb1_clock clock frequency of APB1 peripherals
 */
void i2c_init(I2C_TypeDef* registerStruct, bool fastMode, uint32_t apb1_clock)
{
    // Enable clock
    if (registerStruct==I2C1)
    {
        SET_BIT(RCC->APB1ENR, RCC_APB1ENR_I2C1EN);
    }
    else if (registerStruct==I2C2)
    {
        SET_BIT(RCC->APB1ENR, RCC_APB1ENR_I2C2EN);
    }

    // Configure timing
    MODIFY_REG(registerStruct->CR2, I2C_CR2_FREQ, apb1_clock/1000000);
    if (fastMode)
    {
        MODIFY_REG(registerStruct->CCR, I2C_CCR_CCR, apb1_clock/800000);
        MODIFY_REG(registerStruct->TRISE, I2C_TRISE_TRISE, apb1_clock/4000000+1);
    }
    else
    {
        MODIFY_REG(registerStruct->CCR, I2C_CCR_CCR, apb1_clock/200000);
        MODIFY_REG(registerStruct->TRISE, I2C_TRISE_TRISE, apb1_clock/1000000+1);
    }

    // Enable the peripheral
    SET_BIT(registerStruct->CR1, I2C_CR1_PE);
}


/**
 * Perform an I²C transaction, which sends 0 or more data bytes, followed by receiving 0 or more data bytes.
 *
 * @param registerStruct May be either I2C1 or I2C2
 * @param slave_addr 7bit slave_addr (will be shifted within this function)
 * @param send_buffer Points to the buffer that contains the data bytes that shall be sent (may be 0 if not used)
 * @param send_size Number of bytes to send
 * @param receive_buffer Points to the buffer that will be filled with the received bytes (may be 0 if not used)
 * @param receive_size Number of bytes to receive
 * @return Number of received data bytes
 */
unsigned int i2c_communicate(I2C_TypeDef* registerStruct, uint8_t slave_addr,
    void* send_buffer, unsigned int send_size, void* receive_buffer, unsigned int receive_size)
{
    unsigned int receive_count=0;

    // shift the 7bit address to the right position
    slave_addr=slave_addr<<1;

    // Send data
    if (send_size>0)
    {
        // Send START and slave address
        SET_BIT(registerStruct->CR1, I2C_CR1_START);                     // send START condition
        while (!READ_BIT(registerStruct->SR1, I2C_SR1_SB));              // wait until START has been generated
        WRITE_REG(registerStruct->DR,slave_addr);                        // send slave address
        while (!READ_BIT(registerStruct->SR1, I2C_SR1_ADDR))             // wait until address has been sent
        {
            if (READ_BIT(registerStruct->SR1, I2C_SR1_AF))
            {
                // did not receive ACK after address
                goto error;
            }
        }

        READ_REG(registerStruct->SR2);                                   // clear ADDR
        while (send_size>0)
        {
            WRITE_REG(registerStruct->DR,*((uint8_t*)send_buffer));      // send 1 byte from buffer
            while (!READ_BIT(registerStruct->SR1, I2C_SR1_TXE))          // wait until Tx register is empty
            {
                if (READ_BIT(registerStruct->SR1, I2C_SR1_AF))
                {
                    // did not receive ACK after data byte
                    goto error;
                }
            }
            send_buffer++;
            send_size--;
        }
        while (!READ_BIT(registerStruct->SR1, I2C_SR1_BTF))              // wait until last byte transfer has finished
        {
            if (READ_BIT(registerStruct->SR1, I2C_SR1_AF))
            {
                // did not receive ACK after data byte
                goto error;
            }
        }
    }

    CLEAR_BIT(registerStruct->CR1, I2C_CR1_POS);                         // POS=0
    SET_BIT(registerStruct->CR1, I2C_CR1_ACK);                           // acknowledge each byte

    // Receive data
    // The procedure includes workaround as described in AN2824
    if (receive_size>0)
    {
        // Send (RE-)START and slave address
        SET_BIT(registerStruct->CR1, I2C_CR1_START);                      // send START condition
        while (!READ_BIT(registerStruct->SR1, I2C_SR1_SB));               // wait until START has been generated
        WRITE_REG(registerStruct->DR,slave_addr+1);                       // send slave address + read mode
        while (!READ_BIT(registerStruct->SR1, I2C_SR1_ADDR))              // wait until address has been sent
        {
            if (READ_BIT(registerStruct->SR1, I2C_SR1_AF))
            {
                // did not receive ACK after address
                goto error;
            }
        }

        if (receive_size>2)
        {
            READ_REG(registerStruct->SR2);                                // clear ADDR
            while (receive_size>3)
            {
                while (!READ_BIT(registerStruct->SR1, I2C_SR1_RXNE));     // wait until a data byte has been received
                *((uint8_t*)receive_buffer)=READ_REG(registerStruct->DR); // read data
                receive_size--;
                receive_count++;
                receive_buffer++;
            }
            while (!READ_BIT(registerStruct->SR1, I2C_SR1_BTF));          // wait until 2 bytes are received
            CLEAR_BIT(registerStruct->CR1, I2C_CR1_ACK);                  // prepare to send a NACK
            *((uint8_t*)receive_buffer)=READ_REG(registerStruct->DR);     // read the penultimate data byte
            receive_size--;
            receive_count++;
            receive_buffer++;
            __disable_irq();
            {
                SET_BIT(registerStruct->CR1, I2C_CR1_STOP);               // prepare to send a STOP condition
                *((uint8_t*)receive_buffer)=READ_REG(registerStruct->DR); // read the last data byte
                receive_size--;
                receive_count++;
                receive_buffer++;
            }
            __enable_irq();
        }
        else if (receive_size==2)
        {
            SET_BIT(registerStruct->CR1, I2C_CR1_POS);                    // NACK shall be applied to the next byte, not the current byte
            __disable_irq();
            {
                READ_REG(registerStruct->SR2);                            // clear ADDR
                CLEAR_BIT(registerStruct->CR1, I2C_CR1_ACK);              // prepare to send a NACK
            }
            __enable_irq();
            while (!READ_BIT(registerStruct->SR1, I2C_SR1_BTF));          // wait until 2 bytes are received
            __disable_irq();
            {
                SET_BIT(registerStruct->CR1, I2C_CR1_STOP);               // prepare to send a STOP condition
                *((uint8_t*)receive_buffer)=READ_REG(registerStruct->DR); // read the penultimate data byte
                receive_size--;
                receive_count++;
                receive_buffer++;
             }
            __enable_irq();
        }
        else if (receive_size==1)
        {
            CLEAR_BIT(registerStruct->CR1, I2C_CR1_ACK);                  // prepare to send a NACK
            __disable_irq();
            {
                READ_REG(registerStruct->SR2);                            // clear ADDR
                SET_BIT(registerStruct->CR1, I2C_CR1_STOP);               // prepare to send a STOP condition
            }
            __enable_irq();
            while (!READ_BIT(registerStruct->SR1, I2C_SR1_RXNE));         // wait until a data byte has been received
        }

        *((uint8_t*)receive_buffer)=READ_REG(registerStruct->DR);         // read the last data byte
        receive_size--;
        receive_count++;
        receive_buffer++;
        while (READ_BIT(registerStruct->SR1, I2C_CR1_STOP));              // wait until STOP has been generated
    }

    else if (receive_size==0 && send_size>0)
    {
        SET_BIT(registerStruct->CR1, I2C_CR1_STOP);                       // send STOP condition
        while (READ_BIT(registerStruct->CR1, I2C_CR1_STOP));              // wait until STOP has been generated
    }

    return receive_count;

    error:
    SET_BIT(registerStruct->CR1, I2C_CR1_STOP);                           // send STOP condition
    while (READ_BIT(registerStruct->CR1, I2C_CR1_STOP));                  // wait until STOP has been generated
    CLEAR_BIT(registerStruct->CR1, I2C_CR1_PE);                           // restart the I2C interface clear all error flags
    SET_BIT(registerStruct->CR1, I2C_CR1_PE);
    //ITM_SendString("I2C bus error!\n");
    return receive_count;
}


int main(void)
{
    // Configure PA5 and PC13 for blinking LED
    SET_BIT(RCC->APB2ENR, RCC_APB2ENR_IOPAEN);
    SET_BIT(RCC->APB2ENR, RCC_APB2ENR_IOPCEN);
    MODIFY_REG(GPIOA->CRL, GPIO_CRL_CNF5 + GPIO_CRL_MODE5, GPIO_CRL_MODE5_0);
    MODIFY_REG(GPIOC->CRH, GPIO_CRH_CNF13 + GPIO_CRH_MODE13, GPIO_CRH_MODE13_0);

    // Enable Port B and alternate functions for I²C
    SET_BIT(RCC->APB2ENR, RCC_APB2ENR_IOPBEN);
    SET_BIT(RCC->APB2ENR, RCC_APB2ENR_AFIOEN);

    // Initialize I2C1, no fast mode, APB1 clock is 8 MHz
    i2c_init(I2C1, false, 8000000);

    // I2C uses PB6=SCL, PB7=SDA, alternate function open-drain 2 MHz
    // Must be done after the I2C initialization otherwise the I/O pins would start with wrong logic level
    MODIFY_REG(GPIOB->CRL, GPIO_CRL_CNF6 + GPIO_CRL_MODE6, GPIO_CRL_CNF6_0 + GPIO_CRL_CNF6_1 + GPIO_CRL_MODE6_1);
    MODIFY_REG(GPIOB->CRL, GPIO_CRL_CNF7 + GPIO_CRL_MODE7, GPIO_CRL_CNF7_0 + GPIO_CRL_CNF7_1 + GPIO_CRL_MODE7_1);

    uint8_t send_buffer[4] = { 10, 20, 30, 40 };
    uint8_t receive_buffer[9];

    for (;;)
    {
        delay(200);

        // Both LED HIGH
        WRITE_REG(GPIOA->BSRR, GPIO_BSRR_BS5);
        WRITE_REG(GPIOC->BSRR, GPIO_BSRR_BS13);

        // Communicate with slave address 8: send 4 bytes, then receive 9 bytes
        int received = i2c_communicate(I2C1, 8, send_buffer, 4, receive_buffer, 9);

        delay(200);

        // Both LED LOW
        WRITE_REG(GPIOA->BSRR, GPIO_BSRR_BR5);
        WRITE_REG(GPIOC->BSRR, GPIO_BSRR_BR13);
    }

}
