/*
 *	This is a monolithic version of the DEHUFF program.
 *	The original source consisted of huff.h, dehuff.c, decode.c and
 *	misc.c
 */

/*
 *	Definitions for the Huffman encoded file structure
 */

#include	<stdio.h>

#ifndef	uchar
#define	uchar	unsigned char
#endif

#ifndef	CHAR_BIT
#define	CHAR_BIT	8
#endif	CHAR_BIT

#define	ALFSIZ		(1<<CHAR_BIT)	/* max no. of distinct characters */
#define	MAXFILENT	500		/* maximum number of files */
#define	MAGIC		0x01BD
#define	HSIZE		10		/* length of header in file */
#define	FSIZE		13		/* length of file entry */

#define	SET(ar, bit)	(ar[(bit)/CHAR_BIT] |= (1<<((bit)%CHAR_BIT)))
#define	CLR(ar, bit)	(ar[(bit)/CHAR_BIT] &= ~(1<<((bit)%CHAR_BIT)))
#define	TST(ar, bit)	((ar[(bit)/CHAR_BIT] & (1<<((bit)%CHAR_BIT)))!=0)

#if	unix
#define	casecmp	strcmp
#endif

typedef struct node
{
	uchar		n_c;
	long		n_f;
	struct node *	n_left, * n_right;
}	node;

typedef	struct
{
	uchar		h_nbits;
	uchar		h_cbits[ALFSIZ/CHAR_BIT];
}	h_char;

typedef struct
{
	uchar		c_chr;
	long		c_freq;
	h_char		c_bits;
}	chent;

typedef	struct
{
	char *		f_name;
	long		f_npos;
	long		f_nchrs;
	long		f_pos;
	char		f_asc;
}	filent;

typedef struct
{
	short		hd_magic;	/* magic number */
	short		hd_nfiles;	/* no of files */
	short		hd_alfsiz;	/* size of alphabet */
	long		hd_hpos;	/* start pos of file hdrs */
}	hdr;


extern node *		root;
extern chent		clist[ALFSIZ];
extern filent		flist[MAXFILENT];
extern short		alfused;
extern uchar		ascii;
extern unsigned long	get4();
extern unsigned short	get2();
extern chent *		cptrs[ALFSIZ];
extern short		factor;

	hdr	hd;
static	short	indent;
static	uchar	listflg;
static	uchar	xtractflg;
static	uchar	debug;
static	char **	av;
static	char *	getname();
static	void	list(), extract();

#if	!unix
extern	char **	_getargs();
extern	int	_argc_;
#endif	unix

main(argc, argv)
char **	argv;
{
	short	i;

#if	!unix
	if(argc == 1) {
		argv = _getargs(0, "dehuff");
		argc = _argc_;
	}
#endif	unix
	argc--;
	argv++;
	while(argc && **argv == '-') {
		switch(argv[0][1]) {

		case 'l':
		case 'L':
			listflg++;
			break;

		case 'x':
		case 'X':
			xtractflg++;
			break;

		case 'd':
		case 'D':
			debug++;
			break;

		default:
			fprintf(stderr, "Unrecognized flag %s ignored\n", argv[0]);
			break;
		}
		argc--;
		argv++;
	}
	if(argc < 1) {
		printf("Usage of dehuff:\n\nTo list the contents of a .HUF file:\n");
		printf("\n	dehuff file.huf\n\nTo extract files from a .HUF file:\n");
		printf("\n	dehuff -x file.huf\n\nIf no extra arguments are supplied, then the entire contents\n");
		printf("of the .HUF file will be listed or extracted, otherwise only files\n");
		printf("matching an argument will be listed or extracted, e.g.\n\n");
		printf("	dehuff -x xyz.huf afile.c\n\nwould extract only the file afile.c.\n");
		printf("The number reported for each file is its length in bytes\n");
		exit(1);
	}
	av = argv+1;
	if(!freopen(*argv, "rb", stdin))
		error("Can't open file %s", *argv);
	hd.hd_magic = get2();
	if(hd.hd_magic != MAGIC)
		error("%s is not a huf file", *argv);
	hd.hd_nfiles = get2();
	alfused = hd.hd_alfsiz = get2();
	hd.hd_hpos = get4();
	bld_tree();
	if(debug)
		prtree(root);
	for(i = 0 ; i < hd.hd_nfiles ; i++)
		if(isarg(getname(i)))
			if(xtractflg)
				extract(i);
			else
				list(i);
}

static
isarg(s)
char *	s;
{
	register char **	p;

	if(*av == 0)		/* default is all members */
		return 1;
	for(p = av ; *p ; p++)
		if(casecmp(*p, s) == 0)
			return 1;
	return 0;
}

static char *
getname(i)
short	i;
{
	static char	fbuf[100];
	register char *	cp;
	filent		hf;

	fseek(stdin, hd.hd_hpos+i*FSIZE, 0);
	hf.f_npos = get4();
	hf.f_nchrs = get4();
	hf.f_pos = get4();
	hf.f_asc = getchar();
	fseek(stdin, hf.f_npos, 0);
	align();
	for(cp = fbuf ; *cp++ = gethch() ; )
		continue;
	return fbuf;
}

static void
list(i)
short	i;
{
	register char *	cp;
	filent		hf;
	char		fbuf[100];

	fseek(stdin, hd.hd_hpos+i*FSIZE, 0);
	hf.f_npos = get4();
	hf.f_nchrs = get4();
	hf.f_pos = get4();
	hf.f_asc = getchar();
	fseek(stdin, hf.f_npos, 0);
	align();
	for(cp = fbuf ; *cp++ = gethch() ; )
		continue;
	fprintf(stderr, "%-20.20s    %ld\n", fbuf, hf.f_nchrs);
}

static void
extract(i)
short	i;
{
	register char *	cp;
	filent		hf;
	FILE *		fp;
	char		fbuf[100];

	fseek(stdin, hd.hd_hpos+i*FSIZE, 0);
	hf.f_npos = get4();
	hf.f_nchrs = get4();
	hf.f_pos = get4();
	hf.f_asc = getchar();
	fseek(stdin, hf.f_npos, 0);
	align();
	for(cp = fbuf ; *cp++ = gethch() ; )
		continue;
	if(!(fp = fopen(fbuf, hf.f_asc ? "w" : "wb"))) {
		fprintf(stderr, "Can't create %s\n", fbuf);
		return;
	}
	fprintf(stderr, "%-20.20s    %ld\n", fbuf, hf.f_nchrs);
	fseek(stdin, hf.f_pos, 0);
	align();
	while(hf.f_nchrs--)
		putc(gethch(), fp);
	fclose(fp);
}
prtree(np)
register node *	np;
{
	short	i;

	for(i = indent ; i-- ; )
		fputc(' ', stderr);
	if(np->n_left) {
		fprintf(stderr, "X\n");
		indent += 4;
		prtree(np->n_left);
		prtree(np->n_right);
		indent -= 4;
	} else
		if(np->n_c >= ' ' && np->n_c <= 0176)
			fprintf(stderr, "'%c'\n", np->n_c);
		else
			fprintf(stderr, "%03o\n", np->n_c);
}


/*	Decode functions - used to be in a separate file
 */
	char	cl[ALFSIZ];
	short	alfused;
static	short	clidx;
static	short	g_bit;
static	short	g_char;
	node *	root;
static	node *	get_tree();


bld_tree()
{
	short	i;

	for(i = 0 ; i < alfused ; i++)
		cl[i] = getchar();
	align();
	clidx = 0;
	root = get_tree();
}

static node *
get_tree()
{
	register node *	tp;

	if(!(tp = (node *)calloc(sizeof(node), 1)))
		error("Out of memory");
	if(get_bit()) {
		tp->n_left = get_tree();
		tp->n_right = get_tree();
	} else
		tp->n_c = cl[clidx++];
	return tp;
}

align()
{
	g_char = 0;
	g_bit = CHAR_BIT;
}

get_bit()
{
	if(g_bit == CHAR_BIT)
		if((g_char = getchar()) == EOF)
			error("Read error or EOF on huf file");
		else
			g_bit = 0;
	return g_char & (1 << g_bit++);
}

gethch()
{
	register node *	tp;

	tp = root;
	while(tp->n_left)
		tp = get_bit() ? tp->n_left : tp->n_right;
	return tp->n_c & 0xFF;
}
#include	<stdio.h>

#if	! unix
/* this is a kludge for a bug in the Venix cpp */
#endif

#if	! unix
#include	<ctype.h>

casecmp(s1, s2)
register char *	s1, * s2;
{
	char	a, b;

	while(a = *s1++) {
		b = *s2++;
		if(isupper(a))
			a = tolower(a);
		if(isupper(b))
			b = tolower(b);
		if(a != b)
			return 1;
	}
	return 0;
}
#endif	unix

error(s, a1, a2)
char *	s;
{
	fprintf(stderr, "huff: fatal error: ");
	fprintf(stderr, s, a1, a2);
	fputc('\n', stderr);
	exit(1);
}

put2(i)
unsigned short	i;
{
	putchar(i & 0xFF);
	putchar(i >> 8);
}

put4(i)
unsigned long	i;
{
	put2(i & 0xFFFFL);
	put2(i >> 16);
}

unsigned long
get4()
{
	unsigned long	i;

	i = (unsigned long)getchar();
	i += (unsigned long)getchar() << 8;
	i += (unsigned long)getchar() << 16;
	i += (unsigned long)getchar() << 24;
	if(feof(stdin))
		error("EOF on input file");
	return i;
}

unsigned short
get2()
{
	unsigned short	i;

	i = (unsigned short)getchar();
	i += (unsigned short)getchar() << 8;
	if(feof(stdin))
		error("EOF on input file");
	return i;
}
