/* NXP  LPC1343:  USB als COM Port betreiben */
#include "StdTypes.h"
#include "LPC1343.h"
#include "usb.h"

/* die Transfer-Puffer fr zeichenweises In und Out ber den USB */
#define txLen  256
char UsbTxBuf[txLen];
int  txr, txw;

#define rxLen  256
char UsbRxBuf[rxLen];
int  rxr, rxw;


/***************************  Konstanten ********************************/

/* USB device (base address 0x4002 0000) */
/* Device interrupt registers */
#define USBDevIntSt  (*(volatile unsigned long *)(0x40020000))
#define USBDevIntEn  (*(volatile unsigned long *)(0x40020004))
#define USBDevIntClr (*(volatile unsigned long *)(0x40020008))
#define USBDevIntSet (*(volatile unsigned long *)(0x4002000C))

/* SIE command registers */
#define USBCmdCode   (*(volatile unsigned long *)(0x40020010))
#define USBCmdData   (*(volatile unsigned long *)(0x40020014))

/* USB data transfer registers */
#define USBRxData    (*(volatile unsigned long *)(0x40020018))
#define USBTxData    (*(volatile unsigned long *)(0x4002001C))
#define USBRxPLen    (*(volatile unsigned long *)(0x40020020))
#define USBTxPLen    (*(volatile unsigned long *)(0x40020024))
#define USBCtrl      (*(volatile unsigned long *)(0x40020028))

/* Miscellaneous registers */
#define USBDevFIQSel (*(volatile unsigned long *)(0x4002002C))


#define SYSAHBCLKCTRL (*(volatile unsigned long *)(0x40048080))
#define PDRUNCFG      (*(volatile unsigned long *)(0x40048238))


/******* Zuordnung physischer Endpunkte 0..9 ********************/
/* ist beim LPC1343 fest zugeordnet!         */
#define  EpCtrlOut  0       // phys. EP fr Control_Out
#define  EpCtrlIn   1       // phys. EP fr Control_In
#define  EpIntOut   2       // phys. EP fr Interrupt vom Host
#define  EpIntIn    3       // phys. EP fr Interrupt zum Host
#define  EpBulkOut  6       // phys. EP Bulk-Daten Host-->LPC
#define  EpBulkIn   7       // phys. EP Bulk-Daten LPC-->Host

#define  logEpCtrl     (EpCtrlOut>>1)
#define  logEpInt      (EpIntOut >>1)
#define  logEpBulkOut  (EpBulkOut>>1)
#define  logEpBulkIn   (EpBulkIn >>1)

/* Packet Size ist fest zugeordnet */
#define   EpCtrlMaxLen  64
#define   EpBulkMaxLen  64
#define   EpIsochronLen 512      /* hier unbenutzt */


/******* Codes der Standard-bRequest's im Setup-Paket ***/
#define GET_STATUS          0x00
#define CLEAR_FEATURE       0x01
#define SET_FEATURE         0x03
#define SET_ADDRESS         0x05
#define GET_DESCRIPTOR      0x06
#define SET_DESCRIPTOR      0x07
#define GET_CONFIGURATION   0x08
#define SET_CONFIGURATION   0x09
#define GET_INTERFACE       0x0A
#define SET_INTERFACE       0x0B
#define SYNC_FRAME          0x0C

/******* zustzliche bRequest-Codes fr virtuelle ComPorts */
#define SET_LINE_CODE               0x20    /* 7 Byte Paket mit Baudrate etc. */
#define GET_LINE_CODE               0x21    /* 7 Byte Paket mit Baudrate etc. */
#define SET_CONTROL_LINE_STATE      0x22    /* 2 Bit  DTR und RTS */
#define SEND_BREAK                  0x23    /* hier unbenutzt */



/******* Struktur des Setup-Paketes *****/
struct TSetupPaket
{ byte bmRequestType;  /* siehe oben */
  byte bRequest;       /* siehe Request-Tafel in USB-Doku */
  word wValue;         /* je nach Request */
  word wIndex;         /* je nach Request */
  word wLength;        /* Anzahl Bytes, wenn Data-Stage vorhanden ist */
};


/******* Struktur des Kommando- und Org-Datenblockes *******************/
struct TCommand
{ struct TSetupPaket SetupPacket;  /* das jeweils letzte Setup-Paket   */
  long   TransferLen;              /* noch zum Host zu sendende Anzahl Bytes */
  long   PacketLen;                /* wie lang das Paket zum Host sein darf */
  byte*  TransferPtr;              /* zeigt auf die noch zu sendenden Bytes */

  bool   RemoteWakeup;
  bool   SelfPowered;
  byte   Configuration;
};


/* Line coding structure
  0-3 BaudRate     Data terminal rate (baudrate), in bits per second
  4   bCharFormat  Stop bits: 0 - 1 Stop bit, 1 - 1.5 Stop bits, 2 - 2 Stop bits
  5   bParityType  Parity:    0 - None, 1 - Odd, 2 - Even, 3 - Mark, 4 - Space
  6   bDataBits    Data bits: 5, 6, 7, 8, 16
 */
struct T_LineCoding
{  dword  BaudRate;     /* Baud rate    */
   byte   Stopbits;     /* stop bit     */
   byte   ParityType;   /* parity       */
   byte   DataBits;     /* data bits    */
};


/************  Variablen *****************************************/

struct TCommand     CMD;
struct T_LineCoding LineCoding;
word                Dtr_Rts;

// #include "debugfkt.inc"

/************ Funktionen zum Starten des virtuellen COM-Portes *****/
void Class_Start (void)
{ LineCoding.BaudRate = 9600;
  LineCoding.Stopbits = 0;
  LineCoding.ParityType = 0;
  LineCoding.DataBits = 8;
  Dtr_Rts = 0;
  txr = txw = rxr = rxw = 0;
}

bool Class_Compare (word aValue)  /* immer true, wird hier nicht gebraucht */
{ return true; }



/************* die Descriptoren ************************************/
#define LEN_DEVICE         18
#define DESC_DEVICE         1
#define VID            0x0416   /* Vendor ID (von RealTek) */
#define PID            0x5011   /* Product ID */

const byte DeviceDescriptor[] =
{ LEN_DEVICE,             /* bLength              */
  DESC_DEVICE,            /* bDescriptorType      */
  0x00, 0x02,             /* bcdUSB               */
  0x02,                   /* bDeviceClass         */
  0x00,                   /* bDeviceSubClass      */
  0x00,                   /* bDeviceProtocol      */
  EpCtrlMaxLen,           /* bMaxPacketSize0      */
  0x16,                   /* Vendor  ID LO        */
  0x04,                   /* Vendor  ID HI        */
  0x11,                   /* Product ID LO        */
  0x50,                   /* Product ID HI        */
  0x00, 0x01,             /* bcdDevice            */
  0x01,                   /* iManufacturer        */
  0x02,                   /* iProduct             */
  0x03,                   /* iSerialNumber        */
  0x01                    /* bNumConfigurations   */
};

#define LEN_CONFIG          9
#define DESC_CONFIG         2

#define LEN_INTERFACE       9
#define DESC_INTERFACE      4

#define LEN_ENDPOINT        7
#define DESC_ENDPOINT       5

const byte ConfigDescriptor[] =
{ LEN_CONFIG,             /* bLength              */
  DESC_CONFIG,            /* bDescriptorType      */
  0x43, 0x00,             /* wTotalLength         */
  0x02,                   /* bNumInterfaces       */
  0x01,                   /* bConfigurationValue  */
  0x00,                   /* iConfiguration       */
  0xC0,                   /* bmAttributes         */
  0x32,                   /* MaxPower             */

                          /* INTERFACE descriptor */
  LEN_INTERFACE,          /* bLength              */
  DESC_INTERFACE,         /* bDescriptorType      */
  0x00,                   /* bInterfaceNumber     */
  0x00,                   /* bAlternateSetting    */
  0x01,                   /* bNumEndpoints        */
  0x02,                   /* bInterfaceClass      */
  0x02,                   /* bInterfaceSubClass   */
  0x01,                   /* bInterfaceProtocol   */
  0x00,                   /* iInterface           */

                          /* Communication Class Specified INTERFACE descriptor */
  0x05,                   /* Size of the descriptor, in bytes */
  0x24,                   /* CS_INTERFACE descriptor type */
  0x00,                   /* Header functional descriptor subtype */
  0x10, 0x01,             /* Communication device compliant to the communication */
                          /* spec. ver. 1.10 */
                          /* Communication Class Specified INTERFACE descriptor */
  0x05,                   /* Size of the descriptor, in bytes */
  0x24,                   /* CS_INTERFACE descriptor type */
  0x01,                   /* Call management functional descriptor */
  0x00,                   /* BIT0: Whether device handle call management itself. */
                          /* BIT1: Whether device can send/receive call  */
                          /* management information over a Data Class Interface 0 */
  0x01,                   /* Interface number of data class interface optionally */
                          /* used for call management */
                          /* Communication Class Specified INTERFACE descriptor */
  0x04,                   /* Size of the descriptor, in bytes */
  0x24,                   /* CS_INTERFACE descriptor type */
  0x02,                   /* Abstract control management funcational descriptor */
                          /* subtype */
  0x00,                   /* bmCapabilities       */
                          /* Communication Class Specified INTERFACE descriptor */
  0x05,                   /* bLength              */
  0x24,                   /* bDescriptorType: CS_INTERFACE descriptor type */
  0x06,                   /* bDescriptorSubType   */
  0x00,                   /* bMasterInterface     */
  0x01,                   /* bSlaveInterface0     */


  /* ENDPOINT descriptor fr Interrupt */
  LEN_ENDPOINT,           /* bLength              */
  DESC_ENDPOINT,          /* bDescriptorType      */
  0x80+logEpInt,          /* bEndpointAddress     */
  3,                      /* Attribute: Interrupt */
  8, 0x00,                /* wMaxPacketSize       */
  0x01,                   /* bInterval            */


  /* INTERFACE descriptor */
  LEN_INTERFACE,          /* bLength              */
  DESC_INTERFACE,         /* bDescriptorType      */
  0x01,                   /* bInterfaceNumber     */
  0x00,                   /* bAlternateSetting    */
  0x02,                   /* bNumEndpoints        */
  0x0A,                   /* bInterfaceClass      */
  0x00,                   /* bInterfaceSubClass   */
  0x00,                   /* bInterfaceProtocol   */
  0x00,                   /* iInterface           */


  /* ENDPOINT descriptor fr Bulk IN */
  LEN_ENDPOINT,           /* bLength              */
  DESC_ENDPOINT,          /* bDescriptorType      */
  0x80+logEpBulkIn,       /* bEndpointAddress     */
  2,                      /* Attribute: Bulk      */
  EpBulkMaxLen, 0x00,     /* wMaxPacketSize       */
  0,                      /* bInterval   2ms probieren         */


  /* ENDPOINT descriptor fr Bulk OUT */
  LEN_ENDPOINT,           /* bLength              */
  DESC_ENDPOINT,          /* bDescriptorType      */
  logEpBulkOut,           /* bEndpointAddress     */
  2,                      /* Attribute: Bulk      */
  EpBulkMaxLen, 0x00,     /* wMaxPacketSize       */
  0,                      /* bInterval   2ms probieren         */
};


#define DESC_STRING         3
const byte StringLang[] =
{ 4,                      /* bLength                  */
  DESC_STRING,            /* bDescriptorType          */
  0x09, 0x04              /* Language ID: USA(0x0409) */
};


const byte VendorStringDescriptor[] =
{ 16,                     /* bLength          */
  DESC_STRING,            /* bDescriptorType  */
  'N', 0,
  'u', 0,
  'v', 0,
  'o', 0,
  't', 0,
  'o', 0,
  'n', 0
};

const byte ProductStringDescriptor[] =
{ 32,                     /* bLength          */
  DESC_STRING,            /* bDescriptorType  */
  'U', 0,
  'S', 0,
  'B', 0,
  ' ', 0,
  'V', 0,
  'i', 0,
  'r', 0,
  't', 0,
  'u', 0,
  'a', 0,
  'l', 0,
  ' ', 0,
  'C', 0,
  'O', 0,
  'M', 0
};

const byte StringSerial[26] =
{ 26,                     /* bLength          */
  DESC_STRING,            /* bDescriptorType  */
  'N', 0,
  'T', 0,
  '2', 0,
  '0', 0,
  '0', 0,
  '9', 0,
  '1', 0,
  '0', 0,
  '1', 0,
  '4', 0,
  '0', 0,
  '0', 0
};


/* um Nullbyte oder ein leeres Paket senden zu knnen */
const byte always0 = 0;

/************  Hilfsroutinen ************************************************/

/* Interrupt-Bits in USBDevIntSt */
#define  FRAME    (1<<0)
#define  EP0      (1<<1)
#define  EP1      (1<<2)
#define  EP2      (1<<3)
#define  EP3      (1<<4)
#define  EP4      (1<<5)
#define  EP5      (1<<6)
#define  EP6      (1<<7)
#define  EP7      (1<<8)
#define  DEV_STAT (1<<9)
#define  CCEMPTY  (1<<10)
#define  CDFULL   (1<<11)
#define  RxENDPKT (1<<12)
#define  TxENDPKT (1<<13)


/* Warten auf Bits in USBDevIntSt (hier ohne Timer)*/

bool WarteAufInt (dword WantedBits)
{ long L;
  L =  1000;
  while (--L)
  { if ((USBDevIntSt & WantedBits)== WantedBits) return true;
  }
  return false;
}


/* SIE Kommandos */
#define  cmSetAddress     (0xD0 << 16)
#define  cmConfigDevice   (0xD8 << 16)
#define  cmClearBuf       (0xF2 << 16)
#define  cmSetMode        (0xF3 << 16)
#define  cmReadIntStatus  (0xF4 << 16)
#define  cmReadFrameNum   (0xF5 << 16)
#define  cmValidateBuf    (0xFA << 16)
#define  cmGetErrorStatus (0xFB << 16)
#define  cmReadChipId     (0xFD << 16)
#define  cmDeviceStatus   (0xFE << 16)
#define  cmGetErrorCode   (0xFF << 16)

#define  SieRD  0x200
#define  SieWR  0x100
#define  SieCMD 0x500

bool SIE_Cmd (dword Kommando)
{ dword L, D;
  Kommando = Kommando & 0x00FF0700; /* Angst? */
  D = Kommando & 0x700;             /* CMD, RD, WR separieren */
  L = CDFULL;                       /* bei Write + Command */
  if (D!=SieRD) L = CCEMPTY;        /* bei Read            */
  USBDevIntClr = L;                 /* Clear Flag          */
  USBCmdCode = Kommando;            /* auslsen            */
  if (!WarteAufInt(L)) return false;
  return true;
}


int GetDeviceStatus (void)
{ byte B;
   if (!SIE_Cmd(cmDeviceStatus | SieCMD))    /* Get Device Status */
     return -1;
   if (!SIE_Cmd(cmDeviceStatus | SieRD))
     return -1;
   B = USBCmdData & 0xFF;                    /* Device Status */
   /* mit jedem Lesen handelt man sich einen neuen sinnlosen Int ein,
      wenn sich der Core im Suspended Mode befindet  */
   return B;
}


int GetEPStatus (int physEpNum)
{ dword D;
  D = physEpNum << 16;
  if (!SIE_Cmd(D | SieCMD)) return -1;  /* Select EP */
  if (!SIE_Cmd(D | SieRD))  return -1;  /* read..  */
  return USBCmdData & 0xFF;
}


int ClearInterrupt (int physEpNum)
{ dword D;
  D = (physEpNum | 0x40) << 16;
  if (!SIE_Cmd(D | SieCMD)) return -1;  /* Select EP */
  if (!SIE_Cmd(D | SieRD))  return -1;  /* read..  */
  return USBCmdData & 0xFF;
}



bool Stall (int physEpNum)
{ dword D;
  D = (0x40 + physEpNum)<<16;
  if (!SIE_Cmd(D | SieCMD)) return false;        /* Set EP Status */
  if (!SIE_Cmd((1<<16) | SieWR))  return false;  /* Stall Bit auf 1 */
  return true;
}

bool StallLogEP (int logEpNum)
{ int i;
  bool e;
  i = logEpNum<<1;
  e = Stall(i);
  e = e && Stall(i+1);
  return e;
}


bool UnStall (int physEpNum)
{ dword D;
  D = (0x40 + physEpNum)<<16;
  if (!SIE_Cmd(D | SieCMD)) return false;     /* Set EP Status */
  if (!SIE_Cmd(0 | SieWR))  return false;     /* Stall Bit auf 0 */
  return true;
}

bool UnStallLogEP (int logEpNum)
{ int i;
  bool e;
  i = logEpNum<<1;
  e = UnStall(i);
  e = e && UnStall(i+1);
  return e;
}


int ClearBuffer (dword logEpNum)
{ int   i;
  int   j;
  dword D;

  i = 0;
  D = logEpNum << 17;                           /* OUT ist even */
  if (!SIE_Cmd(D | SieCMD))          i = -1;    /* Select EP */
  if (!SIE_Cmd(D | SieRD))           i = -1;    /* read Status */
  j = USBCmdData & 0x7F;

  if (!SIE_Cmd(cmClearBuf | SieCMD)) i = -1;    /* Clear Buffer */
  if (!SIE_Cmd(cmClearBuf | SieRD))  i = -1;    /* read Status  */
  j = (j<<8) | (USBCmdData & 0x7F);
  if (i<0) return i;
  return j;
}


int ValidateBuffer (dword logEpNum)
{ int   i;
  dword D;

  i = 0;
  D = (logEpNum<<17) | (1<<16);                 /* IN ist odd */
  if (!SIE_Cmd(D | SieCMD))          i = -1;    /* Select EP */
  if (!SIE_Cmd(D | SieRD ))          i = -1;
  if(!i) i = USBCmdData & 0xFF;
  if (!SIE_Cmd(cmValidateBuf | SieCMD)) return -1; /* Validate Buffer */
  return i;
}

int GetErrorCode (void)
{ int i;
  if (!SIE_Cmd(cmGetErrorCode | SieCMD)) return -1; /* Select EP */
  if (!SIE_Cmd(cmGetErrorCode | SieRD )) return -1; /* Select EP */
  i = USBCmdData & 0xFF;
  return i;
}

bool USB_SetAddress (byte adr)
{ dword D;
  D = adr | 128;
  D = D << 16;
  if (!SIE_Cmd(cmSetAddress | SieCMD)) return false;
  if (!SIE_Cmd(D | SieWR)) return false;
  return true;
}

bool USB_ConfigDevice (bool obConf)
{ dword D;
  D = 0;
  if (obConf)  D = (1<<16);
  if (!SIE_Cmd(cmConfigDevice | SieCMD)) return false;
  if (!SIE_Cmd(D | SieWR)) return false;
  return true;
}



/* physische Endpunkte aufsetzen (bei Reset-Kommando usw.) */

void InitEndpoints (void)
{ dword D;

  CMD.Configuration = 0;            /* vor "CONFIGURED" ist hier nix */
  CMD.TransferLen = 0;              /* es stehen ab hier auch */
  CMD.PacketLen   = 0;              /* keine Transfers an */
  CMD.TransferPtr = 0;

  USB_SetAddress(0);
  USBDevIntClr = 0xFFFFFFFF;
  USBDevIntEn  = FRAME | EP0 | EP1 | EP2 | EP3 | EP4 | DEV_STAT;

/* Mode setzen */
  D = (1<<3) |                 /* INAK_AI (BULK-IN: Int bei NAK)  */
      (1<<4) |                 /* INAK_AO (BULK-OUT: Int bei NAK) */
      1;                       /* AP_CLK = 1 (48 MHz immer an) */
  SIE_Cmd(cmSetMode | SieCMD); /* Device: Set Mode */
  SIE_Cmd((D<<16) | SieWR);    /* BulkIO mit NAK + immer CLK */
}


void Nop (dword count)
{ while (count) --count; }

#define RD_EN   (1<<0)
#define WR_EN   (1<<1)


int ReadControlBlock (byte* PBuffer, int maxlen)
{ int count, i, n;
  dword D;

  USBCtrl = ((logEpCtrl & 15) << 2) | RD_EN;
  Nop(10);
  count = USBRxPLen & 0x3FF;
  if (count > maxlen) count = maxlen;

  if(count)
  { n = 4;
    i = count;
    D = USBRxData;
    while (i > 0)
    { *PBuffer = D & 0xFF;
      D = D >> 8;
      --n;
      if (!n) { D = USBRxData; n = 4; }
      --i;
      ++PBuffer;
    }
  }
  else D = USBRxData;   /* dummy ? */

  ClearBuffer(logEpCtrl);
  USBCtrl = 0;
  return count;
}



int WriteControlBlock (byte* PBuffer, int count)
{ dword A, L;
  int  i, n;

  USBCtrl   = ((logEpCtrl & 15) << 2) | WR_EN;
  Nop(10);

  if (count > EpCtrlMaxLen) count = EpCtrlMaxLen;
  USBTxPLen = count;

  if (count)
  { A = 0; i = 0; n = 0;
    while (i < count)
    { L = *PBuffer++;
      A = A | (L << n);
      n += 8;
      if (n > 24)
      { n = 0;
        USBTxData = A;
        A = 0;
      }
      ++i;
    }
    if (n) USBTxData = A;
  }
  if (!count)
  { USBTxData = 0;  // UM10375: 10.10.3.4 S.152
  }

  ValidateBuffer(logEpCtrl);
  USBCtrl = 0;
  return count;
}


void ACK (void)
{ WriteControlBlock((byte*) &always0, 0);
}


/* Request-Typ im Setup-Packet testen (Standard, Class, Vendor) */
bool IsStandardRequest (void)
{ return (CMD.SetupPacket.bmRequestType & 0x60)==0; }

bool IsClassRequest (void)
{ return (CMD.SetupPacket.bmRequestType & 0x60)==0x20; }

bool IsVendorRequest (void)
{ return (CMD.SetupPacket.bmRequestType & 0x60)==0x40; }



/******* anstehende Control-Transfers zum Host blockweise starten *******/
void DescriptorBlockwiseIn (void)
{ int i, j;
  byte* Q;

  if ((CMD.SetupPacket.bmRequestType & 0x80)==0) return;
  i = CMD.TransferLen;
  if (i > CMD.PacketLen) i = CMD.PacketLen;
  Q = CMD.TransferPtr;                   /* Quelle */
  j = WriteControlBlock(Q, i);
  CMD.TransferPtr = Q + j;               /* Zeiger auf evt. Rest merken */
  CMD.TransferLen = CMD.TransferLen - j; /* restliche Anzahl Bytes */
  if (CMD.TransferLen < 0) CMD.TransferLen = 0;
}

/**********************************************************************/
/************ Bearbeitung eingegangener Requests **********************/
/**********************************************************************/



/********** USB-Request "SET FEATURE" und "CLEAR FEATURE" behandeln ****/
void DoSetClearFeature (bool value)
{ int Feature;
  int FuerWen;
  int EP;

  Feature = CMD.SetupPacket.wValue;
  FuerWen = CMD.SetupPacket.bmRequestType;
  EP      = CMD.SetupPacket.wIndex;

  switch(FuerWen)
  { case 0:          /* fr Device */
        if (Feature==1) CMD.RemoteWakeup = value;
        break;

    case 1:          /* fr Interface */
        break;

    case 2:          /* fr einen Endpoint */
        if (Feature==0)
        { if (value==false)
              StallLogEP(EP);
          else
              UnStallLogEP(EP);
        }
        break;

    default: Stall(1);  /* quasi NAK senden */
  }
}


/******** USB-Request "GET STATUS" behandeln ***************************/
void DoGetStatus (void)
{ byte Buf[4];
  int FuerWen;
  int  EP;

  FuerWen = CMD.SetupPacket.bmRequestType;
  EP = CMD.SetupPacket.wIndex;

  Buf[0] = 0;

  switch(FuerWen)
  { case 0x80:          /* fr Device */
         if (CMD.RemoteWakeup) Buf[0] |= 2;
         if (CMD.SelfPowered)  Buf[0] |= 1;
         break;

    case 0x81:          /* fr Interface */
         break;

    case 0x82:          /* fr einen Endpoint */
         if ((EP==logEpCtrl)   ||
             (EP==logEpInt)    ||
             (EP==logEpBulkIn) ||
             (EP==logEpBulkIn))  Buf[0] = 1;
         break;

    default:          /* hier eigentlich NAK senden */
         Stall(0);
         return;
  }

  Buf[1] = 0;
  CMD.PacketLen   = EpCtrlMaxLen;
  CMD.TransferLen = 2;
  CMD.TransferPtr = Buf;
  DescriptorBlockwiseIn();
}


/******** Descriptoren zum Host senden *********************************/
void DoGetDescriptor(void)
{ word Type, Index;
  int  aLen;
  const byte* P;

  Type    = CMD.SetupPacket.wValue >> 8;
  Index   = CMD.SetupPacket.wValue & 0xFF;
  aLen    = -1;
  P       = 0;

  switch (Type)
  { case DESC_DEVICE:                            /* Get Device Descriptor */
         { aLen = LEN_DEVICE;
           P    = DeviceDescriptor;
	   break;
         }

    case DESC_CONFIG:                            /* Get Configuration Descriptor    */
         { aLen = ConfigDescriptor[3];           /* Total-Lnge ist WORD            */
           aLen = (aLen << 8) | ConfigDescriptor[2];
           P    = ConfigDescriptor;
           break;
         }


    case DESC_STRING:                            /* Get String Descriptor */
         { switch (Index)                        /* Get String Descriptor */
           { case 0: aLen = 4;
                     P    = StringLang;
		     break;
             case 1: aLen = VendorStringDescriptor[0];
                     P    = VendorStringDescriptor;
		     break;
             case 2: aLen = ProductStringDescriptor[0];
                     P    = ProductStringDescriptor;
		     break;
             case 3: aLen = StringSerial[0];
                     P    = StringSerial;
		     break;
             default: Stall(1);       /* kennen wir nicht. Stall. */
                      aLen = -1;
           }
           break;
         }
     default: { Stall(1);    /* kennen wir nicht. Stall. */
                aLen = -1;
              }
     }

  if (aLen < 0 ) return;

  /* nicht mehr senden wollen, als der Host haben will */
  if (aLen > CMD.SetupPacket.wLength) aLen = CMD.SetupPacket.wLength;
  CMD.PacketLen   = EpCtrlMaxLen;
  CMD.TransferLen = aLen;
  CMD.TransferPtr = (byte*) P;
  DescriptorBlockwiseIn();
}


/********** haben Adresse empfangen ***********************************/
void DoSetAddress (void)
{ USB_SetAddress(CMD.SetupPacket.wValue);
  ACK();
}


/*********** USB-Request "SET CONFIGURATION" behandeln *************/
void DoSetConfiguration(void)
{ bool haveConfig;

  haveConfig = Class_Compare(CMD.SetupPacket.wValue);
  if (CMD.SetupPacket.wValue == 0)
        { CMD.Configuration = CMD.SetupPacket.wValue & 0xFF;
        }
  else
  if(haveConfig)
        { USB_ConfigDevice(true);
	  Class_Start();
          CMD.Configuration = CMD.SetupPacket.wValue &0xFF;
          ACK();
        }
   else  Stall(0);
}


/************* CDC Spezifisches **************************/

/*  "SET LINE CODING" behandeln */
void VCOM_SetLineCoding(void)
{ ACK();

    /* Vorbereitung auf Empfang von genau 7 Bytes vom Host   ???*/
}


/* Datenausgabe fr CDC-spezifischen USB-Request "SET LINE CODING" */
/* hier werden die empfangenen 7 Bytes aus dem USB-Puffer gelesen
   und im RAM gemerkt
*/
void SetLineCodingDataOut(void)
{ ReadControlBlock((byte*) &LineCoding, 7);
  ACK();
}

/* Zustand von DTR und RTS vom Host zum Gert merken */
void VCOM_Read_DTR_RTS (void)
{ Dtr_Rts = CMD.SetupPacket.wValue>>8;
  ACK();
}

/* CDC-spezifischer USB-Request "GET LINE CODING" behandeln */
void VCOM_GetLineCoding(void)
{ CMD.PacketLen   = EpCtrlMaxLen;
  CMD.TransferLen = 7;
  CMD.TransferPtr = (byte*) &LineCoding;
  DescriptorBlockwiseIn();    /* hier werden die 7 Bytes zum Host geschickt */
}


/************************** Setup-Event ***********************************/
/*
Merke:
1. Paket abholen und dann ClearBuffer.
2. Bei Setup Paketen, bei denen..
- ..nix hinterher kommt, also wo es keine Datenphase gibt,
  beantwortet man mit Senden eines leeren Paketes (ist ACK)
- ..anschlieend etwas zum Host gesendet werden mu, sendet man
  dies direkt danach. Wenn das zu Sendende grer ist als
  die EpBuffer-Lnge, dann nur ein Stck senden. Der Host holt es sich ab
  und der USB-Core gibt dann einen gewhnlichen Int auf Control-In, wo man
  dann das nchste Stck senden kann. Wiederholt sich, bis man nix mehr
  zu senden hat. Ob man als Abschlu ein leeres Paket senden mu oder nicht,
  ist ungewi.
  Wenn der Host zufrieden ist, sendet er ein leeres Paket als ACK.
- ..man anschlieend noch etwas vom Host bekommt, dann gibt es dafr ein
  anschlieendes Int auf Control-Out. Man liest das Paket und sendet dann
  als ACK ein leeres Paket.
*/

void OnSetup (byte EpCtrlStatus)
{ ReadControlBlock(&CMD.SetupPacket.bmRequestType, 8);
  if (IsStandardRequest())         /* wenn Type = Standard */
    { switch (CMD.SetupPacket.bRequest)
      { case SET_ADDRESS:       DoSetAddress();
                                return;

        case CLEAR_FEATURE:     DoSetClearFeature(false);
                                return;

        case SET_FEATURE:       DoSetClearFeature(true);
                                return;

        case GET_CONFIGURATION: CMD.PacketLen   = EpCtrlMaxLen;
                                CMD.TransferLen = 1;
                                CMD.TransferPtr = (byte*) &CMD.Configuration;
                                DescriptorBlockwiseIn();
                                return;

        case GET_STATUS:        DoGetStatus();
                                return;

        case GET_INTERFACE:     CMD.TransferLen = 1;
                                CMD.TransferPtr = (byte*) &always0;
                                DescriptorBlockwiseIn();
                                return;

        case SET_INTERFACE:     Class_Start();
                                ACK();
                                return;

        case GET_DESCRIPTOR:    DoGetDescriptor();
                                return;

        case SET_CONFIGURATION: DoSetConfiguration();
                                return;
      }
      /* auf alles andere reagieren wir mit Stall. siehe unten. */
    }

  if (IsClassRequest())        /* wenn Type = Class */
    { switch (CMD.SetupPacket.bRequest)
      { case SET_LINE_CODE:          VCOM_SetLineCoding();  return;
        case GET_LINE_CODE:          VCOM_GetLineCoding();  return;
        case SET_CONTROL_LINE_STATE: VCOM_Read_DTR_RTS();   return;

        /* falls es hier noch mehr Class-spezifische Requests
           geben sollte, dann Behandlung hier hinein.
         */

      }

    }

  if (IsVendorRequest())       /* wenn Type = Vendor */
    {
    }

    /* wenn keiner zustndig war, dann Stall! */
  Stall(0);
}



/******* die diversen Endpoint-Interrupts ************************************/


void  OnEpCtrlOut (byte EpCtrlStatus)                  /* Control-EP OUT */
{ byte tbuf[EpCtrlMaxLen];

  if (IsStandardRequest())                             /* wenn Type = Standard */
  { /* eigentlich nur leere Pakete, also ACK vom Host,
       aber mglich (nie gesehen) bRequest=7 = SET_DESCRIPTOR
     */
    ReadControlBlock(tbuf, EpCtrlMaxLen);
    return;
  }

  if (IsClassRequest())                                /* wenn Type = Class */
  { switch (CMD.SetupPacket.bRequest)
    { case SET_LINE_CODE:
               SetLineCodingDataOut();
               ACK();
	       return;

      default: ACK();
    }
    return;
  }

  /* nach Vendor-Request fragen wir hier garnicht erst */
  ACK();
}


void OnEpCtrlIn (byte EpCtrlStatus)                   /* Control-EP IN */
{ if ((EpCtrlStatus & (1<<4)) == 0)                   /* war kein NAK */
  { if (IsStandardRequest())                          /* wenn Type = Standard */
    { switch (CMD.SetupPacket.bRequest)
      { case GET_DESCRIPTOR:
                 DescriptorBlockwiseIn();
	         break;
        case GET_LINE_CODE:
                 ACK();
		 break;
	default: ACK();
      }
     return;
    }

    ACK();
    return;
  }
  ACK();
}


/********* BULK IN und OUT Interrupts **********/

void OnEpBulkIn (void)                  /* Bulk-EP IN */
{ int i, n;
  dword L, A;

 /* ohne INAK_AI gibt es kein Datensenden zum Host, weil der Int ja nur 1x
    kommt, wenn der Host das letzte Paket mit ACK beantwortet hat und man
    ab da dumm dasteht ohne ein weiteres Int.

    mit INAK_AI gibt es alle 11 Mikrosekunden !!! einen Int, weil die SIE
    zu bld ist. Unbrauchbar, da zu hufig.

    also:
    Wir nutzen den FRAME-Int, der INAK_BI alle Millisekunde einschaltet
    und das OnEpBulkIn schaltet ihn gleich wieder aus. Das geht und macht
    einen Int pro Millisekunde. Kostet ca. 0.2% der Rechenzeit.
  */
  SIE_Cmd(cmSetMode | SieCMD);          /* Device: Set Mode */
  SIE_Cmd((((1<<4)|1)<<16) | SieWR);    /* kein INAK_AI, aber INAK_AO + AP_CLK */


  if (txr==txw)  return;
  USBCtrl  = (logEpBulkIn << 2) | WR_EN;
  i = txw - txr;
  if (i<0) i += txLen;                  /* i = Anzahl zu sendender Bytes */
  if (i>EpBulkMaxLen) i = EpBulkMaxLen;
  A = 0;
  n = 0;
  USBTxPLen = i;
  while (i)
  { L = UsbTxBuf[txr];
    txr = (txr+1) & (txLen-1);
    A = A | (L << n);
    n += 8;
    if (n > 24)
    { USBTxData = A;
      n = 0;
      A = 0;
    }
    --i;
  }
  if (n) USBTxData = A;                 /* ggf. restliche Bytes ausgeben */
  ValidateBuffer(logEpBulkIn);
  USBCtrl = 0;
}


void OnEpBulkOut (void)                   /* Bulk-EP OUT */
{ int  i, n, hdroom, avail;
  long D;
  char c;

  /* Bulk EP anwhlen und Anzahl der Bytes ermittlen */
  USBCtrl = (logEpBulkOut << 2) | RD_EN;
  Nop(10);
  avail = USBRxPLen & 0x3FF;
  if ((USBRxPLen & (1<<10))==0)            /* wenn Paket invalid */
  { ClearBuffer(logEpBulkOut);             /* wir wollen's nicht */
    USBCtrl = 0;
    return;
  }


  i = rxw - rxr;
  if (i<0) i += rxLen;
  hdroom = rxLen - i;
  if (hdroom <= avail)
  { USBCtrl = 0;
    return;
  }
  i = avail;
  D = n = 0;
  while (i)
  { if (!n)
    { D = USBRxData;                       /* 4 Byte laden */
      n = 4;
    }
    c = D & 0xFF;                          /* LSB zuerst   */
    if (i)
    { UsbRxBuf[rxw] = c;
      rxw = (rxw+1) & (rxLen-1);
    }
    D = D >> 8;
    --n;
    --i;
  }
  ClearBuffer(logEpBulkOut);               /* wir haben's gelesen */
  USBCtrl = 0;
}



void OnEpIntOut (byte EpCtrlStatus)     /* Int-EP OUT */
{
  /* erstmal nix */
}

void OnEpIntIn (byte EpCtrlStatus)     /* Int-EP IN */
{
  /* erstmal nix */
}


/**************** USB-Interrupt-Handler **************************************/

void __irq USB_IRQHandler(void)
{ dword I;
  int   St;
  byte  B;

  I  = USBDevIntSt;                        /* Interrupt-Status nach I  */

  if (I & DEV_STAT)                          /* Busangelegenheiten */
    { USBDevIntClr = DEV_STAT;               /* Int lschen */
      St = GetDeviceStatus();
      if (St & 16)                           /* Bus Reset */
        { InitEndpoints();
	  return;
        }
       return;
    }


  /* Start of Frame Interrupt, nur bei Isochronous, alle 1 ms */
  /* dieses Bit kommt immer, egal ob enabled oder nicht */
  if (I & FRAME)
  { USBDevIntClr = FRAME;           /* Int lschen */
    SIE_Cmd(cmSetMode | SieCMD);    /* jede Millisekunde das INAK_BI wieder einschalten */
    SIE_Cmd((((1<<3)|(1<<4)|1)<<16) | SieWR);
  }

/* Endpoint Interrupts: Bit 1 = EP0, 2 = EP1 usw. */
/* Der Status mu VOR dem Int-Lschen gelesen werden, weil dort EPN und PO gelscht werden! */
  if (I & (2<<EpCtrlOut))           /* EP Control Out   */
    { B = GetEPStatus(EpCtrlOut);   /* Status lesen (bevor Int lschen) */
      USBDevIntClr = 2<<EpCtrlOut;  /* EP Event Flag lschen und.. */
      ClearInterrupt(EpCtrlOut);    /* ..per SIE lschen (es sind BEIDE Aktionen ntig!) */
      if (B & 4)
      OnSetup(B);                   /* Handle the Setup-Packet   */
      else
      OnEpCtrlOut(B);
    }

  if (I & (2<<EpCtrlIn))            /* EP Control In    */
    { B = GetEPStatus(EpCtrlIn);    /* Status lesen (bevor Int lschen) */
      USBDevIntClr = 2<<EpCtrlIn;   /* Clear event flag */
      ClearInterrupt(EpCtrlIn);
      OnEpCtrlIn(B);
    }

  if (I & (2<<EpIntOut))            /* EP Int Out    */
    { B = GetEPStatus(EpIntOut);    /* Status lesen (bevor Int lschen) */
      USBDevIntClr = 2<<EpIntOut;   /* Clear event flag */
      ClearInterrupt(EpIntOut);
      OnEpIntOut(B);
    }

  if (I & (2<<EpIntIn))             /* EP Int In    */
    { B = GetEPStatus(EpIntIn);     /* Status lesen (bevor Int lschen) */
      USBDevIntClr = 2<<EpIntIn;    /* Clear event flag */
      ClearInterrupt(EpIntIn);
      OnEpIntOut(B);
    }

  if (I & (2<<EpBulkOut))           /* EP Bulk Out, Host-->Device */
    { USBDevIntClr = 2<<EpBulkOut;  /* Clear event flag */
      ClearInterrupt(EpBulkOut);
      OnEpBulkOut();
    }

  if (I & (2<<EpBulkIn))            /* EP Bulk In, Device-->Host  */
    { USBDevIntClr = 2<<EpBulkIn;   /* Clear event flag */
      ClearInterrupt(EpBulkIn);
      OnEpBulkIn();
    }

   // ... weitere nach Bedarf
}



/************  USB-Setup **********************************/
/* Die Pins mssen bereits richtig gesetzt sein (PINSELx) */
/* ebenso Clock ein und Powerdown aus.                    */
/* Interrupt 47 ist USB-IRQ                               */
/**********************************************************/
word UsbSetup (void)
{ word b1, b2;

/* nur einbinden, wenn ntig..
  SYSAHBCLKCTRL |= (1<<14);
  PDRUNCFG      &= ~(1<<10);
*/

  Class_Start();                         /* LineCoding-Block aufsetzen mit unseren Defaultwerten */
  USBDevIntClr = 0xFFFFFFFF;
  USBDevFIQSel = 0;                      /* kein FIQ berall */
  NVIC_ISER1  |= (1<<15);                /* Interrupt 47=USB, enable */
  USB_SetAddress(0);
  USBDevIntEn  = FRAME | EP0 | EP1 | EP2 | EP3 | EP4 | DEV_STAT;

  SIE_Cmd(cmReadChipId | SieCMD);        /* Testregister lesen */
  SIE_Cmd(cmReadChipId | SieRD);
  b1 = USBCmdData;
  SIE_Cmd(cmReadChipId | SieRD);
  b2 = USBCmdData;
  SIE_Cmd(cmDeviceStatus | SieCMD);      /* SIE Reset */
  SIE_Cmd((1<<16) | SieWR);              /* CON setzen, sonst luft SIE nicht an */
  InitEndpoints();
  return (b2<<8)|b1;
}


/********** zeichenweises I/O und Pufferung und Kommunikation ***************/
/*
Diese Routinen werden von auerhalb im Usermode
aufgerufen und haben mit dem interrupt-gesteuerten
USB-Betrieb nichts zu tun.
*/

/* liefert true, wenn ein Zeichen abholbereit ist */
bool UsbRxAvail (void)
{ if (rxr!=rxw) return true;
  return false;
}

/* holt ein Zeichen vom USB ab */
/*
Achtung: wenn nix abzuholen ist, wird 0 zurckgeliefert
*/
char UsbGetChar (void)
{ char c;

  c = 0;
  if (rxr!=rxw)
  { c = UsbRxBuf[rxr];
    rxr = (rxr+1) & (rxLen-1);
  }
  return c;
}


/* liefert true, wenn noch ein Zeichen in den Tx-Buffer passt */
bool UsbTxReady (void)
{ int i;
  i = (txw+1) & (txLen-1);
  if (i==txr) return false;
  return true;
}

/* liefert true, wenn Tx-Buffer leer ist */
bool UsbTxEmpty (void)
{ return (txw==txr); }


/* sendet ein Zeichen (d.h. schreibt es in den Tx-Buffer) */
char UsbCharOut (char c)
{ int i;
  i = (txw+1) & (txLen-1);
  while (!UsbTxReady());   /* trampeln auf der Stelle!! */
  UsbTxBuf[txw] = c;
  txw = i;
  return c;
}

/* asciiz zum USB senden */
void UsbStrOut (char* S)
{ while (*S) UsbCharOut(*S++); }


