/* NXP  LPC2478:  USB als COM Port betreiben */
/*
Dieser Treiber implementiert einen virtuellen COM-Port auf dem LPC2478.
Er ist prinzipiell auch fr die LPC17xx geeignet, da diese den gleichen
USB-Core enthalten.

Einen Treiber gleicher Funktionalitt gibt es auch fr Nuvoton NUC120,
obwohl deren USB-Core gnzlich anders ist. Als Identitt wird in beiden
Fllen der Descriptorsatz von Nuvoton benutzt. Das Device hat eine
feste Serial-Nummer, wodurch es den gleichen COMxx benutzt, egal in
welchen USB-Slot es am PC gesteckt wird.

Installation am PC: siehe usb.h

Der eigentliche USB-Verkehr erfolgt komplett interrupt-gesteuert und
bentigt nach Aufruf von UsbSetup() keinerlei Hilfe von auen,
auer T0TC. Dieser Timer soll pro Millisekunde 1x inkrementieren, das
wird zur Zeitbegrenzung gebraucht.


Benutzung:
ber die 6 Funktionen im Headerfile "usb.h" genau so wie ein
gewhnlicher UART. Lediglich die Baudrate, Paritt usw. sind egal.
*/


#include "StdTypes.h"
#include "LPC17xx.h"
#include "usb.h"

/* die Transfer-Puffer fr zeichenweises In und Out ber den USB */
#define txLen  4096     /* immer 2er Potenz */
char UsbTxBuf[txLen];
int  txr, txw;

#define rxLen  256      /* immer 2er Potenz */
char UsbRxBuf[rxLen];
int  rxr, rxw;


/***************************  Konstanten ********************************/
#ifdef LPC17XXINCLUDED

#define USBClkCtrl       (*(volatile unsigned long *)(0x5000CFF4))
#define USBClkSt         (*(volatile unsigned long *)(0x5000CFF8))

/* Device interrupt registers */
#define USBIntSt        (*(volatile unsigned long *)(0x400FC1C0))
#define USBDevIntSt     (*(volatile unsigned long *)(0x5000C200))
#define USBDevIntEn     (*(volatile unsigned long *)(0x5000C204))
#define USBDevIntClr    (*(volatile unsigned long *)(0x5000C208))
#define USBDevIntSet    (*(volatile unsigned long *)(0x5000C20C))
#define USBDevIntPri    (*(volatile unsigned long *)(0x5000C22C))

/* Endpoint interrupt registers */
#define USBEpIntSt      (*(volatile unsigned long *)(0x5000C230))
#define USBEpIntEn      (*(volatile unsigned long *)(0x5000C234))
#define USBEpIntClr     (*(volatile unsigned long *)(0x5000C238))
#define USBEpIntSet     (*(volatile unsigned long *)(0x5000C23C))
#define USBEpIntPri     (*(volatile unsigned long *)(0x5000C240))

/* Endpoint realization registers */
#define USBReEp         (*(volatile unsigned long *)(0x5000C244))
#define USBEpIn         (*(volatile unsigned long *)(0x5000C248))
#define USBMaxPSize     (*(volatile unsigned long *)(0x5000C24C))

/* USB transfer registers */
#define USBRxData       (*(volatile unsigned long *)(0x5000C218))
#define USBRxPLen       (*(volatile unsigned long *)(0x5000C220))
#define USBTxData       (*(volatile unsigned long *)(0x5000C21C))
#define USBTxPLen       (*(volatile unsigned long *)(0x5000C224))
#define USBCtrl         (*(volatile unsigned long *)(0x5000C228))

/* SIE Command registers */
#define USBCmdCode      (*(volatile unsigned long *)(0x5000C210))
#define USBCmdData      (*(volatile unsigned long *)(0x5000C214))

/* DMA Register */
#define USBDMARSt       (*(volatile unsigned long *)(0x5000C250))
#define USBDMARClr      (*(volatile unsigned long *)(0x5000C254))
#define USBDMARSet      (*(volatile unsigned long *)(0x5000C258))
#define USBUDCAH        (*(volatile unsigned long *)(0x5000C280))
#define USBEpDMASt      (*(volatile unsigned long *)(0x5000C284))
#define USBEpDMAEn      (*(volatile unsigned long *)(0x5000C288))
#define USBEpDMADis     (*(volatile unsigned long *)(0x5000C28C))
#define USBDMAIntSt     (*(volatile unsigned long *)(0x5000C290))
#define USBDMAIntEn     (*(volatile unsigned long *)(0x5000C294))
#define USBEoTIntSt     (*(volatile unsigned long *)(0x5000C2A0))
#define USBEoTIntClr    (*(volatile unsigned long *)(0x5000C2A4))
#define USBEoTIntSet    (*(volatile unsigned long *)(0x5000C2A8))
#define USBNDDRIntSt    (*(volatile unsigned long *)(0x5000C2AC))
#define USBNDDRIntClr   (*(volatile unsigned long *)(0x5000C2B0))
#define USBNDDRIntSet   (*(volatile unsigned long *)(0x5000C2B4))
#define USBSysErrIntSt  (*(volatile unsigned long *)(0x5000C2B8))
#define USBSysErrIntClr (*(volatile unsigned long *)(0x5000C2BC))
#define USBSysErrIntSet (*(volatile unsigned long *)(0x5000C2C0))
#endif


#ifdef LPC24XXINCLUDED

#define USBClkCtrl      (*(volatile unsigned long *)(0xFFE0CFF4))
#define USBClkSt        (*(volatile unsigned long *)(0xFFE0CFF8))

/* Device interrupt registers */
#define USBIntSt        (*(volatile unsigned long *)(0xE01FC1C0))
#define USBPortSel      (*(volatile unsigned long *)(0xFFE0C110))
#define USBDevIntSt     (*(volatile unsigned long *)(0xFFE0C200))
#define USBDevIntEn     (*(volatile unsigned long *)(0xFFE0C204))
#define USBDevIntClr    (*(volatile unsigned long *)(0xFFE0C208))
#define USBDevIntSet    (*(volatile unsigned long *)(0xFFE0C20C))
#define USBCmdCode      (*(volatile unsigned long *)(0xFFE0C210))
#define USBCmdData      (*(volatile unsigned long *)(0xFFE0C214))
#define USBRxData       (*(volatile unsigned long *)(0xFFE0C218))
#define USBTxData       (*(volatile unsigned long *)(0xFFE0C21C))
#define USBRxPLen       (*(volatile unsigned long *)(0xFFE0C220))
#define USBTxPLen       (*(volatile unsigned long *)(0xFFE0C224))
#define USBCtrl         (*(volatile unsigned long *)(0xFFE0C228))
#define USBDevIntPri    (*(volatile unsigned long *)(0xFFE0C22C))

#define USBEpIntSt      (*(volatile unsigned long *)(0xFFE0C230))
#define USBEpIntEn      (*(volatile unsigned long *)(0xFFE0C234))
#define USBEpIntClr     (*(volatile unsigned long *)(0xFFE0C238))
#define USBEpIntSet     (*(volatile unsigned long *)(0xFFE0C23C))
#define USBEpIntPri     (*(volatile unsigned long *)(0xFFE0C240))

#define USBReEp         (*(volatile unsigned long *)(0xFFE0C244))
#define USBEpIn         (*(volatile unsigned long *)(0xFFE0C248))
#define USBMaxPSize     (*(volatile unsigned long *)(0xFFE0C24C))


#define USBDMARSt       (*(volatile unsigned long *)(0xFFE0C250))
#define USBDMARClr      (*(volatile unsigned long *)(0xFFE0C254))
#define USBDMARSet      (*(volatile unsigned long *)(0xFFE0C258))
#define USBUDCAH        (*(volatile unsigned long *)(0xFFE0C280))
#define USBEpDMASt      (*(volatile unsigned long *)(0xFFE0C284))
#define USBEpDMAEn      (*(volatile unsigned long *)(0xFFE0C288))
#define USBEpDMADis     (*(volatile unsigned long *)(0xFFE0C28C))
#define USBDMAIntSt     (*(volatile unsigned long *)(0xFFE0C290))
#define USBDMAIntEn     (*(volatile unsigned long *)(0xFFE0C294))
#define USBEoTIntSt     (*(volatile unsigned long *)(0xFFE0C2A0))
#define USBEoTIntClr    (*(volatile unsigned long *)(0xFFE0C2A4))
#define USBEoTIntSet    (*(volatile unsigned long *)(0xFFE0C2A8))
#define USBNDDRIntSt    (*(volatile unsigned long *)(0xFFE0C2AC))
#define USBNDDRIntClr   (*(volatile unsigned long *)(0xFFE0C2B0))
#define USBNDDRIntSet   (*(volatile unsigned long *)(0xFFE0C2B4))
#define USBSysErrIntSt  (*(volatile unsigned long *)(0xFFE0C2B8))
#define USBSysErrIntClr (*(volatile unsigned long *)(0xFFE0C2BC))
#define USBSysErrIntSet (*(volatile unsigned long *)(0xFFE0C2C0))
#endif



/******* Zuordnung physischer Endpunkte fr Realize: USBReEp ******/
/* ist beim LPC2478 fest zugeordnet!         */
#define  EpCtrlOut  0       /* phys. EP fr Control_Out        */
#define  EpCtrlIn   1       /* phys. EP fr Control_In         */
#define  EpIntOut   2       /* phys. EP fr Interrupt vom Host */
#define  EpIntIn    3       /* phys. EP fr Interrupt zum Host */
#define  EpBulkOut  4       /* phys. EP Bulk-Daten Host-->LPC  */
#define  EpBulkIn   5       /* phys. EP Bulk-Daten LPC-->Host  */

#define  logEpCtrl     (EpCtrlOut>>1)
#define  logEpInt      (EpIntOut >>1)
#define  logEpBulkOut  (EpBulkOut>>1)
#define  logEpBulkIn   (EpBulkIn >>1)

#define   EpCtrlMaxLen  64
#define   EpBulkMaxLen  64


/******* Codes der Standard-bRequest's im Setup-Paket ***/
#define GET_STATUS          0x00
#define CLEAR_FEATURE       0x01
#define SET_FEATURE         0x03
#define SET_ADDRESS         0x05
#define GET_DESCRIPTOR      0x06
#define SET_DESCRIPTOR      0x07
#define GET_CONFIGURATION   0x08
#define SET_CONFIGURATION   0x09
#define GET_INTERFACE       0x0A
#define SET_INTERFACE       0x0B
#define SYNC_FRAME          0x0C

/******* zustzliche bRequest-Codes fr virtuelle ComPorts */
#define SET_LINE_CODE               0x20    /* 7 Byte Paket mit Baudrate etc. */
#define GET_LINE_CODE               0x21    /* 7 Byte Paket mit Baudrate etc. */
#define SET_CONTROL_LINE_STATE      0x22    /* 2 Bit  DTR und RTS */
#define SEND_BREAK                  0x23    /* hier unbenutzt */



/******* Struktur des Setup-Paketes *****/
struct TSetupPaket
{ byte bmRequestType;  /* siehe oben */
  byte bRequest;       /* siehe Request-Tafel in USB-Doku */
  word wValue;         /* je nach Request */
  word wIndex;         /* je nach Request */
  word wLength;        /* Anzahl Bytes, wenn Data-Stage vorhanden ist */
};


/******* Struktur des internen Kommando- und Org-Datenblockes ****************/
struct TCommand
{ struct TSetupPaket SetupPacket;  /* das jeweils letzte Setup-Paket         */
  long   TransferLen;              /* noch zum Host zu sendende Anzahl Bytes */
  long   PacketLen;                /* wie lang das Paket zum Host sein darf  */
  byte*  TransferPtr;              /* zeigt auf die noch zu sendenden Bytes  */

  bool   RemoteWakeup;
  bool   SelfPowered;
  byte   Configuration;
};


/******* Struktur des Line-Coding-Blockes vom/zum PC ****************/
struct TLineCoding
{  dword  BaudRate;     /* Baud rate    */
   byte   Stopbits;     /* stop bits: 0: 1, 1: 1.5, 2: 2 Stop bits */
   byte   ParityType;   /* parity  0 - None, 1 - Odd, 2 - Even, 3 - Mark, 4 - Space */
   byte   DataBits;     /* data bits: 5, 6, 7, 8, 16  */
};


/************  Variablen *****************************************/
struct TCommand     CMD;
struct TLineCoding  LineCoding;
word                Dtr_Rts;

/************ Funktionen zum Starten des virtuellen COM-Portes *****/
void Class_Start (void)
{ LineCoding.BaudRate = 9600;
  LineCoding.Stopbits = 0;
  LineCoding.ParityType = 0;
  LineCoding.DataBits = 8;
  Dtr_Rts = 0;
  txr = txw = rxr = rxw = 0;
}

bool Class_Compare (word aValue)  /* immer true, wird hier nicht gebraucht */
{ return true; }



/************* die Descriptoren ************************************/
#define LEN_DEVICE         18
#define DESC_DEVICE         1
#define VID            0x0416   /* Vendor ID (von RealTek) */
#define PID            0x5011   /* Product ID */

const byte DeviceDescriptor[] =
{ LEN_DEVICE,             /* bLength              */
  DESC_DEVICE,            /* bDescriptorType      */
  0x00, 0x02,             /* bcdUSB               */
  0x02,                   /* bDeviceClass         */
  0x00,                   /* bDeviceSubClass      */
  0x00,                   /* bDeviceProtocol      */
  EpCtrlMaxLen,           /* bMaxPacketSize0      */
  0x16,                   /* Vendor  ID LO        */
  0x04,                   /* Vendor  ID HI        */
  0x11,                   /* Product ID LO        */
  0x50,                   /* Product ID HI        */
  0x00, 0x01,             /* bcdDevice            */
  0x01,                   /* iManufacturer        */
  0x02,                   /* iProduct             */
  0x03,                   /* iSerialNumber        */
  0x01                    /* bNumConfigurations   */
};

#define LEN_CONFIG          9
#define DESC_CONFIG         2

#define LEN_INTERFACE       9
#define DESC_INTERFACE      4

#define LEN_ENDPOINT        7
#define DESC_ENDPOINT       5

const byte ConfigDescriptor[] =
{ LEN_CONFIG,             /* bLength              */
  DESC_CONFIG,            /* bDescriptorType      */
  0x43, 0x00,             /* wTotalLength         */
  0x02,                   /* bNumInterfaces       */
  0x01,                   /* bConfigurationValue  */
  0x00,                   /* iConfiguration       */
  0xC0,                   /* bmAttributes         */
  0x32,                   /* MaxPower             */

                          /* INTERFACE descriptor */
  LEN_INTERFACE,          /* bLength              */
  DESC_INTERFACE,         /* bDescriptorType      */
  0x00,                   /* bInterfaceNumber     */
  0x00,                   /* bAlternateSetting    */
  0x01,                   /* bNumEndpoints        */
  0x02,                   /* bInterfaceClass      */
  0x02,                   /* bInterfaceSubClass   */
  0x01,                   /* bInterfaceProtocol   */
  0x00,                   /* iInterface           */

                          /* Communication Class Specified INTERFACE descriptor */
  0x05,                   /* Size of the descriptor, in bytes */
  0x24,                   /* CS_INTERFACE descriptor type */
  0x00,                   /* Header functional descriptor subtype */
  0x10, 0x01,             /* Communication device compliant to the communication */
                          /* spec. ver. 1.10 */
                          /* Communication Class Specified INTERFACE descriptor */
  0x05,                   /* Size of the descriptor, in bytes */
  0x24,                   /* CS_INTERFACE descriptor type */
  0x01,                   /* Call management functional descriptor */
  0x00,                   /* BIT0: Whether device handle call management itself. */
                          /* BIT1: Whether device can send/receive call  */
                          /* management information over a Data Class Interface 0 */
  0x01,                   /* Interface number of data class interface optionally */
                          /* used for call management */
                          /* Communication Class Specified INTERFACE descriptor */
  0x04,                   /* Size of the descriptor, in bytes */
  0x24,                   /* CS_INTERFACE descriptor type */
  0x02,                   /* Abstract control management funcational descriptor */
                          /* subtype */
  0x00,                   /* bmCapabilities       */
                          /* Communication Class Specified INTERFACE descriptor */
  0x05,                   /* bLength              */
  0x24,                   /* bDescriptorType: CS_INTERFACE descriptor type */
  0x06,                   /* bDescriptorSubType   */
  0x00,                   /* bMasterInterface     */
  0x01,                   /* bSlaveInterface0     */


  /* ENDPOINT descriptor fr Interrupt */
  LEN_ENDPOINT,           /* bLength              */
  DESC_ENDPOINT,          /* bDescriptorType      */
  0x80+logEpInt,          /* bEndpointAddress     */
  3,                      /* Attribute: Interrupt */
  8, 0x00,                /* wMaxPacketSize       */
  5,                      /* bInterval  5 ms ?    */


  /* INTERFACE descriptor */
  LEN_INTERFACE,          /* bLength              */
  DESC_INTERFACE,         /* bDescriptorType      */
  0x01,                   /* bInterfaceNumber     */
  0x00,                   /* bAlternateSetting    */
  0x02,                   /* bNumEndpoints        */
  0x0A,                   /* bInterfaceClass      */
  0x00,                   /* bInterfaceSubClass   */
  0x00,                   /* bInterfaceProtocol   */
  0x00,                   /* iInterface           */


  /* ENDPOINT descriptor fr Bulk IN */
  LEN_ENDPOINT,           /* bLength              */
  DESC_ENDPOINT,          /* bDescriptorType      */
  0x80+logEpBulkIn,       /* bEndpointAddress     */
  2,                      /* Attribute: Bulk      */
  EpBulkMaxLen, 0x00,     /* wMaxPacketSize       */
  0,                      /* bInterval in ms      */


  /* ENDPOINT descriptor fr Bulk OUT */
  LEN_ENDPOINT,           /* bLength              */
  DESC_ENDPOINT,          /* bDescriptorType      */
  logEpBulkOut,           /* bEndpointAddress     */
  2,                      /* Attribute: Bulk      */
  EpBulkMaxLen, 0x00,     /* wMaxPacketSize       */
  0,                      /* bInterval in 2ms     */
};


#define DESC_STRING         3
const byte StringLang[] =
{ 4,                      /* bLength                  */
  DESC_STRING,            /* bDescriptorType          */
  0x09, 0x04              /* Language ID: USA(0x0409) */
};


const byte VendorStringDescriptor[] =
{ 16,                     /* bLength          */
  DESC_STRING,            /* bDescriptorType  */
  'N', 0,
  'u', 0,
  'v', 0,
  'o', 0,
  't', 0,
  'o', 0,
  'n', 0
};

const byte ProductStringDescriptor[] =
{ 32,                     /* bLength          */
  DESC_STRING,            /* bDescriptorType  */
  'U', 0,
  'S', 0,
  'B', 0,
  ' ', 0,
  'V', 0,
  'i', 0,
  'r', 0,
  't', 0,
  'u', 0,
  'a', 0,
  'l', 0,
  ' ', 0,
  'C', 0,
  'O', 0,
  'M', 0
};

const byte StringSerial[26] =
{ 26,                     /* bLength          */
  DESC_STRING,            /* bDescriptorType  */
  'N', 0,
  'T', 0,
  '2', 0,
  '0', 0,
  '0', 0,
  '9', 0,
  '1', 0,
  '0', 0,
  '1', 0,
  '4', 0,
  '0', 0,
  '0', 0
};


/* um Nullbyte oder ein leeres Paket senden zu knnen */
const byte always0 = 0;

/************  Hilfsroutinen ************************************************/

/* Interrupt-Bits in USBDevIntSt */
#define  FRAME    (1<<0)
#define  EP_FAST  (1<<1)
#define  EP_SLOW  (1<<2)
#define  DEV_STAT (1<<3)
#define  CCEMPTY  (1<<4)
#define  CDFULL   (1<<5)
#define  RxENDPKT (1<<6)
#define  TxENDPKT (1<<7)
#define  EP_RLZD  (1<<8)
#define  ERR_INT  (1<<9)


/* Warten auf Bits in USBDevIntSt */
bool WarteAufInt (dword WantedBits)
{ long L;
  L =  1000;
  while (--L)
  { if ((USBDevIntSt & WantedBits)== WantedBits) return true;
  }
  return false;
}


/* SIE Kommandos */
#define  cmSetAddress     (0xD0 << 16)
#define  cmConfigDevice   (0xD8 << 16)
#define  cmClearBuf       (0xF2 << 16)
#define  cmSetMode        (0xF3 << 16)
#define  cmReadFrameNum   (0xF5 << 16)
#define  cmValidateBuf    (0xFA << 16)
#define  cmGetErrorStatus (0xFB << 16)
#define  cmReadTestReg    (0xFD << 16)
#define  cmDeviceStatus   (0xFE << 16)
#define  cmGetErrorCode   (0xFF << 16)

/* bler Fehler im UM10237: SieRD und SieWR sind vertauscht!
  (ist bei NXP seit Jahren bekannt!!)
 */
#define  SieRD  0x200
#define  SieWR  0x100
#define  SieCMD 0x500


/* Kommando, Read und Write zur SIE senden */
bool SIE_Cmd (dword Kommando)
{ dword L, D;
  Kommando = Kommando & 0x00FF0700; /* Angst? */
  D = Kommando & 0x700;             /* CMD, RD, WR separieren */
  L = CDFULL;                       /* bei Write + Command */
  if (D!=SieRD) L = CCEMPTY;        /* bei Read            */
  USBDevIntClr = L;                 /* Clear Flag          */
  USBCmdCode = Kommando;            /* auslsen            */
  if (!WarteAufInt(L)) return false;
  return true;
}


int GetDeviceStatus (void)
{ byte B;
   if (!SIE_Cmd(cmDeviceStatus | SieCMD))    /* Get Device Status */
     return -1;
   if (!SIE_Cmd(cmDeviceStatus | SieRD))
     return -1;
   B = USBCmdData & 0xFF;                    /* Device Status */
   /* mit jedem Lesen handelt man sich einen neuen sinnlosen Int ein  */
   return B;
}


int GetEPStatus (int physEpNum)
{ dword D;
  D = physEpNum << 16;
  if (!SIE_Cmd(D | SieCMD)) return -1;  /* Select EP */
  if (!SIE_Cmd(D | SieRD))  return -1;  /* read..  */
  return USBCmdData & 0xFF;
}



bool Stall (int physEpNum)
{ dword D;
  D = (0x40 + physEpNum)<<16;
  if (!SIE_Cmd(D | SieCMD)) return false;        /* Set EP Status */
  if (!SIE_Cmd((1<<16) | SieWR))  return false;  /* Stall Bit auf 1 */
  return true;
}

bool StallLogEP (int logEpNum)
{ int i;
  bool e;
  i = logEpNum<<1;
  e = Stall(i);
  e = e && Stall(i+1);
  return e;
}


bool UnStall (int physEpNum)
{ dword D;
  D = (0x40 + physEpNum)<<16;
  if (!SIE_Cmd(D | SieCMD)) return false;     /* Set EP Status */
  if (!SIE_Cmd(0 | SieWR))  return false;     /* Stall Bit auf 0 */
  return true;
}

bool UnStallLogEP (int logEpNum)
{ int i;
  bool e;
  i = logEpNum<<1;
  e = UnStall(i);
  e = e && UnStall(i+1);
  return e;
}


int ClearBuffer (dword logEpNum)
{ int   i;
  dword D;

  i = 0;
  D = logEpNum << 17;                           /* OUT ist even */
  if (!SIE_Cmd(D | SieCMD))          i = -1;    /* Select EP */
  if (!SIE_Cmd(cmClearBuf | SieCMD)) i = -1;    /* Clear Buffer */
  if (!SIE_Cmd(cmClearBuf | SieRD))  i = -1;    /* read Status  */
  if (i==0) i = USBCmdData & 0xFF;
  return i;
}


int ValidateBuffer (dword logEPNum)
{ int   i;
  dword D;

  D = (logEPNum<<17) | (1<<16);        /* IN ist odd */
  if (!SIE_Cmd(D | SieCMD)) return -1; /* Select EP */
  if (!SIE_Cmd(D | SieRD )) return -1; /* Select EP */
  i = USBCmdData & 0xFF;
  if (!SIE_Cmd(cmValidateBuf | SieCMD)) return -1; /* Validate Buffer */
  return i;
}


bool USB_SetAddress (byte adr)
{ dword D;
  D = adr | 128;
  D = D << 16;
  if (!SIE_Cmd(cmSetAddress | SieCMD)) return false;
  if (!SIE_Cmd(D | SieWR)) return false;
  return true;
}

bool USB_ConfigDevice (bool obConf)
{ dword D;
  D = 0;
  if (obConf)  D = (1<<16);
  if (!SIE_Cmd(cmConfigDevice | SieCMD)) return false;
  if (!SIE_Cmd(D | SieWR)) return false;
  return true;
}



/* physische Endpunkte aufsetzen (bei Reset-Kommando usw.) */

void InitEndpoints (void)
{ dword D;

  CMD.Configuration = 0;            /* vor "CONFIGURED" ist hier nix */
  CMD.TransferLen = 0;              /* es stehen ab hier auch */
  CMD.PacketLen   = 0;              /* keine Transfers an */
  CMD.TransferPtr = 0;


/* Control Out */
  USBDevIntClr = 1<<8;        /* Clear EP_RLZD in USBDevIntSt  */
  USBReEp = 1<<EpCtrlOut;     /* Enable Bit setzen */
  USBEpIn = EpCtrlOut;        /* phys. EP */
  USBMaxPSize = EpCtrlMaxLen; /* 64 Byte Paket */
  WarteAufInt(EP_RLZD);

/* Control In */
  USBDevIntClr = 1<<8;         /* Clear EP_RLZD in USBDevIntSt  */
  USBReEp |= 1<<EpCtrlIn;      /* Enable Bit setzen */
  USBEpIn = EpCtrlIn;          /* phys. EP */
  USBMaxPSize = EpCtrlMaxLen;  /* 64 Byte Paket */
  WarteAufInt(EP_RLZD);

/* Interrupt In */
  USBDevIntClr = 1<<8;         /* Clear EP_RLZD in USBDevIntSt  */
  USBReEp |= 1<<EpIntIn;       /* Enable Bit setzen */
  USBEpIn = EpIntIn;           /* phys. EP */
  USBMaxPSize = EpBulkMaxLen;  /* 64 Byte Paket */
  WarteAufInt(EP_RLZD);

/* Bulk Out */
  USBDevIntClr = 1<<8;         /* Clear EP_RLZD in USBDevIntSt  */
  USBReEp |= 1<<EpBulkOut;     /* Enable Bit setzen */
  USBEpIn = EpBulkOut;         /* phys. EP */
  USBMaxPSize = EpBulkMaxLen;  /* 64 Byte Paket */
  WarteAufInt(EP_RLZD);

/* Bulk In */
  USBDevIntClr = 1<<8;         /* Clear EP_RLZD in USBDevIntSt  */
  USBReEp |= 1<<EpBulkIn;      /* Enable Bit setzen */
  USBEpIn = EpBulkIn;          /* phys. EP */
  USBMaxPSize = EpBulkMaxLen;  /* 64 Byte Paket */
  WarteAufInt(EP_RLZD);


/* Mode setzen, Vorsicht mit INAK_BI, Trick dafr siehe OnEpBulkIn() */
  D = (1<<5) |                 /* INAK_BI (BULK-IN: Int bei NAK)  */
      (1<<6) |                 /* INAK_BO (BULK-OUT: Int bei NAK) */
      1;                       /* AP_CLK = 1 (48 MHz immer an) */

  SIE_Cmd(cmSetMode | SieCMD); /* Device: Set Mode */
  SIE_Cmd((D<<16) | SieWR);    /* BulkIO mit NAK + immer CLK */

  USBEpIntClr  = 0xFFFFFFFF;
  USBEpIntEn   = (1<<EpCtrlOut) |      /* Control_Out */
                 (1<<EpCtrlIn)  |      /* Control_In  */
                 (1<<EpBulkOut) |      /* Bulk_Out    */
                 (1<<EpBulkIn);        /* Bulk_In     */

  USBDevIntClr = 0xFFFFFFFF;
  USBDevIntEn  = FRAME    |
                 EP_SLOW  |
                 DEV_STAT |
                 ERR_INT;
}




int ReadControlBlock (byte* PBuffer, int maxlen)
{ int count, i, n;
  dword D, L;

  USBCtrl = ((logEpCtrl & 15) << 2) | 1;         /* CTRL_RD_EN */
  L = 2000;
  while ((--L) && (0==(USBRxPLen & (1<<11))));   /* warten auf Paket Ready */
  count = USBRxPLen & 0x3FF;
  if (count > maxlen) count = maxlen;

  if(count)
  { n = 4;
    i = count;
    D = USBRxData;
    while (i > 0)
    { *PBuffer = D & 0xFF;
      D = D >> 8;
      --n;
      if (!n) { D = USBRxData; n = 4; }
      --i;
      ++PBuffer;
    }
  }
  USBCtrl = 0;
  ClearBuffer(logEpCtrl);
  return count;
}



int WriteControlBlock (byte* PBuffer, int count)
{ dword A, L;
  int  i, n;

  if (count > EpCtrlMaxLen) count = EpCtrlMaxLen;
  USBCtrl   = ((logEpCtrl & 15) << 2) | 2;    /* CTRL_WR_EN */
  USBTxPLen = count;

  if (count)
  { A = 0; i = 0; n = 0;
    while (i < count)
    { L = *PBuffer++;
      A = A | (L << n);
      n += 8;
      if (n > 24)
      { n = 0;
        USBTxData = A;
        A = 0;
      }
      ++i;
    }
    if (n) USBTxData = A;
  }
  USBCtrl = 0;
  ValidateBuffer(logEpCtrl);
  return count;
}


void ACK (void)
{ WriteControlBlock((byte*) &always0, 0); }


/* Request-Typ im Setup-Packet testen (Standard, Class, Vendor) */
bool IsStandardRequest (void)
{ return (CMD.SetupPacket.bmRequestType & 0x60)==0; }

bool IsClassRequest (void)
{ return (CMD.SetupPacket.bmRequestType & 0x60)==0x20; }

bool IsVendorRequest (void)
{ return (CMD.SetupPacket.bmRequestType & 0x60)==0x40; }



/******* anstehende Control-Transfers zum Host blockweise starten *******/
void DescriptorBlockweiseIn (void)
{ int i, j;
  byte* Q;

  if ((CMD.SetupPacket.bmRequestType & 0x80)==0) return;
  i = CMD.TransferLen;
  if (i > CMD.PacketLen) i = CMD.PacketLen;
  Q = CMD.TransferPtr;                   /* Quelle */
  j = WriteControlBlock(Q, i);
  CMD.TransferPtr = Q + j;               /* Zeiger auf evt. Rest merken */
  CMD.TransferLen = CMD.TransferLen - j; /* restliche Anzahl Bytes */
  if (CMD.TransferLen < 0) CMD.TransferLen = 0;
}

/**********************************************************************/
/************ Bearbeitung eingegangener Requests **********************/
/**********************************************************************/



/********** USB-Request "SET FEATURE" und "CLEAR FEATURE" behandeln ****/
void DoSetClearFeature (bool value)
{ int Feature;
  int FuerWen;
  int EP;

  Feature = CMD.SetupPacket.wValue;
  FuerWen = CMD.SetupPacket.bmRequestType;
  EP      = CMD.SetupPacket.wIndex;

  switch(FuerWen)
  { case 0:          /* fr Device */
        if (Feature==1) CMD.RemoteWakeup = value;
        break;

    case 1:          /* fr Interface */
        break;

    case 2:          /* fr einen Endpoint */
        if (Feature==0)
        { if (value==false)
              StallLogEP(EP);
          else
              UnStallLogEP(EP);
        }
        break;

    default: Stall(1);  /* quasi NAK senden */
  }
}


/******** USB-Request "GET STATUS" behandeln ***************************/
void DoGetStatus (void)
{ byte Buf[4];
  int FuerWen;
  int  EP;

  FuerWen = CMD.SetupPacket.bmRequestType;
  EP = CMD.SetupPacket.wIndex;

  Buf[0] = 0;

  switch(FuerWen)
  { case 0x80:          /* fr Device */
         if (CMD.RemoteWakeup) Buf[0] |= 2;
         if (CMD.SelfPowered)  Buf[0] |= 1;
         break;

    case 0x81:          /* fr Interface */
         break;

    case 0x82:          /* fr einen Endpoint */
         if ((EP==logEpCtrl)   ||
             (EP==logEpInt)    ||
             (EP==logEpBulkIn) ||
             (EP==logEpBulkIn))  Buf[0] = 1;
         break;

    default:          /* hier eigentlich NAK senden */
         Stall(0);
         return;
  }

  Buf[1] = 0;
  CMD.PacketLen   = EpCtrlMaxLen;
  CMD.TransferLen = 2;
  CMD.TransferPtr = Buf;
  DescriptorBlockweiseIn();
}


/******** Descriptoren zum Host senden *********************************/
void DoGetDescriptor(void)
{ word Type, Index;
  int  aLen;
  const byte* P;

  Type    = CMD.SetupPacket.wValue >> 8;
  Index   = CMD.SetupPacket.wValue & 0xFF;
  aLen    = -1;
  P       = 0;

  switch (Type)
  { case DESC_DEVICE:                            /* Get Device Descriptor */
         { aLen = LEN_DEVICE;
           P    = DeviceDescriptor;
           break;
         }

    case DESC_CONFIG:                            /* Get Configuration Descriptor    */
         { aLen = ConfigDescriptor[3];           /* Total-Lnge ist WORD            */
           aLen = (aLen << 8) | ConfigDescriptor[2];
	   P    = ConfigDescriptor;
           break;
         }


    case DESC_STRING:                            /* Get String Descriptor */
         { switch (Index)                        /* Get String Descriptor */
           { case 0: aLen = 4;
                     P    = StringLang;
                     break;
             case 1: aLen = VendorStringDescriptor[0];
                     P    = VendorStringDescriptor;
                     break;
             case 2: aLen = ProductStringDescriptor[0];
                     P    = ProductStringDescriptor;
                     break;
             case 3: aLen = StringSerial[0];
                     P    = StringSerial;
                     break;
             default: Stall(1);       /* kennen wir nicht. Stall. */
                      aLen = -1;
           }
           break;
         }
     default: { Stall(1);    /* kennen wir nicht. Stall. */
                aLen = -1;
              }
     }

  if (aLen < 0 ) return;
  /* nicht mehr senden wollen, als der Host haben will */
  if (aLen > CMD.SetupPacket.wLength) aLen = CMD.SetupPacket.wLength;
  CMD.PacketLen   = EpCtrlMaxLen;
  CMD.TransferLen = aLen;
  CMD.TransferPtr = (byte*) P;
  DescriptorBlockweiseIn();
}


/********** haben Adresse empfangen ***********************************/
void DoSetAddress (void)
{ USB_SetAddress(CMD.SetupPacket.wValue);
  ACK();
}


/*********** USB-Request "SET CONFIGURATION" behandeln *************/
void DoSetConfiguration(void)
{ bool haveConfig;

  haveConfig = Class_Compare(CMD.SetupPacket.wValue);
  if (CMD.SetupPacket.wValue == 0)
        { CMD.Configuration = CMD.SetupPacket.wValue & 0xFF;
        }
  else
  if(haveConfig)
        { USB_ConfigDevice(true);
	  Class_Start();
          CMD.Configuration = CMD.SetupPacket.wValue &0xFF;
          ACK();
        }
   else  Stall(0);
}


/************* CDC Spezifisches **************************/

/*  "SET LINE CODING" behandeln */
void VCOM_SetLineCoding(void)
{ ACK();   /* Vorbereitung auf Empfang von genau 7 Bytes vom Host */
}


/* Datenausgabe fr CDC-spezifischen USB-Request "SET LINE CODING" */
/* hier werden die empfangenen 7 Bytes aus dem USB-Puffer gelesen
   und im RAM gemerkt
*/
void SetLineCodingDataOut(void)
{ ReadControlBlock((byte*) &LineCoding, 7);
  ACK();
}

/* Zustand von DTR und RTS vom Host zum Gert merken */
void VCOM_Read_DTR_RTS (void)
{ Dtr_Rts = CMD.SetupPacket.wValue>>8;
  ACK();
}

/* CDC-spezifischer USB-Request "GET LINE CODING" behandeln */
void VCOM_GetLineCoding(void)
{ CMD.PacketLen   = EpCtrlMaxLen;
  CMD.TransferLen = 7;
  CMD.TransferPtr = (byte*) &LineCoding;
  DescriptorBlockweiseIn();    /* hier werden die 7 Bytes zum Host geschickt */
}


/************************** Setup-Event ***********************************/
/*
Merke:
1. Paket abholen und dann ClearBuffer.
2. Bei Setup Paketen, bei denen..
- ..nix hinterher kommt, also wo es keine Datenphase gibt,
  beantwortet man mit Senden eines leeren Paketes (ist ACK)

- ..anschlieend etwas zum Host gesendet werden mu, sendet man
  dies direkt danach. Wenn das zu Sendende grer ist als
  die EpBuffer-Lnge, dann nur ein Stck senden. Der Host holt es sich ab
  und der USB-Core gibt dann einen gewhnlichen Int auf Control-In, wo man
  dann das nchste Stck senden kann. Wiederholt sich, bis man nix mehr
  zu senden hat. Ob man als Abschlu ein leeres Paket senden mu oder nicht,
  ist ungewi. Hier wird es gemacht. Wenn der Host zufrieden ist, sendet er
  ein leeres Paket als ACK.

- ..man anschlieend noch etwas vom Host bekommt, dann gibt es dafr ein
  anschlieendes Int auf Control-Out. Man liest das Paket und sendet dann
  als ACK ein leeres Paket.
*/

void OnSetup (byte EpCtrlStatus)
{ ReadControlBlock(&CMD.SetupPacket.bmRequestType, 8); /* Paket abholen */

  if (IsStandardRequest())         /* wenn Type = Standard */
    { switch (CMD.SetupPacket.bRequest)
      { case SET_ADDRESS:       DoSetAddress();  return;
        case CLEAR_FEATURE:     DoSetClearFeature(false); return;
        case SET_FEATURE:       DoSetClearFeature(true);  return;
        case GET_CONFIGURATION: CMD.PacketLen   = EpCtrlMaxLen;
                                CMD.TransferLen = 1;
                                CMD.TransferPtr = (byte*) &CMD.Configuration;
                                DescriptorBlockweiseIn();
                                return;
        case GET_STATUS:        DoGetStatus();  return;
        case GET_INTERFACE:     CMD.TransferLen = 1;
                                CMD.TransferPtr = (byte*) &always0;
                                DescriptorBlockweiseIn();
                                return;
        case SET_INTERFACE:     Class_Start(); ACK(); return;
        case GET_DESCRIPTOR:    DoGetDescriptor(); return;
        case SET_CONFIGURATION: DoSetConfiguration(); return;
      }
      /* auf alles andere reagieren wir mit Stall. siehe unten. */
    }

  if (IsClassRequest())        /* wenn Type = Class */
    { switch (CMD.SetupPacket.bRequest)
      { case SET_LINE_CODE:          VCOM_SetLineCoding();  return;
        case GET_LINE_CODE:          VCOM_GetLineCoding();  return;
        case SET_CONTROL_LINE_STATE: VCOM_Read_DTR_RTS();   return;

        /* falls es hier noch mehr Class-spezifische Requests
           geben sollte, dann Behandlung hier hinein.
         */

      }

    }

  if (IsVendorRequest())       /* wenn Type = Vendor */
    {  /* wird hier nicht gebraucht */
    }

    /* wenn keiner zustndig war, dann Stall! */
  Stall(0);
}



/******* die diversen Endpoint-Interrupts ************************************/

void  OnEpCtrlOut (byte EpCtrlStatus)                  /* Control-EP OUT */
{ byte tbuf[EpCtrlMaxLen];

  if (IsStandardRequest())                             /* wenn Type = Standard */
  { /* eigentlich nur leere Pakete, also ACK vom Host,
       aber mglich (nie gesehen) bRequest=7 = SET_DESCRIPTOR
     */
    ReadControlBlock(tbuf, EpCtrlMaxLen);
    return;
  }

  if (IsClassRequest())                                /* wenn Type = Class */
  { switch (CMD.SetupPacket.bRequest)
    { case SET_LINE_CODE:
               SetLineCodingDataOut();
               ACK();
	       return;

      default: ACK();
    }
    return;
  }

  /* nach Vendor-Request fragen wir hier garnicht erst */
  ACK();
}


void OnEpCtrlIn (byte EpCtrlStatus)                   /* Control-EP IN */
{ if ((EpCtrlStatus & (1<<4)) == 0)                   /* war kein NAK */
  { if (IsStandardRequest())                          /* wenn Type = Standard */
    { switch (CMD.SetupPacket.bRequest)
      { case GET_DESCRIPTOR:
                 DescriptorBlockweiseIn();
	         break;
        case GET_LINE_CODE:
                 ACK();
		 break;
	default: ACK();
      }
     return;
    }

    ACK();  return;
  }
  ACK();
}


/********* BULK IN und OUT Interrupts **********/

void OnEpBulkIn (void)                  /* Bulk-EP IN */
{ int i, n;
  dword L, A;

 /* ohne INAK_BI gibt es kein Datensenden zum Host, weil der Int ja nur 1x
    kommt, wenn der Host das letzte Paket mit ACK beantwortet hat und man
    ab da dumm dasteht ohne ein weiteres Int.

    mit INAK_BI gibt es alle 11 Mikrosekunden !!! einen Int, weil die SIE
    zu bld ist. Unbrauchbar, da zu hufig.

    also:
    Wir nutzen den FRAME-Int, der INAK_BI alle Millisekunde einschaltet
    und das OnEpBulkIn schaltet ihn gleich wieder aus. Das geht und macht
    einen Int pro Millisekunde. Kostet ca. 0.2% der Rechenzeit.
  */
  SIE_Cmd(cmSetMode | SieCMD);          /* Device: Set Mode */
  SIE_Cmd((((1<<6)|1)<<16) | SieWR);    /* BulkIO mit NAK + immer CLK */

  if (txr==txw)  return;
  USBCtrl  = (logEpBulkIn << 2) | 2;    /* CTRL_WR_EN */
  i = txw - txr;
  if (i<0) i += txLen;                  /* i = Anzahl zu sendender Bytes */
  if (i>EpBulkMaxLen) i = EpBulkMaxLen;
  A = 0;
  n = 0;
  USBTxPLen = i;
  while (i)
  { L = UsbTxBuf[txr];
    txr = (txr+1) & (txLen-1);
    A = A | (L << n);
    n += 8;
    if (n > 24)
    { USBTxData = A;
      n = 0;
      A = 0;
    }
    --i;
  }
  if (n) USBTxData = A;                 /* ggf. restliche Bytes ausgeben */
  ValidateBuffer(logEpBulkIn);
  USBCtrl = 0;
}


void OnEpBulkOut (void)                 /* Bulk-EP OUT */
{ int  i, n, hdroom, avail;
  long D;
  char c;

  /* Bulk EP anwhlen und Anzahl der Bytes ermittlen */
  USBCtrl = (logEpBulkOut << 2) | 1;    /* CTRL_RD_EN */
  i = 100;
  while (--i)
  { if (USBRxPLen & (1<<11))            /* warten auf Paket Ready */
    goto lenvalid;
  }
  USBCtrl = 0;
  return;

  lenvalid:                             /* Daten auslesen */
  avail = USBRxPLen & 0x3FF;
  if ((USBRxPLen & (1<<10))==0)         /* wenn Paket invalid */
  { USBCtrl = 0;
    ClearBuffer(logEpBulkOut);          /* wir wollen's nicht */
    return;
  }


  i = rxw - rxr;
  if (i<0) i += rxLen;
  hdroom = rxLen - i;
  if (hdroom <= avail)
  { USBCtrl = 0;
    return;
  }
  i = avail;
  D = n = 0;
  while (i)
  { if (!n)
    { D = USBRxData;                       /* 4 Byte laden */
      n = 4;
    }
    c = D & 0xFF;                          /* LSB zuerst   */
    if (i)
    { UsbRxBuf[rxw] = c;
      rxw = (rxw+1) & (rxLen-1);
    }
    D = D >> 8;
    --n;
    --i;
  }
  ClearBuffer(logEpBulkOut);               /* wir haben's gelesen */
  USBCtrl = 0;
}



void OnEpIntOut (byte EpCtrlStatus)     /* Int-EP OUT */
{ /* erstmal nix */
}

void OnEpIntIn (byte EpCtrlStatus)     /* Int-EP IN */
{  /* erstmal nix */
}


/**************** USB-Interrupt-Handler **************************************/

void __irq USB_IRQHandler(void)
{ dword I, E;
  int   St;
  byte  B;

  I  = USBDevIntSt;                       /* Interrupt-Status nach I  */
  E  = USBEpIntSt;                        /* EP Event Flags lesen     */


  /* Error Interrupt, nur zum Debuggen */
  if (I & ERR_INT)
  { USBDevIntClr = ERR_INT;
    SIE_Cmd(cmGetErrorCode | SieCMD);     /* Get Err Code */
    SIE_Cmd(cmGetErrorCode | SieRD);
    B = USBCmdData & 0xFF;                /* Error Code   */

    SIE_Cmd(cmGetErrorStatus | SieCMD);   /* Get Err Status */
    SIE_Cmd(cmGetErrorStatus | SieRD);
    B = USBCmdData & 0xFF;                /* Error Status   */

    /* Auswertung ggf. hier */

  }


  if (I & DEV_STAT)                       /* Busangelegenheiten */
    { USBDevIntClr = DEV_STAT;            /* Int lschen */
      St = GetDeviceStatus();
      if (St & 16) InitEndpoints();       /* Bus Reset */
      return;
    }


  /* Start of Frame Interrupt, nur bei Isochronous, alle 1 ms */
  /* dieses Bit kommt immer, egal ob sein Int enabled ist oder nicht */
  if (I & FRAME)
  { USBDevIntClr = FRAME;           /* Int lschen */
    SIE_Cmd(cmSetMode | SieCMD);    /* jede Millisekunde das INAK_BI wieder einschalten */
    SIE_Cmd((((1<<5)|(1<<6)|1)<<16) | SieWR);
  }


  if (I & EP_SLOW)                  /* Endpoint Int's    */
  { if (E & (1<<EpCtrlOut))         /* EP Control Out    */
    { B = GetEPStatus(EpCtrlOut);   /* Status lesen (bevor Int lschen) */
      USBEpIntClr = 1<<EpCtrlOut;   /* EP Event Flag lschen   */
      if (B & 4)
      OnSetup(B);                   /* Handle the Setup-Packet */
      else
      OnEpCtrlOut(B);
    }

    if (E & (1<<EpCtrlIn))          /* EP Control In    */
    { B = GetEPStatus(EpCtrlIn);    /* Status lesen (bevor Int lschen) */
      USBEpIntClr = 1<<EpCtrlIn;    /* Clear event flag */
      OnEpCtrlIn(B);
    }

    if (E & (1<<EpIntOut))          /* EP Int Out    */
    { B = GetEPStatus(EpIntOut);    /* Status lesen (bevor Int lschen) */
      USBEpIntClr = 1<<EpIntOut;    /* Clear event flag */
      OnEpIntOut(B);
    }

    if (E & (1<<EpIntIn))           /* EP Int In    */
    { B = GetEPStatus(EpIntIn);     /* Status lesen (bevor Int lschen) */
      USBEpIntClr = 1<<EpIntIn;     /* Clear event flag */
      OnEpIntOut(B);
    }

    if (E & (1<<EpBulkOut))         /* EP Bulk Out, Host-->Device */
    { USBEpIntClr = 1<<EpBulkOut;   /* Clear event flag */
      OnEpBulkOut();
    }

    if (E & (1<<EpBulkIn))          /* EP Bulk In, Device-->Host  */
    { USBEpIntClr = 1<<EpBulkIn;    /* Clear event flag */
      OnEpBulkIn();
    }

   /* ... weitere Ep nach Bedarf */

    USBDevIntClr = EP_SLOW;         /* zum Schluss Int lschen     */
  }
}



/************  USB-Setup **********************************/
/* Die Pins mssen bereits richtig gesetzt sein (PINSELx) */
/* die Taktfrequenz ebenfalls. (48 MHz)                   */
/**********************************************************/
word UsbSetup (void)
{ word b1, b2;

/* nur einbinden, wenn ntig..
  SYSAHBCLKCTRL |= (1<<14);
  PDRUNCFG      &= ~(1<<10);
*/

  Class_Start();          /* LineCoding-Block aufsetzen mit unseren Defaultwerten */
  CMD.Configuration = 0;                 /* alles erstmal austragen */

#ifdef LPC17XXINCLUDED
  USBClkCtrl = 0x12;
  while ((USBClkSt & 0x12) != 0x12);     /* warten auf USB Clocks */
#endif

#ifdef LPC24XXINCLUDED
  while ((USBClkSt & 0x1A) != 0x1A);     /* warten auf USB Clocks */
  USBPortSel   = 3;                      /* USB --> USB2 Anschluss */
#endif

  USBEpIntClr  = 0xFFFFFFFF;
  USBDevIntClr = 0xFFFFFFFF;
  USBEpIntEn   = 3 | (3<<2) | (3<<4);       /* Control, Int, Bulk */
  USBEpDMADis  = 0xFFFFFFFF;             /* alle DMA Kanle aus */
#ifdef LPC24XXINCLUDED
  NVIC_ISER1  |= (1<<15);                /* Interrupt 47=USB, enable */
#endif
#ifdef LPC17XXINCLUDED
  NVIC_ISER0  |= (1<<24);                /* Interrupt 24=USB, enable */
#endif

  USBIntSt     = 0x80000000;             /* alle Int's erlauben */
  InitEndpoints();

  SIE_Cmd(cmReadTestReg | SieCMD);       /* Testregister lesen */
  SIE_Cmd(cmReadTestReg | SieRD);
  b1 = USBCmdData;
  SIE_Cmd(cmReadTestReg | SieRD);
  b2 = USBCmdData;
  SIE_Cmd(cmDeviceStatus | SieCMD);      /* SIE Reset */
  SIE_Cmd((1<<16) | SieWR);              /* CON setzen, sonst luft SIE nicht an */

  return (b2<<8)|b1;                     /* 'magic' 0xA50F aus Testregister */
}


/********** zeichenweises I/O und Pufferung und Kommunikation ***************/
/*
Diese Routinen werden von auerhalb im Usermode
aufgerufen und haben mit dem interrupt-gesteuerten
USB-Betrieb nichts zu tun.
*/

/* liefert true, wenn ein Zeichen abholbereit ist */
bool UsbRxAvail (void)
{ if (rxr!=rxw) return true;
  return false;
}

/* holt ein Zeichen vom USB ab */
/*
Achtung: wenn nix abzuholen ist, wird 0 zurckgeliefert
*/
char UsbGetChar (void)
{ char c;

  c = 0;
  if (rxr!=rxw)
  { c = UsbRxBuf[rxr];
    rxr = (rxr+1) & (rxLen-1);
  }
  return c;
}


/* liefert true, wenn noch ein Zeichen in den Tx-Buffer passt */
bool UsbTxReady (void)
{ int i;
  i = (txw+1) & (txLen-1);
  if (i==txr) return false;
  return true;
}

/* liefert true, wenn Tx-Buffer leer ist */
bool UsbTxEmpty (void)
{ return (txw==txr); }

/* Anzahl freier Pltze im Tx-Buffer liefern */
int UsbTxFree (void)
{ int i;
  i = txw - txr;    /* i = belegte Pltze */
  if (i<0) i = i + txLen;
  return txLen - i;
}



/* sendet ein Zeichen (d.h. schreibt es in den Tx-Buffer) */
/* Achtung: hier wird gewartet! */
char UsbCharOut (char c)
{ int i;
  i = (txw+1) & (txLen-1);
  while (!UsbTxReady());   /* trampeln auf der Stelle!! */
  UsbTxBuf[txw] = c;
  txw = i;
  return c;
}

/* String zum USB senden */
void UsbStrOut (char* S)
{ while (*S) UsbCharOut(*S++); }

  /****ende****/

