/* Nuvoton NUC120RE3AN:  USB als COM Port betreiben */
#include "StdTypes.h"
#include "nuc120.h"
#include "usb.h"

extern void ShortDelay (dword i);

/* die Transfer-Puffer fr zeichenweises In und Out ber den USB */
#define txLen  256
char UsbTxBuf[txLen];
int  txr, txw;

#define rxLen  256
char UsbRxBuf[rxLen];
int  rxr, rxw;


/***************************  Konstanten ********************************/

/******* USB-Zustnde ******/
#define  isDETACHED      0
#define  isATTACHED      1
#define  isPOWERED       3
#define  isDEFAULT       7
#define  isADDRESSED   0xF
#define  isCONFIGURED 0x1F
#define  isSUSPENDED  0x20


/******* Zuordnung physischer Endpunkte ******/
#define  EpCtrlIn   0       // Control_In  ist physischer EP 0
#define  EpCtrlOut  1       // Control_Out ist physischer EP 1
#define  EpBulkIn   2       // Bulk-Daten zum Host   ist  EP 2
#define  EpBulkOut  3       // Bulk-Daten zum NUC120 ist  EP 3
#define  EpIntIn    4       // Interrupt zum Host    ist  EP 4

/******* Adressen im USB_SRAM ****************/
/* es ist genug RAM da, also kriegt jeder EP 64 Byte davon ab. */
#define  Ep0Buf         0
#define  Ep1Buf        64
#define  Ep2Buf       128
#define  Ep3Buf       192
#define  Ep4Buf       256
#define  Ep5Buf       320
#define  StBuf        384

#define  Ep0MaxLen     64
#define  Ep1MaxLen     64
#define  Ep2MaxLen     64
#define  Ep3MaxLen     64
#define  Ep4MaxLen     64
#define  Ep5MaxLen     64
#define  StMaxLen       8
#define  EpBufMask  0x1F8    // alle Adressen auf 8 Byte aligned

/******* Zeiger auf die Puffer im USB_SRAM ***/
#define  Pt0Buf   (byte*) (USB_BA + 0x100 + Ep0Buf)
#define  Pt1Buf   (byte*) (USB_BA + 0x100 + Ep1Buf)
#define  Pt2Buf   (byte*) (USB_BA + 0x100 + Ep2Buf)
#define  Pt3Buf   (byte*) (USB_BA + 0x100 + Ep3Buf)
#define  Pt4Buf   (byte*) (USB_BA + 0x100 + Ep4Buf)
#define  Pt5Buf   (byte*) (USB_BA + 0x100 + Ep5Buf)
#define  PtStBuf  (byte*) (USB_BA + 0x100 + StBuf)


/******* Codes der Standard-bRequest's im Setup-Paket ***/
#define GET_STATUS          0x00
#define CLEAR_FEATURE       0x01
#define SET_FEATURE         0x03
#define SET_ADDRESS         0x05
#define GET_DESCRIPTOR      0x06
#define SET_DESCRIPTOR      0x07
#define GET_CONFIGURATION   0x08
#define SET_CONFIGURATION   0x09
#define GET_INTERFACE       0x0A
#define SET_INTERFACE       0x0B
#define SYNC_FRAME          0x0C

/******* zustzliche bRequest-Codes fr virtuelle ComPorts */
#define SET_LINE_CODE               0x20
#define GET_LINE_CODE               0x21
#define SET_CONTROL_LINE_STATE      0x22
#define SEND_BREAK                  0x23



/******* Struktur des Setup-Paketes *****/
struct TSetupPaket
{ byte bmRequestType;  /* siehe oben */
  byte bRequest;       /* siehe Request-Tafel in USB-Doku */
  word wValue;         /* je nach Request */
  word wIndex;         /* je nach Request */
  word wLength;        /* Anzahl Bytes, wenn Data-Stage vorhanden ist */
};


/******* Struktur des Kommando- und Org-Datenblockes *******************/
struct TCommand
{ struct TSetupPaket SetupPacket;  /* das jeweils letzte Setup-Paket   */
  dword  USBState;                 /* USB-Status (isDETACHED...isXXXX  */
  long   TransferLen;              /* noch zum Host zu sendende Anzahl Bytes */
  long   PacketLen;                /* wie lang das Paket zum Host sein darf */
  byte*  TransferPtr;              /* zeigt auf die noch zu sendenden Bytes */

  bool   Toggle[6];                /* fr DATA0/DATA1 der 6 EP's  */
 /* ich bin mir nicht sicher, ob man diese Toggle's wirklich braucht, denn
    die Hardware kann selber zwischen DATA0 und DATA1 toggeln */
  bool   RemoteWakeup;
  bool   SelfPowered;
  byte   MyFADDR;                  /* vom Host zugeteilte Adresse */
  byte   Configuration;

};


/* Line coding structure
  0-3 BaudRate     Data terminal rate (baudrate), in bits per second
  4   bCharFormat  Stop bits: 0 - 1 Stop bit, 1 - 1.5 Stop bits, 2 - 2 Stop bits
  5   bParityType  Parity:    0 - None, 1 - Odd, 2 - Even, 3 - Mark, 4 - Space
  6   bDataBits    Data bits: 5, 6, 7, 8, 16
 */
struct T_LineCoding
{  dword  BaudRate;     /* Baud rate    */
   byte   Stopbits;     /* stop bit     */
   byte   ParityType;   /* parity       */
   byte   DataBits;     /* data bits    */
};



/************  Variablen *****************************************/

struct TCommand     CMD;
struct T_LineCoding LineCoding;
word                Dtr_Rts;

/************ Funktionen zum Starten des virtuellen COM-Portes *****/
void Class_Start (void)
{ LineCoding.BaudRate = 9600;
  LineCoding.Stopbits = 0;
  LineCoding.ParityType = 0;
  LineCoding.DataBits = 8;
  Dtr_Rts = 0;
  txr = txw = rxr = rxw = 0;
}

bool Class_Compare (word aValue)  /* immer true, wird hier nicht gebraucht */
{ return true; }



/************* die Descriptoren ************************************/
#define LEN_DEVICE         18
#define DESC_DEVICE         1
#define VID            0x0416   /* Vendor ID (von RealTek) */
#define PID            0x5011   /* Product ID */

const byte DeviceDescriptor[] =
{ LEN_DEVICE,             /* bLength              */
  DESC_DEVICE,            /* bDescriptorType      */
  0x00, 0x02,             /* bcdUSB               */
  0x02,                   /* bDeviceClass         */
  0x00,                   /* bDeviceSubClass      */
  0x00,                   /* bDeviceProtocol      */
  64,                     /* bMaxPacketSize0      */
  0x16,                   /* Vendor  ID LO        */
  0x04,                   /* Vendor  ID HI        */
  0x11,                   /* Product ID LO        */
  0x50,                   /* Product ID HI        */
  0x00, 0x01,             /* bcdDevice            */
  0x01,                   /* iManufacturer        */
  0x02,                   /* iProduct             */
  0x03,                   /* iSerialNumber        */
  0x01                    /* bNumConfigurations   */
};

#define LEN_CONFIG          9
#define DESC_CONFIG         2

#define LEN_INTERFACE       9
#define DESC_INTERFACE      4

#define LEN_ENDPOINT        7
#define DESC_ENDPOINT       5

const byte ConfigDescriptor[] =
{ LEN_CONFIG,             /* bLength              */
  DESC_CONFIG,            /* bDescriptorType      */
  0x43, 0x00,             /* wTotalLength         */
  0x02,                   /* bNumInterfaces       */
  0x01,                   /* bConfigurationValue  */
  0x00,                   /* iConfiguration       */
  0xC0,                   /* bmAttributes         */
  0x32,                   /* MaxPower             */

                          /* INTERFACE descriptor */
  LEN_INTERFACE,          /* bLength              */
  DESC_INTERFACE,         /* bDescriptorType      */
  0x00,                   /* bInterfaceNumber     */
  0x00,                   /* bAlternateSetting    */
  0x01,                   /* bNumEndpoints        */
  0x02,                   /* bInterfaceClass      */
  0x02,                   /* bInterfaceSubClass   */
  0x01,                   /* bInterfaceProtocol   */
  0x00,                   /* iInterface           */

                          /* Communication Class Specified INTERFACE descriptor */
  0x05,                   /* Size of the descriptor, in bytes */
  0x24,                   /* CS_INTERFACE descriptor type */
  0x00,                   /* Header functional descriptor subtype */
  0x10, 0x01,             /* Communication device compliant to the communication spec. ver. 1.10 */

                          /* Communication Class Specified INTERFACE descriptor */
  0x05,                   /* Size of the descriptor, in bytes */
  0x24,                   /* CS_INTERFACE descriptor type */
  0x01,                   /* Call management functional descriptor */
  0x00,                   /* BIT0: Whether device handle call management itself. */
                          /* BIT1: Whether device can send/receive call management information over a Data Class Interface 0 */
  0x01,                   /* Interface number of data class interface optionally used for call management */

                          /* Communication Class Specified INTERFACE descriptor */
  0x04,                   /* Size of the descriptor, in bytes */
  0x24,                   /* CS_INTERFACE descriptor type */
  0x02,                   /* Abstract control management funcational descriptor subtype */
  0x00,                   /* bmCapabilities       */

                          /* Communication Class Specified INTERFACE descriptor */
  0x05,                   /* bLength              */
  0x24,                   /* bDescriptorType: CS_INTERFACE descriptor type */
  0x06,                   /* bDescriptorSubType   */
  0x00,                   /* bMasterInterface     */
  0x01,                   /* bSlaveInterface0     */

                          /* ENDPOINT descriptor  */
  LEN_ENDPOINT,           /* bLength              */
  DESC_ENDPOINT,          /* bDescriptorType      */
  0x83,                   /* bEndpointAddress     */
  3,                      /* Attribute: Interrupt */
  8, 0x00,                /* wMaxPacketSize       */
  0x01,                   /* bInterval            */

                             /* INTERFACE descriptor */
  LEN_INTERFACE,          /* bLength              */
  DESC_INTERFACE,         /* bDescriptorType      */
  0x01,                   /* bInterfaceNumber     */
  0x00,                   /* bAlternateSetting    */
  0x02,                   /* bNumEndpoints        */
  0x0A,                   /* bInterfaceClass      */
  0x00,                   /* bInterfaceSubClass   */
  0x00,                   /* bInterfaceProtocol   */
  0x00,                   /* iInterface           */

                          /* ENDPOINT descriptor  */
  LEN_ENDPOINT,           /* bLength              */
  DESC_ENDPOINT,          /* bDescriptorType      */
  0x81,                   /* bEndpointAddress     */
  2,                      /* Attribute: Bulk_In   */
  64, 0x00,               /* wMaxPacketSize       */
  0x00,                   /* bInterval            */

                          /* ENDPOINT descriptor  */
  LEN_ENDPOINT,           /* bLength              */
  DESC_ENDPOINT,          /* bDescriptorType      */
  2,                      /* bEndpointAddress     */
  2,                      /* Attribute: Bulk_Out  */
  64, 0x00,               /* wMaxPacketSize       */
  0x00,                   /* bInterval            */
};


#define DESC_STRING         3
const byte StringLang[] =
{ 4,                      /* bLength                  */
  DESC_STRING,            /* bDescriptorType          */
  0x09, 0x04              /* Language ID: USA(0x0409) */
};


const byte VendorStringDescriptor[] =
{ 16,                     /* bLength          */
  DESC_STRING,            /* bDescriptorType  */
  'N', 0,
  'u', 0,
  'v', 0,
  'o', 0,
  't', 0,
  'o', 0,
  'n', 0
};

const byte ProductStringDescriptor[] =
{ 32,                     /* bLength          */
  DESC_STRING,            /* bDescriptorType  */
  'U', 0,
  'S', 0,
  'B', 0,
  ' ', 0,
  'V', 0,
  'i', 0,
  'r', 0,
  't', 0,
  'u', 0,
  'a', 0,
  'l', 0,
  ' ', 0,
  'C', 0,
  'O', 0,
  'M', 0
};

const byte StringSerial[26] =
{ 26,                     /* bLength          */
  DESC_STRING,            /* bDescriptorType  */
  'N', 0,
  'T', 0,
  '2', 0,
  '0', 0,
  '0', 0,
  '9', 0,
  '1', 0,
  '0', 0,
  '1', 0,
  '4', 0,
  '0', 0,
  '0', 0
};


/************  Hilfsroutinen ************************************************/

/* physische Endpunkte aufsetzen (bei Reset-Kommando usw.) */
void InitEndpoints (void)
{ CMD.MyFADDR = 0;                  /* bis "DEFAULT" haben wir keine Adresse */
  CMD.Configuration = 0;            /* vor "CONFIGURED" ist hier nix */
  USB_FADDR   = 0;                  /* Hardware auf "keine Adresse" setzen */
  USB_INTEN   = 0x010F;             /* Interruptquellen aufsetzen */
  CMD.TransferLen = 0;              /* es stehen ab hier auch */
  CMD.PacketLen   = 0;              /* keine Transfers an */
  CMD.TransferPtr = 0;

  if(CMD.USBState > isDEFAULT) CMD.USBState = isDEFAULT;

/* EP0: Control, Input, 64 Bytes */
  USB_CFG0     = (1<<9)|(2<<5)|0;   /* STALL, IN, EpNummer=0 */
  USB_CFGP0    = 3;
  USB_BUFSEG0  = Ep0Buf;            /* EP0 Puffer festlegen */

/* EP1: Control, Output, 64 Bytes */
  USB_CFG1     = (1<<9)|(1<<5)|0;   /* STALL, OUT, EpNummer=0 */
  USB_CFGP1    = 3;
  USB_BUFSEG1  = Ep1Buf;            /* EP1 Puffer festlegen */

/* EP2: Bulk_in, Input, 64 Bytes */
  USB_CFG2     = (2<<5)|1;          /* IN, EpNummer=1 */
  USB_CFGP2    = 1;
  USB_BUFSEG2  = Ep2Buf;            /* EP2 Puffer festlegen */

/* EP3: Bulk_out, Output, 64 Bytes */
  USB_CFG3     = (1<<5)|2;          /* OUT, EpNummer=2 */
  USB_CFGP3    = 1;
  USB_BUFSEG3  = Ep3Buf;            /* EP2 Puffer festlegen */

/* EP4: Int_in, Input, 8 Bytes */
  USB_CFG4     = (2<<5)|3;          /* IN, EpNummer=3 */
  USB_CFGP4    = 1;
  USB_BUFSEG4  = Ep4Buf;            /* EP2 Puffer festlegen */

/* EP5: nix */
  USB_CFG5     = 0;                 /* AUSGESCHALTET  */
  USB_BUFSEG5  = Ep5Buf;            /* EP5 Puffer festlegen */

/* fr Setup_Block vom Host */
  USB_STBUFSEG = StBuf;
}


/* Request-Typ im Setup-Packet testen (Standard, Class, Vendor) */
bool IsStandardRequest (void)
{ return (CMD.SetupPacket.bmRequestType & 0x60)==0; }

bool IsClassRequest (void)
{ return (CMD.SetupPacket.bmRequestType & 0x60)==0x20; }

bool IsVendorRequest (void)
{ return (CMD.SetupPacket.bmRequestType & 0x60)==0x40; }


/* leeres Paket vom Control-EP zum Host senden, quasi "ACK" */
void SendEmptyPacket (void)
{ CMD.Toggle[0] = 1;
  USB_CFG0   |= (1<<7);
  USB_BUFSEG0 = Ep0Buf;
  USB_MXPLD0  = 0;
}



/******* anstehende Control-Transfers zum Host blockweise starten *******/
void DescriptorBlockwiseIn (void)
{ int i, j;
  byte* Q;
  byte* Z;

  if ((CMD.SetupPacket.bmRequestType & 0x80)==0) return;
  if (CMD.TransferLen==0) { USB_MXPLD1 = 0; return; }

  i = CMD.TransferLen;
  if (i > CMD.PacketLen) i = CMD.PacketLen;
  Q = CMD.TransferPtr;                   /* Quelle */
  Z = Pt0Buf;                            /* Ziel   */
  j = 0;
  while (j<i) { *Z++ = *Q++; ++j; }      /* Block zum USB_SRAM kopieren */
  CMD.TransferPtr = Q;                   /* Zeiger auf evt. Rest merken */
  CMD.TransferLen = CMD.TransferLen - i; /* restliche Anzahl Bytes */
  if (CMD.TransferLen < 0) CMD.TransferLen = 0;

  if (CMD.Toggle[0])
      USB_CFG0 |= (1<<7);
  else
      USB_CFG0 &= ~(1<<7);
  CMD.Toggle[0] = !CMD.Toggle[0];
  USB_BUFSEG0 = Ep0Buf;
  USB_MXPLD0  = i;

  /* wenn der gleiche Puffer fr Ep0In und Ep0Out benutzt wird,
     mu man hier ein
        if (CMD.TransferLen == 0) USB_MXPLD1 = 0;
     einfgen.
   */

}


/************ Bearbeitung eingegangener Requests **********************/

/* die empfangene Adresse in die Hardware schreiben */
void DoSetFADDR (void)
{ if (CMD.MyFADDR == 0)          /* war wohl nix.. */
      CMD.USBState = isDEFAULT;
  else
    { CMD.USBState = isADDRESSED;
      USB_FADDR = CMD.MyFADDR;   /* Adresse in Hardware schreiben */
    }
}

/* die empfangene Adresse erstmal im CMD-Block merken */
void DoBecomeAddressed (void)
{ if (CMD.USBState == isDEFAULT)
    { CMD.MyFADDR = CMD.SetupPacket.wValue;
      USB_CFG0 |= (1<<7);              /* set to DATA1 */
      USB_MXPLD0 = 0;
      CMD.USBState = isADDRESSED;
    }
    else
    { USB_CFGP0 = 2 | 1;
      USB_CFGP1 = 2 | 1;
    }
}


/* USB-Request "CLEAR FEATURE" behandeln */
void DoClearFeature (void)
{ int Feature;
  int FuerWen;
  int EP;

  Feature = CMD.SetupPacket.wValue;
  FuerWen = CMD.SetupPacket.bmRequestType;
  EP      = CMD.SetupPacket.wIndex;

  switch(FuerWen)
  { case 0:          /* fr Device */
        if (Feature==1) CMD.RemoteWakeup = 0;
        break;

    case 1:          /* fr Interface */
        break;

    case 3:          /* fr einen Endpoint */
        if (Feature==0)
        {            /* fr alle EP>=2 ist logEP=physEP */
          switch(EP)
          {  case 2:  USB_CFG2  &= 0x7F;  /* CSTALL=0 */
                      USB_CFGP2 &= 1;     /* SSTALL=0 */
                      break;
             case 3:  USB_CFG3  &= 0x7F;
                      USB_CFGP3 &= 1;
                      break;
             case 4:  USB_CFG4  &= 0x7F;
                      USB_CFGP3 &= 1;
                      break;
             default: USB_CFGP0 = 2 | 1;
                      USB_CFGP1 = 2 | 1;
           }
        }
        else
        { USB_CFGP0 = 2 | 1;
          USB_CFGP1 = 2 | 1;
        }
        break;
    default:   USB_CFGP0 = 2 | 1;
               USB_CFGP1 = 2 | 1;
  }
  SendEmptyPacket();
}


/* USB-Request "SET FEATURE" behandeln */
void DoSetFeature (void)
{ int Feature;
  int FuerWen;
  int EP;

  Feature = CMD.SetupPacket.wValue;
  FuerWen = CMD.SetupPacket.bmRequestType;
  EP      = CMD.SetupPacket.wIndex;


  switch(FuerWen)
  { case 0:          /* fr Device */
        if (Feature==1) CMD.RemoteWakeup = 0;
        break;

    case 1:          /* fr Interface */
        break;

    case 3:          /* fr einen Endpoint */
        if (Feature==0)
        {            /* fr alle EP>=2 ist logEP=physEP */
          switch(EP)
          {  case 2:  USB_CFG2  &= 0x7F;  /* CSTALL=0 */
                      USB_CFGP2 |= 2;     /* SSTALL=1 */
                      break;
             case 3:  USB_CFG3  &= 0x7F;
                      USB_CFGP3 |= 2;
                      break;
             case 4:  USB_CFG4  &= 0x7F;
                      USB_CFGP3 |= 2;
                      break;
             default: USB_CFGP0 = 2 | 1;
                      USB_CFGP1 = 2 | 1;
           }
        }
        else
        { USB_CFGP0 = 2 | 1;
          USB_CFGP1 = 2 | 1;
        }
        break;
    default:   USB_CFGP0 = 2 | 1;
               USB_CFGP1 = 2 | 1;
  }
  SendEmptyPacket();
}


/* USB-Request "GET STATUS" behandeln */
void DoGetStatus (void)
{ byte Buf[4];
  int  EP;
  long L;

  EP = CMD.SetupPacket.wIndex;   /* ab EP2 ist logEP=physEP */
  Buf[0] = 0;

  if (CMD.SetupPacket.bmRequestType == 0x80)
  { if (CMD.RemoteWakeup) Buf[0] |= 2;
    if (CMD.SelfPowered)  Buf[0] |= 1;
  }

  else
  if (CMD.SetupPacket.bmRequestType == 0x81)
  { Buf[0] = 0;
  }

  else
  if (CMD.SetupPacket.bmRequestType == 0x82)
  {   L = 0;
      switch(EP)         /* Interrupt-In Endpoint */
      { case 0: L = USB_CFG0; break;
        case 1: L = USB_CFG1; break;
        case 2: L = USB_CFG2; break;
        case 3: L = USB_CFG3; break;
        case 4: L = USB_CFG4; break;
        case 5: L = USB_CFG5; break;
        default: USB_CFGP0 = 2 | 1;
                 USB_CFGP1 = 2 | 1;
                 return;
       }
      if (L & 0x60) Buf[0] = 1;
  }
  else
  { USB_CFGP0 = 2 | 1;
    USB_CFGP1 = 2 | 1;
    return;
  }
  Buf[1] = 0;
  CMD.PacketLen   = Ep0MaxLen;
  CMD.TransferLen = 2;
  CMD.TransferPtr = Buf;
  DescriptorBlockwiseIn();
}


/* Descriptoren zum Host senden */
void DoGetDescriptor(void)
{ word Type, Index;
  int aLen;
  const byte* P;

  Type    = CMD.SetupPacket.wValue >> 8;
  Index   = CMD.SetupPacket.wValue & 0xFF;
  aLen    = -1;
  P       = 0;

  switch (Type)
  { case DESC_DEVICE:                            /* Get Device Descriptor */
         { aLen = LEN_DEVICE;
           P    = DeviceDescriptor;
           break;
         }

    case DESC_CONFIG:                            /* Get Configuration Descriptor    */
         { aLen = ConfigDescriptor[3];           /* Total-Lnge ist WORD            */
           aLen = (aLen << 8) | ConfigDescriptor[2];
           P    = ConfigDescriptor;
           break;
         }


    case DESC_STRING:                            /* Get String Descriptor */
         { switch (Index)                        /* Get String Descriptor */
           { case 0: aLen = 4;
                     P    = StringLang;
                     break;
             case 1: aLen = VendorStringDescriptor[0];
                     P    = VendorStringDescriptor;
                     break;
             case 2: aLen = ProductStringDescriptor[0];
                     P    = ProductStringDescriptor;
                     break;
             case 3: aLen = StringSerial[0];
                     P    = StringSerial;
                     break;
             default: USB_CFGP0 = 2 | 1;  // kennen wir nicht. Stall.
                      USB_CFGP1 = 2 | 1;
                      aLen = -1;
           }
           break;
         }
     default: { USB_CFGP0 = 2 | 1;  // kennen wir nicht. Stall.
                USB_CFGP1 = 2 | 1;
                aLen = -1;
              }
     }

     if (aLen < 0 ) return;
     if (aLen > CMD.SetupPacket.wLength) aLen = CMD.SetupPacket.wLength;

  CMD.PacketLen   = Ep0MaxLen;
  CMD.TransferLen = aLen;
  CMD.TransferPtr = (byte*) P;
  DescriptorBlockwiseIn();
}



/* USB-Request "SET CONFIGURATION" behandeln */
void DoSetConfiguration(void)
{ bool haveConfig;

  haveConfig = Class_Compare(CMD.SetupPacket.wValue);
  if (CMD.SetupPacket.wValue == 0)
        { CMD.USBState = isADDRESSED;
          CMD.Configuration = CMD.SetupPacket.wValue & 0xFF;
          SendEmptyPacket();
        }
  else
  if(haveConfig)
        { CMD.USBState = isCONFIGURED;
          Class_Start();
          CMD.Configuration = CMD.SetupPacket.wValue &0xFF;
          SendEmptyPacket();
          USB_MXPLD3    = 64;
          CMD.Toggle[2] = 0;
          USB_CFG2     |= (1<<7);
          USB_MXPLD2    = 0;
        }
   else
        { USB_CFGP0 = 2 | 1;  // kennen wir nicht. Stall.
          USB_CFGP1 = 2 | 1;
        }
}


/************* CDC Spezifisches **************************/

/*  "SET LINE CODING" behandeln */
void VCOM_SetLineCoding(void)
{ CMD.Toggle[0] = false;
  USB_MXPLD1 = 7;  /* Vorbereitung auf Empfang von genau 7 Bytes vom Host*/
}


/* CDC-spezifischer USB-Request "GET LINE CODING" behandeln */
void VCOM_GetLineCoding(void)
{ CMD.Toggle[0] = true;
  CMD.PacketLen   = Ep0MaxLen;
  CMD.TransferLen = 7;
  CMD.TransferPtr = (byte*) &LineCoding;
  DescriptorBlockwiseIn();    /* hier werden die 7 Bytes zum Host geschickt */
}



/* Datenausgabe fr CDC-spezifischen USB-Request "SET LINE CODING" */
/* hier werden die empfangenen 7 Bytes aus dem USB-Puffer gelesen
   und im RAM gemerkt
*/
void SetLineCodingDataOut(void)
{ byte* Q;
  byte* Z;
  int Count;

  Count = USB_MXPLD1;
  Q = Pt1Buf;
  Z = (byte*)&LineCoding;
  while(Count--)  { *Z++ = *Q++;  }

  /* ACK to end the transfer */
  CMD.Toggle[0] = false;
  SendEmptyPacket();
}

/* Zustand von DTR und RTS vom Host zum Gert merken */
void VCOM_ReadLineCoding (void)
  { Dtr_Rts = CMD.SetupPacket.wValue>>8;
    SendEmptyPacket();
}



/************************* Wakeup-Event ************************************/
void OnWakeup      (void)                 /* hier nix */
{
}

/************************* Float-Detect-Event ******************************/
void OnFloatDetect (void)
{ if (USB_FLDET & 1)                   /* if attached */
  { if (CMD.USBState == isDETACHED)
    { CMD.USBState = isATTACHED;
      USB_ATTR = 0x7D0;                /* EnableUsb   */
    }
  }
  else                                 /* if detached */
  { CMD.USBState = isDETACHED;
    USB_ATTR = 0x760;                  /* DisableUsb  */
  }
}


/************************** Setup-Event ***********************************/
const byte always0 = 0;

void OnSetup (void)
{ byte* Q;
  byte* Z;
  int Count;

  CMD.Toggle[0] = 1;                         /* Assign the toggle bit for data phase */
  if (CMD.USBState < isDEFAULT)              /* wenigstens DEFAULT, sonst reagieren wir nicht */
    { USB_CFGP0 = 2 | 1;
      USB_CFGP1 = 2 | 1;
      return;
    }

  Q = PtStBuf;
  Z = (byte*)&CMD.SetupPacket.bmRequestType;
  Count = 8;
  while(Count--) *Z++ = *Q++;

  if ((CMD.SetupPacket.bmRequestType & 0x60)==0)         /* wenn Type = Standard */
    { switch (CMD.SetupPacket.bRequest)
      { case SET_ADDRESS:       DoBecomeAddressed();
                                return;

        case CLEAR_FEATURE:     DoClearFeature();
                                return;

        case SET_FEATURE:       DoSetFeature();
                                return;

        case GET_CONFIGURATION: CMD.PacketLen   = Ep0MaxLen;
                                CMD.TransferLen = 1;
                                CMD.TransferPtr = (byte*) &CMD.Configuration;
                                DescriptorBlockwiseIn();
                                return;

        case GET_STATUS:        DoGetStatus();
                                return;

        case GET_INTERFACE:     CMD.TransferLen = 1;
                                CMD.TransferPtr = (byte*) &always0;
                                DescriptorBlockwiseIn();
                                return;

        case SET_INTERFACE:     Class_Start();
                                SendEmptyPacket();
                                return;

        case GET_DESCRIPTOR:    DoGetDescriptor();
                                return;

        case SET_CONFIGURATION: DoSetConfiguration();
                                return;
      }
      // auf alles andere reagieren wir mit Stall. siehe unten.
    }

  if ((CMD.SetupPacket.bmRequestType & 0x60)==0x20)       /* wenn Type = Class */
    { switch (CMD.SetupPacket.bRequest)
      { case SET_LINE_CODE:          VCOM_SetLineCoding();  return;
        case GET_LINE_CODE:          VCOM_GetLineCoding();  return;
        case SET_CONTROL_LINE_STATE: VCOM_ReadLineCoding(); return;
        /* falls es hier noch mehr Class-spezifische Requests
           geben sollte, dann Behandlung hier hinein.
	 */
      }
    }

  if ((CMD.SetupPacket.bmRequestType & 0x60)==0x40)       /* wenn Type = Vendor */
    {  return;
    }

    /* wenn keiner zustndig war, dann Stall! */
  USB_CFGP0 = 2 | 1;
  USB_CFGP1 = 2 | 1;
}


/******* Control IN ACK default handler *************************************/
void Default_In (void)
{ if (CMD.SetupPacket.bmRequestType & 0x80) USB_MXPLD1 = 0; }


/******* Control OUT ACK default handler ************************************/
void Default_Out(void)
{ if ((CMD.SetupPacket.bmRequestType & 0x80) == 0)  SendEmptyPacket(); }


/******* die diversen Endpoint-Interrupts ************************************/

void OnEP0Int (int PacketType)         /* Control-EP IN */
{ int   ISOPacketType;

  if (CMD.USBState == isDETACHED) return;

  if (PacketType == 0)                                /* EPSTS_IN_ACK */
  { if (IsStandardRequest())                          /* wenn Type = Standard */
    { switch (CMD.SetupPacket.bRequest)
      { case SET_ADDRESS:       DoSetFADDR();            break;
        case GET_DESCRIPTOR:    DescriptorBlockwiseIn(); break;
        default: Default_In();
      }
     return;
    }
    Default_In();
    return;
  }

  /* fr Isochrone Transfers: wird hier erstmal nicht bentigt. */
  if(PacketType == 7)                                  /* EPSTS_ISO */
    { ISOPacketType = USB_CFG0 & 0x70;
      CMD.Toggle[0] = 0;
      if (ISOPacketType == 0x30)             /* This is isochronous out translation end. Call relative handler */
        { // ggf. passende Aktion hier
          Default_Out();
        }
        else
        if(ISOPacketType == 0x50)              /* This is isochronous in translation end. Call relative handler */
        { // ggf. passende Aktion hier
          Default_In();
        }
      return;
    }
  Default_In();
}



void OnEP1Int (int PacketType)         /* Control-EP OUT */
{ if (CMD.USBState == isDETACHED) return;

  if (IsStandardRequest())                             /* wenn Type = Standard */
  { Default_Out();
    if (PacketType == 2) CMD.Toggle[1] = 1;
    if (PacketType == 6) CMD.Toggle[1] = 0;
    return;
  }

  if (IsClassRequest())                                /* wenn Type = Class */
  { switch (CMD.SetupPacket.bRequest)
    { case SET_LINE_CODE: SetLineCodingDataOut(); break;
      default: Default_Out();
    }
    return;
  }

  /* nach Vendor-Request fragen wir hier garnicht erst */
  Default_Out();
}



void OnEP2Int (int PacketType)         /* Bulk-EP IN */
{ int i;
  byte* P;

  if (CMD.USBState == isDETACHED) return;

  /* wenn es was im Tx-Puffer gibt, senden wir es an den Host */
  i = 0;
  P = Pt2Buf;
  if (txr!=txw)
  { i = 0;
    while ((txr!=txw) && (i < Ep2MaxLen))
    { *P++ =  UsbTxBuf[txr];
       txr = (txr+1) & (txLen-1);
       ++i;
    }
   }
  USB_MXPLD2 = i;
}


void OnEP3Int (int PacketType)         /* Bulk-EP OUT */
{ int i, j;
  char c;
  byte* P;

/*
 Strategie: wir holen erstmal das Paket ab und verstauen es
 im Rx-Puffer. Dafr MUSS Platz sein!
 Dann schauen wir nach, ob noch genug Platz im Rx-Puffer ist,
 um ein weiteres Paket unterzubringen.
 Wenn ja, setzen wir MXPLD3 auf Ep3MaxLen.
 Wenn nicht, dann setzen wir MXPLD3 auf 0, womit der Host
 begreifen soll, da wir zwar ein neues Paket (mit 0 Bytes Inhalt)
 annehmen werden, aber im Moment keine weiteren Daten vertragen knnen.
 */

  if (CMD.USBState == isDETACHED) return;
  if (PacketType == 2) CMD.Toggle[3] = 1;
  if (PacketType == 6) CMD.Toggle[3] = 0;

  /* Fllstand im UsbRxBuffer ermitteln */
  i = rxw - rxr;  if (i<0) i += rxLen;

  /* Anzahl freier Stellen im UsbRxBuf ermitteln */
  i = rxLen - i;

  /* wenn zu wenig Platz ist, Stop */
  if (i < USB_MXPLD3)
  { USB_CFGP3 = 1; return; }

  j = USB_MXPLD3;
  i = i - j;            /* wieviel danach noch frei sind */
  P = Pt3Buf;
  while (j)
  { c = *P++;
    --j;
    UsbRxBuf[rxw] = c;
    rxw = (rxw+1) & (rxLen-1);
  }

  if (i > Ep3MaxLen)
    USB_MXPLD3 = Ep3MaxLen;
  else
    USB_MXPLD3 = 0;
}


void OnEP4Int (int PacketType)         /* Int-EP IN */
{ if (CMD.USBState == isDETACHED) return;
  // erstmal nix
}

void OnEP5Int (int PacketType)         /* unbenutzter EP */
{ if (CMD.USBState == isDETACHED) return;
  // erstmal nix
}




/**************** USB-Interrupt-Handler **************************************/
void USB_IRQHandler(void)
{ dword D, I;

  I = USB_INTSTS;
  if (I & 8)                                 /* WAKEUP */
    { USB_INTSTS = 8;                        /* Clear wakeup event. write one clear */
      OnWakeup();                            /* Pre-dispatch wakeup event. */
    }

  else
  if (I & 4)                                 /* FLDET */
    { USB_INTSTS = 4;                        /* Clear float-detection event. Write one clear */
      OnFloatDetect();                       /* Pre-dispatch float-detection event. */
    }

  else
  if (I & 1)                                 /* INTSTS_BUS */
    { D = USB_ATTR;
      USB_INTSTS = 1;                        /* Clear bus event. Write one clear */
      if (CMD.USBState != isDETACHED)
      { if (D & 1)                           /* Bus reset */
        { USB_ATTR = 0x7D0;                  /* enable USB & PHY */
          InitEndpoints();
          CMD.USBState = isDEFAULT;
        }

        else
        if (D & 2)                           /* Bus suspend */
        { USB_ATTR = 0x7C0;                  /* disable PHY */
          if (CMD.USBState >= isATTACHED)
              CMD.USBState |= isSUSPENDED;
        }

        else
        if (D & 4)                           /* Bus resume */
        { USB_ATTR = 0x7D0;                  /* enable PHY */
          if (CMD.USBState >= isATTACHED)
              CMD.USBState &= ~isSUSPENDED;
        }

        else
        if (D & 8)                           /* Bus resume */
        { // was tun, wenn timeout??
        }
      }
    }

  else
  if (I & 2)                                 /* INTSTS_USB */
    { if (I & 0x80000000)
      { USB_INTSTS = 0x80000000;             /* Clear setup event flag */
        OnSetup();                           /* Handle the Setup-Packet   */
      }
      else
      { USB_INTSTS = I & 0x00FF0000;                     /* und lschen  */
        if (I & (1<<16)) OnEP0Int((USB_EPSTS>> 8) & 7);  /* EP0 Interrupt */
        if (I & (1<<17)) OnEP1Int((USB_EPSTS>>11) & 7);  /* EP1 Interrupt */
        if (I & (1<<18)) OnEP2Int((USB_EPSTS>>14) & 7);  /* EP2 Interrupt */
        if (I & (1<<19)) OnEP3Int((USB_EPSTS>>17) & 7);  /* EP3 Interrupt */
        if (I & (1<<20)) OnEP4Int((USB_EPSTS>>20) & 7);  /* EP4 Interrupt */
        if (I & (1<<21)) OnEP5Int((USB_EPSTS>>23) & 7);  /* EP5 Interrupt */
      }

    }
}



/************  USB-Setup *********************************/

word UsbSetup (void)
{ int i;
  byte* P;

  Class_Start();           /* LineCoding-Block aufsetzen mit unseren Defaultwerten */

  CMD.Configuration = 0;   /* alles erstmal austragen */
  CMD.USBState = 0;
  CMD.MyFADDR  = 0;

  APBCLK  |= (1<<27);      /* Enable USB Clock */
  IPRSTC2 |= (1<<27);      /* Reset USB ein */
  ShortDelay(1000);
  IPRSTC2 &= ~(1<<27);     /* Reset wieder aus */

  USB_ATTR = 0x760;

  USB_DRVSE0   = 1;                         /* D+ und D- auf low */
  ShortDelay(300);
  USB_DRVSE0   = 0;                         /* D+ und D- wieder normal */

  USB_FADDR    = 0;                         /* erstmal 0, kriegen wir spter vom Host */
  i = 512;
  P = (byte*) (USB_BA + 0x100);
  while (i--) *P++ = 0xFF;

  InitEndpoints();
  OnFloatDetect();                          /* Controller ein oder aus */
  USB_INTEN  = 0x10F;                       /* enable Interrupts BUS, USB und FLDET */
  NVIC_IPR5  = (NVIC_IPR5 & 0x00FFFFFF) | 0x80000000;   /* Pri23 = 2 */
  NVIC_ISER |= (1<<23);                     /* Enable USB-Interrupts in NVIC */
  return PID;                               /* nur Dummy... */
}


/********** zeichenweises I/O und Pufferung und Kommunikation ***************/
/*
Diese Routinen werden von auerhalb im Usermode
aufgerufen und haben mit dem interrupt-gesteuerten
USB-Betrieb nichts zu tun
*/

/* liefert true, wenn ein Zeichen abholbereit ist */
bool UsbRxAvail (void)
{ if (rxr!=rxw) return true;
  return false;
}

/* holt ein Zeichen vom USB ab */
/*
Achtung: wenn nix abzuholen ist, wird 0 zurckgeliefert
*/
char UsbGetChar (void)
{ char c;

  c = 0;
  if (rxr!=rxw)
  { c = UsbRxBuf[rxr];
    rxr = (rxr+1) & (rxLen-1);
  }
  return c;
}


/* liefert true, wenn noch ein Zeichen in den Tx-Buffer passt */
bool UsbTxReady (void)
{ int i;
  i = (txw+1) & (txLen-1);
  if (i==txr) return false;
  return true;
}

/* sendet ein Zeichen (d.h. schreibt es in den Tx-Buffer) */
char UsbCharOut (char c)
{ int i;
  i = (txw+1) & (txLen-1);
  while (!UsbTxReady());   /* trampeln auf der Stelle!! */
  UsbTxBuf[txw] = c;
  txw = i;
  return c;
}

/* asciiz zum USB senden */
void UsbStrOut (char* S)
{ while (*S) UsbCharOut(*S++); }


