/* ST  STM32F103C8T6:  USB als COM Port betreiben */
#include "StdTypes.h"
#include "STM32F103all.h"
#include "usb.h"

/* die Transfer-Puffer fr zeichenweises In und Out ber den USB */
#define txLen  256
char UsbTxBuf[txLen];
int  txr, txw;

#define rxLen  256
char UsbRxBuf[rxLen];
int  rxr, rxw;

/***************************  Konstanten ********************************/

/* alle USB-Register sind 16 Bit breit, mssen aber 32 bittig
   gelesen und geschrieben werden
   Auch der USB-RAM von 512 Bytes ist 32 bittig zu behandeln
   (aber mit Vorsicht!)
 */

/* Bits in USB_CNTR */
#define  FRES     (1<<0)
#define  PDWN     (1<<1)
#define  LP_MODE  (1<<2)
#define  FSUSP    (1<<3)
#define  RESUME   (1<<4)

#define  ESOFM    (1<<8)
#define  SOFM     (1<<9)
#define  RESETM   (1<<10)
#define  SUSPM    (1<<11)
#define  WKUPM    (1<<12)
#define  ERRM     (1<<13)
#define  PMAOVRM  (1<<14)
#define  CTRM     (1<<15)

/* Bits in USB_ISTR */
#define  DIR      (1<<4)
#define  ESOF     (1<<8)
#define  SOF      (1<<9)
#define  RESET    (1<<10)
#define  SUSP     (1<<11)
#define  WKUP     (1<<12)
#define  ERR      (1<<13)
#define  PMAOVR   (1<<14)
#define  CTR      (1<<15)

/* Bits in den USB_EPnR */
#define  CTR_RX   (1<<15)
#define  DTOG_RX  (1<<14)
#define  STAT_RX  (3<<12)
#define  SETUP    (1<<11)
#define  EP_TYPE  (3<<9)
#define  EP_KIND  (1<<8)
#define  CTR_TX   (1<<7)
#define  DTOG_TX  (1<<6)
#define  STAT_TX  (3<<4)
#define  MASK_EA  15

/* EndPoint Register Mask (No Toggle Fields) */
#define EP_NoToggleBits  (CTR_RX|SETUP|EP_TYPE|EP_KIND|CTR_TX|MASK_EA)



/******* Zuordnung physischer Endpunkte 0..7 ********************/
#define  logEpCtrl      0
#define  logEpBulkIn    1
#define  logEpBulkOut   2
#define  logEpInt       3

/* fr Stall, Unstall usw. */
#define  phys_In        0x80
#define  physEpCtrlIn   (0 + 0x80)
#define  physEpCtrlOut  0
#define  physEpBulkIn   (1 + 0x80)
#define  physEpBulkOut  2
#define  physEpIntIn    (3 + 0x80)
#define  physEpIntOut   3



/* Layout des USB RAM's (512 Bytes)
   ================================
Der RAM geht aus Sicht der CPU von 0x40006000 bis 0x400063FF,
also 0x400 Bytes gleich 1024 Bytes und das Layout ist krtig,
weil die Hlfte nicht implementiert ist und zu 0 gelesen wird!
Er ist NUR 16 bitweise les- und schreibbar! NICHT byteweise
und auch nicht wirklich 32 bitweise.
Beispiel:
Text sei "Hello-World"
0x40006000: 48 65 00 00 6C 6C 00 00 6F 2D 00 00 57 6F 00 00 72 6C 00 00 64 ...
             H  e        l  l        o  -        W  o        r  l        d ...

ab Offset 0:
Control_In   64 Bytes
Control_Out  64 Bytes
Bulk_In_A    64 Bytes (evtl. DoubleBuffered)
Bulk_In_B    64 Bytes
Bulk_Out_A   64 Bytes (evtl. DoubleBuffered)
Bulk_Out_B   64 Bytes
Int_In        8 Bytes (hier nicht benutzt)
Int_Out       8 Bytes (hier nicht benutzt)
-----------------------
macht       400 Bytes

anschlieend EpTable (USB_BTABLE zeigt drauf)
mit 4 Eintrgen (Control, BulkIn, BuklOut, Int) zu je 4 dwords
also 4*4 dwords = 64 Bytes
macht in Summe 464 Bytes
*/

#define USB_RAM       0x40006000

#define EpCtrlMaxLen  64
#define EpCtrlLenId   ((1<<15)|(1<<10))

#define EpBulkMaxLen  64
#define EpBulkLenId   ((1<<15)|(1<<10))

#define EpIntMaxLen    8
#define EpIntLenId    (4<<10)

/******* Pufferlngen und Lngen-Codes *******/

/* EP0 = control */
#define Ep0TxOffset   0      /* 64 Bytes ab   0 */
#define Ep0RxOffset   64     /* 64 Bytes ab  64 */

/* EP1 = Bulk-IN */
#define Ep1TxAOffset  128    /* 64 Bytes ab 128 */
#define Ep1TxBOffset  192    /* 64 Bytes ab 192 */

/* EP2 = Bulk-OUT */
#define Ep2RxAOffset  256    /* 64 Bytes ab 256 */
#define Ep2RxBOffset  320    /* 64 Bytes ab 320 */

/* EP3 = Int (unbenutzt) */
#define Ep3TxOffset   384    /* 8 Bytes ab 384 */
#define Ep3RxOffset   292    /* 8 Bytes ab 392 */

/* EP-Tafel */
#define EpTableOffset 400    /* 64 Bytes ab 400 */

#define EPControlTxBuffer (USB_RAM + (Ep0TxOffset<<1))
#define EPControlRxBuffer (USB_RAM + (Ep0RxOffset<<1))

#define EP1TxABuffer      (USB_RAM + (Ep1TxAOffset<<1))
#define EP1TxBBuffer      (USB_RAM + (Ep1TxBOffset<<1))

#define EP2RxABuffer      (USB_RAM + (Ep2RxAOffset<<1))
#define EP2RxBBuffer      (USB_RAM + (Ep2RxBOffset<<1))

#define EP3TxBuffer       (USB_RAM + (Ep3TxOffset<<1))
#define EP3RxBuffer       (USB_RAM + (Ep3RxOffset<<1))

struct TEpTableEntry
{ dword TxOffset;
  dword TxCount;
  dword RxOffset;
  dword RxCount;
};

#define EpTable   ((struct TEpTableEntry *) (USB_RAM + (EpTableOffset<<1)))



/******* Codes der Standard-bRequest's im Setup-Paket ***/
#define GET_STATUS          0x00
#define CLEAR_FEATURE       0x01
#define SET_FEATURE         0x03
#define SET_ADDRESS         0x05
#define GET_DESCRIPTOR      0x06
#define SET_DESCRIPTOR      0x07
#define GET_CONFIGURATION   0x08
#define SET_CONFIGURATION   0x09
#define GET_INTERFACE       0x0A
#define SET_INTERFACE       0x0B
#define SYNC_FRAME          0x0C

/******* zustzliche bRequest-Codes fr virtuelle ComPorts */
#define SET_LINE_CODE               0x20    /* 7 Byte Paket mit Baudrate etc. */
#define GET_LINE_CODE               0x21    /* 7 Byte Paket mit Baudrate etc. */
#define SET_CONTROL_LINE_STATE      0x22    /* 2 Bit  DTR und RTS */
#define SEND_BREAK                  0x23    /* hier unbenutzt */



/******* Struktur des Setup-Paketes *****/
struct TSetupPaket
{ byte bmRequestType;  /* siehe oben */
  byte bRequest;       /* siehe Request-Tafel in USB-Doku */
  word wValue;         /* je nach Request */
  word wIndex;         /* je nach Request */
  word wLength;        /* Anzahl Bytes, wenn Data-Stage vorhanden ist */
};


/******* Struktur des Kommando- und Org-Datenblockes *******************/
struct TCommand
{ struct TSetupPaket SetupPacket;  /* das jeweils letzte Setup-Paket   */
  long   TransferLen;              /* noch zum Host zu sendende Anzahl Bytes */
  long   PacketLen;                /* wie lang das Paket zum Host sein darf */
  byte*  TransferPtr;              /* zeigt auf die noch zu sendenden Bytes */

  bool   RemoteWakeup;
  bool   SelfPowered;
  byte   Configuration;
};


/* Line coding structure
  0-3 BaudRate     Data terminal rate (baudrate), in bits per second
  4   bCharFormat  Stop bits: 0 - 1 Stop bit, 1 - 1.5 Stop bits, 2 - 2 Stop bits
  5   bParityType  Parity:    0 - None, 1 - Odd, 2 - Even, 3 - Mark, 4 - Space
  6   bDataBits    Data bits: 5, 6, 7, 8, 16
 */
struct T_LineCoding
{  dword  BaudRate;     /* Baud rate    */
   byte   Stopbits;     /* stop bit     */
   byte   ParityType;   /* parity       */
   byte   DataBits;     /* data bits    */
};


/************  Variablen *****************************************/

struct TCommand     CMD;
struct T_LineCoding LineCoding;
word                Dtr_Rts;

// #include "debugfkt.inc"

/************ Funktionen zum Starten des virtuellen COM-Portes *****/
void Class_Start (void)
{ LineCoding.BaudRate = 9600;
  LineCoding.Stopbits = 0;
  LineCoding.ParityType = 0;
  LineCoding.DataBits = 8;
  Dtr_Rts = 0;
  txr = txw = rxr = rxw = 0;
}

bool Class_Compare (word aValue)  /* immer true, wird hier nicht gebraucht */
{ return true; }



/************* die Descriptoren ************************************/
#define LEN_DEVICE         18
#define DESC_DEVICE         1
#define VID            0x0416   /* Vendor ID (von RealTek) */
#define PID            0x5011   /* Product ID */

const byte DeviceDescriptor[] =
{ LEN_DEVICE,             /* bLength              */
  DESC_DEVICE,            /* bDescriptorType      */
  0x00, 0x02,             /* bcdUSB               */
  0x02,                   /* bDeviceClass         */
  0x00,                   /* bDeviceSubClass      */
  0x00,                   /* bDeviceProtocol      */
  EpCtrlMaxLen,           /* bMaxPacketSize0      */
  0x16,                   /* Vendor  ID LO        */
  0x04,                   /* Vendor  ID HI        */
  0x11,                   /* Product ID LO        */
  0x50,                   /* Product ID HI        */
  0x00, 0x01,             /* bcdDevice            */
  0x01,                   /* iManufacturer        */
  0x02,                   /* iProduct             */
  0x03,                   /* iSerialNumber        */
  0x01                    /* bNumConfigurations   */
};

#define LEN_CONFIG          9
#define DESC_CONFIG         2

#define LEN_INTERFACE       9
#define DESC_INTERFACE      4

#define LEN_ENDPOINT        7
#define DESC_ENDPOINT       5

const byte ConfigDescriptor[] =
{ LEN_CONFIG,             /* bLength              */
  DESC_CONFIG,            /* bDescriptorType      */
  0x43, 0x00,             /* wTotalLength         */
  0x02,                   /* bNumInterfaces       */
  0x01,                   /* bConfigurationValue  */
  0x00,                   /* iConfiguration       */
  0xC0,                   /* bmAttributes         */
  0x32,                   /* MaxPower             */

                          /* INTERFACE descriptor */
  LEN_INTERFACE,          /* bLength              */
  DESC_INTERFACE,         /* bDescriptorType      */
  0x00,                   /* bInterfaceNumber     */
  0x00,                   /* bAlternateSetting    */
  0x01,                   /* bNumEndpoints        */
  0x02,                   /* bInterfaceClass      */
  0x02,                   /* bInterfaceSubClass   */
  0x01,                   /* bInterfaceProtocol   */
  0x00,                   /* iInterface           */

                          /* Communication Class Specified INTERFACE descriptor */
  0x05,                   /* Size of the descriptor, in bytes */
  0x24,                   /* CS_INTERFACE descriptor type */
  0x00,                   /* Header functional descriptor subtype */
  0x10, 0x01,             /* Communication device compliant to the communication */
                          /* spec. ver. 1.10 */
                          /* Communication Class Specified INTERFACE descriptor */
  0x05,                   /* Size of the descriptor, in bytes */
  0x24,                   /* CS_INTERFACE descriptor type */
  0x01,                   /* Call management functional descriptor */
  0x00,                   /* BIT0: Whether device handle call management itself. */
                          /* BIT1: Whether device can send/receive call  */
                          /* management information over a Data Class Interface 0 */
  0x01,                   /* Interface number of data class interface optionally */
                          /* used for call management */
                          /* Communication Class Specified INTERFACE descriptor */
  0x04,                   /* Size of the descriptor, in bytes */
  0x24,                   /* CS_INTERFACE descriptor type */
  0x02,                   /* Abstract control management funcational descriptor */
                          /* subtype */
  0x00,                   /* bmCapabilities       */
                          /* Communication Class Specified INTERFACE descriptor */
  0x05,                   /* bLength              */
  0x24,                   /* bDescriptorType: CS_INTERFACE descriptor type */
  0x06,                   /* bDescriptorSubType   */
  0x00,                   /* bMasterInterface     */
  0x01,                   /* bSlaveInterface0     */


  /* ENDPOINT descriptor fr Interrupt */
  LEN_ENDPOINT,           /* bLength              */
  DESC_ENDPOINT,          /* bDescriptorType      */
  0x80+logEpInt,          /* bEndpointAddress     */
  3,                      /* Attribute: Interrupt */
  EpIntMaxLen, 0x00,      /* wMaxPacketSize       */
  0x01,                   /* bInterval            */


  /* INTERFACE descriptor */
  LEN_INTERFACE,          /* bLength              */
  DESC_INTERFACE,         /* bDescriptorType      */
  0x01,                   /* bInterfaceNumber     */
  0x00,                   /* bAlternateSetting    */
  0x02,                   /* bNumEndpoints        */
  0x0A,                   /* bInterfaceClass      */
  0x00,                   /* bInterfaceSubClass   */
  0x00,                   /* bInterfaceProtocol   */
  0x00,                   /* iInterface           */


  /* ENDPOINT descriptor fr Bulk IN */
  LEN_ENDPOINT,           /* bLength              */
  DESC_ENDPOINT,          /* bDescriptorType      */
  0x80+logEpBulkIn,       /* bEndpointAddress     */
  2,                      /* Attribute: Bulk      */
  EpBulkMaxLen, 0x00,     /* wMaxPacketSize       */
  0,                      /* bInterval   2ms probieren         */


  /* ENDPOINT descriptor fr Bulk OUT */
  LEN_ENDPOINT,           /* bLength              */
  DESC_ENDPOINT,          /* bDescriptorType      */
  logEpBulkOut,           /* bEndpointAddress     */
  2,                      /* Attribute: Bulk      */
  EpBulkMaxLen, 0x00,     /* wMaxPacketSize       */
  0,                      /* bInterval   2ms probieren         */
};


#define DESC_STRING         3
const byte StringLang[] =
{ 4,                      /* bLength                  */
  DESC_STRING,            /* bDescriptorType          */
  0x09, 0x04              /* Language ID: USA(0x0409) */
};


const byte VendorStringDescriptor[] =
{ 16,                     /* bLength          */
  DESC_STRING,            /* bDescriptorType  */
  'N', 0,
  'u', 0,
  'v', 0,
  'o', 0,
  't', 0,
  'o', 0,
  'n', 0
};

const byte ProductStringDescriptor[] =
{ 32,                     /* bLength          */
  DESC_STRING,            /* bDescriptorType  */
  'U', 0,
  'S', 0,
  'B', 0,
  ' ', 0,
  'V', 0,
  'i', 0,
  'r', 0,
  't', 0,
  'u', 0,
  'a', 0,
  'l', 0,
  ' ', 0,
  'C', 0,
  'O', 0,
  'M', 0
};

const byte StringSerial[26] =
{ 26,                     /* bLength          */
  DESC_STRING,            /* bDescriptorType  */
  'N', 0,
  'T', 0,
  '2', 0,
  '0', 0,
  '0', 0,
  '9', 0,
  '1', 0,
  '0', 0,
  '1', 0,
  '4', 0,
  '0', 0,
  '0', 0
};


/* um Nullbyte oder ein leeres Paket senden zu knnen */
const byte always0 = 0;

/************  Hilfsroutinen ************************************************/

void Stall (int physEpNum)
{ dword D, S, Maske;
  int logEpNum;

  logEpNum = physEpNum & 15;
  if (logEpNum != physEpNum)
  { Maske = EP_NoToggleBits | STAT_RX;    /* ohne STAT_TX und ohne beide DTOG_x */
    S = 1<<12;
  }
  else
  { Maske = EP_NoToggleBits | STAT_TX;    /* ohne STAT_RX und ohne beide DTOG_x */
    S = 1<<4;
  }
  D = USB_EpRegs(logEpNum);
  USB_EpRegs(logEpNum) = (D ^ S) & Maske;
}

void UnStall (int physEpNum)
{ dword D, S, Maske;
  int logEpNum;

  logEpNum = physEpNum & 15;
  if (logEpNum != physEpNum)
  { Maske = EP_NoToggleBits | STAT_RX;    /* ohne STAT_TX und ohne beide DTOG_x */
    S = 3<<12;                            /* ergibt fr RX = VALID */
  }
  else
  { Maske = EP_NoToggleBits | STAT_TX;    /* ohne STAT_RX und ohne beide DTOG_x */
    S = 2<<4;                             /* ergibt fr TX = NAK */
  }
  D = USB_EpRegs(logEpNum);
  USB_EpRegs(logEpNum) = (D ^ S) & Maske;
}

void StallLogEP (int logEpNum)
{ Stall(logEpNum);
  Stall(logEpNum | phys_In);
}



void UnStallLogEP (int logEpNum)
{ UnStall(logEpNum);
  UnStall(logEpNum | phys_In);
}


/* Endpoint empfangsbereit machen, also STAT_RX auf 11 setzen per Toggle */
void ClearBuffer (int logEpNum)
{ dword D, Maske;
  Maske = EP_NoToggleBits | STAT_RX;    /* ohne STAT_TX und ohne beide DTOG_x */
  D = USB_EpRegs(logEpNum);
  USB_EpRegs(logEpNum) = (D ^ STAT_RX) & Maske;
}

/* Endpoint sendebereit machen, also STAT_TX auf 11 setzen per Toggle */
void ValidateBuffer (int logEpNum)
{ dword D, Maske;
  Maske = EP_NoToggleBits | STAT_TX;    /* ohne STAT_RX und ohne beide DTOG_x */
  D = USB_EpRegs(logEpNum);
  USB_EpRegs(logEpNum) = (D ^ STAT_TX) & Maske;
}


bool USB_SetAddress (byte adr)
{ USB_DADDR = 0x80 | adr;
  return true;
}


bool USB_ConfigDevice (bool obConf)
{ return true;  // nix bei diesem Core zu tun.
}



/* physische Endpunkte aufsetzen (bei Reset-Kommando usw.) */

void InitEndpoints (void)
{ USB_CNTR = 1;                     /* erstmal Reset und alle Ints aus */
  CMD.Configuration = 0;            /* vor "CONFIGURED" ist hier nix */
  CMD.TransferLen = 0;              /* es stehen ab hier auch */
  CMD.PacketLen   = 0;              /* keine Transfers an */
  CMD.TransferPtr = 0;
  USB_CNTR = 0;                     /* alle Ints aus */

  /* EP0 = Control, IN und OUT */
  EpTable[0].TxOffset = Ep0TxOffset;
  EpTable[0].TxCount  = 0;
  EpTable[0].RxOffset = Ep0RxOffset;
  EpTable[0].RxCount  = EpCtrlLenId;

  /* EP1 = Bulk IN (nur IN) */
  EpTable[1].TxOffset = Ep1TxAOffset;
  EpTable[1].TxCount  = 0;
  EpTable[1].RxOffset = Ep1TxBOffset;  /* hier 2. TxPuffer */
  EpTable[1].RxCount  = EpBulkLenId;   /* erstmal.. */

  /* EP2 = Bulk OUT (nur OUT) */
  EpTable[2].TxOffset = Ep2RxAOffset;
  EpTable[2].TxCount  = EpBulkLenId;
  EpTable[2].RxOffset = Ep2RxBOffset;
  EpTable[2].RxCount  = EpBulkLenId;

  /* EP3 = Int, IN und OUT */
  EpTable[3].TxOffset = Ep3TxOffset;
  EpTable[3].TxCount  = EpIntLenId;
  EpTable[3].RxOffset = Ep3RxOffset;
  EpTable[3].RxCount  = EpIntLenId;

  USB_BTABLE = EpTableOffset;

  USB_EP0R = (3<<12) |    /* STAT_RX = 3, also Empfang enabled */
             (2<<4)  |    /* STAT_TX = 2, also NAK erstmal     */
             (1<<9)  |    /* EP_TYPE = 1, also Control         */
             logEpCtrl;

  USB_EP1R = (0<<12) |    /* STAT_RX = 0, also Empfang disabled */
             (2<<4)  |    /* STAT_TX = 2, also NAK erstmal     */
             (0<<9)  |    /* EP_TYPE = 0, also Bulk            */
             logEpBulkIn;

  USB_EP2R = (3<<12) |    /* STAT_RX = 3, also Empfang enabled */
             (0<<4)  |    /* STAT_TX = 0, also Senden disabled */
             (0<<9)  |    /* EP_TYPE = 0, also Bulk            */
             logEpBulkOut;

  USB_EP3R = (3<<12) |    /* STAT_RX = 3, also Empfang enabled */
             (2<<4)  |    /* STAT_TX = 2, also NAK erstmal     */
             (3<<9)  |    /* EP_TYPE = 0, also Bulk            */
             logEpInt;

  USB_ISTR = 0;           /* pending Interrupts beseitigen */
  USB_CNTR = CTRM |       /* Int bei ACKed Paketen in oder out */
             RESETM |     /* Int bei Reset */
             SOFM;        /* Int bei 1 ms Frame */
  USB_SetAddress(0);
}


void Nop (dword count)
{ while (count) --count; }


int ReadControlBlock (byte* PBuffer, int maxlen)
{ int count, i, n;
  dword D;
  dword* P;

  count = EpTable[0].RxCount & 0x3FF;
  if (count > maxlen) count = maxlen;

  if(count)
  { P = (dword*) EPControlRxBuffer;
    n = 2;
    i = count;
    D = *P++;
    while (i > 0)
    { *PBuffer = D & 0xFF;
      D = D >> 8;
      --n;
      if (!n) { D = *P++; n = 2; }
      --i;
      ++PBuffer;
    }
  }
  ClearBuffer(logEpCtrl);
  return count;
}



int WriteControlBlock (byte* PBuffer, int count)
{ dword A, L;
  dword* P;
  int  i, n;

  if (count > EpCtrlMaxLen) count = EpCtrlMaxLen;
  EpTable[0].TxCount = count;

  if (count)
  { A = 0; i = 0; n = 0;
    P = (dword*) EPControlTxBuffer;
    while (i < count)
    { L = *PBuffer++;
      A = A | (L << n);
      n += 8;
      if (n > 8)
      { n = 0;
        *P++ = A;
        A = 0;
      }
      ++i;
    }
    if (n) *P = A;
  }
  ValidateBuffer(logEpCtrl);
  Nop(100);                      /* mu sein, sonst keine Enumeration! */
  return count;
}


void ACK (void)
{ WriteControlBlock((byte*) &always0, 0);
}


/* Request-Typ im Setup-Packet testen (Standard, Class, Vendor) */
bool IsStandardRequest (void)
{ return (CMD.SetupPacket.bmRequestType & 0x60)==0; }

bool IsClassRequest (void)
{ return (CMD.SetupPacket.bmRequestType & 0x60)==0x20; }

bool IsVendorRequest (void)
{ return (CMD.SetupPacket.bmRequestType & 0x60)==0x40; }



/******* anstehende Control-Transfers zum Host blockweise starten *******/
void DescriptorBlockwiseIn (void)
{ int i, j;
  byte* Q;

  if ((CMD.SetupPacket.bmRequestType & 0x80)==0) return;
  i = CMD.TransferLen;
  if (i > CMD.PacketLen) i = CMD.PacketLen;
  Q = CMD.TransferPtr;                   /* Quelle */
  j = WriteControlBlock(Q, i);
  CMD.TransferPtr = Q + j;               /* Zeiger auf evt. Rest merken */
  CMD.TransferLen = CMD.TransferLen - j; /* restliche Anzahl Bytes */
  if (CMD.TransferLen < 0) CMD.TransferLen = 0;
}

/**********************************************************************/
/************ Bearbeitung eingegangener Requests **********************/
/**********************************************************************/


/********** USB-Request "SET FEATURE" und "CLEAR FEATURE" behandeln ****/
void DoSetClearFeature (bool value)
{ int Feature;
  int FuerWen;
  int EP;

  Feature = CMD.SetupPacket.wValue;
  FuerWen = CMD.SetupPacket.bmRequestType;
  EP      = CMD.SetupPacket.wIndex;

  switch(FuerWen)
  { case 0:          /* fr Device */
        if (Feature==1) CMD.RemoteWakeup = value;
        break;

    case 1:          /* fr Interface */
        break;

    case 2:          /* fr einen Endpoint */
        if (Feature==0)
        { if (value==false)
              StallLogEP(EP);
          else
              UnStallLogEP(EP);
        }
        break;

    default: Stall(1);  /* quasi NAK senden */
  }
}


/******** USB-Request "GET STATUS" behandeln ***************************/
void DoGetStatus (void)
{ byte Buf[4];
  int FuerWen;
  int  EP;

  FuerWen = CMD.SetupPacket.bmRequestType;
  EP = CMD.SetupPacket.wIndex;

  Buf[0] = 0;

  switch(FuerWen)
  { case 0x80:          /* fr Device */
         if (CMD.RemoteWakeup) Buf[0] |= 2;
         if (CMD.SelfPowered)  Buf[0] |= 1;
         break;

    case 0x81:          /* fr Interface */
         break;

    case 0x82:          /* fr einen Endpoint */
         if ((EP==logEpCtrl)   ||
             (EP==logEpInt)    ||
             (EP==logEpBulkIn) ||
             (EP==logEpBulkIn))  Buf[0] = 1;
         break;

    default:          /* hier eigentlich NAK senden */
         Stall(physEpCtrlIn);
         return;
  }

  Buf[1] = 0;
  CMD.PacketLen   = EpCtrlMaxLen;
  CMD.TransferLen = 2;
  CMD.TransferPtr = Buf;
  DescriptorBlockwiseIn();
}


/******** Descriptoren zum Host senden *********************************/
void DoGetDescriptor(void)
{ word Type, Index;
  int  aLen;
  const byte* P;

  Type    = CMD.SetupPacket.wValue >> 8;
  Index   = CMD.SetupPacket.wValue & 0xFF;
  aLen    = -1;
  P       = 0;

  switch (Type)
  { case DESC_DEVICE:                            /* Get Device Descriptor */
         { aLen = LEN_DEVICE;
           P    = DeviceDescriptor;
           break;
         }

    case DESC_CONFIG:                            /* Get Configuration Descriptor    */
         { aLen = ConfigDescriptor[3];           /* Total-Lnge ist WORD            */
           aLen = (aLen << 8) | ConfigDescriptor[2];
           P    = ConfigDescriptor;
           break;
         }


    case DESC_STRING:                            /* Get String Descriptor */
         { switch (Index)                        /* Get String Descriptor */
           { case 0: aLen = 4;
                     P    = StringLang;
                     break;
             case 1: aLen = VendorStringDescriptor[0];
                     P    = VendorStringDescriptor;
                     break;
             case 2: aLen = ProductStringDescriptor[0];
                     P    = ProductStringDescriptor;
                     break;
             case 3: aLen = StringSerial[0];
                     P    = StringSerial;
                     break;
             default: Stall(1);       /* kennen wir nicht. Stall. */
                      aLen = -1;
           }
           break;
         }
     default: { StallLogEP(logEpCtrl);  /* kennen wir nicht. Stall. */
                aLen = -1;
              }
     }

  if (aLen < 0 ) return;

  /* nicht mehr senden wollen, als der Host haben will */
  if (aLen > CMD.SetupPacket.wLength) aLen = CMD.SetupPacket.wLength;
  CMD.PacketLen   = EpCtrlMaxLen;
  CMD.TransferLen = aLen;
  CMD.TransferPtr = (byte*) P;
  DescriptorBlockwiseIn();
}


/********** haben Adresse empfangen ***********************************/
void DoSetAddress (void)
{ ACK();                       // das mu bei diesem Core davor!
  USB_SetAddress(CMD.SetupPacket.wValue);
}


/*********** USB-Request "SET CONFIGURATION" behandeln *************/
void DoSetConfiguration(void)
{ bool haveConfig;

  haveConfig = Class_Compare(CMD.SetupPacket.wValue);
  if (CMD.SetupPacket.wValue == 0)
        { CMD.Configuration = CMD.SetupPacket.wValue & 0xFF;
        }
  else
  if(haveConfig)
        { USB_ConfigDevice(true);
          Class_Start();
          CMD.Configuration = CMD.SetupPacket.wValue &0xFF;
          ACK();
        }
   else  Stall(0);
}


/*************************** CDC Spezifisches **************************/


/************* "SET LINE CODING" behandeln *****************************/
void VCOM_SetLineCoding(void)
{ ACK();

    /* Vorbereitung auf Empfang von genau 7 Bytes vom Host   ???*/
}


/* Datenausgabe fr CDC-spezifischen USB-Request "SET LINE CODING" */
/* hier werden die empfangenen 7 Bytes aus dem USB-Puffer gelesen
   und im RAM gemerkt
*/
void SetLineCodingDataOut(void)
{ ReadControlBlock((byte*) &LineCoding, 7);
  ACK();
}

/* Zustand von DTR und RTS vom Host zum Gert merken */
void VCOM_Read_DTR_RTS (void)
{ Dtr_Rts = CMD.SetupPacket.wValue>>8;
  ACK();
}

/* CDC-spezifischer USB-Request "GET LINE CODING" behandeln */
void VCOM_GetLineCoding(void)
{ CMD.PacketLen   = EpCtrlMaxLen;
  CMD.TransferLen = 7;
  CMD.TransferPtr = (byte*) &LineCoding;
  DescriptorBlockwiseIn();    /* hier werden die 7 Bytes zum Host geschickt */
}


/************************** Setup-Event ***********************************/
/*
Merke:
1. Paket abholen und dann ClearBuffer.
2. Bei Setup Paketen, bei denen..
- ..nix hinterher kommt, also wo es keine Datenphase gibt,
  beantwortet man mit Senden eines leeren Paketes (ist ACK)
- ..anschlieend etwas zum Host gesendet werden mu, sendet man
  dies direkt danach. Wenn das zu Sendende grer ist als
  die EpBuffer-Lnge, dann nur ein Stck senden. Der Host holt es sich ab
  und der USB-Core gibt dann einen gewhnlichen Int auf Control-In, wo man
  dann das nchste Stck senden kann. Wiederholt sich, bis man nix mehr
  zu senden hat. Ob man als Abschlu ein leeres Paket senden mu oder nicht,
  ist ungewi.
  Wenn der Host zufrieden ist, sendet er ein leeres Paket als ACK.
- ..man anschlieend noch etwas vom Host bekommt, dann gibt es dafr ein
  anschlieendes Int auf Control-Out. Man liest das Paket und sendet dann
  als ACK ein leeres Paket.
*/

void OnSetup (word EpCtrlStatus)
{ ReadControlBlock(&CMD.SetupPacket.bmRequestType, 8);

// report_SetupPacket();  /* nur debug via debugfkt.inc */

  if (IsStandardRequest())         /* wenn Type = Standard */
    { switch (CMD.SetupPacket.bRequest)
      { case SET_ADDRESS:       DoSetAddress();
                                return;

        case CLEAR_FEATURE:     DoSetClearFeature(false);
                                return;

        case SET_FEATURE:       DoSetClearFeature(true);
                                return;

        case GET_CONFIGURATION: CMD.PacketLen   = EpCtrlMaxLen;
                                CMD.TransferLen = 1;
                                CMD.TransferPtr = (byte*) &CMD.Configuration;
                                DescriptorBlockwiseIn();
                                return;

        case GET_STATUS:        DoGetStatus();
                                return;

        case GET_INTERFACE:     CMD.TransferLen = 1;
                                CMD.TransferPtr = (byte*) &always0;
                                DescriptorBlockwiseIn();
                                return;

        case SET_INTERFACE:     Class_Start();
                                ACK();
                                return;

        case GET_DESCRIPTOR:    DoGetDescriptor();
                                return;

        case SET_CONFIGURATION: DoSetConfiguration();
                                return;
      }
      /* auf alles andere reagieren wir mit Stall. siehe unten. */
    }

  if (IsClassRequest())        /* wenn Type = Class */
    { switch (CMD.SetupPacket.bRequest)
      { case SET_LINE_CODE:          VCOM_SetLineCoding();  return;
        case GET_LINE_CODE:          VCOM_GetLineCoding();  return;
        case SET_CONTROL_LINE_STATE: VCOM_Read_DTR_RTS();   return;

        /* falls es hier noch mehr Class-spezifische Requests
           geben sollte, dann Behandlung hier hinein.
         */

      }

    }

  if (IsVendorRequest())       /* wenn Type = Vendor */
    {
    }

    /* wenn keiner zustndig war, dann Stall! */
  Stall(0);
}



/******* die diversen Endpoint-Interrupts ************************************/


void  OnEpCtrlOut (word EpCtrlStatus)                  /* Control-EP OUT */
{ byte tbuf[EpCtrlMaxLen];

  if (IsStandardRequest())                             /* wenn Type = Standard */
  { /* eigentlich nur leere Pakete, also ACK vom Host,
       aber mglich (nie gesehen) bRequest=7 = SET_DESCRIPTOR
     */
    ReadControlBlock(tbuf, EpCtrlMaxLen);
    return;
  }

  if (IsClassRequest())                                /* wenn Type = Class */
  { switch (CMD.SetupPacket.bRequest)
    { case SET_LINE_CODE:
               SetLineCodingDataOut();
               ACK();
               return;

      default: ACK();
    }
    return;
  }

  /* nach Vendor-Request fragen wir hier garnicht erst */
  ACK();
}


void OnEpCtrlIn (word EpCtrlStatus)                   /* Control-EP IN */
{ if (IsStandardRequest())                            /* wenn Type = Standard */
    { switch (CMD.SetupPacket.bRequest)
      { case GET_DESCRIPTOR:
                 if (CMD.TransferLen > 0)
		     DescriptorBlockwiseIn();
                 break;
        case GET_LINE_CODE:
                 ACK();
                 break;
        default: ACK();
      }
     return;
    }
}


/********* BULK IN und OUT Interrupts **********/

void OnEpBulkIn (void)                  /* EP1 = Bulk-EP IN */
{ int i, n;
  dword L, A;
  dword* P;

 /* bei STM32 mu man offenbar den Frame-Int benutzen,
   um alle 1 ms nach dem Rechten zu schauen.
  */

  if (txr==txw)  return;
  P = (dword*) EP1TxABuffer;
  i = txw - txr;
  if (i<0) i += txLen;                  /* i = Anzahl zu sendender Bytes */
  if (i>EpBulkMaxLen) i = EpBulkMaxLen;
  A = 0;
  n = 0;
  EpTable[1].TxCount = (i & 0x3FF) | EpBulkLenId;

  while (i)
  { L = UsbTxBuf[txr];
    txr = (txr+1) & (txLen-1);
    A = A | (L << n);
    n += 8;
    if (n > 8)
    { *P++ = A;
      n = 0;
      A = 0;
    }
    --i;
  }
  if (n) *P = A;                 /* ggf. restliche Bytes ausgeben */
  ValidateBuffer(logEpBulkIn);
}


void OnEpBulkOut (void)                   /* EP2 = Bulk-EP OUT */
{ int  i, n, hdroom, avail;
  long D;
  char c;
  dword* P;


  /* Bulk EP anwhlen und Anzahl der Bytes ermittlen */
  avail = EpTable[2].RxCount & 0x3FF;

  i = rxw - rxr;
  if (i<0) i += rxLen;
  hdroom = rxLen - i;
  if (hdroom <= avail)  return;

  P = (dword*) EP2RxBBuffer;
  n = 2;
  i = avail;
  D = *P++;                               /* 2 Byte laden */
  while (i>0)
  { c = D & 0xFF;                         /* LSB zuerst   */
    UsbRxBuf[rxw] = c;
    rxw = (rxw+1) & (rxLen-1);
    D = D >> 8;
    --n;
    if (!n) { D = *P++; n = 2; }
    --i;
  }
  ClearBuffer(logEpBulkOut);               /* wir haben's gelesen */
}


void OnEpIntIn (void)                      /* Int-EP IN */
{
  /* erstmal nix */
}

void OnEpIntOut (void)                     /* Int-EP IN */
{
  /* erstmal nix */
}

/**************** USB-Interrupt-Handler **************************************/

void __irq USB_LP_CAN1_RX0_IRQHandler (void)
{ dword I;
  int   EpNum;
  word  EpStatus;


  I  = USB_ISTR;                             /* Interrupt-Status nach I  */

  if (I & PMAOVR)                            /* interner Timeout...  */
  { USB_ISTR = ~PMAOVR;                      /* Int lschen */
  }

  if (I & ERR)                               /* Datenfehler bei Transaction */
  { USB_ISTR = ~ERR;                         /* Int lschen */
  }

  if (I & WKUP)                              /* Suspend-->Resume */
  { USB_ISTR = ~WKUP;                        /* Int lschen */
  }

  if (I & SUSP)                              /* nach 3 ms Pause -->Suspend */
  { USB_ISTR = ~SUSP;                        /* Int lschen */
  }

  if (I & RESET)                             /* Bus Reset */
  { InitEndpoints();
    USB_ISTR = ~RESET;                       /* Int lschen */
    return;
  }

  if (I & SOF)                               /* Start of Frame, alle 1 ms */
  { USB_ISTR = ~SOF;                         /* Int lschen */
    OnEpBulkIn();                            /* immer mal nachschauen... */
  }

  if (I & ESOF)                              /* Wenn ein SOF Paket fehlt */
  { USB_ISTR = ~ESOF;                        /* Int lschen */
  }

  /* Endpoint Interrupts */
  if (I & CTR)
  { USB_ISTR = ~CTR;                         /* Interruptbit lschen */
    EpNum    = I & MASK_EA;                  /* welcher EP?          */
    EpStatus = USB_EpRegs(EpNum);            /* EP Status lesen      */

    if (I & DIR)                             /* OUT, also Paket wurde empfangen */
    { USB_EpRegs(EpNum) = EpStatus & ~CTR_RX & EP_NoToggleBits;
      if (EpNum == logEpCtrl)
      { if (EpStatus & SETUP)
            OnSetup(EpStatus);               /* Handle the Setup-Packet   */
        else
            OnEpCtrlOut(EpStatus);           /* eigentlich nur Class-spezifisches */
      }
      if (EpNum == logEpBulkOut) OnEpBulkOut();
      if (EpNum == logEpInt)     OnEpIntOut();
    }
    else                                     /* IN, also Paket wurde gesendet */
    { USB_EpRegs(EpNum) = EpStatus & ~CTR_TX & EP_NoToggleBits;
      if (EpNum == logEpCtrl)    OnEpCtrlIn(EpStatus);
      if (EpNum == logEpBulkOut) OnEpBulkIn();
      if (EpNum == logEpInt)     OnEpIntIn();
    }
  }
}



/************  USB-Setup **********************************/
/* Die Pins mssen bereits richtig gesetzt sein           */
/* ebenso Clock ein und Powerdown aus.                    */
/* Interrupt 20 ist USB-IRQ (USB_LP_CANRX_Handler)        */
/* PortA.15 ist USB-On-Schalter (1k5)                     */
/**********************************************************/

/* im STM32F103C8T6 vorhandene Vektoren:
 USB_HP_CAN1_TX_IRQHandler  = Int19
 USB_LP_CAN1_RX0_IRQHandler = Int20
 USBWakeUp_IRQHandler       = Int42

 also welcher denn nun?
*/

word UsbSetup (void)
{ dword* P;

//  lbw = lbr = 0;                 /* nur debug via debugfkt.inc */

  P = (dword*) USB_RAM;          /* RAM ablschen  */
  while ((dword)P < (USB_RAM+1024)) *P++ = 0;

  Class_Start();                 /* LineCoding-Block aufsetzen mit unseren Defaultwerten */
  USB_CNTR = 3;                  /* Powerdown+Reset */
  Nop(10);                       /* warten */
  USB_CNTR = 1;                  /* Reset  */
  USB_ISTR = 0;                  /* spurious Ints beseitigen */
  Nop(1000);                     /* warten */
  NVIC_ISER0  |= (1<<(20));      /* Interrupt 20=USB, enable */
  InitEndpoints();
  GPIOA_BSRR = (1UL<<(15+16));   /* PA15 von Hand auf Low setzen wg. Transistor + 1k5 an D+ */
  return 0;
}


/********** zeichenweises I/O und Pufferung und Kommunikation ***************/
/*
Diese Routinen werden von auerhalb im Usermode
aufgerufen und haben mit dem interrupt-gesteuerten
USB-Betrieb nichts zu tun.
*/

/* liefert true, wenn ein Zeichen abholbereit ist */
bool UsbRxAvail (void)
{ if (rxr!=rxw) return true;
  return false;
}

/* holt ein Zeichen vom USB ab */
/*
Achtung: wenn nix abzuholen ist, wird 0 zurckgeliefert
*/
char UsbGetChar (void)
{ char c;

  c = 0;
  if (rxr!=rxw)
  { c = UsbRxBuf[rxr];
    rxr = (rxr+1) & (rxLen-1);
  }
  return c;
}


/* liefert true, wenn noch ein Zeichen in den Tx-Buffer passt */
bool UsbTxReady (void)
{ int i;
  i = (txw+1) & (txLen-1);
  if (i==txr) return false;
  return true;
}


/* liefert true, wenn Tx-Buffer leer ist */
bool UsbTxEmpty (void)
{ return (txw==txr); }


/* Anzahl freier Pltze im Tx-Buffer liefern */
int UsbTxFree (void)
{ int i;
  i = txw - txr;    /* i = belegte Pltze */
  if (i<0) i = i + txLen;
  return txLen - i;
}


/* sendet ein Zeichen (d.h. schreibt es in den Tx-Buffer) */
char UsbCharOut (char c)
{ int i;
  i = (txw+1) & (txLen-1);
  while (!UsbTxReady());   /* trampeln auf der Stelle!! */
  UsbTxBuf[txw] = c;
  txw = i;
  return c;
}


/* asciiz zum USB senden */
void UsbStrOut (char* S)
{ while (*S) UsbCharOut(*S++); }


